//
// Syd: rock-solid application kernel
// src/kernel/sysinfo.rs: sysinfo(2) handler
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use libseccomp::ScmpNotifResp;
use nix::errno::Errno;

use crate::{config::MMAP_MIN_ADDR, hook::UNotifyEventRequest, sysinfo::SysInfo};

pub(crate) fn sys_sysinfo(request: UNotifyEventRequest) -> ScmpNotifResp {
    syscall_handler!(request, |request: UNotifyEventRequest| {
        let req = request.scmpreq;

        // SAFETY: Check pointer against mmap_min_addr.
        if req.data.args[0] < *MMAP_MIN_ADDR {
            return Err(Errno::EFAULT);
        }

        // Generate a randomized sysinfo(2) structure.
        let info = SysInfo::new()?.as_raw();

        // SAFETY: The use of `from_raw_parts` here is safe because
        // `info` is fully initialized at this point by the preceding
        // `fillrandom` call, ensuring that the memory region from
        // `&info` up to the size of `libc::sysinfo` is valid. The
        // conversion to a byte slice is done to facilitate copying the
        // structure to another memory location without altering its
        // contents. This operation does not extend the lifetime of
        // `info` beyond this function, nor does it modify the content
        // of `info`, adhering to Rust's safety and borrowing rules.
        let info = unsafe {
            std::slice::from_raw_parts(
                std::ptr::addr_of!(info) as *const u8,
                std::mem::size_of_val(&info),
            )
        };

        request.write_mem(info, req.data.args[0])?;
        Ok(request.return_syscall(0))
    })
}
