//
// Syd: rock-solid application kernel
// src/kernel/stat.rs: stat syscall handlers
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::{
    collections::HashSet,
    fs::File,
    io::BufReader,
    os::{
        fd::{AsFd, AsRawFd},
        unix::ffi::OsStrExt,
    },
};

use libseccomp::ScmpNotifResp;
use nix::{errno::Errno, NixPath};

use crate::{
    compat::{fstatat64, statx, STATX_BASIC_STATS, STATX_MODE, STATX_TYPE},
    config::{MAGIC_PREFIX, MMAP_MIN_ADDR},
    fs::{is_sidechannel_device, parse_fd, CanonicalPath, FileInfo, FileType, FsFlags},
    hook::{SysArg, SysFlags, UNotifyEventRequest},
    kernel::sandbox_path,
    path::XPath,
    sandbox::Capability,
    scmp_arch_bits,
};

pub(crate) fn sys_stat(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    let is32 = scmp_arch_bits(req.data.arch) == 32;

    let arg = SysArg {
        path: Some(0),
        fsflags: FsFlags::MUST_PATH,
        ..Default::default()
    };

    syscall_stat_handler(request, arg, 1, is32)
}

pub(crate) fn sys_stat64(request: UNotifyEventRequest) -> ScmpNotifResp {
    let arg = SysArg {
        path: Some(0),
        fsflags: FsFlags::MUST_PATH,
        ..Default::default()
    };

    syscall_stat_handler(request, arg, 1, false)
}

pub(crate) fn sys_fstat(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    let is32 = scmp_arch_bits(req.data.arch) == 32;

    let arg = SysArg {
        dirfd: Some(0),
        ..Default::default()
    };

    syscall_stat_handler(request, arg, 1, is32)
}

pub(crate) fn sys_fstat64(request: UNotifyEventRequest) -> ScmpNotifResp {
    let arg = SysArg {
        dirfd: Some(0),
        ..Default::default()
    };

    syscall_stat_handler(request, arg, 1, false)
}

pub(crate) fn sys_lstat(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    let is32 = scmp_arch_bits(req.data.arch) == 32;

    let arg = SysArg {
        path: Some(0),
        fsflags: FsFlags::MUST_PATH | FsFlags::NO_FOLLOW_LAST,
        ..Default::default()
    };

    syscall_stat_handler(request, arg, 1, is32)
}

pub(crate) fn sys_lstat64(request: UNotifyEventRequest) -> ScmpNotifResp {
    let arg = SysArg {
        path: Some(0),
        fsflags: FsFlags::MUST_PATH | FsFlags::NO_FOLLOW_LAST,
        ..Default::default()
    };

    syscall_stat_handler(request, arg, 1, false)
}

pub(crate) fn sys_statx(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    let empty = req.data.args[2] & libc::AT_EMPTY_PATH as u64 != 0;
    let follow = req.data.args[2] & libc::AT_SYMLINK_NOFOLLOW as u64 == 0;

    let mut flags = SysFlags::empty();
    let mut fsflags = FsFlags::MUST_PATH;

    if empty {
        flags |= SysFlags::EMPTY_PATH;
    }

    if !follow {
        fsflags |= FsFlags::NO_FOLLOW_LAST;
    }

    let arg = SysArg {
        dirfd: Some(0),
        path: Some(1),
        flags,
        fsflags,
        ..Default::default()
    };

    syscall_stat_handler(request, arg, 4, false)
}

pub(crate) fn sys_newfstatat(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    let empty = req.data.args[3] & libc::AT_EMPTY_PATH as u64 != 0;
    let follow = req.data.args[3] & libc::AT_SYMLINK_NOFOLLOW as u64 == 0;
    let mut flags = SysFlags::empty();
    let mut fsflags = FsFlags::MUST_PATH;

    if empty {
        flags |= SysFlags::EMPTY_PATH;
    }

    if !follow {
        fsflags |= FsFlags::NO_FOLLOW_LAST;
    }

    let arg = SysArg {
        dirfd: Some(0),
        path: Some(1),
        flags,
        fsflags,
        ..Default::default()
    };

    syscall_stat_handler(request, arg, 2, false)
}

#[allow(clippy::cognitive_complexity)]
fn syscall_stat_handler(
    request: UNotifyEventRequest,
    arg: SysArg,
    arg_stat: usize,
    is32: bool,
) -> ScmpNotifResp {
    syscall_handler!(request, |request: UNotifyEventRequest| {
        // Note: This is a virtual call handler,
        // `sandbox` is an upgradable read lock with exclusive access.
        // We'll either upgrade it or downgrade it based on magic lock.
        // Exception: Sandbox lock had been set and there's no turning back.
        let req = request.scmpreq;
        let is_fd = arg.path.is_none();
        let sandbox = request.get_sandbox();
        let is_lock = sandbox.locked_for(req.pid());

        let is_crypt = sandbox.enabled(Capability::CAP_CRYPT);
        let is_stat = sandbox.enabled(Capability::CAP_STAT);

        // Check for chroot.
        if sandbox.is_chroot() {
            return Err(if is_fd { Errno::EACCES } else { Errno::ENOENT });
        }

        // Read the remote path.
        // If lock is on do not check for magic path.
        let (mut path, magic) = request.read_path(&sandbox, arg, !is_lock)?;

        // SAFETY: For magic calls we allow NULL as stat argument, see syd(2).
        // For other calls, return EFAULT here for invalid pointers.
        let is_magic = !is_lock && magic;
        if !is_magic && req.data.args[arg_stat] < *MMAP_MIN_ADDR {
            return Err(Errno::EFAULT);
        }

        if is_magic {
            drop(sandbox); // release the read-lock.

            // Handle magic prefix (ie /dev/syd)
            let mut cmd = path
                .abs()
                .strip_prefix(MAGIC_PREFIX)
                .unwrap_or_else(|| XPath::from_bytes(&path.abs().as_bytes()[MAGIC_PREFIX.len()..]))
                .to_owned();
            // Careful here, Path::strip_prefix removes trailing slashes.
            if path.abs().ends_with_slash() {
                cmd.push(b"");
            }

            // Acquire a write lock to the sandbox.
            let mut sandbox = request.get_mut_sandbox();

            // Execute magic command.
            match cmd.as_os_str().as_bytes() {
                b"ghost" => {
                    // SAFETY: Reset sandbox to ensure no run-away execs.
                    sandbox.reset()?;

                    // Signal the poll process to exit.
                    return Err(Errno::EOWNERDEAD);
                }
                b"panic" => sandbox.panic()?,
                _ => {}
            }

            if cmd.is_empty() || cmd.is_equal(b".el") || cmd.is_equal(b".sh") {
                sandbox.config("")?;
            } else if let Some(cmd) = cmd.strip_prefix(b"load") {
                // We handle load specially here as it involves process access.
                // 1. Attempt to parse as FD, pidfd_getfd and load it.
                // 2. Attempt to parse as profile name if (1) fails.
                match parse_fd(cmd) {
                    Ok(remote_fd) => {
                        let fd = request.get_fd(remote_fd)?;
                        let file = BufReader::new(File::from(fd));
                        let mut imap = HashSet::default();
                        // SAFETY: parse_config() checks for the file name
                        // /dev/syd/load and disables config file include
                        // feature depending on this check.
                        if sandbox
                            .parse_config(file, XPath::from_bytes(b"/dev/syd/load"), &mut imap)
                            .is_err()
                        {
                            return Ok(request.fail_syscall(Errno::EINVAL));
                        }
                        // Fall through to emulate as /dev/null.
                    }
                    Err(Errno::EBADF) => {
                        if sandbox.parse_profile(&cmd.to_string()).is_err() {
                            return Ok(request.fail_syscall(Errno::EINVAL));
                        }
                        // Fall through to emulate as /dev/null.
                    }
                    Err(errno) => {
                        return Ok(request.fail_syscall(errno));
                    }
                }
            } else if let Ok(cmd) = std::str::from_utf8(cmd.as_bytes()) {
                sandbox.config(cmd)?;
            } else {
                // SAFETY: Invalid UTF-8 is not permitted.
                // To include non-UTF-8, hex-encode them.
                return Err(Errno::EINVAL);
            }
            drop(sandbox); // release the write-lock.

            // If the stat buffer is NULL, return immediately.
            if req.data.args[arg_stat] == 0 {
                return Ok(request.return_syscall(0));
            }
        } else {
            // Handle fstat for files with encryption in progress.
            let mut crypt_stat = false;
            if is_crypt && is_fd {
                // SAFETY: SysArg.path is None asserting dirfd is Some fd!=AT_FDCWD.
                #[allow(clippy::disallowed_methods)]
                let fd = path.dir.as_ref().unwrap();
                if let Ok(info) = FileInfo::from_fd(fd) {
                    #[allow(clippy::disallowed_methods)]
                    let files = request.crypt_map.as_ref().unwrap();
                    for (enc_path, map) in
                        files.read().unwrap_or_else(|err| err.into_inner()).iter()
                    {
                        if info == map.info {
                            // Found underlying encrypted file for the memory fd.
                            // Note, we only ever attempt to encrypt regular files.
                            path =
                                CanonicalPath::new(enc_path.clone(), FileType::Reg, arg.fsflags)?;
                            crypt_stat = true;
                            break;
                        }
                    }
                }
            }

            // SAFETY:
            // 1. Allow access to fd-only calls.
            // 2. Allow access to files with encryption in progress.
            // 3. Allow access to /memfd:syd-*. This prefix is internal
            //    to Syd and sandbox process cannot create memory file
            //    descriptors with this name prefix.
            if is_stat
                && !crypt_stat
                && arg.path.is_some()
                && !path.abs().starts_with(b"/memfd:syd-")
            {
                sandbox_path(
                    Some(&request),
                    &sandbox,
                    request.scmpreq.pid(), // Unused when request.is_some()
                    path.abs(),
                    Capability::CAP_STAT,
                    false,
                    "stat",
                )?;
            }

            drop(sandbox); // release the read-lock.
        }

        // SAFETY: Path hiding is done, now it is safe to:
        //
        // Return ENOTDIR for non-directories with trailing slash.
        if let Some(file_type) = &path.typ {
            if !matches!(file_type, FileType::Dir | FileType::MagicLnk(_, _))
                && path.abs().last() == Some(b'/')
            {
                return Err(Errno::ENOTDIR);
            }
        }

        let mut flags = if path.base.is_empty() {
            libc::AT_EMPTY_PATH
        } else {
            // SAFETY: After this point we are not permitted to resolve
            // symbolic links any longer or else we risk TOCTOU.
            libc::AT_SYMLINK_NOFOLLOW
        };

        #[allow(clippy::cast_possible_truncation)]
        if arg_stat == 4 {
            // statx

            // Support AT_STATX_* flags.
            flags |= req.data.args[2] as libc::c_int
                & !(libc::AT_SYMLINK_NOFOLLOW | libc::AT_EMPTY_PATH);

            // SAFETY: The sidechannel check below requires the mask
            // to have the following items:
            // 1. STATX_TYPE (to check for char/block device)
            // 2. STATX_MODE (to check for world readable/writable)
            // To ensure that here, we inject these two flags into
            // mask noting if they were set originally. This can be
            // in three ways,
            // (a) Explicitly setting STATX_{TYPE,MODE}.
            // (b) Explicitly setting STATX_BASIC_STATS.
            // (c) Setting the catch-all STATX_ALL flag.
            // After the statx call if the flags STATX_{TYPE,MODE}
            // were not set we clear stx_mode's type and mode bits
            // as necessary and also remove STATX_{TYPE,MODE} from
            // stx_mask as necessary.
            let mut mask = req.data.args[3] as libc::c_uint;
            let orig_mask = mask;
            let basic_stx = (orig_mask & STATX_BASIC_STATS) != 0;
            if !basic_stx {
                mask |= STATX_TYPE | STATX_MODE;
            }

            // Note, unlike statfs, stat does not EINTR.
            let mut statx = statx(
                path.dir.as_ref().map(|fd| fd.as_fd()).ok_or(Errno::EBADF)?,
                path.base,
                flags,
                mask,
            )?;

            // SAFETY: Check if the file is a sidechannel device and
            // update its access and modification times to match the
            // creation time if it is. This prevents timing attacks on
            // block or character devices like /dev/ptmx using stat.
            if is_sidechannel_device(statx.stx_mode.into()) {
                statx.stx_atime = statx.stx_ctime;
                statx.stx_mtime = statx.stx_ctime;
            }

            // SAFETY: Restore mask, type and mode, see the comment above.
            #[allow(clippy::cast_possible_truncation)]
            if !basic_stx {
                if (orig_mask & STATX_TYPE) == 0 {
                    statx.stx_mode &= !libc::S_IFMT as u16;
                    statx.stx_mask &= !STATX_TYPE;
                }
                if (orig_mask & STATX_MODE) == 0 {
                    statx.stx_mode &= libc::S_IFMT as u16;
                    statx.stx_mask &= !STATX_MODE;
                }
            }

            // SAFETY: The following block creates an immutable byte
            // slice representing the memory of `statx`. We ensure that
            // the slice covers the entire memory of `statx` using
            // `std::mem::size_of_val`. Since `statx` is a stack
            // variable and we're only borrowing its memory for the
            // duration of the slice, there's no risk of `statx` being
            // deallocated while the slice exists. Additionally, we
            // ensure that the slice is not used outside of its valid
            // lifetime.
            let statx = unsafe {
                std::slice::from_raw_parts(
                    std::ptr::addr_of!(statx) as *const u8,
                    std::mem::size_of_val(&statx),
                )
            };
            let addr = req.data.args[4];
            if addr != 0 {
                request.write_mem(statx, addr)?;
            }
        } else {
            // "stat" | "fstat" | "lstat" | "newfstatat"

            // SAFETY: In libc we trust.
            // Note, unlike statfs, stat does not EINTR.
            let mut stat = fstatat64(path.dir.as_ref().map(|fd| fd.as_raw_fd()), path.base, flags)?;

            // SAFETY: Check if the file is a sidechannel device and
            // update its access and modification times to match the
            // creation time if it is. This prevents timing attacks on
            // block or character devices like /dev/ptmx using stat.
            if is_sidechannel_device(stat.st_mode) {
                stat.st_atime = stat.st_ctime;
                stat.st_mtime = stat.st_ctime;
                stat.st_atime_nsec = stat.st_ctime_nsec;
                stat.st_mtime_nsec = stat.st_ctime_nsec;
            }

            let addr = req.data.args[arg_stat];
            if addr != 0 {
                if is32 {
                    let stat32: crate::compat::stat32 = stat.into();

                    // SAFETY: The following block creates an immutable
                    // byte slice representing the memory of `stat`.  We
                    // ensure that the slice covers the entire memory of
                    // `stat` using `std::mem::size_of_val`. Since
                    // `stat` is a stack variable and we're only
                    // borrowing its memory for the duration of the
                    // slice, there's no risk of `stat` being
                    // deallocated while the slice exists.
                    // Additionally, we ensure that the slice is not
                    // used outside of its valid lifetime.
                    let stat = unsafe {
                        std::slice::from_raw_parts(
                            std::ptr::addr_of!(stat32) as *const u8,
                            std::mem::size_of_val(&stat32),
                        )
                    };
                    request.write_mem(stat, addr)?;
                } else {
                    // SAFETY: The following block creates an immutable
                    // byte slice representing the memory of `stat`.  We
                    // ensure that the slice covers the entire memory of
                    // `stat` using `std::mem::size_of_val`. Since
                    // `stat` is a stack variable and we're only
                    // borrowing its memory for the duration of the
                    // slice, there's no risk of `stat` being
                    // deallocated while the slice exists.
                    // Additionally, we ensure that the slice is not
                    // used outside of its valid lifetime.
                    let stat = unsafe {
                        std::slice::from_raw_parts(
                            std::ptr::addr_of!(stat) as *const u8,
                            std::mem::size_of_val(&stat),
                        )
                    };
                    request.write_mem(stat, addr)?;
                }
            }
        }

        // stat system call successfully emulated.
        Ok(request.return_syscall(0))
    })
}
