//
// Syd: rock-solid application kernel
// src/kernel/fcntl.rs: fcntl{,64}(2) handler
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::os::fd::RawFd;

use libseccomp::ScmpNotifResp;
use nix::errno::Errno;

use crate::{
    config::PROC_FILE,
    fs::{readlinkat, FileInfo},
    hook::UNotifyEventRequest,
    path::XPathBuf,
    sandbox::Capability,
};

pub(crate) fn sys_fcntl(request: UNotifyEventRequest) -> ScmpNotifResp {
    // Note, we only hook into F_SETFL requests
    // which do not have O_APPEND set!
    let req = request.scmpreq;
    let fd = if let Ok(fd) = RawFd::try_from(req.data.args[0]) {
        if fd < 0 {
            return request.fail_syscall(Errno::EBADF);
        }
        fd
    } else {
        return request.fail_syscall(Errno::EBADF);
    };

    let mut pfd = XPathBuf::from_pid(req.pid());
    pfd.push(b"fd");
    pfd.push_fd(fd);
    let path = match readlinkat(PROC_FILE(), &pfd) {
        Ok(path) => {
            if !request.is_valid() {
                return request.fail_syscall(Errno::ESRCH);
            }
            path
        }
        Err(_) => return request.fail_syscall(Errno::EBADF),
    };

    let sandbox = request.get_sandbox();
    let is_crypt = sandbox.enabled(Capability::CAP_CRYPT);
    let is_append = sandbox.is_append(&path);
    drop(sandbox); // release the read-lock.

    if is_append {
        // Deny silently.
        return request.return_syscall(0);
    }

    if is_crypt {
        let fd = if let Ok(fd) = request.get_fd(fd) {
            fd
        } else {
            return request.fail_syscall(Errno::EBADF);
        };
        if let Ok(info) = FileInfo::from_fd(&fd) {
            #[allow(clippy::disallowed_methods)]
            let files = request.crypt_map.as_ref().unwrap();
            for map in files.read().unwrap_or_else(|err| err.into_inner()).values() {
                if info == map.info {
                    // Deny with EACCES, caller should know.
                    return request.fail_syscall(Errno::EACCES);
                }
            }
        }
    }

    // SAFETY: fcntl is fd-only.
    // No pointer dereference in access check.
    unsafe { request.continue_syscall() }
}
