//
// Syd: rock-solid application kernel
// src/kernel/fanotify.rs: fanotify_mark(2) handler
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::os::fd::{AsFd, AsRawFd, FromRawFd, IntoRawFd, RawFd};

use libseccomp::ScmpNotifResp;
use nix::{
    errno::Errno,
    sys::fanotify::{Fanotify, MarkFlags, MaskFlags},
};

use crate::{
    config::PROC_FILE,
    fs::{fd_mode, is_sidechannel_device, FsFlags},
    hook::{SysArg, UNotifyEventRequest},
    kernel::syscall_path_handler,
    path::XPathBuf,
    scmp_arch_bits,
};

pub(crate) fn sys_fanotify_mark(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Reject undefined flags.
    let flags: libc::c_uint = match req.data.args[1].try_into() {
        Ok(flags) => flags,
        Err(_) => return request.fail_syscall(Errno::EINVAL),
    };

    // SAFETY: Reject invalid flags.
    let flags = match MarkFlags::from_bits(flags) {
        Some(flags) => flags,
        None => return request.fail_syscall(Errno::EINVAL),
    };

    // SAFETY: Reject undefined/invalid masks.
    let mask = match MaskFlags::from_bits(req.data.args[2]) {
        Some(mask) => mask,
        None => return request.fail_syscall(Errno::EINVAL),
    };

    // fanotify(7) requires read access to the file or directory,
    // with the exception of symbolic links. However, due to
    // the `/proc' indirection in the handler we must not pass
    // WANT_READ here. See LTP fanotify tests.
    let mut fsflags = FsFlags::MUST_PATH;
    if flags.contains(MarkFlags::FAN_MARK_DONT_FOLLOW) {
        fsflags |= FsFlags::NO_FOLLOW_LAST;
    }

    let is32 = scmp_arch_bits(req.data.arch) == 32;
    let pidx = if is32 { 5 } else { 4 };
    let argv = &[SysArg {
        dirfd: Some(if is32 { 4 } else { 3 }),
        path: if req.data.args[pidx] != 0 {
            Some(pidx)
        } else {
            None
        },
        fsflags,
        ..Default::default()
    }];

    syscall_path_handler(
        request,
        "fanotify_mark",
        argv,
        |path_args, request, sandbox| {
            drop(sandbox); // release the read-lock.

            // SAFETY: SysArg has one element.
            #[allow(clippy::disallowed_methods)]
            let path = path_args.0.as_ref().unwrap();

            // SAFETY: MUST_PATH asserts dir is Some.
            #[allow(clippy::disallowed_methods)]
            let fd = path.dir.as_ref().map(|fd| fd.as_fd()).unwrap();

            // SAFETY:
            // 1. Strip FAN_{ACCESS,ACCESS_PERM,MODIFY} if we're marking a sidechannel device.
            // 2. Strip FAN_MARK_DONT_FOLLOW which has already been handled during canonicalization.
            let mut mask = mask;
            if is_sidechannel_device(fd_mode(fd)?) {
                mask.remove(MaskFlags::FAN_ACCESS);
                mask.remove(MaskFlags::FAN_ACCESS_PERM);
                mask.remove(MaskFlags::FAN_MODIFY);
            }
            let mut flags = flags;
            flags.remove(MarkFlags::FAN_MARK_DONT_FOLLOW);

            // SAFETY: We open a FD to the path and then use the
            // proc path $PROC_FILE/thread-self/fd/$fd in address'
            // path argument to avoid symlink TOCTOU.
            let pfd = XPathBuf::from_self_fd(fd.as_raw_fd());

            // Validate the FANotify FD.
            let notify_fd = RawFd::try_from(req.data.args[0]).or(Err(Errno::EBADF))?;
            if notify_fd < 0 {
                return Err(Errno::EBADF);
            }

            // Get the FANotify FD.
            let notify_fd = request.get_fd(notify_fd).map(|notify_fd| {
                // SAFETY: pidfd_getfd(2) returns valid FD on success.
                unsafe { Fanotify::from_raw_fd(notify_fd.into_raw_fd()) }
            })?;

            // Call fanotify_mark(2) through type-safe interface.
            notify_fd
                .mark(flags, mask, PROC_FILE(), Some(&pfd))
                .map(|_| request.return_syscall(0))
        },
    )
}
