use crate::co;
use crate::prelude::*;

const_basic_decl! { HRESULT: u32;
	/// A native
	/// [COM error code](https://learn.microsoft.com/en-us/openspecs/windows_protocols/ms-erref/0642cb2f-2075-4469-918c-4441e69c548a)
	/// (`u32`), which can be considered a superset of
	/// [`ERROR`](crate::co::ERROR).
	///
	/// The [`Result` alias](crate#errors-and-result-aliases) for this type is
	/// [`HrResult`](crate::HrResult).
	///
	/// Implements the standard [`Error`](std::error::Error) trait.
	///
	/// Implements the [`Debug`](std::fmt::Debug) and
	/// [`Display`](std::fmt::Display) traits to show the error code along with
	/// the error description, taken from
	/// [`FormattedError`](crate::prelude::FormattedError) trait. For example,
	/// the code below:
	///
	/// ```no_run
	/// use winsafe::{self as w, prelude::*, co};
	///
	/// println!("{}", co::HRESULT::E_INVALIDARG);
	/// ```
	///
	/// Will print:
	///
	/// ```text
	/// [0x80070057 2147942487] The parameter is incorrect.
	/// ```
	///
	/// The numbers inside the brackets are the system error code, in
	/// hexadecimal and decimal formats for convenience.
	///
	/// An [`ERROR`](crate::co::ERROR) can be seamlessly converted into an
	/// `HRESULT`. This operation is equivalent to the
	/// [`HRESULT_FROM_WIN32`](https://learn.microsoft.com/en-us/windows/win32/api/winerror/nf-winerror-hresult_from_win32)
	/// macro:
	///
	/// ```no_run
	/// use winsafe::{self as w, prelude::*, co};
	///
	/// let err = co::ERROR::FILE_NOT_FOUND;
	///
	/// let hr = err.to_hresult();
	/// ```
}

impl std::error::Error for HRESULT {
	fn source(&self) -> Option<&(dyn std::error::Error + 'static)> {
		None
	}
}

impl std::fmt::Display for HRESULT {
	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		if self.0 > 0xffff {
			write!(f, "[{:#010x}] {}", self.0, self.FormatMessage().trim_end())
		} else {
			write!(f, "[{:#06x}] {}", self.0, self.FormatMessage().trim_end())
		}
	}
}
impl std::fmt::Debug for HRESULT {
	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		if self.0 > 0xffff {
			write!(f, "HRESULT({:#010x} {}) {}",
				self.0, self.0, self.FormatMessage().trim_end())
		} else {
			write!(f, "HRESULT({:#06x} {}) {}",
				self.0, self.0, self.FormatMessage().trim_end())
		}
	}
}

impl FormattedError for HRESULT {}

impl co::ERROR {
	/// [`HRESULT_FROM_WIN32`](https://learn.microsoft.com/en-us/windows/win32/api/winerror/nf-winerror-hresult_from_win32)
	/// macro.
	pub const fn to_hresult(self) -> co::HRESULT {
		unsafe {
			if self.raw() as i32 <= 0 {
				co::HRESULT::from_raw(self.raw())
			} else {
				co::HRESULT::from_raw(
					(self.raw() & 0x0000_ffff) |
					co::FACILITY::WIN32.raw() << 16 |
					0x8000_0000,
				)
			}
		}
	}
}

impl HRESULT {
	/// [`HRESULT_CODE`](https://learn.microsoft.com/en-us/windows/win32/api/winerror/nf-winerror-hresult_code)
	/// macro.
	pub const fn code(self) -> u16 {
		(self.0 & 0xffff) as u16
	}

	/// [`HRESULT_FACILITY`](https://learn.microsoft.com/en-us/windows/win32/api/winerror/nf-winerror-hresult_facility)
	/// macro.
	pub const fn facility(self) -> co::FACILITY {
		unsafe { co::FACILITY::from_raw((self.0 >> 16) & 0x1fff) }
	}

	/// [`HRESULT_SEVERITY`](https://learn.microsoft.com/en-us/windows/win32/api/winerror/nf-winerror-hresult_severity)
	/// macro.
	pub const fn severity(self) -> co::SEVERITY {
		unsafe { co::SEVERITY::from_raw(((self.0 >> 31) & 0x1) as _) }
	}
}

const_values_pub! { HRESULT;
	/// Operation successful.
	S_OK 0
	S_FALSE 1

	E_UNEXPECTED 0x8000_ffff
	E_NOTIMPL 0x8000_4001
	E_OUTOFMEMORY 0x8007_000e
	E_INVALIDARG 0x8007_0057
	E_NOINTERFACE 0x8000_4002
	E_POINTER 0x8000_4003
	E_HANDLE 0x8007_0006
	E_ABORT 0x8000_4004
	E_FAIL 0x8000_4005
	E_ACCESSDENIED 0x8007_0005
	E_PENDING 0x8000_000a
	E_BOUNDS 0x8000_000b
	E_CHANGED_STATE 0x8000_000c
	E_ILLEGAL_STATE_CHANGE 0x8000_000d
	E_ILLEGAL_METHOD_CALL 0x8000_000e
	RO_E_METADATA_NAME_NOT_FOUND 0x8000_000f
	RO_E_METADATA_NAME_IS_NAMESPACE 0x8000_0010
	RO_E_METADATA_INVALID_TYPE_FORMAT 0x8000_0011
	RO_E_INVALID_METADATA_FILE 0x8000_0012
	RO_E_CLOSED 0x8000_0013
	RO_E_EXCLUSIVE_WRITE 0x8000_0014
	RO_E_CHANGE_NOTIFICATION_IN_PROGRESS 0x8000_0015
	RO_E_ERROR_STRING_NOT_FOUND 0x8000_0016
	E_STRING_NOT_NULL_TERMINATED 0x8000_0017
	E_ILLEGAL_DELEGATE_ASSIGNMENT 0x8000_0018
	E_ASYNC_OPERATION_NOT_STARTED 0x8000_0019
	E_APPLICATION_EXITING 0x8000_001a
	E_APPLICATION_VIEW_EXITING 0x8000_001b
	RO_E_MUST_BE_AGILE 0x8000_001c
	RO_E_UNSUPPORTED_FROM_MTA 0x8000_001d
	RO_E_COMMITTED 0x8000_001e
	RO_E_BLOCKED_CROSS_ASTA_CALL 0x8000_001f
	RO_E_CANNOT_ACTIVATE_FULL_TRUST_SERVER 0x8000_0020
	RO_E_CANNOT_ACTIVATE_UNIVERSAL_APPLICATION_SERVER 0x8000_0021
	CO_E_INIT_TLS 0x8000_4006
	CO_E_INIT_SHARED_ALLOCATOR 0x8000_4007
	CO_E_INIT_MEMORY_ALLOCATOR 0x8000_4008
	CO_E_INIT_CLASS_CACHE 0x8000_4009
	CO_E_INIT_RPC_CHANNEL 0x8000_400a
	CO_E_INIT_TLS_SET_CHANNEL_CONTROL 0x8000_400b
	CO_E_INIT_TLS_CHANNEL_CONTROL 0x8000_400c
	CO_E_INIT_UNACCEPTED_USER_ALLOCATOR 0x8000_400d
	CO_E_INIT_SCM_MUTEX_EXISTS 0x8000_400e
	CO_E_INIT_SCM_FILE_MAPPING_EXISTS 0x8000_400f
	CO_E_INIT_SCM_MAP_VIEW_OF_FILE 0x8000_4010
	CO_E_INIT_SCM_EXEC_FAILURE 0x8000_4011
	CO_E_INIT_ONLY_SINGLE_THREADED 0x8000_4012
	CO_E_CANT_REMOTE 0x8000_4013
	CO_E_BAD_SERVER_NAME 0x8000_4014
	CO_E_WRONG_SERVER_IDENTITY 0x8000_4015
	CO_E_OLE1DDE_DISABLED 0x8000_4016
	CO_E_RUNAS_SYNTAX 0x8000_4017
	CO_E_CREATEPROCESS_FAILURE 0x8000_4018
	CO_E_RUNAS_CREATEPROCESS_FAILURE 0x8000_4019
	CO_E_RUNAS_LOGON_FAILURE 0x8000_401a
	CO_E_LAUNCH_PERMSSION_DENIED 0x8000_401b
	CO_E_START_SERVICE_FAILURE 0x8000_401c
	CO_E_REMOTE_COMMUNICATION_FAILURE 0x8000_401d
	CO_E_SERVER_START_TIMEOUT 0x8000_401e
	CO_E_CLSREG_INCONSISTENT 0x8000_401f
	CO_E_IIDREG_INCONSISTENT 0x8000_4020
	CO_E_NOT_SUPPORTED 0x8000_4021
	CO_E_RELOAD_DLL 0x8000_4022
	CO_E_MSI_ERROR 0x8000_4023
	CO_E_ATTEMPT_TO_CREATE_OUTSIDE_CLIENT_CONTEXT 0x8000_4024
	CO_E_SERVER_PAUSED 0x8000_4025
	CO_E_SERVER_NOT_PAUSED 0x8000_4026
	CO_E_CLASS_DISABLED 0x8000_4027
	CO_E_CLRNOTAVAILABLE 0x8000_4028
	CO_E_ASYNC_WORK_REJECTED 0x8000_4029
	CO_E_SERVER_INIT_TIMEOUT 0x8000_402a
	CO_E_NO_SECCTX_IN_ACTIVATE 0x8000_402b
	CO_E_TRACKER_CONFIG 0x8000_4030
	CO_E_THREADPOOL_CONFIG 0x8000_4031
	CO_E_SXS_CONFIG 0x8000_4032
	CO_E_MALFORMED_SPN 0x8000_4033
	CO_E_UNREVOKED_REGISTRATION_ON_APARTMENT_SHUTDOWN 0x8000_4034
	CO_E_PREMATURE_STUB_RUNDOWN 0x8000_4035
	OLE_E_OLEVERB 0x8004_0000
	OLE_E_ADVF 0x8004_0001
	OLE_E_ENUM_NOMORE 0x8004_0002
	OLE_E_ADVISENOTSUPPORTED 0x8004_0003
	OLE_E_NOCONNECTION 0x8004_0004
	OLE_E_NOTRUNNING 0x8004_0005
	OLE_E_NOCACHE 0x8004_0006
	OLE_E_BLANK 0x8004_0007
	OLE_E_CLASSDIFF 0x8004_0008
	OLE_E_CANT_GETMONIKER 0x8004_0009
	OLE_E_CANT_BINDTOSOURCE 0x8004_000a
	OLE_E_STATIC 0x8004_000b
	OLE_E_PROMPTSAVECANCELLED 0x8004_000c
	OLE_E_INVALIDRECT 0x8004_000d
	OLE_E_WRONGCOMPOBJ 0x8004_000e
	OLE_E_INVALIDHWND 0x8004_000f
	OLE_E_NOT_INPLACEACTIVE 0x8004_0010
	OLE_E_CANTCONVERT 0x8004_0011
	OLE_E_NOSTORAGE 0x8004_0012
	DV_E_FORMATETC 0x8004_0064
	DV_E_DVTARGETDEVICE 0x8004_0065
	DV_E_STGMEDIUM 0x8004_0066
	DV_E_STATDATA 0x8004_0067
	DV_E_LINDEX 0x8004_0068
	DV_E_TYMED 0x8004_0069
	DV_E_CLIPFORMAT 0x8004_006a
	DV_E_DVASPECT 0x8004_006b
	DV_E_DVTARGETDEVICE_SIZE 0x8004_006c
	DV_E_NOIVIEWOBJECT 0x8004_006d
	DRAGDROP_E_NOTREGISTERED 0x8004_0100
	DRAGDROP_E_ALREADYREGISTERED 0x8004_0101
	DRAGDROP_E_INVALIDHWND 0x8004_0102
	DRAGDROP_E_CONCURRENT_DRAG_ATTEMPTED 0x8004_0103
	CLASS_E_NOAGGREGATION 0x8004_0110
	CLASS_E_CLASSNOTAVAILABLE 0x8004_0111
	CLASS_E_NOTLICENSED 0x8004_0112
	VIEW_E_DRAW 0x8004_0140
	REGDB_E_READREGDB 0x8004_0150
	REGDB_E_WRITEREGDB 0x8004_0151
	REGDB_E_KEYMISSING 0x8004_0152
	REGDB_E_INVALIDVALUE 0x8004_0153
	REGDB_E_CLASSNOTREG 0x8004_0154
	REGDB_E_IIDNOTREG 0x8004_0155
	REGDB_E_BADTHREADINGMODEL 0x8004_0156
	REGDB_E_PACKAGEPOLICYVIOLATION 0x8004_0157
	CAT_E_CATIDNOEXIST 0x8004_0160
	CAT_E_NODESCRIPTION 0x8004_0161
	CS_E_PACKAGE_NOTFOUND 0x8004_0164
	CS_E_NOT_DELETABLE 0x8004_0165
	CS_E_CLASS_NOTFOUND 0x8004_0166
	CS_E_INVALID_VERSION 0x8004_0167
	CS_E_NO_CLASSSTORE 0x8004_0168
	CS_E_OBJECT_NOTFOUND 0x8004_0169
	CS_E_OBJECT_ALREADY_EXISTS 0x8004_016a
	CS_E_INVALID_PATH 0x8004_016b
	CS_E_NETWORK_ERROR 0x8004_016c
	CS_E_ADMIN_LIMIT_EXCEEDED 0x8004_016d
	CS_E_SCHEMA_MISMATCH 0x8004_016e
	CS_E_INTERNAL_ERROR 0x8004_016f
	CACHE_E_NOCACHE_UPDATED 0x8004_0170
	OLEOBJ_E_NOVERBS 0x8004_0180
	OLEOBJ_E_INVALIDVERB 0x8004_0181
	INPLACE_E_NOTUNDOABLE 0x8004_01a0
	INPLACE_E_NOTOOLSPACE 0x8004_01a1
	CONVERT10_E_OLESTREAM_GET 0x8004_01c0
	CONVERT10_E_OLESTREAM_PUT 0x8004_01c1
	CONVERT10_E_OLESTREAM_FMT 0x8004_01c2
	CONVERT10_E_OLESTREAM_BITMAP_TO_DIB 0x8004_01c3
	CONVERT10_E_STG_FMT 0x8004_01c4
	CONVERT10_E_STG_NO_STD_STREAM 0x8004_01c5
	CONVERT10_E_STG_DIB_TO_BITMAP 0x8004_01c6
	CLIPBRD_E_CANT_OPEN 0x8004_01d0
	CLIPBRD_E_CANT_EMPTY 0x8004_01d1
	CLIPBRD_E_CANT_SET 0x8004_01d2
	CLIPBRD_E_BAD_DATA 0x8004_01d3
	CLIPBRD_E_CANT_CLOSE 0x8004_01d4
	MK_E_CONNECTMANUALLY 0x8004_01e0
	MK_E_EXCEEDEDDEADLINE 0x8004_01e1
	MK_E_NEEDGENERIC 0x8004_01e2
	MK_E_UNAVAILABLE 0x8004_01e3
	MK_E_SYNTAX 0x8004_01e4
	MK_E_NOOBJECT 0x8004_01e5
	MK_E_INVALIDEXTENSION 0x8004_01e6
	MK_E_INTERMEDIATEINTERFACENOTSUPPORTED 0x8004_01e7
	MK_E_NOTBINDABLE 0x8004_01e8
	MK_E_NOTBOUND 0x8004_01e9
	MK_E_CANTOPENFILE 0x8004_01ea
	MK_E_MUSTBOTHERUSER 0x8004_01eb
	MK_E_NOINVERSE 0x8004_01ec
	MK_E_NOSTORAGE 0x8004_01ed
	MK_E_NOPREFIX 0x8004_01ee
	MK_E_ENUMERATION_FAILED 0x8004_01ef
	CO_E_NOTINITIALIZED 0x8004_01f0
	CO_E_ALREADYINITIALIZED 0x8004_01f1
	CO_E_CANTDETERMINECLASS 0x8004_01f2
	CO_E_CLASSSTRING 0x8004_01f3
	CO_E_IIDSTRING 0x8004_01f4
	CO_E_APPNOTFOUND 0x8004_01f5
	CO_E_APPSINGLEUSE 0x8004_01f6
	CO_E_ERRORINAPP 0x8004_01f7
	CO_E_DLLNOTFOUND 0x8004_01f8
	CO_E_ERRORINDLL 0x8004_01f9
	CO_E_WRONGOSFORAPP 0x8004_01fa
	CO_E_OBJNOTREG 0x8004_01fb
	CO_E_OBJISREG 0x8004_01fc
	CO_E_OBJNOTCONNECTED 0x8004_01fd
	CO_E_APPDIDNTREG 0x8004_01fe
	CO_E_RELEASED 0x8004_01ff
	EVENT_S_SOME_SUBSCRIBERS_FAILED 0x0004_0200
	EVENT_E_ALL_SUBSCRIBERS_FAILED 0x8004_0201
	EVENT_S_NOSUBSCRIBERS 0x0004_0202
	EVENT_E_QUERYSYNTAX 0x8004_0203
	EVENT_E_QUERYFIELD 0x8004_0204
	EVENT_E_INTERNALEXCEPTION 0x8004_0205
	EVENT_E_INTERNALERROR 0x8004_0206
	EVENT_E_INVALID_PER_USER_SID 0x8004_0207
	EVENT_E_USER_EXCEPTION 0x8004_0208
	EVENT_E_TOO_MANY_METHODS 0x8004_0209
	EVENT_E_MISSING_EVENTCLASS 0x8004_020a
	EVENT_E_NOT_ALL_REMOVED 0x8004_020b
	EVENT_E_COMPLUS_NOT_INSTALLED 0x8004_020c
	EVENT_E_CANT_MODIFY_OR_DELETE_UNCONFIGURED_OBJECT 0x8004_020d
	EVENT_E_CANT_MODIFY_OR_DELETE_CONFIGURED_OBJECT 0x8004_020e
	EVENT_E_INVALID_EVENT_CLASS_PARTITION 0x8004_020f
	EVENT_E_PER_USER_SID_NOT_LOGGED_ON 0x8004_0210
	TPC_E_INVALID_PROPERTY 0x8004_0241
	TPC_E_NO_DEFAULT_TABLET 0x8004_0212
	TPC_E_UNKNOWN_PROPERTY 0x8004_021b
	TPC_E_INVALID_INPUT_RECT 0x8004_0219
	TPC_E_INVALID_STROKE 0x8004_0222
	TPC_E_INITIALIZE_FAIL 0x8004_0223
	TPC_E_NOT_RELEVANT 0x8004_0232
	TPC_E_INVALID_PACKET_DESCRIPTION 0x8004_0233
	TPC_E_RECOGNIZER_NOT_REGISTERED 0x8004_0235
	TPC_E_INVALID_RIGHTS 0x8004_0236
	TPC_E_OUT_OF_ORDER_CALL 0x8004_0237
	TPC_E_QUEUE_FULL 0x8004_0238
	TPC_E_INVALID_CONFIGURATION 0x8004_0239
	TPC_E_INVALID_DATA_FROM_RECOGNIZER 0x8004_023a
	TPC_S_TRUNCATED 0x0004_0252
	TPC_S_INTERRUPTED 0x0004_0253
	TPC_S_NO_DATA_TO_PROCESS 0x0004_0254
	XACT_E_ALREADYOTHERSINGLEPHASE 0x8004_d000
	XACT_E_CANTRETAIN 0x8004_d001
	XACT_E_COMMITFAILED 0x8004_d002
	XACT_E_COMMITPREVENTED 0x8004_d003
	XACT_E_HEURISTICABORT 0x8004_d004
	XACT_E_HEURISTICCOMMIT 0x8004_d005
	XACT_E_HEURISTICDAMAGE 0x8004_d006
	XACT_E_HEURISTICDANGER 0x8004_d007
	XACT_E_ISOLATIONLEVEL 0x8004_d008
	XACT_E_NOASYNC 0x8004_d009
	XACT_E_NOENLIST 0x8004_d00a
	XACT_E_NOISORETAIN 0x8004_d00b
	XACT_E_NORESOURCE 0x8004_d00c
	XACT_E_NOTCURRENT 0x8004_d00d
	XACT_E_NOTRANSACTION 0x8004_d00e
	XACT_E_NOTSUPPORTED 0x8004_d00f
	XACT_E_UNKNOWNRMGRID 0x8004_d010
	XACT_E_WRONGSTATE 0x8004_d011
	XACT_E_WRONGUOW 0x8004_d012
	XACT_E_XTIONEXISTS 0x8004_d013
	XACT_E_NOIMPORTOBJECT 0x8004_d014
	XACT_E_INVALIDCOOKIE 0x8004_d015
	XACT_E_INDOUBT 0x8004_d016
	XACT_E_NOTIMEOUT 0x8004_d017
	XACT_E_ALREADYINPROGRESS 0x8004_d018
	XACT_E_ABORTED 0x8004_d019
	XACT_E_LOGFULL 0x8004_d01a
	XACT_E_TMNOTAVAILABLE 0x8004_d01b
	XACT_E_CONNECTION_DOWN 0x8004_d01c
	XACT_E_CONNECTION_DENIED 0x8004_d01d
	XACT_E_REENLISTTIMEOUT 0x8004_d01e
	XACT_E_TIP_CONNECT_FAILED 0x8004_d01f
	XACT_E_TIP_PROTOCOL_ERROR 0x8004_d020
	XACT_E_TIP_PULL_FAILED 0x8004_d021
	XACT_E_DEST_TMNOTAVAILABLE 0x8004_d022
	XACT_E_TIP_DISABLED 0x8004_d023
	XACT_E_NETWORK_TX_DISABLED 0x8004_d024
	XACT_E_PARTNER_NETWORK_TX_DISABLED 0x8004_d025
	XACT_E_XA_TX_DISABLED 0x8004_d026
	XACT_E_UNABLE_TO_READ_DTC_CONFIG 0x8004_d027
	XACT_E_UNABLE_TO_LOAD_DTC_PROXY 0x8004_d028
	XACT_E_ABORTING 0x8004_d029
	XACT_E_PUSH_COMM_FAILURE 0x8004_d02a
	XACT_E_PULL_COMM_FAILURE 0x8004_d02b
	XACT_E_LU_TX_DISABLED 0x8004_d02c
	XACT_E_CLERKNOTFOUND 0x8004_d080
	XACT_E_CLERKEXISTS 0x8004_d081
	XACT_E_RECOVERYINPROGRESS 0x8004_d082
	XACT_E_TRANSACTIONCLOSED 0x8004_d083
	XACT_E_INVALIDLSN 0x8004_d084
	XACT_E_REPLAYREQUEST 0x8004_d085
	XACT_S_ASYNC 0x0004_d000
	XACT_S_DEFECT 0x0004_d001
	XACT_S_READONLY 0x0004_d002
	XACT_S_SOMENORETAIN 0x0004_d003
	XACT_S_OKINFORM 0x0004_d004
	XACT_S_MADECHANGESCONTENT 0x0004_d005
	XACT_S_MADECHANGESINFORM 0x0004_d006
	XACT_S_ALLNORETAIN 0x0004_d007
	XACT_S_ABORTING 0x0004_d008
	XACT_S_SINGLEPHASE 0x0004_d009
	XACT_S_LOCALLY_OK 0x0004_d00a
	XACT_S_LASTRESOURCEMANAGER 0x0004_d010
	CONTEXT_E_ABORTED 0x8004_e002
	CONTEXT_E_ABORTING 0x8004_e003
	CONTEXT_E_NOCONTEXT 0x8004_e004
	CONTEXT_E_WOULD_DEADLOCK 0x8004_e005
	CONTEXT_E_SYNCH_TIMEOUT 0x8004_e006
	CONTEXT_E_OLDREF 0x8004_e007
	CONTEXT_E_ROLENOTFOUND 0x8004_e00c
	CONTEXT_E_TMNOTAVAILABLE 0x8004_e00f
	CO_E_ACTIVATIONFAILED 0x8004_e021
	CO_E_ACTIVATIONFAILED_EVENTLOGGED 0x8004_e022
	CO_E_ACTIVATIONFAILED_CATALOGERROR 0x8004_e023
	CO_E_ACTIVATIONFAILED_TIMEOUT 0x8004_e024
	CO_E_INITIALIZATIONFAILED 0x8004_e025
	CONTEXT_E_NOJIT 0x8004_e026
	CONTEXT_E_NOTRANSACTION 0x8004_e027
	CO_E_THREADINGMODEL_CHANGED 0x8004_e028
	CO_E_NOIISINTRINSICS 0x8004_e029
	CO_E_NOCOOKIES 0x8004_e02a
	CO_E_DBERROR 0x8004_e02b
	CO_E_NOTPOOLED 0x8004_e02c
	CO_E_NOTCONSTRUCTED 0x8004_e02d
	CO_E_NOSYNCHRONIZATION 0x8004_e02e
	CO_E_ISOLEVELMISMATCH 0x8004_e02f
	CO_E_CALL_OUT_OF_TX_SCOPE_NOT_ALLOWED 0x8004_e030
	CO_E_EXIT_TRANSACTION_SCOPE_NOT_CALLED 0x8004_e031
	OLE_S_USEREG 0x0004_0000
	OLE_S_STATIC 0x0004_0001
	OLE_S_MAC_CLIPFORMAT 0x0004_0002
	DRAGDROP_S_DROP 0x0004_0100
	DRAGDROP_S_CANCEL 0x0004_0101
	DRAGDROP_S_USEDEFAULTCURSORS 0x0004_0102
	DATA_S_SAMEFORMATETC 0x0004_0130
	VIEW_S_ALREADY_FROZEN 0x0004_0140
	CACHE_S_FORMATETC_NOTSUPPORTED 0x0004_0170
	CACHE_S_SAMECACHE 0x0004_0171
	CACHE_S_SOMECACHES_NOTUPDATED 0x0004_0172
	OLEOBJ_S_INVALIDVERB 0x0004_0180
	OLEOBJ_S_CANNOT_DOVERB_NOW 0x0004_0181
	OLEOBJ_S_INVALIDHWND 0x0004_0182
	INPLACE_S_TRUNCATED 0x0004_01a0
	CONVERT10_S_NO_PRESENTATION 0x0004_01c0
	MK_S_REDUCED_TO_SELF 0x0004_01e2
	MK_S_ME 0x0004_01e4
	MK_S_HIM 0x0004_01e5
	MK_S_US 0x0004_01e6
	MK_S_MONIKERALREADYREGISTERED 0x0004_01e7
	SCHED_S_TASK_READY 0x0004_1300
	SCHED_S_TASK_RUNNING 0x0004_1301
	SCHED_S_TASK_DISABLED 0x0004_1302
	SCHED_S_TASK_HAS_NOT_RUN 0x0004_1303
	SCHED_S_TASK_NO_MORE_RUNS 0x0004_1304
	SCHED_S_TASK_NOT_SCHEDULED 0x0004_1305
	SCHED_S_TASK_TERMINATED 0x0004_1306
	SCHED_S_TASK_NO_VALID_TRIGGERS 0x0004_1307
	SCHED_S_EVENT_TRIGGER 0x0004_1308
	SCHED_E_TRIGGER_NOT_FOUND 0x8004_1309
	SCHED_E_TASK_NOT_READY 0x8004_130a
	SCHED_E_TASK_NOT_RUNNING 0x8004_130b
	SCHED_E_SERVICE_NOT_INSTALLED 0x8004_130c
	SCHED_E_CANNOT_OPEN_TASK 0x8004_130d
	SCHED_E_INVALID_TASK 0x8004_130e
	SCHED_E_ACCOUNT_INFORMATION_NOT_SET 0x8004_130f
	SCHED_E_ACCOUNT_NAME_NOT_FOUND 0x8004_1310
	SCHED_E_ACCOUNT_DBASE_CORRUPT 0x8004_1311
	SCHED_E_NO_SECURITY_SERVICES 0x8004_1312
	SCHED_E_UNKNOWN_OBJECT_VERSION 0x8004_1313
	SCHED_E_UNSUPPORTED_ACCOUNT_OPTION 0x8004_1314
	SCHED_E_SERVICE_NOT_RUNNING 0x8004_1315
	SCHED_E_UNEXPECTEDNODE 0x8004_1316
	SCHED_E_NAMESPACE 0x8004_1317
	SCHED_E_INVALIDVALUE 0x8004_1318
	SCHED_E_MISSINGNODE 0x8004_1319
	SCHED_E_MALFORMEDXML 0x8004_131a
	SCHED_S_SOME_TRIGGERS_FAILED 0x0004_131b
	SCHED_S_BATCH_LOGON_PROBLEM 0x0004_131c
	SCHED_E_TOO_MANY_NODES 0x8004_131d
	SCHED_E_PAST_END_BOUNDARY 0x8004_131e
	SCHED_E_ALREADY_RUNNING 0x8004_131f
	SCHED_E_USER_NOT_LOGGED_ON 0x8004_1320
	SCHED_E_INVALID_TASK_HASH 0x8004_1321
	SCHED_E_SERVICE_NOT_AVAILABLE 0x8004_1322
	SCHED_E_SERVICE_TOO_BUSY 0x8004_1323
	SCHED_E_TASK_ATTEMPTED 0x8004_1324
	SCHED_S_TASK_QUEUED 0x0004_1325
	SCHED_E_TASK_DISABLED 0x8004_1326
	SCHED_E_TASK_NOT_V1_COMPAT 0x8004_1327
	SCHED_E_START_ON_DEMAND 0x8004_1328
	SCHED_E_TASK_NOT_UBPM_COMPAT 0x8004_1329
	SCHED_E_DEPRECATED_FEATURE_USED 0x8004_1330
	CO_E_CLASS_CREATE_FAILED 0x8008_0001
	CO_E_SCM_ERROR 0x8008_0002
	CO_E_SCM_RPC_FAILURE 0x8008_0003
	CO_E_BAD_PATH 0x8008_0004
	CO_E_SERVER_EXEC_FAILURE 0x8008_0005
	CO_E_OBJSRV_RPC_FAILURE 0x8008_0006
	MK_E_NO_NORMALIZED 0x8008_0007
	CO_E_SERVER_STOPPING 0x8008_0008
	MEM_E_INVALID_ROOT 0x8008_0009
	MEM_E_INVALID_LINK 0x8008_0010
	MEM_E_INVALID_SIZE 0x8008_0011
	CO_S_NOTALLINTERFACES 0x0008_0012
	CO_S_MACHINENAMENOTFOUND 0x0008_0013
	CO_E_MISSING_DISPLAYNAME 0x8008_0015
	CO_E_RUNAS_VALUE_MUST_BE_AAA 0x8008_0016
	CO_E_ELEVATION_DISABLED 0x8008_0017
	APPX_E_PACKAGING_INTERNAL 0x8008_0200
	APPX_E_INTERLEAVING_NOT_ALLOWED 0x8008_0201
	APPX_E_RELATIONSHIPS_NOT_ALLOWED 0x8008_0202
	APPX_E_MISSING_REQUIRED_FILE 0x8008_0203
	APPX_E_INVALID_MANIFEST 0x8008_0204
	APPX_E_INVALID_BLOCKMAP 0x8008_0205
	APPX_E_CORRUPT_CONTENT 0x8008_0206
	APPX_E_BLOCK_HASH_INVALID 0x8008_0207
	APPX_E_REQUESTED_RANGE_TOO_LARGE 0x8008_0208
	APPX_E_INVALID_SIP_CLIENT_DATA 0x8008_0209
	APPX_E_INVALID_KEY_INFO 0x8008_020a
	APPX_E_INVALID_CONTENTGROUPMAP 0x8008_020b
	APPX_E_INVALID_APPINSTALLER 0x8008_020c
	APPX_E_DELTA_BASELINE_VERSION_MISMATCH 0x8008_020d
	APPX_E_DELTA_PACKAGE_MISSING_FILE 0x8008_020e
	APPX_E_INVALID_DELTA_PACKAGE 0x8008_020f
	APPX_E_DELTA_APPENDED_PACKAGE_NOT_ALLOWED 0x8008_0210
	APPX_E_INVALID_PACKAGING_LAYOUT 0x8008_0211
	APPX_E_INVALID_PACKAGESIGNCONFIG 0x8008_0212
	APPX_E_RESOURCESPRI_NOT_ALLOWED 0x8008_0213
	APPX_E_FILE_COMPRESSION_MISMATCH 0x8008_0214
	APPX_E_INVALID_PAYLOAD_PACKAGE_EXTENSION 0x8008_0215
	APPX_E_INVALID_ENCRYPTION_EXCLUSION_FILE_LIST 0x8008_0216
	BT_E_SPURIOUS_ACTIVATION 0x8008_0300
	DISP_E_UNKNOWNINTERFACE 0x8002_0001
	DISP_E_MEMBERNOTFOUND 0x8002_0003
	DISP_E_PARAMNOTFOUND 0x8002_0004
	DISP_E_TYPEMISMATCH 0x8002_0005
	DISP_E_UNKNOWNNAME 0x8002_0006
	DISP_E_NONAMEDARGS 0x8002_0007
	DISP_E_BADVARTYPE 0x8002_0008
	DISP_E_EXCEPTION 0x8002_0009
	DISP_E_OVERFLOW 0x8002_000a
	DISP_E_BADINDEX 0x8002_000b
	DISP_E_UNKNOWNLCID 0x8002_000c
	DISP_E_ARRAYISLOCKED 0x8002_000d
	DISP_E_BADPARAMCOUNT 0x8002_000e
	DISP_E_PARAMNOTOPTIONAL 0x8002_000f
	DISP_E_BADCALLEE 0x8002_0010
	DISP_E_NOTACOLLECTION 0x8002_0011
	DISP_E_DIVBYZERO 0x8002_0012
	DISP_E_BUFFERTOOSMALL 0x8002_0013
	TYPE_E_BUFFERTOOSMALL 0x8002_8016
	TYPE_E_FIELDNOTFOUND 0x8002_8017
	TYPE_E_INVDATAREAD 0x8002_8018
	TYPE_E_UNSUPFORMAT 0x8002_8019
	TYPE_E_REGISTRYACCESS 0x8002_801c
	TYPE_E_LIBNOTREGISTERED 0x8002_801d
	TYPE_E_UNDEFINEDTYPE 0x8002_8027
	TYPE_E_QUALIFIEDNAMEDISALLOWED 0x8002_8028
	TYPE_E_INVALIDSTATE 0x8002_8029
	TYPE_E_WRONGTYPEKIND 0x8002_802a
	TYPE_E_ELEMENTNOTFOUND 0x8002_802b
	TYPE_E_AMBIGUOUSNAME 0x8002_802c
	TYPE_E_NAMECONFLICT 0x8002_802d
	TYPE_E_UNKNOWNLCID 0x8002_802e
	TYPE_E_DLLFUNCTIONNOTFOUND 0x8002_802f
	TYPE_E_BADMODULEKIND 0x8002_88bd
	TYPE_E_SIZETOOBIG 0x8002_88c5
	TYPE_E_DUPLICATEID 0x8002_88c6
	TYPE_E_INVALIDID 0x8002_88cf
	TYPE_E_TYPEMISMATCH 0x8002_8ca0
	TYPE_E_OUTOFBOUNDS 0x8002_8ca1
	TYPE_E_IOERROR 0x8002_8ca2
	TYPE_E_CANTCREATETMPFILE 0x8002_8ca3
	TYPE_E_CANTLOADLIBRARY 0x8002_9c4a
	TYPE_E_INCONSISTENTPROPFUNCS 0x8002_9c83
	TYPE_E_CIRCULARTYPE 0x8002_9c84
	STG_E_INVALIDFUNCTION 0x8003_0001
	STG_E_FILENOTFOUND 0x8003_0002
	STG_E_PATHNOTFOUND 0x8003_0003
	STG_E_TOOMANYOPENFILES 0x8003_0004
	STG_E_ACCESSDENIED 0x8003_0005
	STG_E_INVALIDHANDLE 0x8003_0006
	STG_E_INSUFFICIENTMEMORY 0x8003_0008
	STG_E_INVALIDPOINTER 0x8003_0009
	STG_E_NOMOREFILES 0x8003_0012
	STG_E_DISKISWRITEPROTECTED 0x8003_0013
	STG_E_SEEKERROR 0x8003_0019
	STG_E_WRITEFAULT 0x8003_001d
	STG_E_READFAULT 0x8003_001e
	STG_E_SHAREVIOLATION 0x8003_0020
	STG_E_LOCKVIOLATION 0x8003_0021
	STG_E_FILEALREADYEXISTS 0x8003_0050
	STG_E_INVALIDPARAMETER 0x8003_0057
	STG_E_MEDIUMFULL 0x8003_0070
	STG_E_PROPSETMISMATCHED 0x8003_00f0
	STG_E_ABNORMALAPIEXIT 0x8003_00fa
	STG_E_INVALIDHEADER 0x8003_00fb
	STG_E_INVALIDNAME 0x8003_00fc
	STG_E_UNKNOWN 0x8003_00fd
	STG_E_UNIMPLEMENTEDFUNCTION 0x8003_00fe
	STG_E_INVALIDFLAG 0x8003_00ff
	STG_E_INUSE 0x8003_0100
	STG_E_NOTCURRENT 0x8003_0101
	STG_E_REVERTED 0x8003_0102
	STG_E_CANTSAVE 0x8003_0103
	STG_E_OLDFORMAT 0x8003_0104
	STG_E_OLDDLL 0x8003_0105
	STG_E_SHAREREQUIRED 0x8003_0106
	STG_E_NOTFILEBASEDSTORAGE 0x8003_0107
	STG_E_EXTANTMARSHALLINGS 0x8003_0108
	STG_E_DOCFILECORRUPT 0x8003_0109
	STG_E_BADBASEADDRESS 0x8003_0110
	STG_E_DOCFILETOOLARGE 0x8003_0111
	STG_E_NOTSIMPLEFORMAT 0x8003_0112
	STG_E_INCOMPLETE 0x8003_0201
	STG_E_TERMINATED 0x8003_0202
	STG_S_CONVERTED 0x0003_0200
	STG_S_BLOCK 0x0003_0201
	STG_S_RETRYNOW 0x0003_0202
	STG_S_MONITORING 0x0003_0203
	STG_S_MULTIPLEOPENS 0x0003_0204
	STG_S_CONSOLIDATIONFAILED 0x0003_0205
	STG_S_CANNOTCONSOLIDATE 0x0003_0206
	STG_S_POWER_CYCLE_REQUIRED 0x0003_0207
	STG_E_FIRMWARE_SLOT_INVALID 0x8003_0208
	STG_E_FIRMWARE_IMAGE_INVALID 0x8003_0209
	STG_E_DEVICE_UNRESPONSIVE 0x8003_020a
	STG_E_STATUS_COPY_PROTECTION_FAILURE 0x8003_0305
	STG_E_CSS_AUTHENTICATION_FAILURE 0x8003_0306
	STG_E_CSS_KEY_NOT_PRESENT 0x8003_0307
	STG_E_CSS_KEY_NOT_ESTABLISHED 0x8003_0308
	STG_E_CSS_SCRAMBLED_SECTOR 0x8003_0309
	STG_E_CSS_REGION_MISMATCH 0x8003_030a
	STG_E_RESETS_EXHAUSTED 0x8003_030b
	RPC_E_CALL_REJECTED 0x8001_0001
	RPC_E_CALL_CANCELED 0x8001_0002
	RPC_E_CANTPOST_INSENDCALL 0x8001_0003
	RPC_E_CANTCALLOUT_INASYNCCALL 0x8001_0004
	RPC_E_CANTCALLOUT_INEXTERNALCALL 0x8001_0005
	RPC_E_CONNECTION_TERMINATED 0x8001_0006
	RPC_E_SERVER_DIED 0x8001_0007
	RPC_E_CLIENT_DIED 0x8001_0008
	RPC_E_INVALID_DATAPACKET 0x8001_0009
	RPC_E_CANTTRANSMIT_CALL 0x8001_000a
	RPC_E_CLIENT_CANTMARSHAL_DATA 0x8001_000b
	RPC_E_CLIENT_CANTUNMARSHAL_DATA 0x8001_000c
	RPC_E_SERVER_CANTMARSHAL_DATA 0x8001_000d
	RPC_E_SERVER_CANTUNMARSHAL_DATA 0x8001_000e
	RPC_E_INVALID_DATA 0x8001_000f
	RPC_E_INVALID_PARAMETER 0x8001_0010
	RPC_E_CANTCALLOUT_AGAIN 0x8001_0011
	RPC_E_SERVER_DIED_DNE 0x8001_0012
	RPC_E_SYS_CALL_FAILED 0x8001_0100
	RPC_E_OUT_OF_RESOURCES 0x8001_0101
	RPC_E_ATTEMPTED_MULTITHREAD 0x8001_0102
	RPC_E_NOT_REGISTERED 0x8001_0103
	RPC_E_FAULT 0x8001_0104
	RPC_E_SERVERFAULT 0x8001_0105
	RPC_E_CHANGED_MODE 0x8001_0106
	RPC_E_INVALIDMETHOD 0x8001_0107
	RPC_E_DISCONNECTED 0x8001_0108
	RPC_E_RETRY 0x8001_0109
	RPC_E_SERVERCALL_RETRYLATER 0x8001_010a
	RPC_E_SERVERCALL_REJECTED 0x8001_010b
	RPC_E_INVALID_CALLDATA 0x8001_010c
	RPC_E_CANTCALLOUT_ININPUTSYNCCALL 0x8001_010d
	RPC_E_WRONG_THREAD 0x8001_010e
	RPC_E_THREAD_NOT_INIT 0x8001_010f
	RPC_E_VERSION_MISMATCH 0x8001_0110
	RPC_E_INVALID_HEADER 0x8001_0111
	RPC_E_INVALID_EXTENSION 0x8001_0112
	RPC_E_INVALID_IPID 0x8001_0113
	RPC_E_INVALID_OBJECT 0x8001_0114
	RPC_S_CALLPENDING 0x8001_0115
	RPC_S_WAITONTIMER 0x8001_0116
	RPC_E_CALL_COMPLETE 0x8001_0117
	RPC_E_UNSECURE_CALL 0x8001_0118
	RPC_E_TOO_LATE 0x8001_0119
	RPC_E_NO_GOOD_SECURITY_PACKAGES 0x8001_011a
	RPC_E_ACCESS_DENIED 0x8001_011b
	RPC_E_REMOTE_DISABLED 0x8001_011c
	RPC_E_INVALID_OBJREF 0x8001_011d
	RPC_E_NO_CONTEXT 0x8001_011e
	RPC_E_TIMEOUT 0x8001_011f
	RPC_E_NO_SYNC 0x8001_0120
	RPC_E_FULLSIC_REQUIRED 0x8001_0121
	RPC_E_INVALID_STD_NAME 0x8001_0122
	CO_E_FAILEDTOIMPERSONATE 0x8001_0123
	CO_E_FAILEDTOGETSECCTX 0x8001_0124
	CO_E_FAILEDTOOPENTHREADTOKEN 0x8001_0125
	CO_E_FAILEDTOGETTOKENINFO 0x8001_0126
	CO_E_TRUSTEEDOESNTMATCHCLIENT 0x8001_0127
	CO_E_FAILEDTOQUERYCLIENTBLANKET 0x8001_0128
	CO_E_FAILEDTOSETDACL 0x8001_0129
	CO_E_ACCESSCHECKFAILED 0x8001_012a
	CO_E_NETACCESSAPIFAILED 0x8001_012b
	CO_E_WRONGTRUSTEENAMESYNTAX 0x8001_012c
	CO_E_INVALIDSID 0x8001_012d
	CO_E_CONVERSIONFAILED 0x8001_012e
	CO_E_NOMATCHINGSIDFOUND 0x8001_012f
	CO_E_LOOKUPACCSIDFAILED 0x8001_0130
	CO_E_NOMATCHINGNAMEFOUND 0x8001_0131
	CO_E_LOOKUPACCNAMEFAILED 0x8001_0132
	CO_E_SETSERLHNDLFAILED 0x8001_0133
	CO_E_FAILEDTOGETWINDIR 0x8001_0134
	CO_E_PATHTOOLONG 0x8001_0135
	CO_E_FAILEDTOGENUUID 0x8001_0136
	CO_E_FAILEDTOCREATEFILE 0x8001_0137
	CO_E_FAILEDTOCLOSEHANDLE 0x8001_0138
	CO_E_EXCEEDSYSACLLIMIT 0x8001_0139
	CO_E_ACESINWRONGORDER 0x8001_013a
	CO_E_INCOMPATIBLESTREAMVERSION 0x8001_013b
	CO_E_FAILEDTOOPENPROCESSTOKEN 0x8001_013c
	CO_E_DECODEFAILED 0x8001_013d
	CO_E_ACNOTINITIALIZED 0x8001_013f
	CO_E_CANCEL_DISABLED 0x8001_0140
	RPC_E_UNEXPECTED 0x8001_ffff
	ERROR_AUDITING_DISABLED 0xc009_0001
	ERROR_ALL_SIDS_FILTERED 0xc009_0002
	ERROR_BIZRULES_NOT_ENABLED 0xc009_0003
	NTE_BAD_UID 0x8009_0001
	NTE_BAD_HASH 0x8009_0002
	NTE_BAD_KEY 0x8009_0003
	NTE_BAD_LEN 0x8009_0004
	NTE_BAD_DATA 0x8009_0005
	NTE_BAD_SIGNATURE 0x8009_0006
	NTE_BAD_VER 0x8009_0007
	NTE_BAD_ALGID 0x8009_0008
	NTE_BAD_FLAGS 0x8009_0009
	NTE_BAD_TYPE 0x8009_000a
	NTE_BAD_KEY_STATE 0x8009_000b
	NTE_BAD_HASH_STATE 0x8009_000c
	NTE_NO_KEY 0x8009_000d
	NTE_NO_MEMORY 0x8009_000e
	NTE_EXISTS 0x8009_000f
	NTE_PERM 0x8009_0010
	NTE_NOT_FOUND 0x8009_0011
	NTE_DOUBLE_ENCRYPT 0x8009_0012
	NTE_BAD_PROVIDER 0x8009_0013
	NTE_BAD_PROV_TYPE 0x8009_0014
	NTE_BAD_PUBLIC_KEY 0x8009_0015
	NTE_BAD_KEYSET 0x8009_0016
	NTE_PROV_TYPE_NOT_DEF 0x8009_0017
	NTE_PROV_TYPE_ENTRY_BAD 0x8009_0018
	NTE_KEYSET_NOT_DEF 0x8009_0019
	NTE_KEYSET_ENTRY_BAD 0x8009_001a
	NTE_PROV_TYPE_NO_MATCH 0x8009_001b
	NTE_SIGNATURE_FILE_BAD 0x8009_001c
	NTE_PROVIDER_DLL_FAIL 0x8009_001d
	NTE_PROV_DLL_NOT_FOUND 0x8009_001e
	NTE_BAD_KEYSET_PARAM 0x8009_001f
	NTE_FAIL 0x8009_0020
	NTE_SYS_ERR 0x8009_0021
	NTE_SILENT_CONTEXT 0x8009_0022
	NTE_TOKEN_KEYSET_STORAGE_FULL 0x8009_0023
	NTE_TEMPORARY_PROFILE 0x8009_0024
	NTE_FIXEDPARAMETER 0x8009_0025
	NTE_INVALID_HANDLE 0x8009_0026
	NTE_INVALID_PARAMETER 0x8009_0027
	NTE_BUFFER_TOO_SMALL 0x8009_0028
	NTE_NOT_SUPPORTED 0x8009_0029
	NTE_NO_MORE_ITEMS 0x8009_002a
	NTE_BUFFERS_OVERLAP 0x8009_002b
	NTE_DECRYPTION_FAILURE 0x8009_002c
	NTE_INTERNAL_ERROR 0x8009_002d
	NTE_UI_REQUIRED 0x8009_002e
	NTE_HMAC_NOT_SUPPORTED 0x8009_002f
	NTE_DEVICE_NOT_READY 0x8009_0030
	NTE_AUTHENTICATION_IGNORED 0x8009_0031
	NTE_VALIDATION_FAILED 0x8009_0032
	NTE_INCORRECT_PASSWORD 0x8009_0033
	NTE_ENCRYPTION_FAILURE 0x8009_0034
	NTE_DEVICE_NOT_FOUND 0x8009_0035
	NTE_USER_CANCELLED 0x8009_0036
	NTE_PASSWORD_CHANGE_REQUIRED 0x8009_0037
	NTE_NOT_ACTIVE_CONSOLE 0x8009_0038
	SEC_E_INSUFFICIENT_MEMORY 0x8009_0300
	SEC_E_INVALID_HANDLE 0x8009_0301
	SEC_E_UNSUPPORTED_FUNCTION 0x8009_0302
	SEC_E_TARGET_UNKNOWN 0x8009_0303
	SEC_E_INTERNAL_ERROR 0x8009_0304
	SEC_E_SECPKG_NOT_FOUND 0x8009_0305
	SEC_E_NOT_OWNER 0x8009_0306
	SEC_E_CANNOT_INSTALL 0x8009_0307
	SEC_E_INVALID_TOKEN 0x8009_0308
	SEC_E_CANNOT_PACK 0x8009_0309
	SEC_E_QOP_NOT_SUPPORTED 0x8009_030a
	SEC_E_NO_IMPERSONATION 0x8009_030b
	SEC_E_LOGON_DENIED 0x8009_030c
	SEC_E_UNKNOWN_CREDENTIALS 0x8009_030d
	SEC_E_NO_CREDENTIALS 0x8009_030e
	SEC_E_MESSAGE_ALTERED 0x8009_030f
	SEC_E_OUT_OF_SEQUENCE 0x8009_0310
	SEC_E_NO_AUTHENTICATING_AUTHORITY 0x8009_0311
	SEC_I_CONTINUE_NEEDED 0x0009_0312
	SEC_I_COMPLETE_NEEDED 0x0009_0313
	SEC_I_COMPLETE_AND_CONTINUE 0x0009_0314
	SEC_I_LOCAL_LOGON 0x0009_0315
	SEC_I_GENERIC_EXTENSION_RECEIVED 0x0009_0316
	SEC_E_BAD_PKGID 0x8009_0316
	SEC_E_CONTEXT_EXPIRED 0x8009_0317
	SEC_I_CONTEXT_EXPIRED 0x0009_0317
	SEC_E_INCOMPLETE_MESSAGE 0x8009_0318
	SEC_E_INCOMPLETE_CREDENTIALS 0x8009_0320
	SEC_E_BUFFER_TOO_SMALL 0x8009_0321
	SEC_I_INCOMPLETE_CREDENTIALS 0x0009_0320
	SEC_I_RENEGOTIATE 0x0009_0321
	SEC_E_WRONG_PRINCIPAL 0x8009_0322
	SEC_I_NO_LSA_CONTEXT 0x0009_0323
	SEC_E_TIME_SKEW 0x8009_0324
	SEC_E_UNTRUSTED_ROOT 0x8009_0325
	SEC_E_ILLEGAL_MESSAGE 0x8009_0326
	SEC_E_CERT_UNKNOWN 0x8009_0327
	SEC_E_CERT_EXPIRED 0x8009_0328
	SEC_E_ENCRYPT_FAILURE 0x8009_0329
	SEC_E_DECRYPT_FAILURE 0x8009_0330
	SEC_E_ALGORITHM_MISMATCH 0x8009_0331
	SEC_E_SECURITY_QOS_FAILED 0x8009_0332
	SEC_E_UNFINISHED_CONTEXT_DELETED 0x8009_0333
	SEC_E_NO_TGT_REPLY 0x8009_0334
	SEC_E_NO_IP_ADDRESSES 0x8009_0335
	SEC_E_WRONG_CREDENTIAL_HANDLE 0x8009_0336
	SEC_E_CRYPTO_SYSTEM_INVALID 0x8009_0337
	SEC_E_MAX_REFERRALS_EXCEEDED 0x8009_0338
	SEC_E_MUST_BE_KDC 0x8009_0339
	SEC_E_STRONG_CRYPTO_NOT_SUPPORTED 0x8009_033a
	SEC_E_TOO_MANY_PRINCIPALS 0x8009_033b
	SEC_E_NO_PA_DATA 0x8009_033c
	SEC_E_PKINIT_NAME_MISMATCH 0x8009_033d
	SEC_E_SMARTCARD_LOGON_REQUIRED 0x8009_033e
	SEC_E_SHUTDOWN_IN_PROGRESS 0x8009_033f
	SEC_E_KDC_INVALID_REQUEST 0x8009_0340
	SEC_E_KDC_UNABLE_TO_REFER 0x8009_0341
	SEC_E_KDC_UNKNOWN_ETYPE 0x8009_0342
	SEC_E_UNSUPPORTED_PREAUTH 0x8009_0343
	SEC_E_DELEGATION_REQUIRED 0x8009_0345
	SEC_E_BAD_BINDINGS 0x8009_0346
	SEC_E_MULTIPLE_ACCOUNTS 0x8009_0347
	SEC_E_NO_KERB_KEY 0x8009_0348
	SEC_E_CERT_WRONG_USAGE 0x8009_0349
	SEC_E_DOWNGRADE_DETECTED 0x8009_0350
	SEC_E_SMARTCARD_CERT_REVOKED 0x8009_0351
	SEC_E_ISSUING_CA_UNTRUSTED 0x8009_0352
	SEC_E_REVOCATION_OFFLINE_C 0x8009_0353
	SEC_E_PKINIT_CLIENT_FAILURE 0x8009_0354
	SEC_E_SMARTCARD_CERT_EXPIRED 0x8009_0355
	SEC_E_NO_S4U_PROT_SUPPORT 0x8009_0356
	SEC_E_CROSSREALM_DELEGATION_FAILURE 0x8009_0357
	SEC_E_REVOCATION_OFFLINE_KDC 0x8009_0358
	SEC_E_ISSUING_CA_UNTRUSTED_KDC 0x8009_0359
	SEC_E_KDC_CERT_EXPIRED 0x8009_035a
	SEC_E_KDC_CERT_REVOKED 0x8009_035b
	SEC_I_SIGNATURE_NEEDED 0x0009_035c
	SEC_E_INVALID_PARAMETER 0x8009_035d
	SEC_E_DELEGATION_POLICY 0x8009_035e
	SEC_E_POLICY_NLTM_ONLY 0x8009_035f
	SEC_I_NO_RENEGOTIATION 0x0009_0360
	SEC_E_NO_CONTEXT 0x8009_0361
	SEC_E_PKU2U_CERT_FAILURE 0x8009_0362
	SEC_E_MUTUAL_AUTH_FAILED 0x8009_0363
	SEC_I_MESSAGE_FRAGMENT 0x0009_0364
	SEC_E_ONLY_HTTPS_ALLOWED 0x8009_0365
	SEC_I_CONTINUE_NEEDED_MESSAGE_OK 0x0009_0366
	SEC_E_APPLICATION_PROTOCOL_MISMATCH 0x8009_0367
	SEC_I_ASYNC_CALL_PENDING 0x0009_0368
	SEC_E_INVALID_UPN_NAME 0x8009_0369
	SEC_E_EXT_BUFFER_TOO_SMALL 0x8009_036a
	SEC_E_INSUFFICIENT_BUFFERS 0x8009_036b
	CRYPT_E_MSG_ERROR 0x8009_1001
	CRYPT_E_UNKNOWN_ALGO 0x8009_1002
	CRYPT_E_OID_FORMAT 0x8009_1003
	CRYPT_E_INVALID_MSG_TYPE 0x8009_1004
	CRYPT_E_UNEXPECTED_ENCODING 0x8009_1005
	CRYPT_E_AUTH_ATTR_MISSING 0x8009_1006
	CRYPT_E_HASH_VALUE 0x8009_1007
	CRYPT_E_INVALID_INDEX 0x8009_1008
	CRYPT_E_ALREADY_DECRYPTED 0x8009_1009
	CRYPT_E_NOT_DECRYPTED 0x8009_100a
	CRYPT_E_RECIPIENT_NOT_FOUND 0x8009_100b
	CRYPT_E_CONTROL_TYPE 0x8009_100c
	CRYPT_E_ISSUER_SERIALNUMBER 0x8009_100d
	CRYPT_E_SIGNER_NOT_FOUND 0x8009_100e
	CRYPT_E_ATTRIBUTES_MISSING 0x8009_100f
	CRYPT_E_STREAM_MSG_NOT_READY 0x8009_1010
	CRYPT_E_STREAM_INSUFFICIENT_DATA 0x8009_1011
	CRYPT_I_NEW_PROTECTION_REQUIRED 0x0009_1012
	CRYPT_E_BAD_LEN 0x8009_2001
	CRYPT_E_BAD_ENCODE 0x8009_2002
	CRYPT_E_FILE_ERROR 0x8009_2003
	CRYPT_E_NOT_FOUND 0x8009_2004
	CRYPT_E_EXISTS 0x8009_2005
	CRYPT_E_NO_PROVIDER 0x8009_2006
	CRYPT_E_SELF_SIGNED 0x8009_2007
	CRYPT_E_DELETED_PREV 0x8009_2008
	CRYPT_E_NO_MATCH 0x8009_2009
	CRYPT_E_UNEXPECTED_MSG_TYPE 0x8009_200a
	CRYPT_E_NO_KEY_PROPERTY 0x8009_200b
	CRYPT_E_NO_DECRYPT_CERT 0x8009_200c
	CRYPT_E_BAD_MSG 0x8009_200d
	CRYPT_E_NO_SIGNER 0x8009_200e
	CRYPT_E_PENDING_CLOSE 0x8009_200f
	CRYPT_E_REVOKED 0x8009_2010
	CRYPT_E_NO_REVOCATION_DLL 0x8009_2011
	CRYPT_E_NO_REVOCATION_CHECK 0x8009_2012
	CRYPT_E_REVOCATION_OFFLINE 0x8009_2013
	CRYPT_E_NOT_IN_REVOCATION_DATABASE 0x8009_2014
	CRYPT_E_INVALID_NUMERIC_STRING 0x8009_2020
	CRYPT_E_INVALID_PRINTABLE_STRING 0x8009_2021
	CRYPT_E_INVALID_IA5_STRING 0x8009_2022
	CRYPT_E_INVALID_X500_STRING 0x8009_2023
	CRYPT_E_NOT_CHAR_STRING 0x8009_2024
	CRYPT_E_FILERESIZED 0x8009_2025
	CRYPT_E_SECURITY_SETTINGS 0x8009_2026
	CRYPT_E_NO_VERIFY_USAGE_DLL 0x8009_2027
	CRYPT_E_NO_VERIFY_USAGE_CHECK 0x8009_2028
	CRYPT_E_VERIFY_USAGE_OFFLINE 0x8009_2029
	CRYPT_E_NOT_IN_CTL 0x8009_202a
	CRYPT_E_NO_TRUSTED_SIGNER 0x8009_202b
	CRYPT_E_MISSING_PUBKEY_PARA 0x8009_202c
	CRYPT_E_OBJECT_LOCATOR_OBJECT_NOT_FOUND 0x8009_202d
	CRYPT_E_OSS_ERROR 0x8009_3000
	OSS_MORE_BUF 0x8009_3001
	OSS_NEGATIVE_UINTEGER 0x8009_3002
	OSS_PDU_RANGE 0x8009_3003
	OSS_MORE_INPUT 0x8009_3004
	OSS_DATA_ERROR 0x8009_3005
	OSS_BAD_ARG 0x8009_3006
	OSS_BAD_VERSION 0x8009_3007
	OSS_OUT_MEMORY 0x8009_3008
	OSS_PDU_MISMATCH 0x8009_3009
	OSS_LIMITED 0x8009_300a
	OSS_BAD_PTR 0x8009_300b
	OSS_BAD_TIME 0x8009_300c
	OSS_INDEFINITE_NOT_SUPPORTED 0x8009_300d
	OSS_MEM_ERROR 0x8009_300e
	OSS_BAD_TABLE 0x8009_300f
	OSS_TOO_LONG 0x8009_3010
	OSS_CONSTRAINT_VIOLATED 0x8009_3011
	OSS_FATAL_ERROR 0x8009_3012
	OSS_ACCESS_SERIALIZATION_ERROR 0x8009_3013
	OSS_NULL_TBL 0x8009_3014
	OSS_NULL_FCN 0x8009_3015
	OSS_BAD_ENCRULES 0x8009_3016
	OSS_UNAVAIL_ENCRULES 0x8009_3017
	OSS_CANT_OPEN_TRACE_WINDOW 0x8009_3018
	OSS_UNIMPLEMENTED 0x8009_3019
	OSS_OID_DLL_NOT_LINKED 0x8009_301a
	OSS_CANT_OPEN_TRACE_FILE 0x8009_301b
	OSS_TRACE_FILE_ALREADY_OPEN 0x8009_301c
	OSS_TABLE_MISMATCH 0x8009_301d
	OSS_TYPE_NOT_SUPPORTED 0x8009_301e
	OSS_REAL_DLL_NOT_LINKED 0x8009_301f
	OSS_REAL_CODE_NOT_LINKED 0x8009_3020
	OSS_OUT_OF_RANGE 0x8009_3021
	OSS_COPIER_DLL_NOT_LINKED 0x8009_3022
	OSS_CONSTRAINT_DLL_NOT_LINKED 0x8009_3023
	OSS_COMPARATOR_DLL_NOT_LINKED 0x8009_3024
	OSS_COMPARATOR_CODE_NOT_LINKED 0x8009_3025
	OSS_MEM_MGR_DLL_NOT_LINKED 0x8009_3026
	OSS_PDV_DLL_NOT_LINKED 0x8009_3027
	OSS_PDV_CODE_NOT_LINKED 0x8009_3028
	OSS_API_DLL_NOT_LINKED 0x8009_3029
	OSS_BERDER_DLL_NOT_LINKED 0x8009_302a
	OSS_PER_DLL_NOT_LINKED 0x8009_302b
	OSS_OPEN_TYPE_ERROR 0x8009_302c
	OSS_MUTEX_NOT_CREATED 0x8009_302d
	OSS_CANT_CLOSE_TRACE_FILE 0x8009_302e
	CRYPT_E_ASN1_ERROR 0x8009_3100
	CRYPT_E_ASN1_INTERNAL 0x8009_3101
	CRYPT_E_ASN1_EOD 0x8009_3102
	CRYPT_E_ASN1_CORRUPT 0x8009_3103
	CRYPT_E_ASN1_LARGE 0x8009_3104
	CRYPT_E_ASN1_CONSTRAINT 0x8009_3105
	CRYPT_E_ASN1_MEMORY 0x8009_3106
	CRYPT_E_ASN1_OVERFLOW 0x8009_3107
	CRYPT_E_ASN1_BADPDU 0x8009_3108
	CRYPT_E_ASN1_BADARGS 0x8009_3109
	CRYPT_E_ASN1_BADREAL 0x8009_310a
	CRYPT_E_ASN1_BADTAG 0x8009_310b
	CRYPT_E_ASN1_CHOICE 0x8009_310c
	CRYPT_E_ASN1_RULE 0x8009_310d
	CRYPT_E_ASN1_UTF8 0x8009_310e
	CRYPT_E_ASN1_PDU_TYPE 0x8009_3133
	CRYPT_E_ASN1_NYI 0x8009_3134
	CRYPT_E_ASN1_EXTENDED 0x8009_3201
	CRYPT_E_ASN1_NOEOD 0x8009_3202
	CERTSRV_E_BAD_REQUESTSUBJECT 0x8009_4001
	CERTSRV_E_NO_REQUEST 0x8009_4002
	CERTSRV_E_BAD_REQUESTSTATUS 0x8009_4003
	CERTSRV_E_PROPERTY_EMPTY 0x8009_4004
	CERTSRV_E_INVALID_CA_CERTIFICATE 0x8009_4005
	CERTSRV_E_SERVER_SUSPENDED 0x8009_4006
	CERTSRV_E_ENCODING_LENGTH 0x8009_4007
	CERTSRV_E_ROLECONFLICT 0x8009_4008
	CERTSRV_E_RESTRICTEDOFFICER 0x8009_4009
	CERTSRV_E_KEY_ARCHIVAL_NOT_CONFIGURED 0x8009_400a
	CERTSRV_E_NO_VALID_KRA 0x8009_400b
	CERTSRV_E_BAD_REQUEST_KEY_ARCHIVAL 0x8009_400c
	CERTSRV_E_NO_CAADMIN_DEFINED 0x8009_400d
	CERTSRV_E_BAD_RENEWAL_CERT_ATTRIBUTE 0x8009_400e
	CERTSRV_E_NO_DB_SESSIONS 0x8009_400f
	CERTSRV_E_ALIGNMENT_FAULT 0x8009_4010
	CERTSRV_E_ENROLL_DENIED 0x8009_4011
	CERTSRV_E_TEMPLATE_DENIED 0x8009_4012
	CERTSRV_E_DOWNLEVEL_DC_SSL_OR_UPGRADE 0x8009_4013
	CERTSRV_E_ADMIN_DENIED_REQUEST 0x8009_4014
	CERTSRV_E_NO_POLICY_SERVER 0x8009_4015
	CERTSRV_E_WEAK_SIGNATURE_OR_KEY 0x8009_4016
	CERTSRV_E_KEY_ATTESTATION_NOT_SUPPORTED 0x8009_4017
	CERTSRV_E_ENCRYPTION_CERT_REQUIRED 0x8009_4018
	CERTSRV_E_UNSUPPORTED_CERT_TYPE 0x8009_4800
	CERTSRV_E_NO_CERT_TYPE 0x8009_4801
	CERTSRV_E_TEMPLATE_CONFLICT 0x8009_4802
	CERTSRV_E_SUBJECT_ALT_NAME_REQUIRED 0x8009_4803
	CERTSRV_E_ARCHIVED_KEY_REQUIRED 0x8009_4804
	CERTSRV_E_SMIME_REQUIRED 0x8009_4805
	CERTSRV_E_BAD_RENEWAL_SUBJECT 0x8009_4806
	CERTSRV_E_BAD_TEMPLATE_VERSION 0x8009_4807
	CERTSRV_E_TEMPLATE_POLICY_REQUIRED 0x8009_4808
	CERTSRV_E_SIGNATURE_POLICY_REQUIRED 0x8009_4809
	CERTSRV_E_SIGNATURE_COUNT 0x8009_480a
	CERTSRV_E_SIGNATURE_REJECTED 0x8009_480b
	CERTSRV_E_ISSUANCE_POLICY_REQUIRED 0x8009_480c
	CERTSRV_E_SUBJECT_UPN_REQUIRED 0x8009_480d
	CERTSRV_E_SUBJECT_DIRECTORY_GUID_REQUIRED 0x8009_480e
	CERTSRV_E_SUBJECT_DNS_REQUIRED 0x8009_480f
	CERTSRV_E_ARCHIVED_KEY_UNEXPECTED 0x8009_4810
	CERTSRV_E_KEY_LENGTH 0x8009_4811
	CERTSRV_E_SUBJECT_EMAIL_REQUIRED 0x8009_4812
	CERTSRV_E_UNKNOWN_CERT_TYPE 0x8009_4813
	CERTSRV_E_CERT_TYPE_OVERLAP 0x8009_4814
	CERTSRV_E_TOO_MANY_SIGNATURES 0x8009_4815
	CERTSRV_E_RENEWAL_BAD_PUBLIC_KEY 0x8009_4816
	CERTSRV_E_INVALID_EK 0x8009_4817
	CERTSRV_E_INVALID_IDBINDING 0x8009_4818
	CERTSRV_E_INVALID_ATTESTATION 0x8009_4819
	CERTSRV_E_KEY_ATTESTATION 0x8009_481a
	CERTSRV_E_CORRUPT_KEY_ATTESTATION 0x8009_481b
	CERTSRV_E_EXPIRED_CHALLENGE 0x8009_481c
	CERTSRV_E_INVALID_RESPONSE 0x8009_481d
	CERTSRV_E_INVALID_REQUESTID 0x8009_481e
	CERTSRV_E_REQUEST_PRECERTIFICATE_MISMATCH 0x8009_481f
	CERTSRV_E_PENDING_CLIENT_RESPONSE 0x8009_4820
	XENROLL_E_KEY_NOT_EXPORTABLE 0x8009_5000
	XENROLL_E_CANNOT_ADD_ROOT_CERT 0x8009_5001
	XENROLL_E_RESPONSE_KA_HASH_NOT_FOUND 0x8009_5002
	XENROLL_E_RESPONSE_UNEXPECTED_KA_HASH 0x8009_5003
	XENROLL_E_RESPONSE_KA_HASH_MISMATCH 0x8009_5004
	XENROLL_E_KEYSPEC_SMIME_MISMATCH 0x8009_5005
	TRUST_E_SYSTEM_ERROR 0x8009_6001
	TRUST_E_NO_SIGNER_CERT 0x8009_6002
	TRUST_E_COUNTER_SIGNER 0x8009_6003
	TRUST_E_CERT_SIGNATURE 0x8009_6004
	TRUST_E_TIME_STAMP 0x8009_6005
	TRUST_E_BAD_DIGEST 0x8009_6010
	TRUST_E_MALFORMED_SIGNATURE 0x8009_6011
	TRUST_E_BASIC_CONSTRAINTS 0x8009_6019
	TRUST_E_FINANCIAL_CRITERIA 0x8009_601e
	MSSIPOTF_E_OUTOFMEMRANGE 0x8009_7001
	MSSIPOTF_E_CANTGETOBJECT 0x8009_7002
	MSSIPOTF_E_NOHEADTABLE 0x8009_7003
	MSSIPOTF_E_BAD_MAGICNUMBER 0x8009_7004
	MSSIPOTF_E_BAD_OFFSET_TABLE 0x8009_7005
	MSSIPOTF_E_TABLE_TAGORDER 0x8009_7006
	MSSIPOTF_E_TABLE_LONGWORD 0x8009_7007
	MSSIPOTF_E_BAD_FIRST_TABLE_PLACEMENT 0x8009_7008
	MSSIPOTF_E_TABLES_OVERLAP 0x8009_7009
	MSSIPOTF_E_TABLE_PADBYTES 0x8009_700a
	MSSIPOTF_E_FILETOOSMALL 0x8009_700b
	MSSIPOTF_E_TABLE_CHECKSUM 0x8009_700c
	MSSIPOTF_E_FILE_CHECKSUM 0x8009_700d
	MSSIPOTF_E_FAILED_POLICY 0x8009_7010
	MSSIPOTF_E_FAILED_HINTS_CHECK 0x8009_7011
	MSSIPOTF_E_NOT_OPENTYPE 0x8009_7012
	MSSIPOTF_E_FILE 0x8009_7013
	MSSIPOTF_E_CRYPT 0x8009_7014
	MSSIPOTF_E_BADVERSION 0x8009_7015
	MSSIPOTF_E_DSIG_STRUCTURE 0x8009_7016
	MSSIPOTF_E_PCONST_CHECK 0x8009_7017
	MSSIPOTF_E_STRUCTURE 0x8009_7018
	ERROR_CRED_REQUIRES_CONFIRMATION 0x8009_7019
	TRUST_E_PROVIDER_UNKNOWN 0x800b_0001
	TRUST_E_ACTION_UNKNOWN 0x800b_0002
	TRUST_E_SUBJECT_FORM_UNKNOWN 0x800b_0003
	TRUST_E_SUBJECT_NOT_TRUSTED 0x800b_0004
	DIGSIG_E_ENCODE 0x800b_0005
	DIGSIG_E_DECODE 0x800b_0006
	DIGSIG_E_EXTENSIBILITY 0x800b_0007
	DIGSIG_E_CRYPTO 0x800b_0008
	PERSIST_E_SIZEDEFINITE 0x800b_0009
	PERSIST_E_SIZEINDEFINITE 0x800b_000a
	PERSIST_E_NOTSELFSIZING 0x800b_000b
	TRUST_E_NOSIGNATURE 0x800b_0100
	CERT_E_EXPIRED 0x800b_0101
	CERT_E_VALIDITYPERIODNESTING 0x800b_0102
	CERT_E_ROLE 0x800b_0103
	CERT_E_PATHLENCONST 0x800b_0104
	CERT_E_CRITICAL 0x800b_0105
	CERT_E_PURPOSE 0x800b_0106
	CERT_E_ISSUERCHAINING 0x800b_0107
	CERT_E_MALFORMED 0x800b_0108
	CERT_E_UNTRUSTEDROOT 0x800b_0109
	CERT_E_CHAINING 0x800b_010a
	TRUST_E_FAIL 0x800b_010b
	CERT_E_REVOKED 0x800b_010c
	CERT_E_UNTRUSTEDTESTROOT 0x800b_010d
	CERT_E_REVOCATION_FAILURE 0x800b_010e
	CERT_E_CN_NO_MATCH 0x800b_010f
	CERT_E_WRONG_USAGE 0x800b_0110
	TRUST_E_EXPLICIT_DISTRUST 0x800b_0111
	CERT_E_UNTRUSTEDCA 0x800b_0112
	CERT_E_INVALID_POLICY 0x800b_0113
	CERT_E_INVALID_NAME 0x800b_0114
	SPAPI_E_EXPECTED_SECTION_NAME 0x800f_0000
	SPAPI_E_BAD_SECTION_NAME_LINE 0x800f_0001
	SPAPI_E_SECTION_NAME_TOO_LONG 0x800f_0002
	SPAPI_E_GENERAL_SYNTAX 0x800f_0003
	SPAPI_E_WRONG_INF_STYLE 0x800f_0100
	SPAPI_E_SECTION_NOT_FOUND 0x800f_0101
	SPAPI_E_LINE_NOT_FOUND 0x800f_0102
	SPAPI_E_NO_BACKUP 0x800f_0103
	SPAPI_E_NO_ASSOCIATED_CLASS 0x800f_0200
	SPAPI_E_CLASS_MISMATCH 0x800f_0201
	SPAPI_E_DUPLICATE_FOUND 0x800f_0202
	SPAPI_E_NO_DRIVER_SELECTED 0x800f_0203
	SPAPI_E_KEY_DOES_NOT_EXIST 0x800f_0204
	SPAPI_E_INVALID_DEVINST_NAME 0x800f_0205
	SPAPI_E_INVALID_CLASS 0x800f_0206
	SPAPI_E_DEVINST_ALREADY_EXISTS 0x800f_0207
	SPAPI_E_DEVINFO_NOT_REGISTERED 0x800f_0208
	SPAPI_E_INVALID_REG_PROPERTY 0x800f_0209
	SPAPI_E_NO_INF 0x800f_020a
	SPAPI_E_NO_SUCH_DEVINST 0x800f_020b
	SPAPI_E_CANT_LOAD_CLASS_ICON 0x800f_020c
	SPAPI_E_INVALID_CLASS_INSTALLER 0x800f_020d
	SPAPI_E_DI_DO_DEFAULT 0x800f_020e
	SPAPI_E_DI_NOFILECOPY 0x800f_020f
	SPAPI_E_INVALID_HWPROFILE 0x800f_0210
	SPAPI_E_NO_DEVICE_SELECTED 0x800f_0211
	SPAPI_E_DEVINFO_LIST_LOCKED 0x800f_0212
	SPAPI_E_DEVINFO_DATA_LOCKED 0x800f_0213
	SPAPI_E_DI_BAD_PATH 0x800f_0214
	SPAPI_E_NO_CLASSINSTALL_PARAMS 0x800f_0215
	SPAPI_E_FILEQUEUE_LOCKED 0x800f_0216
	SPAPI_E_BAD_SERVICE_INSTALLSECT 0x800f_0217
	SPAPI_E_NO_CLASS_DRIVER_LIST 0x800f_0218
	SPAPI_E_NO_ASSOCIATED_SERVICE 0x800f_0219
	SPAPI_E_NO_DEFAULT_DEVICE_INTERFACE 0x800f_021a
	SPAPI_E_DEVICE_INTERFACE_ACTIVE 0x800f_021b
	SPAPI_E_DEVICE_INTERFACE_REMOVED 0x800f_021c
	SPAPI_E_BAD_INTERFACE_INSTALLSECT 0x800f_021d
	SPAPI_E_NO_SUCH_INTERFACE_CLASS 0x800f_021e
	SPAPI_E_INVALID_REFERENCE_STRING 0x800f_021f
	SPAPI_E_INVALID_MACHINENAME 0x800f_0220
	SPAPI_E_REMOTE_COMM_FAILURE 0x800f_0221
	SPAPI_E_MACHINE_UNAVAILABLE 0x800f_0222
	SPAPI_E_NO_CONFIGMGR_SERVICES 0x800f_0223
	SPAPI_E_INVALID_PROPPAGE_PROVIDER 0x800f_0224
	SPAPI_E_NO_SUCH_DEVICE_INTERFACE 0x800f_0225
	SPAPI_E_DI_POSTPROCESSING_REQUIRED 0x800f_0226
	SPAPI_E_INVALID_COINSTALLER 0x800f_0227
	SPAPI_E_NO_COMPAT_DRIVERS 0x800f_0228
	SPAPI_E_NO_DEVICE_ICON 0x800f_0229
	SPAPI_E_INVALID_INF_LOGCONFIG 0x800f_022a
	SPAPI_E_DI_DONT_INSTALL 0x800f_022b
	SPAPI_E_INVALID_FILTER_DRIVER 0x800f_022c
	SPAPI_E_NON_WINDOWS_NT_DRIVER 0x800f_022d
	SPAPI_E_NON_WINDOWS_DRIVER 0x800f_022e
	SPAPI_E_NO_CATALOG_FOR_OEM_INF 0x800f_022f
	SPAPI_E_DEVINSTALL_QUEUE_NONNATIVE 0x800f_0230
	SPAPI_E_NOT_DISABLEABLE 0x800f_0231
	SPAPI_E_CANT_REMOVE_DEVINST 0x800f_0232
	SPAPI_E_INVALID_TARGET 0x800f_0233
	SPAPI_E_DRIVER_NONNATIVE 0x800f_0234
	SPAPI_E_IN_WOW64 0x800f_0235
	SPAPI_E_SET_SYSTEM_RESTORE_POINT 0x800f_0236
	SPAPI_E_INCORRECTLY_COPIED_INF 0x800f_0237
	SPAPI_E_SCE_DISABLED 0x800f_0238
	SPAPI_E_UNKNOWN_EXCEPTION 0x800f_0239
	SPAPI_E_PNP_REGISTRY_ERROR 0x800f_023a
	SPAPI_E_REMOTE_REQUEST_UNSUPPORTED 0x800f_023b
	SPAPI_E_NOT_AN_INSTALLED_OEM_INF 0x800f_023c
	SPAPI_E_INF_IN_USE_BY_DEVICES 0x800f_023d
	SPAPI_E_DI_FUNCTION_OBSOLETE 0x800f_023e
	SPAPI_E_NO_AUTHENTICODE_CATALOG 0x800f_023f
	SPAPI_E_AUTHENTICODE_DISALLOWED 0x800f_0240
	SPAPI_E_AUTHENTICODE_TRUSTED_PUBLISHER 0x800f_0241
	SPAPI_E_AUTHENTICODE_TRUST_NOT_ESTABLISHED 0x800f_0242
	SPAPI_E_AUTHENTICODE_PUBLISHER_NOT_TRUSTED 0x800f_0243
	SPAPI_E_SIGNATURE_OSATTRIBUTE_MISMATCH 0x800f_0244
	SPAPI_E_ONLY_VALIDATE_VIA_AUTHENTICODE 0x800f_0245
	SPAPI_E_DEVICE_INSTALLER_NOT_READY 0x800f_0246
	SPAPI_E_DRIVER_STORE_ADD_FAILED 0x800f_0247
	SPAPI_E_DEVICE_INSTALL_BLOCKED 0x800f_0248
	SPAPI_E_DRIVER_INSTALL_BLOCKED 0x800f_0249
	SPAPI_E_WRONG_INF_TYPE 0x800f_024a
	SPAPI_E_FILE_HASH_NOT_IN_CATALOG 0x800f_024b
	SPAPI_E_DRIVER_STORE_DELETE_FAILED 0x800f_024c
	SPAPI_E_UNRECOVERABLE_STACK_OVERFLOW 0x800f_0300
	SPAPI_E_ERROR_NOT_INSTALLED 0x800f_1000
	SCARD_F_INTERNAL_ERROR 0x8010_0001
	SCARD_E_CANCELLED 0x8010_0002
	SCARD_E_INVALID_HANDLE 0x8010_0003
	SCARD_E_INVALID_PARAMETER 0x8010_0004
	SCARD_E_INVALID_TARGET 0x8010_0005
	SCARD_E_NO_MEMORY 0x8010_0006
	SCARD_F_WAITED_TOO_LONG 0x8010_0007
	SCARD_E_INSUFFICIENT_BUFFER 0x8010_0008
	SCARD_E_UNKNOWN_READER 0x8010_0009
	SCARD_E_TIMEOUT 0x8010_000a
	SCARD_E_SHARING_VIOLATION 0x8010_000b
	SCARD_E_NO_SMARTCARD 0x8010_000c
	SCARD_E_UNKNOWN_CARD 0x8010_000d
	SCARD_E_CANT_DISPOSE 0x8010_000e
	SCARD_E_PROTO_MISMATCH 0x8010_000f
	SCARD_E_NOT_READY 0x8010_0010
	SCARD_E_INVALID_VALUE 0x8010_0011
	SCARD_E_SYSTEM_CANCELLED 0x8010_0012
	SCARD_F_COMM_ERROR 0x8010_0013
	SCARD_F_UNKNOWN_ERROR 0x8010_0014
	SCARD_E_INVALID_ATR 0x8010_0015
	SCARD_E_NOT_TRANSACTED 0x8010_0016
	SCARD_E_READER_UNAVAILABLE 0x8010_0017
	SCARD_P_SHUTDOWN 0x8010_0018
	SCARD_E_PCI_TOO_SMALL 0x8010_0019
	SCARD_E_READER_UNSUPPORTED 0x8010_001a
	SCARD_E_DUPLICATE_READER 0x8010_001b
	SCARD_E_CARD_UNSUPPORTED 0x8010_001c
	SCARD_E_NO_SERVICE 0x8010_001d
	SCARD_E_SERVICE_STOPPED 0x8010_001e
	SCARD_E_UNEXPECTED 0x8010_001f
	SCARD_E_ICC_INSTALLATION 0x8010_0020
	SCARD_E_ICC_CREATEORDER 0x8010_0021
	SCARD_E_UNSUPPORTED_FEATURE 0x8010_0022
	SCARD_E_DIR_NOT_FOUND 0x8010_0023
	SCARD_E_FILE_NOT_FOUND 0x8010_0024
	SCARD_E_NO_DIR 0x8010_0025
	SCARD_E_NO_FILE 0x8010_0026
	SCARD_E_NO_ACCESS 0x8010_0027
	SCARD_E_WRITE_TOO_MANY 0x8010_0028
	SCARD_E_BAD_SEEK 0x8010_0029
	SCARD_E_INVALID_CHV 0x8010_002a
	SCARD_E_UNKNOWN_RES_MNG 0x8010_002b
	SCARD_E_NO_SUCH_CERTIFICATE 0x8010_002c
	SCARD_E_CERTIFICATE_UNAVAILABLE 0x8010_002d
	SCARD_E_NO_READERS_AVAILABLE 0x8010_002e
	SCARD_E_COMM_DATA_LOST 0x8010_002f
	SCARD_E_NO_KEY_CONTAINER 0x8010_0030
	SCARD_E_SERVER_TOO_BUSY 0x8010_0031
	SCARD_E_PIN_CACHE_EXPIRED 0x8010_0032
	SCARD_E_NO_PIN_CACHE 0x8010_0033
	SCARD_E_READ_ONLY_CARD 0x8010_0034
	SCARD_W_UNSUPPORTED_CARD 0x8010_0065
	SCARD_W_UNRESPONSIVE_CARD 0x8010_0066
	SCARD_W_UNPOWERED_CARD 0x8010_0067
	SCARD_W_RESET_CARD 0x8010_0068
	SCARD_W_REMOVED_CARD 0x8010_0069
	SCARD_W_SECURITY_VIOLATION 0x8010_006a
	SCARD_W_WRONG_CHV 0x8010_006b
	SCARD_W_CHV_BLOCKED 0x8010_006c
	SCARD_W_EOF 0x8010_006d
	SCARD_W_CANCELLED_BY_USER 0x8010_006e
	SCARD_W_CARD_NOT_AUTHENTICATED 0x8010_006f
	SCARD_W_CACHE_ITEM_NOT_FOUND 0x8010_0070
	SCARD_W_CACHE_ITEM_STALE 0x8010_0071
	SCARD_W_CACHE_ITEM_TOO_BIG 0x8010_0072
	COMADMIN_E_OBJECTERRORS 0x8011_0401
	COMADMIN_E_OBJECTINVALID 0x8011_0402
	COMADMIN_E_KEYMISSING 0x8011_0403
	COMADMIN_E_ALREADYINSTALLED 0x8011_0404
	COMADMIN_E_APP_FILE_WRITEFAIL 0x8011_0407
	COMADMIN_E_APP_FILE_READFAIL 0x8011_0408
	COMADMIN_E_APP_FILE_VERSION 0x8011_0409
	COMADMIN_E_BADPATH 0x8011_040a
	COMADMIN_E_APPLICATIONEXISTS 0x8011_040b
	COMADMIN_E_ROLEEXISTS 0x8011_040c
	COMADMIN_E_CANTCOPYFILE 0x8011_040d
	COMADMIN_E_NOUSER 0x8011_040f
	COMADMIN_E_INVALIDUSERIDS 0x8011_0410
	COMADMIN_E_NOREGISTRYCLSID 0x8011_0411
	COMADMIN_E_BADREGISTRYPROGID 0x8011_0412
	COMADMIN_E_AUTHENTICATIONLEVEL 0x8011_0413
	COMADMIN_E_USERPASSWDNOTVALID 0x8011_0414
	COMADMIN_E_CLSIDORIIDMISMATCH 0x8011_0418
	COMADMIN_E_REMOTEINTERFACE 0x8011_0419
	COMADMIN_E_DLLREGISTERSERVER 0x8011_041a
	COMADMIN_E_NOSERVERSHARE 0x8011_041b
	COMADMIN_E_DLLLOADFAILED 0x8011_041d
	COMADMIN_E_BADREGISTRYLIBID 0x8011_041e
	COMADMIN_E_APPDIRNOTFOUND 0x8011_041f
	COMADMIN_E_REGISTRARFAILED 0x8011_0423
	COMADMIN_E_COMPFILE_DOESNOTEXIST 0x8011_0424
	COMADMIN_E_COMPFILE_LOADDLLFAIL 0x8011_0425
	COMADMIN_E_COMPFILE_GETCLASSOBJ 0x8011_0426
	COMADMIN_E_COMPFILE_CLASSNOTAVAIL 0x8011_0427
	COMADMIN_E_COMPFILE_BADTLB 0x8011_0428
	COMADMIN_E_COMPFILE_NOTINSTALLABLE 0x8011_0429
	COMADMIN_E_NOTCHANGEABLE 0x8011_042a
	COMADMIN_E_NOTDELETEABLE 0x8011_042b
	COMADMIN_E_SESSION 0x8011_042c
	COMADMIN_E_COMP_MOVE_LOCKED 0x8011_042d
	COMADMIN_E_COMP_MOVE_BAD_DEST 0x8011_042e
	COMADMIN_E_REGISTERTLB 0x8011_0430
	COMADMIN_E_SYSTEMAPP 0x8011_0433
	COMADMIN_E_COMPFILE_NOREGISTRAR 0x8011_0434
	COMADMIN_E_COREQCOMPINSTALLED 0x8011_0435
	COMADMIN_E_SERVICENOTINSTALLED 0x8011_0436
	COMADMIN_E_PROPERTYSAVEFAILED 0x8011_0437
	COMADMIN_E_OBJECTEXISTS 0x8011_0438
	COMADMIN_E_COMPONENTEXISTS 0x8011_0439
	COMADMIN_E_REGFILE_CORRUPT 0x8011_043b
	COMADMIN_E_PROPERTY_OVERFLOW 0x8011_043c
	COMADMIN_E_NOTINREGISTRY 0x8011_043e
	COMADMIN_E_OBJECTNOTPOOLABLE 0x8011_043f
	COMADMIN_E_APPLID_MATCHES_CLSID 0x8011_0446
	COMADMIN_E_ROLE_DOES_NOT_EXIST 0x8011_0447
	COMADMIN_E_START_APP_NEEDS_COMPONENTS 0x8011_0448
	COMADMIN_E_REQUIRES_DIFFERENT_PLATFORM 0x8011_0449
	COMADMIN_E_CAN_NOT_EXPORT_APP_PROXY 0x8011_044a
	COMADMIN_E_CAN_NOT_START_APP 0x8011_044b
	COMADMIN_E_CAN_NOT_EXPORT_SYS_APP 0x8011_044c
	COMADMIN_E_CANT_SUBSCRIBE_TO_COMPONENT 0x8011_044d
	COMADMIN_E_EVENTCLASS_CANT_BE_SUBSCRIBER 0x8011_044e
	COMADMIN_E_LIB_APP_PROXY_INCOMPATIBLE 0x8011_044f
	COMADMIN_E_BASE_PARTITION_ONLY 0x8011_0450
	COMADMIN_E_START_APP_DISABLED 0x8011_0451
	COMADMIN_E_CAT_DUPLICATE_PARTITION_NAME 0x8011_0457
	COMADMIN_E_CAT_INVALID_PARTITION_NAME 0x8011_0458
	COMADMIN_E_CAT_PARTITION_IN_USE 0x8011_0459
	COMADMIN_E_FILE_PARTITION_DUPLICATE_FILES 0x8011_045a
	COMADMIN_E_CAT_IMPORTED_COMPONENTS_NOT_ALLOWED 0x8011_045b
	COMADMIN_E_AMBIGUOUS_APPLICATION_NAME 0x8011_045c
	COMADMIN_E_AMBIGUOUS_PARTITION_NAME 0x8011_045d
	COMADMIN_E_REGDB_NOTINITIALIZED 0x8011_0472
	COMADMIN_E_REGDB_NOTOPEN 0x8011_0473
	COMADMIN_E_REGDB_SYSTEMERR 0x8011_0474
	COMADMIN_E_REGDB_ALREADYRUNNING 0x8011_0475
	COMADMIN_E_MIG_VERSIONNOTSUPPORTED 0x8011_0480
	COMADMIN_E_MIG_SCHEMANOTFOUND 0x8011_0481
	COMADMIN_E_CAT_BITNESSMISMATCH 0x8011_0482
	COMADMIN_E_CAT_UNACCEPTABLEBITNESS 0x8011_0483
	COMADMIN_E_CAT_WRONGAPPBITNESS 0x8011_0484
	COMADMIN_E_CAT_PAUSE_RESUME_NOT_SUPPORTED 0x8011_0485
	COMADMIN_E_CAT_SERVERFAULT 0x8011_0486
	COMQC_E_APPLICATION_NOT_QUEUED 0x8011_0600
	COMQC_E_NO_QUEUEABLE_INTERFACES 0x8011_0601
	COMQC_E_QUEUING_SERVICE_NOT_AVAILABLE 0x8011_0602
	COMQC_E_NO_IPERSISTSTREAM 0x8011_0603
	COMQC_E_BAD_MESSAGE 0x8011_0604
	COMQC_E_UNAUTHENTICATED 0x8011_0605
	COMQC_E_UNTRUSTED_ENQUEUER 0x8011_0606
	MSDTC_E_DUPLICATE_RESOURCE 0x8011_0701
	COMADMIN_E_OBJECT_PARENT_MISSING 0x8011_0808
	COMADMIN_E_OBJECT_DOES_NOT_EXIST 0x8011_0809
	COMADMIN_E_APP_NOT_RUNNING 0x8011_080a
	COMADMIN_E_INVALID_PARTITION 0x8011_080b
	COMADMIN_E_SVCAPP_NOT_POOLABLE_OR_RECYCLABLE 0x8011_080d
	COMADMIN_E_USER_IN_SET 0x8011_080e
	COMADMIN_E_CANTRECYCLELIBRARYAPPS 0x8011_080f
	COMADMIN_E_CANTRECYCLESERVICEAPPS 0x8011_0811
	COMADMIN_E_PROCESSALREADYRECYCLED 0x8011_0812
	COMADMIN_E_PAUSEDPROCESSMAYNOTBERECYCLED 0x8011_0813
	COMADMIN_E_CANTMAKEINPROCSERVICE 0x8011_0814
	COMADMIN_E_PROGIDINUSEBYCLSID 0x8011_0815
	COMADMIN_E_DEFAULT_PARTITION_NOT_IN_SET 0x8011_0816
	COMADMIN_E_RECYCLEDPROCESSMAYNOTBEPAUSED 0x8011_0817
	COMADMIN_E_PARTITION_ACCESSDENIED 0x8011_0818
	COMADMIN_E_PARTITION_MSI_ONLY 0x8011_0819
	COMADMIN_E_LEGACYCOMPS_NOT_ALLOWED_IN_1_0_FORMAT 0x8011_081a
	COMADMIN_E_LEGACYCOMPS_NOT_ALLOWED_IN_NONBASE_PARTITIONS 0x8011_081b
	COMADMIN_E_COMP_MOVE_SOURCE 0x8011_081c
	COMADMIN_E_COMP_MOVE_DEST 0x8011_081d
	COMADMIN_E_COMP_MOVE_PRIVATE 0x8011_081e
	COMADMIN_E_BASEPARTITION_REQUIRED_IN_SET 0x8011_081f
	COMADMIN_E_CANNOT_ALIAS_EVENTCLASS 0x8011_0820
	COMADMIN_E_PRIVATE_ACCESSDENIED 0x8011_0821
	COMADMIN_E_SAFERINVALID 0x8011_0822
	COMADMIN_E_REGISTRY_ACCESSDENIED 0x8011_0823
	COMADMIN_E_PARTITIONS_DISABLED 0x8011_0824
	WER_S_REPORT_DEBUG 0x001b_0000
	WER_S_REPORT_UPLOADED 0x001b_0001
	WER_S_REPORT_QUEUED 0x001b_0002
	WER_S_DISABLED 0x001b_0003
	WER_S_SUSPENDED_UPLOAD 0x001b_0004
	WER_S_DISABLED_QUEUE 0x001b_0005
	WER_S_DISABLED_ARCHIVE 0x001b_0006
	WER_S_REPORT_ASYNC 0x001b_0007
	WER_S_IGNORE_ASSERT_INSTANCE 0x001b_0008
	WER_S_IGNORE_ALL_ASSERTS 0x001b_0009
	WER_S_ASSERT_CONTINUE 0x001b_000a
	WER_S_THROTTLED 0x001b_000b
	WER_S_REPORT_UPLOADED_CAB 0x001b_000c
	WER_E_CRASH_FAILURE 0x801b_8000
	WER_E_CANCELED 0x801b_8001
	WER_E_NETWORK_FAILURE 0x801b_8002
	WER_E_NOT_INITIALIZED 0x801b_8003
	WER_E_ALREADY_REPORTING 0x801b_8004
	WER_E_DUMP_THROTTLED 0x801b_8005
	WER_E_INSUFFICIENT_CONSENT 0x801b_8006
	WER_E_TOO_HEAVY 0x801b_8007
	ERROR_FLT_IO_COMPLETE 0x001f_0001
	ERROR_FLT_NO_HANDLER_DEFINED 0x801f_0001
	ERROR_FLT_CONTEXT_ALREADY_DEFINED 0x801f_0002
	ERROR_FLT_INVALID_ASYNCHRONOUS_REQUEST 0x801f_0003
	ERROR_FLT_DISALLOW_FAST_IO 0x801f_0004
	ERROR_FLT_INVALID_NAME_REQUEST 0x801f_0005
	ERROR_FLT_NOT_SAFE_TO_POST_OPERATION 0x801f_0006
	ERROR_FLT_NOT_INITIALIZED 0x801f_0007
	ERROR_FLT_FILTER_NOT_READY 0x801f_0008
	ERROR_FLT_POST_OPERATION_CLEANUP 0x801f_0009
	ERROR_FLT_INTERNAL_ERROR 0x801f_000a
	ERROR_FLT_DELETING_OBJECT 0x801f_000b
	ERROR_FLT_MUST_BE_NONPAGED_POOL 0x801f_000c
	ERROR_FLT_DUPLICATE_ENTRY 0x801f_000d
	ERROR_FLT_CBDQ_DISABLED 0x801f_000e
	ERROR_FLT_DO_NOT_ATTACH 0x801f_000f
	ERROR_FLT_DO_NOT_DETACH 0x801f_0010
	ERROR_FLT_INSTANCE_ALTITUDE_COLLISION 0x801f_0011
	ERROR_FLT_INSTANCE_NAME_COLLISION 0x801f_0012
	ERROR_FLT_FILTER_NOT_FOUND 0x801f_0013
	ERROR_FLT_VOLUME_NOT_FOUND 0x801f_0014
	ERROR_FLT_INSTANCE_NOT_FOUND 0x801f_0015
	ERROR_FLT_CONTEXT_ALLOCATION_NOT_FOUND 0x801f_0016
	ERROR_FLT_INVALID_CONTEXT_REGISTRATION 0x801f_0017
	ERROR_FLT_NAME_CACHE_MISS 0x801f_0018
	ERROR_FLT_NO_DEVICE_OBJECT 0x801f_0019
	ERROR_FLT_VOLUME_ALREADY_MOUNTED 0x801f_001a
	ERROR_FLT_ALREADY_ENLISTED 0x801f_001b
	ERROR_FLT_CONTEXT_ALREADY_LINKED 0x801f_001c
	ERROR_FLT_NO_WAITER_FOR_REPLY 0x801f_0020
	ERROR_FLT_REGISTRATION_BUSY 0x801f_0023
	ERROR_HUNG_DISPLAY_DRIVER_THREAD 0x8026_0001
	DWM_E_COMPOSITIONDISABLED 0x8026_3001
	DWM_E_REMOTING_NOT_SUPPORTED 0x8026_3002
	DWM_E_NO_REDIRECTION_SURFACE_AVAILABLE 0x8026_3003
	DWM_E_NOT_QUEUING_PRESENTS 0x8026_3004
	DWM_E_ADAPTER_NOT_FOUND 0x8026_3005
	DWM_S_GDI_REDIRECTION_SURFACE 0x0026_3005
	DWM_E_TEXTURE_TOO_LARGE 0x8026_3007
	DWM_S_GDI_REDIRECTION_SURFACE_BLT_VIA_GDI 0x0026_3008
	ERROR_MONITOR_NO_DESCRIPTOR 0x0026_1001
	ERROR_MONITOR_UNKNOWN_DESCRIPTOR_FORMAT 0x0026_1002
	ERROR_MONITOR_INVALID_DESCRIPTOR_CHECKSUM 0xc026_1003
	ERROR_MONITOR_INVALID_STANDARD_TIMING_BLOCK 0xc026_1004
	ERROR_MONITOR_WMI_DATABLOCK_REGISTRATION_FAILED 0xc026_1005
	ERROR_MONITOR_INVALID_SERIAL_NUMBER_MONDSC_BLOCK 0xc026_1006
	ERROR_MONITOR_INVALID_USER_FRIENDLY_MONDSC_BLOCK 0xc026_1007
	ERROR_MONITOR_NO_MORE_DESCRIPTOR_DATA 0xc026_1008
	ERROR_MONITOR_INVALID_DETAILED_TIMING_BLOCK 0xc026_1009
	ERROR_MONITOR_INVALID_MANUFACTURE_DATE 0xc026_100a
	ERROR_GRAPHICS_NOT_EXCLUSIVE_MODE_OWNER 0xc026_2000
	ERROR_GRAPHICS_INSUFFICIENT_DMA_BUFFER 0xc026_2001
	ERROR_GRAPHICS_INVALID_DISPLAY_ADAPTER 0xc026_2002
	ERROR_GRAPHICS_ADAPTER_WAS_RESET 0xc026_2003
	ERROR_GRAPHICS_INVALID_DRIVER_MODEL 0xc026_2004
	ERROR_GRAPHICS_PRESENT_MODE_CHANGED 0xc026_2005
	ERROR_GRAPHICS_PRESENT_OCCLUDED 0xc026_2006
	ERROR_GRAPHICS_PRESENT_DENIED 0xc026_2007
	ERROR_GRAPHICS_CANNOTCOLORCONVERT 0xc026_2008
	ERROR_GRAPHICS_DRIVER_MISMATCH 0xc026_2009
	ERROR_GRAPHICS_PARTIAL_DATA_POPULATED 0x4026_200a
	ERROR_GRAPHICS_PRESENT_REDIRECTION_DISABLED 0xc026_200b
	ERROR_GRAPHICS_PRESENT_UNOCCLUDED 0xc026_200c
	ERROR_GRAPHICS_WINDOWDC_NOT_AVAILABLE 0xc026_200d
	ERROR_GRAPHICS_WINDOWLESS_PRESENT_DISABLED 0xc026_200e
	ERROR_GRAPHICS_PRESENT_INVALID_WINDOW 0xc026_200f
	ERROR_GRAPHICS_PRESENT_BUFFER_NOT_BOUND 0xc026_2010
	ERROR_GRAPHICS_VAIL_STATE_CHANGED 0xc026_2011
	ERROR_GRAPHICS_INDIRECT_DISPLAY_ABANDON_SWAPCHAIN 0xc026_2012
	ERROR_GRAPHICS_INDIRECT_DISPLAY_DEVICE_STOPPED 0xc026_2013
	ERROR_GRAPHICS_NO_VIDEO_MEMORY 0xc026_2100
	ERROR_GRAPHICS_CANT_LOCK_MEMORY 0xc026_2101
	ERROR_GRAPHICS_ALLOCATION_BUSY 0xc026_2102
	ERROR_GRAPHICS_TOO_MANY_REFERENCES 0xc026_2103
	ERROR_GRAPHICS_TRY_AGAIN_LATER 0xc026_2104
	ERROR_GRAPHICS_TRY_AGAIN_NOW 0xc026_2105
	ERROR_GRAPHICS_ALLOCATION_INVALID 0xc026_2106
	ERROR_GRAPHICS_UNSWIZZLING_APERTURE_UNAVAILABLE 0xc026_2107
	ERROR_GRAPHICS_UNSWIZZLING_APERTURE_UNSUPPORTED 0xc026_2108
	ERROR_GRAPHICS_CANT_EVICT_PINNED_ALLOCATION 0xc026_2109
	ERROR_GRAPHICS_INVALID_ALLOCATION_USAGE 0xc026_2110
	ERROR_GRAPHICS_CANT_RENDER_LOCKED_ALLOCATION 0xc026_2111
	ERROR_GRAPHICS_ALLOCATION_CLOSED 0xc026_2112
	ERROR_GRAPHICS_INVALID_ALLOCATION_INSTANCE 0xc026_2113
	ERROR_GRAPHICS_INVALID_ALLOCATION_HANDLE 0xc026_2114
	ERROR_GRAPHICS_WRONG_ALLOCATION_DEVICE 0xc026_2115
	ERROR_GRAPHICS_ALLOCATION_CONTENT_LOST 0xc026_2116
	ERROR_GRAPHICS_GPU_EXCEPTION_ON_DEVICE 0xc026_2200
	ERROR_GRAPHICS_SKIP_ALLOCATION_PREPARATION 0x4026_2201
	ERROR_GRAPHICS_INVALID_VIDPN_TOPOLOGY 0xc026_2300
	ERROR_GRAPHICS_VIDPN_TOPOLOGY_NOT_SUPPORTED 0xc026_2301
	ERROR_GRAPHICS_VIDPN_TOPOLOGY_CURRENTLY_NOT_SUPPORTED 0xc026_2302
	ERROR_GRAPHICS_INVALID_VIDPN 0xc026_2303
	ERROR_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE 0xc026_2304
	ERROR_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET 0xc026_2305
	ERROR_GRAPHICS_VIDPN_MODALITY_NOT_SUPPORTED 0xc026_2306
	ERROR_GRAPHICS_MODE_NOT_PINNED 0x0026_2307
	ERROR_GRAPHICS_INVALID_VIDPN_SOURCEMODESET 0xc026_2308
	ERROR_GRAPHICS_INVALID_VIDPN_TARGETMODESET 0xc026_2309
	ERROR_GRAPHICS_INVALID_FREQUENCY 0xc026_230a
	ERROR_GRAPHICS_INVALID_ACTIVE_REGION 0xc026_230b
	ERROR_GRAPHICS_INVALID_TOTAL_REGION 0xc026_230c
	ERROR_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE_MODE 0xc026_2310
	ERROR_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET_MODE 0xc026_2311
	ERROR_GRAPHICS_PINNED_MODE_MUST_REMAIN_IN_SET 0xc026_2312
	ERROR_GRAPHICS_PATH_ALREADY_IN_TOPOLOGY 0xc026_2313
	ERROR_GRAPHICS_MODE_ALREADY_IN_MODESET 0xc026_2314
	ERROR_GRAPHICS_INVALID_VIDEOPRESENTSOURCESET 0xc026_2315
	ERROR_GRAPHICS_INVALID_VIDEOPRESENTTARGETSET 0xc026_2316
	ERROR_GRAPHICS_SOURCE_ALREADY_IN_SET 0xc026_2317
	ERROR_GRAPHICS_TARGET_ALREADY_IN_SET 0xc026_2318
	ERROR_GRAPHICS_INVALID_VIDPN_PRESENT_PATH 0xc026_2319
	ERROR_GRAPHICS_NO_RECOMMENDED_VIDPN_TOPOLOGY 0xc026_231a
	ERROR_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGESET 0xc026_231b
	ERROR_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE 0xc026_231c
	ERROR_GRAPHICS_FREQUENCYRANGE_NOT_IN_SET 0xc026_231d
	ERROR_GRAPHICS_NO_PREFERRED_MODE 0x0026_231e
	ERROR_GRAPHICS_FREQUENCYRANGE_ALREADY_IN_SET 0xc026_231f
	ERROR_GRAPHICS_STALE_MODESET 0xc026_2320
	ERROR_GRAPHICS_INVALID_MONITOR_SOURCEMODESET 0xc026_2321
	ERROR_GRAPHICS_INVALID_MONITOR_SOURCE_MODE 0xc026_2322
	ERROR_GRAPHICS_NO_RECOMMENDED_FUNCTIONAL_VIDPN 0xc026_2323
	ERROR_GRAPHICS_MODE_ID_MUST_BE_UNIQUE 0xc026_2324
	ERROR_GRAPHICS_EMPTY_ADAPTER_MONITOR_MODE_SUPPORT_INTERSECTION 0xc026_2325
	ERROR_GRAPHICS_VIDEO_PRESENT_TARGETS_LESS_THAN_SOURCES 0xc026_2326
	ERROR_GRAPHICS_PATH_NOT_IN_TOPOLOGY 0xc026_2327
	ERROR_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_SOURCE 0xc026_2328
	ERROR_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_TARGET 0xc026_2329
	ERROR_GRAPHICS_INVALID_MONITORDESCRIPTORSET 0xc026_232a
	ERROR_GRAPHICS_INVALID_MONITORDESCRIPTOR 0xc026_232b
	ERROR_GRAPHICS_MONITORDESCRIPTOR_NOT_IN_SET 0xc026_232c
	ERROR_GRAPHICS_MONITORDESCRIPTOR_ALREADY_IN_SET 0xc026_232d
	ERROR_GRAPHICS_MONITORDESCRIPTOR_ID_MUST_BE_UNIQUE 0xc026_232e
	ERROR_GRAPHICS_INVALID_VIDPN_TARGET_SUBSET_TYPE 0xc026_232f
	ERROR_GRAPHICS_RESOURCES_NOT_RELATED 0xc026_2330
	ERROR_GRAPHICS_SOURCE_ID_MUST_BE_UNIQUE 0xc026_2331
	ERROR_GRAPHICS_TARGET_ID_MUST_BE_UNIQUE 0xc026_2332
	ERROR_GRAPHICS_NO_AVAILABLE_VIDPN_TARGET 0xc026_2333
	ERROR_GRAPHICS_MONITOR_COULD_NOT_BE_ASSOCIATED_WITH_ADAPTER 0xc026_2334
	ERROR_GRAPHICS_NO_VIDPNMGR 0xc026_2335
	ERROR_GRAPHICS_NO_ACTIVE_VIDPN 0xc026_2336
	ERROR_GRAPHICS_STALE_VIDPN_TOPOLOGY 0xc026_2337
	ERROR_GRAPHICS_MONITOR_NOT_CONNECTED 0xc026_2338
	ERROR_GRAPHICS_SOURCE_NOT_IN_TOPOLOGY 0xc026_2339
	ERROR_GRAPHICS_INVALID_PRIMARYSURFACE_SIZE 0xc026_233a
	ERROR_GRAPHICS_INVALID_VISIBLEREGION_SIZE 0xc026_233b
	ERROR_GRAPHICS_INVALID_STRIDE 0xc026_233c
	ERROR_GRAPHICS_INVALID_PIXELFORMAT 0xc026_233d
	ERROR_GRAPHICS_INVALID_COLORBASIS 0xc026_233e
	ERROR_GRAPHICS_INVALID_PIXELVALUEACCESSMODE 0xc026_233f
	ERROR_GRAPHICS_TARGET_NOT_IN_TOPOLOGY 0xc026_2340
	ERROR_GRAPHICS_NO_DISPLAY_MODE_MANAGEMENT_SUPPORT 0xc026_2341
	ERROR_GRAPHICS_VIDPN_SOURCE_IN_USE 0xc026_2342
	ERROR_GRAPHICS_CANT_ACCESS_ACTIVE_VIDPN 0xc026_2343
	ERROR_GRAPHICS_INVALID_PATH_IMPORTANCE_ORDINAL 0xc026_2344
	ERROR_GRAPHICS_INVALID_PATH_CONTENT_GEOMETRY_TRANSFORMATION 0xc026_2345
	ERROR_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_SUPPORTED 0xc026_2346
	ERROR_GRAPHICS_INVALID_GAMMA_RAMP 0xc026_2347
	ERROR_GRAPHICS_GAMMA_RAMP_NOT_SUPPORTED 0xc026_2348
	ERROR_GRAPHICS_MULTISAMPLING_NOT_SUPPORTED 0xc026_2349
	ERROR_GRAPHICS_MODE_NOT_IN_MODESET 0xc026_234a
	ERROR_GRAPHICS_DATASET_IS_EMPTY 0x0026_234b
	ERROR_GRAPHICS_NO_MORE_ELEMENTS_IN_DATASET 0x0026_234c
	ERROR_GRAPHICS_INVALID_VIDPN_TOPOLOGY_RECOMMENDATION_REASON 0xc026_234d
	ERROR_GRAPHICS_INVALID_PATH_CONTENT_TYPE 0xc026_234e
	ERROR_GRAPHICS_INVALID_COPYPROTECTION_TYPE 0xc026_234f
	ERROR_GRAPHICS_UNASSIGNED_MODESET_ALREADY_EXISTS 0xc026_2350
	ERROR_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_PINNED 0x0026_2351
	ERROR_GRAPHICS_INVALID_SCANLINE_ORDERING 0xc026_2352
	ERROR_GRAPHICS_TOPOLOGY_CHANGES_NOT_ALLOWED 0xc026_2353
	ERROR_GRAPHICS_NO_AVAILABLE_IMPORTANCE_ORDINALS 0xc026_2354
	ERROR_GRAPHICS_INCOMPATIBLE_PRIVATE_FORMAT 0xc026_2355
	ERROR_GRAPHICS_INVALID_MODE_PRUNING_ALGORITHM 0xc026_2356
	ERROR_GRAPHICS_INVALID_MONITOR_CAPABILITY_ORIGIN 0xc026_2357
	ERROR_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE_CONSTRAINT 0xc026_2358
	ERROR_GRAPHICS_MAX_NUM_PATHS_REACHED 0xc026_2359
	ERROR_GRAPHICS_CANCEL_VIDPN_TOPOLOGY_AUGMENTATION 0xc026_235a
	ERROR_GRAPHICS_INVALID_CLIENT_TYPE 0xc026_235b
	ERROR_GRAPHICS_CLIENTVIDPN_NOT_SET 0xc026_235c
	ERROR_GRAPHICS_SPECIFIED_CHILD_ALREADY_CONNECTED 0xc026_2400
	ERROR_GRAPHICS_CHILD_DESCRIPTOR_NOT_SUPPORTED 0xc026_2401
	ERROR_GRAPHICS_UNKNOWN_CHILD_STATUS 0x4026_242f
	ERROR_GRAPHICS_NOT_A_LINKED_ADAPTER 0xc026_2430
	ERROR_GRAPHICS_LEADLINK_NOT_ENUMERATED 0xc026_2431
	ERROR_GRAPHICS_CHAINLINKS_NOT_ENUMERATED 0xc026_2432
	ERROR_GRAPHICS_ADAPTER_CHAIN_NOT_READY 0xc026_2433
	ERROR_GRAPHICS_CHAINLINKS_NOT_STARTED 0xc026_2434
	ERROR_GRAPHICS_CHAINLINKS_NOT_POWERED_ON 0xc026_2435
	ERROR_GRAPHICS_INCONSISTENT_DEVICE_LINK_STATE 0xc026_2436
	ERROR_GRAPHICS_LEADLINK_START_DEFERRED 0x4026_2437
	ERROR_GRAPHICS_NOT_POST_DEVICE_DRIVER 0xc026_2438
	ERROR_GRAPHICS_POLLING_TOO_FREQUENTLY 0x4026_2439
	ERROR_GRAPHICS_START_DEFERRED 0x4026_243a
	ERROR_GRAPHICS_ADAPTER_ACCESS_NOT_EXCLUDED 0xc026_243b
	ERROR_GRAPHICS_DEPENDABLE_CHILD_STATUS 0x4026_243c
	ERROR_GRAPHICS_OPM_NOT_SUPPORTED 0xc026_2500
	ERROR_GRAPHICS_COPP_NOT_SUPPORTED 0xc026_2501
	ERROR_GRAPHICS_UAB_NOT_SUPPORTED 0xc026_2502
	ERROR_GRAPHICS_OPM_INVALID_ENCRYPTED_PARAMETERS 0xc026_2503
	ERROR_GRAPHICS_OPM_NO_VIDEO_OUTPUTS_EXIST 0xc026_2505
	ERROR_GRAPHICS_OPM_INTERNAL_ERROR 0xc026_250b
	ERROR_GRAPHICS_OPM_INVALID_HANDLE 0xc026_250c
	ERROR_GRAPHICS_PVP_INVALID_CERTIFICATE_LENGTH 0xc026_250e
	ERROR_GRAPHICS_OPM_SPANNING_MODE_ENABLED 0xc026_250f
	ERROR_GRAPHICS_OPM_THEATER_MODE_ENABLED 0xc026_2510
	ERROR_GRAPHICS_PVP_HFS_FAILED 0xc026_2511
	ERROR_GRAPHICS_OPM_INVALID_SRM 0xc026_2512
	ERROR_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_HDCP 0xc026_2513
	ERROR_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_ACP 0xc026_2514
	ERROR_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_CGMSA 0xc026_2515
	ERROR_GRAPHICS_OPM_HDCP_SRM_NEVER_SET 0xc026_2516
	ERROR_GRAPHICS_OPM_RESOLUTION_TOO_HIGH 0xc026_2517
	ERROR_GRAPHICS_OPM_ALL_HDCP_HARDWARE_ALREADY_IN_USE 0xc026_2518
	ERROR_GRAPHICS_OPM_VIDEO_OUTPUT_NO_LONGER_EXISTS 0xc026_251a
	ERROR_GRAPHICS_OPM_SESSION_TYPE_CHANGE_IN_PROGRESS 0xc026_251b
	ERROR_GRAPHICS_OPM_VIDEO_OUTPUT_DOES_NOT_HAVE_COPP_SEMANTICS 0xc026_251c
	ERROR_GRAPHICS_OPM_INVALID_INFORMATION_REQUEST 0xc026_251d
	ERROR_GRAPHICS_OPM_DRIVER_INTERNAL_ERROR 0xc026_251e
	ERROR_GRAPHICS_OPM_VIDEO_OUTPUT_DOES_NOT_HAVE_OPM_SEMANTICS 0xc026_251f
	ERROR_GRAPHICS_OPM_SIGNALING_NOT_SUPPORTED 0xc026_2520
	ERROR_GRAPHICS_OPM_INVALID_CONFIGURATION_REQUEST 0xc026_2521
	ERROR_GRAPHICS_I2C_NOT_SUPPORTED 0xc026_2580
	ERROR_GRAPHICS_I2C_DEVICE_DOES_NOT_EXIST 0xc026_2581
	ERROR_GRAPHICS_I2C_ERROR_TRANSMITTING_DATA 0xc026_2582
	ERROR_GRAPHICS_I2C_ERROR_RECEIVING_DATA 0xc026_2583
	ERROR_GRAPHICS_DDCCI_VCP_NOT_SUPPORTED 0xc026_2584
	ERROR_GRAPHICS_DDCCI_INVALID_DATA 0xc026_2585
	ERROR_GRAPHICS_DDCCI_MONITOR_RETURNED_INVALID_TIMING_STATUS_BYTE 0xc026_2586
	ERROR_GRAPHICS_MCA_INVALID_CAPABILITIES_STRING 0xc026_2587
	ERROR_GRAPHICS_MCA_INTERNAL_ERROR 0xc026_2588
	ERROR_GRAPHICS_DDCCI_INVALID_MESSAGE_COMMAND 0xc026_2589
	ERROR_GRAPHICS_DDCCI_INVALID_MESSAGE_LENGTH 0xc026_258a
	ERROR_GRAPHICS_DDCCI_INVALID_MESSAGE_CHECKSUM 0xc026_258b
	ERROR_GRAPHICS_INVALID_PHYSICAL_MONITOR_HANDLE 0xc026_258c
	ERROR_GRAPHICS_MONITOR_NO_LONGER_EXISTS 0xc026_258d
	ERROR_GRAPHICS_DDCCI_CURRENT_CURRENT_VALUE_GREATER_THAN_MAXIMUM_VALUE 0xc026_25d8
	ERROR_GRAPHICS_MCA_INVALID_VCP_VERSION 0xc026_25d9
	ERROR_GRAPHICS_MCA_MONITOR_VIOLATES_MCCS_SPECIFICATION 0xc026_25da
	ERROR_GRAPHICS_MCA_MCCS_VERSION_MISMATCH 0xc026_25db
	ERROR_GRAPHICS_MCA_UNSUPPORTED_MCCS_VERSION 0xc026_25dc
	ERROR_GRAPHICS_MCA_INVALID_TECHNOLOGY_TYPE_RETURNED 0xc026_25de
	ERROR_GRAPHICS_MCA_UNSUPPORTED_COLOR_TEMPERATURE 0xc026_25df
	ERROR_GRAPHICS_ONLY_CONSOLE_SESSION_SUPPORTED 0xc026_25e0
	ERROR_GRAPHICS_NO_DISPLAY_DEVICE_CORRESPONDS_TO_NAME 0xc026_25e1
	ERROR_GRAPHICS_DISPLAY_DEVICE_NOT_ATTACHED_TO_DESKTOP 0xc026_25e2
	ERROR_GRAPHICS_MIRRORING_DEVICES_NOT_SUPPORTED 0xc026_25e3
	ERROR_GRAPHICS_INVALID_POINTER 0xc026_25e4
	ERROR_GRAPHICS_NO_MONITORS_CORRESPOND_TO_DISPLAY_DEVICE 0xc026_25e5
	ERROR_GRAPHICS_PARAMETER_ARRAY_TOO_SMALL 0xc026_25e6
	ERROR_GRAPHICS_INTERNAL_ERROR 0xc026_25e7
	ERROR_GRAPHICS_SESSION_TYPE_CHANGE_IN_PROGRESS 0xc026_05e8
	NAP_E_INVALID_PACKET 0x8027_0001
	NAP_E_MISSING_SOH 0x8027_0002
	NAP_E_CONFLICTING_ID 0x8027_0003
	NAP_E_NO_CACHED_SOH 0x8027_0004
	NAP_E_STILL_BOUND 0x8027_0005
	NAP_E_NOT_REGISTERED 0x8027_0006
	NAP_E_NOT_INITIALIZED 0x8027_0007
	NAP_E_MISMATCHED_ID 0x8027_0008
	NAP_E_NOT_PENDING 0x8027_0009
	NAP_E_ID_NOT_FOUND 0x8027_000a
	NAP_E_MAXSIZE_TOO_SMALL 0x8027_000b
	NAP_E_SERVICE_NOT_RUNNING 0x8027_000c
	NAP_S_CERT_ALREADY_PRESENT 0x0027_000d
	NAP_E_ENTITY_DISABLED 0x8027_000e
	NAP_E_NETSH_GROUPPOLICY_ERROR 0x8027_000f
	NAP_E_TOO_MANY_CALLS 0x8027_0010
	NAP_E_SHV_CONFIG_EXISTED 0x8027_0011
	NAP_E_SHV_CONFIG_NOT_FOUND 0x8027_0012
	NAP_E_SHV_TIMEOUT 0x8027_0013
	TPM_E_ERROR_MASK 0x8028_0000
	TPM_E_AUTHFAIL 0x8028_0001
	TPM_E_BADINDEX 0x8028_0002
	TPM_E_BAD_PARAMETER 0x8028_0003
	TPM_E_AUDITFAILURE 0x8028_0004
	TPM_E_CLEAR_DISABLED 0x8028_0005
	TPM_E_DEACTIVATED 0x8028_0006
	TPM_E_DISABLED 0x8028_0007
	TPM_E_DISABLED_CMD 0x8028_0008
	TPM_E_FAIL 0x8028_0009
	TPM_E_BAD_ORDINAL 0x8028_000a
	TPM_E_INSTALL_DISABLED 0x8028_000b
	TPM_E_INVALID_KEYHANDLE 0x8028_000c
	TPM_E_KEYNOTFOUND 0x8028_000d
	TPM_E_INAPPROPRIATE_ENC 0x8028_000e
	TPM_E_MIGRATEFAIL 0x8028_000f
	TPM_E_INVALID_PCR_INFO 0x8028_0010
	TPM_E_NOSPACE 0x8028_0011
	TPM_E_NOSRK 0x8028_0012
	TPM_E_NOTSEALED_BLOB 0x8028_0013
	TPM_E_OWNER_SET 0x8028_0014
	TPM_E_RESOURCES 0x8028_0015
	TPM_E_SHORTRANDOM 0x8028_0016
	TPM_E_SIZE 0x8028_0017
	TPM_E_WRONGPCRVAL 0x8028_0018
	TPM_E_BAD_PARAM_SIZE 0x8028_0019
	TPM_E_SHA_THREAD 0x8028_001a
	TPM_E_SHA_ERROR 0x8028_001b
	TPM_E_FAILEDSELFTEST 0x8028_001c
	TPM_E_AUTH2FAIL 0x8028_001d
	TPM_E_BADTAG 0x8028_001e
	TPM_E_IOERROR 0x8028_001f
	TPM_E_ENCRYPT_ERROR 0x8028_0020
	TPM_E_DECRYPT_ERROR 0x8028_0021
	TPM_E_INVALID_AUTHHANDLE 0x8028_0022
	TPM_E_NO_ENDORSEMENT 0x8028_0023
	TPM_E_INVALID_KEYUSAGE 0x8028_0024
	TPM_E_WRONG_ENTITYTYPE 0x8028_0025
	TPM_E_INVALID_POSTINIT 0x8028_0026
	TPM_E_INAPPROPRIATE_SIG 0x8028_0027
	TPM_E_BAD_KEY_PROPERTY 0x8028_0028
	TPM_E_BAD_MIGRATION 0x8028_0029
	TPM_E_BAD_SCHEME 0x8028_002a
	TPM_E_BAD_DATASIZE 0x8028_002b
	TPM_E_BAD_MODE 0x8028_002c
	TPM_E_BAD_PRESENCE 0x8028_002d
	TPM_E_BAD_VERSION 0x8028_002e
	TPM_E_NO_WRAP_TRANSPORT 0x8028_002f
	TPM_E_AUDITFAIL_UNSUCCESSFUL 0x8028_0030
	TPM_E_AUDITFAIL_SUCCESSFUL 0x8028_0031
	TPM_E_NOTRESETABLE 0x8028_0032
	TPM_E_NOTLOCAL 0x8028_0033
	TPM_E_BAD_TYPE 0x8028_0034
	TPM_E_INVALID_RESOURCE 0x8028_0035
	TPM_E_NOTFIPS 0x8028_0036
	TPM_E_INVALID_FAMILY 0x8028_0037
	TPM_E_NO_NV_PERMISSION 0x8028_0038
	TPM_E_REQUIRES_SIGN 0x8028_0039
	TPM_E_KEY_NOTSUPPORTED 0x8028_003a
	TPM_E_AUTH_CONFLICT 0x8028_003b
	TPM_E_AREA_LOCKED 0x8028_003c
	TPM_E_BAD_LOCALITY 0x8028_003d
	TPM_E_READ_ONLY 0x8028_003e
	TPM_E_PER_NOWRITE 0x8028_003f
	TPM_E_FAMILYCOUNT 0x8028_0040
	TPM_E_WRITE_LOCKED 0x8028_0041
	TPM_E_BAD_ATTRIBUTES 0x8028_0042
	TPM_E_INVALID_STRUCTURE 0x8028_0043
	TPM_E_KEY_OWNER_CONTROL 0x8028_0044
	TPM_E_BAD_COUNTER 0x8028_0045
	TPM_E_NOT_FULLWRITE 0x8028_0046
	TPM_E_CONTEXT_GAP 0x8028_0047
	TPM_E_MAXNVWRITES 0x8028_0048
	TPM_E_NOOPERATOR 0x8028_0049
	TPM_E_RESOURCEMISSING 0x8028_004a
	TPM_E_DELEGATE_LOCK 0x8028_004b
	TPM_E_DELEGATE_FAMILY 0x8028_004c
	TPM_E_DELEGATE_ADMIN 0x8028_004d
	TPM_E_TRANSPORT_NOTEXCLUSIVE 0x8028_004e
	TPM_E_OWNER_CONTROL 0x8028_004f
	TPM_E_DAA_RESOURCES 0x8028_0050
	TPM_E_DAA_INPUT_DATA0 0x8028_0051
	TPM_E_DAA_INPUT_DATA1 0x8028_0052
	TPM_E_DAA_ISSUER_SETTINGS 0x8028_0053
	TPM_E_DAA_TPM_SETTINGS 0x8028_0054
	TPM_E_DAA_STAGE 0x8028_0055
	TPM_E_DAA_ISSUER_VALIDITY 0x8028_0056
	TPM_E_DAA_WRONG_W 0x8028_0057
	TPM_E_BAD_HANDLE 0x8028_0058
	TPM_E_BAD_DELEGATE 0x8028_0059
	TPM_E_BADCONTEXT 0x8028_005a
	TPM_E_TOOMANYCONTEXTS 0x8028_005b
	TPM_E_MA_TICKET_SIGNATURE 0x8028_005c
	TPM_E_MA_DESTINATION 0x8028_005d
	TPM_E_MA_SOURCE 0x8028_005e
	TPM_E_MA_AUTHORITY 0x8028_005f
	TPM_E_PERMANENTEK 0x8028_0061
	TPM_E_BAD_SIGNATURE 0x8028_0062
	TPM_E_NOCONTEXTSPACE 0x8028_0063
	TPM_20_E_ASYMMETRIC 0x8028_0081
	TPM_20_E_ATTRIBUTES 0x8028_0082
	TPM_20_E_HASH 0x8028_0083
	TPM_20_E_VALUE 0x8028_0084
	TPM_20_E_HIERARCHY 0x8028_0085
	TPM_20_E_KEY_SIZE 0x8028_0087
	TPM_20_E_MGF 0x8028_0088
	TPM_20_E_MODE 0x8028_0089
	TPM_20_E_TYPE 0x8028_008a
	TPM_20_E_HANDLE 0x8028_008b
	TPM_20_E_KDF 0x8028_008c
	TPM_20_E_RANGE 0x8028_008d
	TPM_20_E_AUTH_FAIL 0x8028_008e
	TPM_20_E_NONCE 0x8028_008f
	TPM_20_E_PP 0x8028_0090
	TPM_20_E_SCHEME 0x8028_0092
	TPM_20_E_SIZE 0x8028_0095
	TPM_20_E_SYMMETRIC 0x8028_0096
	TPM_20_E_TAG 0x8028_0097
	TPM_20_E_SELECTOR 0x8028_0098
	TPM_20_E_INSUFFICIENT 0x8028_009a
	TPM_20_E_SIGNATURE 0x8028_009b
	TPM_20_E_KEY 0x8028_009c
	TPM_20_E_POLICY_FAIL 0x8028_009d
	TPM_20_E_INTEGRITY 0x8028_009f
	TPM_20_E_TICKET 0x8028_00a0
	TPM_20_E_RESERVED_BITS 0x8028_00a1
	TPM_20_E_BAD_AUTH 0x8028_00a2
	TPM_20_E_EXPIRED 0x8028_00a3
	TPM_20_E_POLICY_CC 0x8028_00a4
	TPM_20_E_BINDING 0x8028_00a5
	TPM_20_E_CURVE 0x8028_00a6
	TPM_20_E_ECC_POINT 0x8028_00a7
	TPM_20_E_INITIALIZE 0x8028_0100
	TPM_20_E_FAILURE 0x8028_0101
	TPM_20_E_SEQUENCE 0x8028_0103
	TPM_20_E_PRIVATE 0x8028_010b
	TPM_20_E_HMAC 0x8028_0119
	TPM_20_E_DISABLED 0x8028_0120
	TPM_20_E_EXCLUSIVE 0x8028_0121
	TPM_20_E_ECC_CURVE 0x8028_0123
	TPM_20_E_AUTH_TYPE 0x8028_0124
	TPM_20_E_AUTH_MISSING 0x8028_0125
	TPM_20_E_POLICY 0x8028_0126
	TPM_20_E_PCR 0x8028_0127
	TPM_20_E_PCR_CHANGED 0x8028_0128
	TPM_20_E_UPGRADE 0x8028_012d
	TPM_20_E_TOO_MANY_CONTEXTS 0x8028_012e
	TPM_20_E_AUTH_UNAVAILABLE 0x8028_012f
	TPM_20_E_REBOOT 0x8028_0130
	TPM_20_E_UNBALANCED 0x8028_0131
	TPM_20_E_COMMAND_SIZE 0x8028_0142
	TPM_20_E_COMMAND_CODE 0x8028_0143
	TPM_20_E_AUTHSIZE 0x8028_0144
	TPM_20_E_AUTH_CONTEXT 0x8028_0145
	TPM_20_E_NV_RANGE 0x8028_0146
	TPM_20_E_NV_SIZE 0x8028_0147
	TPM_20_E_NV_LOCKED 0x8028_0148
	TPM_20_E_NV_AUTHORIZATION 0x8028_0149
	TPM_20_E_NV_UNINITIALIZED 0x8028_014a
	TPM_20_E_NV_SPACE 0x8028_014b
	TPM_20_E_NV_DEFINED 0x8028_014c
	TPM_20_E_BAD_CONTEXT 0x8028_0150
	TPM_20_E_CPHASH 0x8028_0151
	TPM_20_E_PARENT 0x8028_0152
	TPM_20_E_NEEDS_TEST 0x8028_0153
	TPM_20_E_NO_RESULT 0x8028_0154
	TPM_20_E_SENSITIVE 0x8028_0155
	TPM_E_COMMAND_BLOCKED 0x8028_0400
	TPM_E_INVALID_HANDLE 0x8028_0401
	TPM_E_DUPLICATE_VHANDLE 0x8028_0402
	TPM_E_EMBEDDED_COMMAND_BLOCKED 0x8028_0403
	TPM_E_EMBEDDED_COMMAND_UNSUPPORTED 0x8028_0404
	TPM_E_RETRY 0x8028_0800
	TPM_E_NEEDS_SELFTEST 0x8028_0801
	TPM_E_DOING_SELFTEST 0x8028_0802
	TPM_E_DEFEND_LOCK_RUNNING 0x8028_0803
	TPM_20_E_CONTEXT_GAP 0x8028_0901
	TPM_20_E_OBJECT_MEMORY 0x8028_0902
	TPM_20_E_SESSION_MEMORY 0x8028_0903
	TPM_20_E_MEMORY 0x8028_0904
	TPM_20_E_SESSION_HANDLES 0x8028_0905
	TPM_20_E_OBJECT_HANDLES 0x8028_0906
	TPM_20_E_LOCALITY 0x8028_0907
	TPM_20_E_YIELDED 0x8028_0908
	TPM_20_E_CANCELED 0x8028_0909
	TPM_20_E_TESTING 0x8028_090a
	TPM_20_E_NV_RATE 0x8028_0920
	TPM_20_E_LOCKOUT 0x8028_0921
	TPM_20_E_RETRY 0x8028_0922
	TPM_20_E_NV_UNAVAILABLE 0x8028_0923
	TBS_E_INTERNAL_ERROR 0x8028_4001
	TBS_E_BAD_PARAMETER 0x8028_4002
	TBS_E_INVALID_OUTPUT_POINTER 0x8028_4003
	TBS_E_INVALID_CONTEXT 0x8028_4004
	TBS_E_INSUFFICIENT_BUFFER 0x8028_4005
	TBS_E_IOERROR 0x8028_4006
	TBS_E_INVALID_CONTEXT_PARAM 0x8028_4007
	TBS_E_SERVICE_NOT_RUNNING 0x8028_4008
	TBS_E_TOO_MANY_TBS_CONTEXTS 0x8028_4009
	TBS_E_TOO_MANY_RESOURCES 0x8028_400a
	TBS_E_SERVICE_START_PENDING 0x8028_400b
	TBS_E_PPI_NOT_SUPPORTED 0x8028_400c
	TBS_E_COMMAND_CANCELED 0x8028_400d
	TBS_E_BUFFER_TOO_LARGE 0x8028_400e
	TBS_E_TPM_NOT_FOUND 0x8028_400f
	TBS_E_SERVICE_DISABLED 0x8028_4010
	TBS_E_NO_EVENT_LOG 0x8028_4011
	TBS_E_ACCESS_DENIED 0x8028_4012
	TBS_E_PROVISIONING_NOT_ALLOWED 0x8028_4013
	TBS_E_PPI_FUNCTION_UNSUPPORTED 0x8028_4014
	TBS_E_OWNERAUTH_NOT_FOUND 0x8028_4015
	TBS_E_PROVISIONING_INCOMPLETE 0x8028_4016
	TPMAPI_E_INVALID_STATE 0x8029_0100
	TPMAPI_E_NOT_ENOUGH_DATA 0x8029_0101
	TPMAPI_E_TOO_MUCH_DATA 0x8029_0102
	TPMAPI_E_INVALID_OUTPUT_POINTER 0x8029_0103
	TPMAPI_E_INVALID_PARAMETER 0x8029_0104
	TPMAPI_E_OUT_OF_MEMORY 0x8029_0105
	TPMAPI_E_BUFFER_TOO_SMALL 0x8029_0106
	TPMAPI_E_INTERNAL_ERROR 0x8029_0107
	TPMAPI_E_ACCESS_DENIED 0x8029_0108
	TPMAPI_E_AUTHORIZATION_FAILED 0x8029_0109
	TPMAPI_E_INVALID_CONTEXT_HANDLE 0x8029_010a
	TPMAPI_E_TBS_COMMUNICATION_ERROR 0x8029_010b
	TPMAPI_E_TPM_COMMAND_ERROR 0x8029_010c
	TPMAPI_E_MESSAGE_TOO_LARGE 0x8029_010d
	TPMAPI_E_INVALID_ENCODING 0x8029_010e
	TPMAPI_E_INVALID_KEY_SIZE 0x8029_010f
	TPMAPI_E_ENCRYPTION_FAILED 0x8029_0110
	TPMAPI_E_INVALID_KEY_PARAMS 0x8029_0111
	TPMAPI_E_INVALID_MIGRATION_AUTHORIZATION_BLOB 0x8029_0112
	TPMAPI_E_INVALID_PCR_INDEX 0x8029_0113
	TPMAPI_E_INVALID_DELEGATE_BLOB 0x8029_0114
	TPMAPI_E_INVALID_CONTEXT_PARAMS 0x8029_0115
	TPMAPI_E_INVALID_KEY_BLOB 0x8029_0116
	TPMAPI_E_INVALID_PCR_DATA 0x8029_0117
	TPMAPI_E_INVALID_OWNER_AUTH 0x8029_0118
	TPMAPI_E_FIPS_RNG_CHECK_FAILED 0x8029_0119
	TPMAPI_E_EMPTY_TCG_LOG 0x8029_011a
	TPMAPI_E_INVALID_TCG_LOG_ENTRY 0x8029_011b
	TPMAPI_E_TCG_SEPARATOR_ABSENT 0x8029_011c
	TPMAPI_E_TCG_INVALID_DIGEST_ENTRY 0x8029_011d
	TPMAPI_E_POLICY_DENIES_OPERATION 0x8029_011e
	TPMAPI_E_NV_BITS_NOT_DEFINED 0x8029_011f
	TPMAPI_E_NV_BITS_NOT_READY 0x8029_0120
	TPMAPI_E_SEALING_KEY_NOT_AVAILABLE 0x8029_0121
	TPMAPI_E_NO_AUTHORIZATION_CHAIN_FOUND 0x8029_0122
	TPMAPI_E_SVN_COUNTER_NOT_AVAILABLE 0x8029_0123
	TPMAPI_E_OWNER_AUTH_NOT_NULL 0x8029_0124
	TPMAPI_E_ENDORSEMENT_AUTH_NOT_NULL 0x8029_0125
	TPMAPI_E_AUTHORIZATION_REVOKED 0x8029_0126
	TPMAPI_E_MALFORMED_AUTHORIZATION_KEY 0x8029_0127
	TPMAPI_E_AUTHORIZING_KEY_NOT_SUPPORTED 0x8029_0128
	TPMAPI_E_INVALID_AUTHORIZATION_SIGNATURE 0x8029_0129
	TPMAPI_E_MALFORMED_AUTHORIZATION_POLICY 0x8029_012a
	TPMAPI_E_MALFORMED_AUTHORIZATION_OTHER 0x8029_012b
	TPMAPI_E_SEALING_KEY_CHANGED 0x8029_012c
	TBSIMP_E_BUFFER_TOO_SMALL 0x8029_0200
	TBSIMP_E_CLEANUP_FAILED 0x8029_0201
	TBSIMP_E_INVALID_CONTEXT_HANDLE 0x8029_0202
	TBSIMP_E_INVALID_CONTEXT_PARAM 0x8029_0203
	TBSIMP_E_TPM_ERROR 0x8029_0204
	TBSIMP_E_HASH_BAD_KEY 0x8029_0205
	TBSIMP_E_DUPLICATE_VHANDLE 0x8029_0206
	TBSIMP_E_INVALID_OUTPUT_POINTER 0x8029_0207
	TBSIMP_E_INVALID_PARAMETER 0x8029_0208
	TBSIMP_E_RPC_INIT_FAILED 0x8029_0209
	TBSIMP_E_SCHEDULER_NOT_RUNNING 0x8029_020a
	TBSIMP_E_COMMAND_CANCELED 0x8029_020b
	TBSIMP_E_OUT_OF_MEMORY 0x8029_020c
	TBSIMP_E_LIST_NO_MORE_ITEMS 0x8029_020d
	TBSIMP_E_LIST_NOT_FOUND 0x8029_020e
	TBSIMP_E_NOT_ENOUGH_SPACE 0x8029_020f
	TBSIMP_E_NOT_ENOUGH_TPM_CONTEXTS 0x8029_0210
	TBSIMP_E_COMMAND_FAILED 0x8029_0211
	TBSIMP_E_UNKNOWN_ORDINAL 0x8029_0212
	TBSIMP_E_RESOURCE_EXPIRED 0x8029_0213
	TBSIMP_E_INVALID_RESOURCE 0x8029_0214
	TBSIMP_E_NOTHING_TO_UNLOAD 0x8029_0215
	TBSIMP_E_HASH_TABLE_FULL 0x8029_0216
	TBSIMP_E_TOO_MANY_TBS_CONTEXTS 0x8029_0217
	TBSIMP_E_TOO_MANY_RESOURCES 0x8029_0218
	TBSIMP_E_PPI_NOT_SUPPORTED 0x8029_0219
	TBSIMP_E_TPM_INCOMPATIBLE 0x8029_021a
	TBSIMP_E_NO_EVENT_LOG 0x8029_021b
	TPM_E_PPI_ACPI_FAILURE 0x8029_0300
	TPM_E_PPI_USER_ABORT 0x8029_0301
	TPM_E_PPI_BIOS_FAILURE 0x8029_0302
	TPM_E_PPI_NOT_SUPPORTED 0x8029_0303
	TPM_E_PPI_BLOCKED_IN_BIOS 0x8029_0304
	TPM_E_PCP_ERROR_MASK 0x8029_0400
	TPM_E_PCP_DEVICE_NOT_READY 0x8029_0401
	TPM_E_PCP_INVALID_HANDLE 0x8029_0402
	TPM_E_PCP_INVALID_PARAMETER 0x8029_0403
	TPM_E_PCP_FLAG_NOT_SUPPORTED 0x8029_0404
	TPM_E_PCP_NOT_SUPPORTED 0x8029_0405
	TPM_E_PCP_BUFFER_TOO_SMALL 0x8029_0406
	TPM_E_PCP_INTERNAL_ERROR 0x8029_0407
	TPM_E_PCP_AUTHENTICATION_FAILED 0x8029_0408
	TPM_E_PCP_AUTHENTICATION_IGNORED 0x8029_0409
	TPM_E_PCP_POLICY_NOT_FOUND 0x8029_040a
	TPM_E_PCP_PROFILE_NOT_FOUND 0x8029_040b
	TPM_E_PCP_VALIDATION_FAILED 0x8029_040c
	TPM_E_PCP_WRONG_PARENT 0x8029_040e
	TPM_E_KEY_NOT_LOADED 0x8029_040f
	TPM_E_NO_KEY_CERTIFICATION 0x8029_0410
	TPM_E_KEY_NOT_FINALIZED 0x8029_0411
	TPM_E_ATTESTATION_CHALLENGE_NOT_SET 0x8029_0412
	TPM_E_NOT_PCR_BOUND 0x8029_0413
	TPM_E_KEY_ALREADY_FINALIZED 0x8029_0414
	TPM_E_KEY_USAGE_POLICY_NOT_SUPPORTED 0x8029_0415
	TPM_E_KEY_USAGE_POLICY_INVALID 0x8029_0416
	TPM_E_SOFT_KEY_ERROR 0x8029_0417
	TPM_E_KEY_NOT_AUTHENTICATED 0x8029_0418
	TPM_E_PCP_KEY_NOT_AIK 0x8029_0419
	TPM_E_KEY_NOT_SIGNING_KEY 0x8029_041a
	TPM_E_LOCKED_OUT 0x8029_041b
	TPM_E_CLAIM_TYPE_NOT_SUPPORTED 0x8029_041c
	TPM_E_VERSION_NOT_SUPPORTED 0x8029_041d
	TPM_E_BUFFER_LENGTH_MISMATCH 0x8029_041e
	TPM_E_PCP_IFX_RSA_KEY_CREATION_BLOCKED 0x8029_041f
	TPM_E_PCP_TICKET_MISSING 0x8029_0420
	TPM_E_PCP_RAW_POLICY_NOT_SUPPORTED 0x8029_0421
	TPM_E_PCP_KEY_HANDLE_INVALIDATED 0x8029_0422
	TPM_E_PCP_UNSUPPORTED_PSS_SALT 0x4029_0423
	TPM_E_ZERO_EXHAUST_ENABLED 0x8029_0500
	PLA_E_DCS_NOT_FOUND 0x8030_0002
	PLA_E_DCS_IN_USE 0x8030_00aa
	PLA_E_TOO_MANY_FOLDERS 0x8030_0045
	PLA_E_NO_MIN_DISK 0x8030_0070
	PLA_E_DCS_ALREADY_EXISTS 0x8030_00b7
	PLA_S_PROPERTY_IGNORED 0x0030_0100
	PLA_E_PROPERTY_CONFLICT 0x8030_0101
	PLA_E_DCS_SINGLETON_REQUIRED 0x8030_0102
	PLA_E_CREDENTIALS_REQUIRED 0x8030_0103
	PLA_E_DCS_NOT_RUNNING 0x8030_0104
	PLA_E_CONFLICT_INCL_EXCL_API 0x8030_0105
	PLA_E_NETWORK_EXE_NOT_VALID 0x8030_0106
	PLA_E_EXE_ALREADY_CONFIGURED 0x8030_0107
	PLA_E_EXE_PATH_NOT_VALID 0x8030_0108
	PLA_E_DC_ALREADY_EXISTS 0x8030_0109
	PLA_E_DCS_START_WAIT_TIMEOUT 0x8030_010a
	PLA_E_DC_START_WAIT_TIMEOUT 0x8030_010b
	PLA_E_REPORT_WAIT_TIMEOUT 0x8030_010c
	PLA_E_NO_DUPLICATES 0x8030_010d
	PLA_E_EXE_FULL_PATH_REQUIRED 0x8030_010e
	PLA_E_INVALID_SESSION_NAME 0x8030_010f
	PLA_E_PLA_CHANNEL_NOT_ENABLED 0x8030_0110
	PLA_E_TASKSCHED_CHANNEL_NOT_ENABLED 0x8030_0111
	PLA_E_RULES_MANAGER_FAILED 0x8030_0112
	PLA_E_CABAPI_FAILURE 0x8030_0113
	FVE_E_LOCKED_VOLUME 0x8031_0000
	FVE_E_NOT_ENCRYPTED 0x8031_0001
	FVE_E_NO_TPM_BIOS 0x8031_0002
	FVE_E_NO_MBR_METRIC 0x8031_0003
	FVE_E_NO_BOOTSECTOR_METRIC 0x8031_0004
	FVE_E_NO_BOOTMGR_METRIC 0x8031_0005
	FVE_E_WRONG_BOOTMGR 0x8031_0006
	FVE_E_SECURE_KEY_REQUIRED 0x8031_0007
	FVE_E_NOT_ACTIVATED 0x8031_0008
	FVE_E_ACTION_NOT_ALLOWED 0x8031_0009
	FVE_E_AD_SCHEMA_NOT_INSTALLED 0x8031_000a
	FVE_E_AD_INVALID_DATATYPE 0x8031_000b
	FVE_E_AD_INVALID_DATASIZE 0x8031_000c
	FVE_E_AD_NO_VALUES 0x8031_000d
	FVE_E_AD_ATTR_NOT_SET 0x8031_000e
	FVE_E_AD_GUID_NOT_FOUND 0x8031_000f
	FVE_E_BAD_INFORMATION 0x8031_0010
	FVE_E_TOO_SMALL 0x8031_0011
	FVE_E_SYSTEM_VOLUME 0x8031_0012
	FVE_E_FAILED_WRONG_FS 0x8031_0013
	FVE_E_BAD_PARTITION_SIZE 0x8031_0014
	FVE_E_NOT_SUPPORTED 0x8031_0015
	FVE_E_BAD_DATA 0x8031_0016
	FVE_E_VOLUME_NOT_BOUND 0x8031_0017
	FVE_E_TPM_NOT_OWNED 0x8031_0018
	FVE_E_NOT_DATA_VOLUME 0x8031_0019
	FVE_E_AD_INSUFFICIENT_BUFFER 0x8031_001a
	FVE_E_CONV_READ 0x8031_001b
	FVE_E_CONV_WRITE 0x8031_001c
	FVE_E_KEY_REQUIRED 0x8031_001d
	FVE_E_CLUSTERING_NOT_SUPPORTED 0x8031_001e
	FVE_E_VOLUME_BOUND_ALREADY 0x8031_001f
	FVE_E_OS_NOT_PROTECTED 0x8031_0020
	FVE_E_PROTECTION_DISABLED 0x8031_0021
	FVE_E_RECOVERY_KEY_REQUIRED 0x8031_0022
	FVE_E_FOREIGN_VOLUME 0x8031_0023
	FVE_E_OVERLAPPED_UPDATE 0x8031_0024
	FVE_E_TPM_SRK_AUTH_NOT_ZERO 0x8031_0025
	FVE_E_FAILED_SECTOR_SIZE 0x8031_0026
	FVE_E_FAILED_AUTHENTICATION 0x8031_0027
	FVE_E_NOT_OS_VOLUME 0x8031_0028
	FVE_E_AUTOUNLOCK_ENABLED 0x8031_0029
	FVE_E_WRONG_BOOTSECTOR 0x8031_002a
	FVE_E_WRONG_SYSTEM_FS 0x8031_002b
	FVE_E_POLICY_PASSWORD_REQUIRED 0x8031_002c
	FVE_E_CANNOT_SET_FVEK_ENCRYPTED 0x8031_002d
	FVE_E_CANNOT_ENCRYPT_NO_KEY 0x8031_002e
	FVE_E_BOOTABLE_CDDVD 0x8031_0030
	FVE_E_PROTECTOR_EXISTS 0x8031_0031
	FVE_E_RELATIVE_PATH 0x8031_0032
	FVE_E_PROTECTOR_NOT_FOUND 0x8031_0033
	FVE_E_INVALID_KEY_FORMAT 0x8031_0034
	FVE_E_INVALID_PASSWORD_FORMAT 0x8031_0035
	FVE_E_FIPS_RNG_CHECK_FAILED 0x8031_0036
	FVE_E_FIPS_PREVENTS_RECOVERY_PASSWORD 0x8031_0037
	FVE_E_FIPS_PREVENTS_EXTERNAL_KEY_EXPORT 0x8031_0038
	FVE_E_NOT_DECRYPTED 0x8031_0039
	FVE_E_INVALID_PROTECTOR_TYPE 0x8031_003a
	FVE_E_NO_PROTECTORS_TO_TEST 0x8031_003b
	FVE_E_KEYFILE_NOT_FOUND 0x8031_003c
	FVE_E_KEYFILE_INVALID 0x8031_003d
	FVE_E_KEYFILE_NO_VMK 0x8031_003e
	FVE_E_TPM_DISABLED 0x8031_003f
	FVE_E_NOT_ALLOWED_IN_SAFE_MODE 0x8031_0040
	FVE_E_TPM_INVALID_PCR 0x8031_0041
	FVE_E_TPM_NO_VMK 0x8031_0042
	FVE_E_PIN_INVALID 0x8031_0043
	FVE_E_AUTH_INVALID_APPLICATION 0x8031_0044
	FVE_E_AUTH_INVALID_CONFIG 0x8031_0045
	FVE_E_FIPS_DISABLE_PROTECTION_NOT_ALLOWED 0x8031_0046
	FVE_E_FS_NOT_EXTENDED 0x8031_0047
	FVE_E_FIRMWARE_TYPE_NOT_SUPPORTED 0x8031_0048
	FVE_E_NO_LICENSE 0x8031_0049
	FVE_E_NOT_ON_STACK 0x8031_004a
	FVE_E_FS_MOUNTED 0x8031_004b
	FVE_E_TOKEN_NOT_IMPERSONATED 0x8031_004c
	FVE_E_DRY_RUN_FAILED 0x8031_004d
	FVE_E_REBOOT_REQUIRED 0x8031_004e
	FVE_E_DEBUGGER_ENABLED 0x8031_004f
	FVE_E_RAW_ACCESS 0x8031_0050
	FVE_E_RAW_BLOCKED 0x8031_0051
	FVE_E_BCD_APPLICATIONS_PATH_INCORRECT 0x8031_0052
	FVE_E_NOT_ALLOWED_IN_VERSION 0x8031_0053
	FVE_E_NO_AUTOUNLOCK_MASTER_KEY 0x8031_0054
	FVE_E_MOR_FAILED 0x8031_0055
	FVE_E_HIDDEN_VOLUME 0x8031_0056
	FVE_E_TRANSIENT_STATE 0x8031_0057
	FVE_E_PUBKEY_NOT_ALLOWED 0x8031_0058
	FVE_E_VOLUME_HANDLE_OPEN 0x8031_0059
	FVE_E_NO_FEATURE_LICENSE 0x8031_005a
	FVE_E_INVALID_STARTUP_OPTIONS 0x8031_005b
	FVE_E_POLICY_RECOVERY_PASSWORD_NOT_ALLOWED 0x8031_005c
	FVE_E_POLICY_RECOVERY_PASSWORD_REQUIRED 0x8031_005d
	FVE_E_POLICY_RECOVERY_KEY_NOT_ALLOWED 0x8031_005e
	FVE_E_POLICY_RECOVERY_KEY_REQUIRED 0x8031_005f
	FVE_E_POLICY_STARTUP_PIN_NOT_ALLOWED 0x8031_0060
	FVE_E_POLICY_STARTUP_PIN_REQUIRED 0x8031_0061
	FVE_E_POLICY_STARTUP_KEY_NOT_ALLOWED 0x8031_0062
	FVE_E_POLICY_STARTUP_KEY_REQUIRED 0x8031_0063
	FVE_E_POLICY_STARTUP_PIN_KEY_NOT_ALLOWED 0x8031_0064
	FVE_E_POLICY_STARTUP_PIN_KEY_REQUIRED 0x8031_0065
	FVE_E_POLICY_STARTUP_TPM_NOT_ALLOWED 0x8031_0066
	FVE_E_POLICY_STARTUP_TPM_REQUIRED 0x8031_0067
	FVE_E_POLICY_INVALID_PIN_LENGTH 0x8031_0068
	FVE_E_KEY_PROTECTOR_NOT_SUPPORTED 0x8031_0069
	FVE_E_POLICY_PASSPHRASE_NOT_ALLOWED 0x8031_006a
	FVE_E_POLICY_PASSPHRASE_REQUIRED 0x8031_006b
	FVE_E_FIPS_PREVENTS_PASSPHRASE 0x8031_006c
	FVE_E_OS_VOLUME_PASSPHRASE_NOT_ALLOWED 0x8031_006d
	FVE_E_INVALID_BITLOCKER_OID 0x8031_006e
	FVE_E_VOLUME_TOO_SMALL 0x8031_006f
	FVE_E_DV_NOT_SUPPORTED_ON_FS 0x8031_0070
	FVE_E_DV_NOT_ALLOWED_BY_GP 0x8031_0071
	FVE_E_POLICY_USER_CERTIFICATE_NOT_ALLOWED 0x8031_0072
	FVE_E_POLICY_USER_CERTIFICATE_REQUIRED 0x8031_0073
	FVE_E_POLICY_USER_CERT_MUST_BE_HW 0x8031_0074
	FVE_E_POLICY_USER_CONFIGURE_FDV_AUTOUNLOCK_NOT_ALLOWED 0x8031_0075
	FVE_E_POLICY_USER_CONFIGURE_RDV_AUTOUNLOCK_NOT_ALLOWED 0x8031_0076
	FVE_E_POLICY_USER_CONFIGURE_RDV_NOT_ALLOWED 0x8031_0077
	FVE_E_POLICY_USER_ENABLE_RDV_NOT_ALLOWED 0x8031_0078
	FVE_E_POLICY_USER_DISABLE_RDV_NOT_ALLOWED 0x8031_0079
	FVE_E_POLICY_INVALID_PASSPHRASE_LENGTH 0x8031_0080
	FVE_E_POLICY_PASSPHRASE_TOO_SIMPLE 0x8031_0081
	FVE_E_RECOVERY_PARTITION 0x8031_0082
	FVE_E_POLICY_CONFLICT_FDV_RK_OFF_AUK_ON 0x8031_0083
	FVE_E_POLICY_CONFLICT_RDV_RK_OFF_AUK_ON 0x8031_0084
	FVE_E_NON_BITLOCKER_OID 0x8031_0085
	FVE_E_POLICY_PROHIBITS_SELFSIGNED 0x8031_0086
	FVE_E_POLICY_CONFLICT_RO_AND_STARTUP_KEY_REQUIRED 0x8031_0087
	FVE_E_CONV_RECOVERY_FAILED 0x8031_0088
	FVE_E_VIRTUALIZED_SPACE_TOO_BIG 0x8031_0089
	FVE_E_POLICY_CONFLICT_OSV_RP_OFF_ADB_ON 0x8031_0090
	FVE_E_POLICY_CONFLICT_FDV_RP_OFF_ADB_ON 0x8031_0091
	FVE_E_POLICY_CONFLICT_RDV_RP_OFF_ADB_ON 0x8031_0092
	FVE_E_NON_BITLOCKER_KU 0x8031_0093
	FVE_E_PRIVATEKEY_AUTH_FAILED 0x8031_0094
	FVE_E_REMOVAL_OF_DRA_FAILED 0x8031_0095
	FVE_E_OPERATION_NOT_SUPPORTED_ON_VISTA_VOLUME 0x8031_0096
	FVE_E_CANT_LOCK_AUTOUNLOCK_ENABLED_VOLUME 0x8031_0097
	FVE_E_FIPS_HASH_KDF_NOT_ALLOWED 0x8031_0098
	FVE_E_ENH_PIN_INVALID 0x8031_0099
	FVE_E_INVALID_PIN_CHARS 0x8031_009a
	FVE_E_INVALID_DATUM_TYPE 0x8031_009b
	FVE_E_EFI_ONLY 0x8031_009c
	FVE_E_MULTIPLE_NKP_CERTS 0x8031_009d
	FVE_E_REMOVAL_OF_NKP_FAILED 0x8031_009e
	FVE_E_INVALID_NKP_CERT 0x8031_009f
	FVE_E_NO_EXISTING_PIN 0x8031_00a0
	FVE_E_PROTECTOR_CHANGE_PIN_MISMATCH 0x8031_00a1
	FVE_E_PIN_PROTECTOR_CHANGE_BY_STD_USER_DISALLOWED 0x8031_00a2
	FVE_E_PROTECTOR_CHANGE_MAX_PIN_CHANGE_ATTEMPTS_REACHED 0x8031_00a3
	FVE_E_POLICY_PASSPHRASE_REQUIRES_ASCII 0x8031_00a4
	FVE_E_FULL_ENCRYPTION_NOT_ALLOWED_ON_TP_STORAGE 0x8031_00a5
	FVE_E_WIPE_NOT_ALLOWED_ON_TP_STORAGE 0x8031_00a6
	FVE_E_KEY_LENGTH_NOT_SUPPORTED_BY_EDRIVE 0x8031_00a7
	FVE_E_NO_EXISTING_PASSPHRASE 0x8031_00a8
	FVE_E_PROTECTOR_CHANGE_PASSPHRASE_MISMATCH 0x8031_00a9
	FVE_E_PASSPHRASE_TOO_LONG 0x8031_00aa
	FVE_E_NO_PASSPHRASE_WITH_TPM 0x8031_00ab
	FVE_E_NO_TPM_WITH_PASSPHRASE 0x8031_00ac
	FVE_E_NOT_ALLOWED_ON_CSV_STACK 0x8031_00ad
	FVE_E_NOT_ALLOWED_ON_CLUSTER 0x8031_00ae
	FVE_E_EDRIVE_NO_FAILOVER_TO_SW 0x8031_00af
	FVE_E_EDRIVE_BAND_IN_USE 0x8031_00b0
	FVE_E_EDRIVE_DISALLOWED_BY_GP 0x8031_00b1
	FVE_E_EDRIVE_INCOMPATIBLE_VOLUME 0x8031_00b2
	FVE_E_NOT_ALLOWED_TO_UPGRADE_WHILE_CONVERTING 0x8031_00b3
	FVE_E_EDRIVE_DV_NOT_SUPPORTED 0x8031_00b4
	FVE_E_NO_PREBOOT_KEYBOARD_DETECTED 0x8031_00b5
	FVE_E_NO_PREBOOT_KEYBOARD_OR_WINRE_DETECTED 0x8031_00b6
	FVE_E_POLICY_REQUIRES_STARTUP_PIN_ON_TOUCH_DEVICE 0x8031_00b7
	FVE_E_POLICY_REQUIRES_RECOVERY_PASSWORD_ON_TOUCH_DEVICE 0x8031_00b8
	FVE_E_WIPE_CANCEL_NOT_APPLICABLE 0x8031_00b9
	FVE_E_SECUREBOOT_DISABLED 0x8031_00ba
	FVE_E_SECUREBOOT_CONFIGURATION_INVALID 0x8031_00bb
	FVE_E_EDRIVE_DRY_RUN_FAILED 0x8031_00bc
	FVE_E_SHADOW_COPY_PRESENT 0x8031_00bd
	FVE_E_POLICY_INVALID_ENHANCED_BCD_SETTINGS 0x8031_00be
	FVE_E_EDRIVE_INCOMPATIBLE_FIRMWARE 0x8031_00bf
	FVE_E_PROTECTOR_CHANGE_MAX_PASSPHRASE_CHANGE_ATTEMPTS_REACHED 0x8031_00c0
	FVE_E_PASSPHRASE_PROTECTOR_CHANGE_BY_STD_USER_DISALLOWED 0x8031_00c1
	FVE_E_LIVEID_ACCOUNT_SUSPENDED 0x8031_00c2
	FVE_E_LIVEID_ACCOUNT_BLOCKED 0x8031_00c3
	FVE_E_NOT_PROVISIONED_ON_ALL_VOLUMES 0x8031_00c4
	FVE_E_DE_FIXED_DATA_NOT_SUPPORTED 0x8031_00c5
	FVE_E_DE_HARDWARE_NOT_COMPLIANT 0x8031_00c6
	FVE_E_DE_WINRE_NOT_CONFIGURED 0x8031_00c7
	FVE_E_DE_PROTECTION_SUSPENDED 0x8031_00c8
	FVE_E_DE_OS_VOLUME_NOT_PROTECTED 0x8031_00c9
	FVE_E_DE_DEVICE_LOCKEDOUT 0x8031_00ca
	FVE_E_DE_PROTECTION_NOT_YET_ENABLED 0x8031_00cb
	FVE_E_INVALID_PIN_CHARS_DETAILED 0x8031_00cc
	FVE_E_DEVICE_LOCKOUT_COUNTER_UNAVAILABLE 0x8031_00cd
	FVE_E_DEVICELOCKOUT_COUNTER_MISMATCH 0x8031_00ce
	FVE_E_BUFFER_TOO_LARGE 0x8031_00cf
	FVE_E_NO_SUCH_CAPABILITY_ON_TARGET 0x8031_00d0
	FVE_E_DE_PREVENTED_FOR_OS 0x8031_00d1
	FVE_E_DE_VOLUME_OPTED_OUT 0x8031_00d2
	FVE_E_DE_VOLUME_NOT_SUPPORTED 0x8031_00d3
	FVE_E_EOW_NOT_SUPPORTED_IN_VERSION 0x8031_00d4
	FVE_E_ADBACKUP_NOT_ENABLED 0x8031_00d5
	FVE_E_VOLUME_EXTEND_PREVENTS_EOW_DECRYPT 0x8031_00d6
	FVE_E_NOT_DE_VOLUME 0x8031_00d7
	FVE_E_PROTECTION_CANNOT_BE_DISABLED 0x8031_00d8
	FVE_E_OSV_KSR_NOT_ALLOWED 0x8031_00d9
	FVE_E_AD_BACKUP_REQUIRED_POLICY_NOT_SET_OS_DRIVE 0x8031_00da
	FVE_E_AD_BACKUP_REQUIRED_POLICY_NOT_SET_FIXED_DRIVE 0x8031_00db
	FVE_E_AD_BACKUP_REQUIRED_POLICY_NOT_SET_REMOVABLE_DRIVE 0x8031_00dc
	FVE_E_KEY_ROTATION_NOT_SUPPORTED 0x8031_00dd
	FVE_E_EXECUTE_REQUEST_SENT_TOO_SOON 0x8031_00de
	FVE_E_KEY_ROTATION_NOT_ENABLED 0x8031_00df
	FVE_E_DEVICE_NOT_JOINED 0x8031_00e0
	FWP_E_CALLOUT_NOT_FOUND 0x8032_0001
	FWP_E_CONDITION_NOT_FOUND 0x8032_0002
	FWP_E_FILTER_NOT_FOUND 0x8032_0003
	FWP_E_LAYER_NOT_FOUND 0x8032_0004
	FWP_E_PROVIDER_NOT_FOUND 0x8032_0005
	FWP_E_PROVIDER_CONTEXT_NOT_FOUND 0x8032_0006
	FWP_E_SUBLAYER_NOT_FOUND 0x8032_0007
	FWP_E_NOT_FOUND 0x8032_0008
	FWP_E_ALREADY_EXISTS 0x8032_0009
	FWP_E_IN_USE 0x8032_000a
	FWP_E_DYNAMIC_SESSION_IN_PROGRESS 0x8032_000b
	FWP_E_WRONG_SESSION 0x8032_000c
	FWP_E_NO_TXN_IN_PROGRESS 0x8032_000d
	FWP_E_TXN_IN_PROGRESS 0x8032_000e
	FWP_E_TXN_ABORTED 0x8032_000f
	FWP_E_SESSION_ABORTED 0x8032_0010
	FWP_E_INCOMPATIBLE_TXN 0x8032_0011
	FWP_E_TIMEOUT 0x8032_0012
	FWP_E_NET_EVENTS_DISABLED 0x8032_0013
	FWP_E_INCOMPATIBLE_LAYER 0x8032_0014
	FWP_E_KM_CLIENTS_ONLY 0x8032_0015
	FWP_E_LIFETIME_MISMATCH 0x8032_0016
	FWP_E_BUILTIN_OBJECT 0x8032_0017
	FWP_E_TOO_MANY_CALLOUTS 0x8032_0018
	FWP_E_NOTIFICATION_DROPPED 0x8032_0019
	FWP_E_TRAFFIC_MISMATCH 0x8032_001a
	FWP_E_INCOMPATIBLE_SA_STATE 0x8032_001b
	FWP_E_NULL_POINTER 0x8032_001c
	FWP_E_INVALID_ENUMERATOR 0x8032_001d
	FWP_E_INVALID_FLAGS 0x8032_001e
	FWP_E_INVALID_NET_MASK 0x8032_001f
	FWP_E_INVALID_RANGE 0x8032_0020
	FWP_E_INVALID_INTERVAL 0x8032_0021
	FWP_E_ZERO_LENGTH_ARRAY 0x8032_0022
	FWP_E_NULL_DISPLAY_NAME 0x8032_0023
	FWP_E_INVALID_ACTION_TYPE 0x8032_0024
	FWP_E_INVALID_WEIGHT 0x8032_0025
	FWP_E_MATCH_TYPE_MISMATCH 0x8032_0026
	FWP_E_TYPE_MISMATCH 0x8032_0027
	FWP_E_OUT_OF_BOUNDS 0x8032_0028
	FWP_E_RESERVED 0x8032_0029
	FWP_E_DUPLICATE_CONDITION 0x8032_002a
	FWP_E_DUPLICATE_KEYMOD 0x8032_002b
	FWP_E_ACTION_INCOMPATIBLE_WITH_LAYER 0x8032_002c
	FWP_E_ACTION_INCOMPATIBLE_WITH_SUBLAYER 0x8032_002d
	FWP_E_CONTEXT_INCOMPATIBLE_WITH_LAYER 0x8032_002e
	FWP_E_CONTEXT_INCOMPATIBLE_WITH_CALLOUT 0x8032_002f
	FWP_E_INCOMPATIBLE_AUTH_METHOD 0x8032_0030
	FWP_E_INCOMPATIBLE_DH_GROUP 0x8032_0031
	FWP_E_EM_NOT_SUPPORTED 0x8032_0032
	FWP_E_NEVER_MATCH 0x8032_0033
	FWP_E_PROVIDER_CONTEXT_MISMATCH 0x8032_0034
	FWP_E_INVALID_PARAMETER 0x8032_0035
	FWP_E_TOO_MANY_SUBLAYERS 0x8032_0036
	FWP_E_CALLOUT_NOTIFICATION_FAILED 0x8032_0037
	FWP_E_INVALID_AUTH_TRANSFORM 0x8032_0038
	FWP_E_INVALID_CIPHER_TRANSFORM 0x8032_0039
	FWP_E_INCOMPATIBLE_CIPHER_TRANSFORM 0x8032_003a
	FWP_E_INVALID_TRANSFORM_COMBINATION 0x8032_003b
	FWP_E_DUPLICATE_AUTH_METHOD 0x8032_003c
	FWP_E_INVALID_TUNNEL_ENDPOINT 0x8032_003d
	FWP_E_L2_DRIVER_NOT_READY 0x8032_003e
	FWP_E_KEY_DICTATOR_ALREADY_REGISTERED 0x8032_003f
	FWP_E_KEY_DICTATION_INVALID_KEYING_MATERIAL 0x8032_0040
	FWP_E_CONNECTIONS_DISABLED 0x8032_0041
	FWP_E_INVALID_DNS_NAME 0x8032_0042
	FWP_E_STILL_ON 0x8032_0043
	FWP_E_IKEEXT_NOT_RUNNING 0x8032_0044
	FWP_E_DROP_NOICMP 0x8032_0104
	WS_S_ASYNC 0x003d_0000
	WS_S_END 0x003d_0001
	WS_E_INVALID_FORMAT 0x803d_0000
	WS_E_OBJECT_FAULTED 0x803d_0001
	WS_E_NUMERIC_OVERFLOW 0x803d_0002
	WS_E_INVALID_OPERATION 0x803d_0003
	WS_E_OPERATION_ABORTED 0x803d_0004
	WS_E_ENDPOINT_ACCESS_DENIED 0x803d_0005
	WS_E_OPERATION_TIMED_OUT 0x803d_0006
	WS_E_OPERATION_ABANDONED 0x803d_0007
	WS_E_QUOTA_EXCEEDED 0x803d_0008
	WS_E_NO_TRANSLATION_AVAILABLE 0x803d_0009
	WS_E_SECURITY_VERIFICATION_FAILURE 0x803d_000a
	WS_E_ADDRESS_IN_USE 0x803d_000b
	WS_E_ADDRESS_NOT_AVAILABLE 0x803d_000c
	WS_E_ENDPOINT_NOT_FOUND 0x803d_000d
	WS_E_ENDPOINT_NOT_AVAILABLE 0x803d_000e
	WS_E_ENDPOINT_FAILURE 0x803d_000f
	WS_E_ENDPOINT_UNREACHABLE 0x803d_0010
	WS_E_ENDPOINT_ACTION_NOT_SUPPORTED 0x803d_0011
	WS_E_ENDPOINT_TOO_BUSY 0x803d_0012
	WS_E_ENDPOINT_FAULT_RECEIVED 0x803d_0013
	WS_E_ENDPOINT_DISCONNECTED 0x803d_0014
	WS_E_PROXY_FAILURE 0x803d_0015
	WS_E_PROXY_ACCESS_DENIED 0x803d_0016
	WS_E_NOT_SUPPORTED 0x803d_0017
	WS_E_PROXY_REQUIRES_BASIC_AUTH 0x803d_0018
	WS_E_PROXY_REQUIRES_DIGEST_AUTH 0x803d_0019
	WS_E_PROXY_REQUIRES_NTLM_AUTH 0x803d_001a
	WS_E_PROXY_REQUIRES_NEGOTIATE_AUTH 0x803d_001b
	WS_E_SERVER_REQUIRES_BASIC_AUTH 0x803d_001c
	WS_E_SERVER_REQUIRES_DIGEST_AUTH 0x803d_001d
	WS_E_SERVER_REQUIRES_NTLM_AUTH 0x803d_001e
	WS_E_SERVER_REQUIRES_NEGOTIATE_AUTH 0x803d_001f
	WS_E_INVALID_ENDPOINT_URL 0x803d_0020
	WS_E_OTHER 0x803d_0021
	WS_E_SECURITY_TOKEN_EXPIRED 0x803d_0022
	WS_E_SECURITY_SYSTEM_FAILURE 0x803d_0023
	HCS_E_TERMINATED_DURING_START 0x8037_0100
	HCS_E_IMAGE_MISMATCH 0x8037_0101
	HCS_E_HYPERV_NOT_INSTALLED 0x8037_0102
	HCS_E_INVALID_STATE 0x8037_0105
	HCS_E_UNEXPECTED_EXIT 0x8037_0106
	HCS_E_TERMINATED 0x8037_0107
	HCS_E_CONNECT_FAILED 0x8037_0108
	HCS_E_CONNECTION_TIMEOUT 0x8037_0109
	HCS_E_CONNECTION_CLOSED 0x8037_010a
	HCS_E_UNKNOWN_MESSAGE 0x8037_010b
	HCS_E_UNSUPPORTED_PROTOCOL_VERSION 0x8037_010c
	HCS_E_INVALID_JSON 0x8037_010d
	HCS_E_SYSTEM_NOT_FOUND 0x8037_010e
	HCS_E_SYSTEM_ALREADY_EXISTS 0x8037_010f
	HCS_E_SYSTEM_ALREADY_STOPPED 0x8037_0110
	HCS_E_PROTOCOL_ERROR 0x8037_0111
	HCS_E_INVALID_LAYER 0x8037_0112
	HCS_E_WINDOWS_INSIDER_REQUIRED 0x8037_0113
	HCS_E_SERVICE_NOT_AVAILABLE 0x8037_0114
	HCS_E_OPERATION_NOT_STARTED 0x8037_0115
	HCS_E_OPERATION_ALREADY_STARTED 0x8037_0116
	HCS_E_OPERATION_PENDING 0x8037_0117
	HCS_E_OPERATION_TIMEOUT 0x8037_0118
	HCS_E_OPERATION_SYSTEM_CALLBACK_ALREADY_SET 0x8037_0119
	HCS_E_OPERATION_RESULT_ALLOCATION_FAILED 0x8037_011a
	HCS_E_ACCESS_DENIED 0x8037_011b
	HCS_E_GUEST_CRITICAL_ERROR 0x8037_011c
	WHV_E_UNKNOWN_CAPABILITY 0x8037_0300
	WHV_E_INSUFFICIENT_BUFFER 0x8037_0301
	WHV_E_UNKNOWN_PROPERTY 0x8037_0302
	WHV_E_UNSUPPORTED_HYPERVISOR_CONFIG 0x8037_0303
	WHV_E_INVALID_PARTITION_CONFIG 0x8037_0304
	WHV_E_GPA_RANGE_NOT_FOUND 0x8037_0305
	WHV_E_VP_ALREADY_EXISTS 0x8037_0306
	WHV_E_VP_DOES_NOT_EXIST 0x8037_0307
	WHV_E_INVALID_VP_STATE 0x8037_0308
	WHV_E_INVALID_VP_REGISTER_NAME 0x8037_0309
	HCN_E_NETWORK_NOT_FOUND 0x803b_0001
	HCN_E_ENDPOINT_NOT_FOUND 0x803b_0002
	HCN_E_LAYER_NOT_FOUND 0x803b_0003
	HCN_E_SWITCH_NOT_FOUND 0x803b_0004
	HCN_E_SUBNET_NOT_FOUND 0x803b_0005
	HCN_E_ADAPTER_NOT_FOUND 0x803b_0006
	HCN_E_PORT_NOT_FOUND 0x803b_0007
	HCN_E_POLICY_NOT_FOUND 0x803b_0008
	HCN_E_VFP_PORTSETTING_NOT_FOUND 0x803b_0009
	HCN_E_INVALID_NETWORK 0x803b_000a
	HCN_E_INVALID_NETWORK_TYPE 0x803b_000b
	HCN_E_INVALID_ENDPOINT 0x803b_000c
	HCN_E_INVALID_POLICY 0x803b_000d
	HCN_E_INVALID_POLICY_TYPE 0x803b_000e
	HCN_E_INVALID_REMOTE_ENDPOINT_OPERATION 0x803b_000f
	HCN_E_NETWORK_ALREADY_EXISTS 0x803b_0010
	HCN_E_LAYER_ALREADY_EXISTS 0x803b_0011
	HCN_E_POLICY_ALREADY_EXISTS 0x803b_0012
	HCN_E_PORT_ALREADY_EXISTS 0x803b_0013
	HCN_E_ENDPOINT_ALREADY_ATTACHED 0x803b_0014
	HCN_E_REQUEST_UNSUPPORTED 0x803b_0015
	HCN_E_MAPPING_NOT_SUPPORTED 0x803b_0016
	HCN_E_DEGRADED_OPERATION 0x803b_0017
	HCN_E_SHARED_SWITCH_MODIFICATION 0x803b_0018
	HCN_E_GUID_CONVERSION_FAILURE 0x803b_0019
	HCN_E_REGKEY_FAILURE 0x803b_001a
	HCN_E_INVALID_JSON 0x803b_001b
	HCN_E_INVALID_JSON_REFERENCE 0x803b_001c
	HCN_E_ENDPOINT_SHARING_DISABLED 0x803b_001d
	HCN_E_INVALID_IP 0x803b_001e
	HCN_E_SWITCH_EXTENSION_NOT_FOUND 0x803b_001f
	HCN_E_MANAGER_STOPPED 0x803b_0020
	GCN_E_MODULE_NOT_FOUND 0x803b_0021
	GCN_E_NO_REQUEST_HANDLERS 0x803b_0022
	GCN_E_REQUEST_UNSUPPORTED 0x803b_0023
	GCN_E_RUNTIMEKEYS_FAILED 0x803b_0024
	GCN_E_NETADAPTER_TIMEOUT 0x803b_0025
	GCN_E_NETADAPTER_NOT_FOUND 0x803b_0026
	GCN_E_NETCOMPARTMENT_NOT_FOUND 0x803b_0027
	GCN_E_NETINTERFACE_NOT_FOUND 0x803b_0028
	GCN_E_DEFAULTNAMESPACE_EXISTS 0x803b_0029
	HCN_E_ICS_DISABLED 0x803b_002a
	HCN_E_ENDPOINT_NAMESPACE_ALREADY_EXISTS 0x803b_002b
	HCN_E_ENTITY_HAS_REFERENCES 0x803b_002c
	HCN_E_INVALID_INTERNAL_PORT 0x803b_002d
	HCN_E_NAMESPACE_ATTACH_FAILED 0x803b_002e
	HCN_E_ADDR_INVALID_OR_RESERVED 0x803b_002f
	WPN_E_CHANNEL_CLOSED 0x803e_0100
	WPN_E_CHANNEL_REQUEST_NOT_COMPLETE 0x803e_0101
	WPN_E_INVALID_APP 0x803e_0102
	WPN_E_OUTSTANDING_CHANNEL_REQUEST 0x803e_0103
	WPN_E_DUPLICATE_CHANNEL 0x803e_0104
	WPN_E_PLATFORM_UNAVAILABLE 0x803e_0105
	WPN_E_NOTIFICATION_POSTED 0x803e_0106
	WPN_E_NOTIFICATION_HIDDEN 0x803e_0107
	WPN_E_NOTIFICATION_NOT_POSTED 0x803e_0108
	WPN_E_CLOUD_DISABLED 0x803e_0109
	WPN_E_CLOUD_INCAPABLE 0x803e_0110
	WPN_E_CLOUD_AUTH_UNAVAILABLE 0x803e_011a
	WPN_E_CLOUD_SERVICE_UNAVAILABLE 0x803e_011b
	WPN_E_FAILED_LOCK_SCREEN_UPDATE_INTIALIZATION 0x803e_011c
	WPN_E_NOTIFICATION_DISABLED 0x803e_0111
	WPN_E_NOTIFICATION_INCAPABLE 0x803e_0112
	WPN_E_INTERNET_INCAPABLE 0x803e_0113
	WPN_E_NOTIFICATION_TYPE_DISABLED 0x803e_0114
	WPN_E_NOTIFICATION_SIZE 0x803e_0115
	WPN_E_TAG_SIZE 0x803e_0116
	WPN_E_ACCESS_DENIED 0x803e_0117
	WPN_E_DUPLICATE_REGISTRATION 0x803e_0118
	WPN_E_PUSH_NOTIFICATION_INCAPABLE 0x803e_0119
	WPN_E_DEV_ID_SIZE 0x803e_0120
	WPN_E_TAG_ALPHANUMERIC 0x803e_012a
	WPN_E_INVALID_HTTP_STATUS_CODE 0x803e_012b
	WPN_E_OUT_OF_SESSION 0x803e_0200
	WPN_E_POWER_SAVE 0x803e_0201
	WPN_E_IMAGE_NOT_FOUND_IN_CACHE 0x803e_0202
	WPN_E_ALL_URL_NOT_COMPLETED 0x803e_0203
	WPN_E_INVALID_CLOUD_IMAGE 0x803e_0204
	WPN_E_NOTIFICATION_ID_MATCHED 0x803e_0205
	WPN_E_CALLBACK_ALREADY_REGISTERED 0x803e_0206
	WPN_E_TOAST_NOTIFICATION_DROPPED 0x803e_0207
	WPN_E_STORAGE_LOCKED 0x803e_0208
	WPN_E_GROUP_SIZE 0x803e_0209
	WPN_E_GROUP_ALPHANUMERIC 0x803e_020a
	WPN_E_CLOUD_DISABLED_FOR_APP 0x803e_020b
	E_MBN_CONTEXT_NOT_ACTIVATED 0x8054_8201
	E_MBN_BAD_SIM 0x8054_8202
	E_MBN_DATA_CLASS_NOT_AVAILABLE 0x8054_8203
	E_MBN_INVALID_ACCESS_STRING 0x8054_8204
	E_MBN_MAX_ACTIVATED_CONTEXTS 0x8054_8205
	E_MBN_PACKET_SVC_DETACHED 0x8054_8206
	E_MBN_PROVIDER_NOT_VISIBLE 0x8054_8207
	E_MBN_RADIO_POWER_OFF 0x8054_8208
	E_MBN_SERVICE_NOT_ACTIVATED 0x8054_8209
	E_MBN_SIM_NOT_INSERTED 0x8054_820a
	E_MBN_VOICE_CALL_IN_PROGRESS 0x8054_820b
	E_MBN_INVALID_CACHE 0x8054_820c
	E_MBN_NOT_REGISTERED 0x8054_820d
	E_MBN_PROVIDERS_NOT_FOUND 0x8054_820e
	E_MBN_PIN_NOT_SUPPORTED 0x8054_820f
	E_MBN_PIN_REQUIRED 0x8054_8210
	E_MBN_PIN_DISABLED 0x8054_8211
	E_MBN_FAILURE 0x8054_8212
	E_MBN_INVALID_PROFILE 0x8054_8218
	E_MBN_DEFAULT_PROFILE_EXIST 0x8054_8219
	E_MBN_SMS_ENCODING_NOT_SUPPORTED 0x8054_8220
	E_MBN_SMS_FILTER_NOT_SUPPORTED 0x8054_8221
	E_MBN_SMS_INVALID_MEMORY_INDEX 0x8054_8222
	E_MBN_SMS_LANG_NOT_SUPPORTED 0x8054_8223
	E_MBN_SMS_MEMORY_FAILURE 0x8054_8224
	E_MBN_SMS_NETWORK_TIMEOUT 0x8054_8225
	E_MBN_SMS_UNKNOWN_SMSC_ADDRESS 0x8054_8226
	E_MBN_SMS_FORMAT_NOT_SUPPORTED 0x8054_8227
	E_MBN_SMS_OPERATION_NOT_ALLOWED 0x8054_8228
	E_MBN_SMS_MEMORY_FULL 0x8054_8229
	PEER_E_IPV6_NOT_INSTALLED 0x8063_0001
	PEER_E_NOT_INITIALIZED 0x8063_0002
	PEER_E_CANNOT_START_SERVICE 0x8063_0003
	PEER_E_NOT_LICENSED 0x8063_0004
	PEER_E_INVALID_GRAPH 0x8063_0010
	PEER_E_DBNAME_CHANGED 0x8063_0011
	PEER_E_DUPLICATE_GRAPH 0x8063_0012
	PEER_E_GRAPH_NOT_READY 0x8063_0013
	PEER_E_GRAPH_SHUTTING_DOWN 0x8063_0014
	PEER_E_GRAPH_IN_USE 0x8063_0015
	PEER_E_INVALID_DATABASE 0x8063_0016
	PEER_E_TOO_MANY_ATTRIBUTES 0x8063_0017
	PEER_E_CONNECTION_NOT_FOUND 0x8063_0103
	PEER_E_CONNECT_SELF 0x8063_0106
	PEER_E_ALREADY_LISTENING 0x8063_0107
	PEER_E_NODE_NOT_FOUND 0x8063_0108
	PEER_E_CONNECTION_FAILED 0x8063_0109
	PEER_E_CONNECTION_NOT_AUTHENTICATED 0x8063_010a
	PEER_E_CONNECTION_REFUSED 0x8063_010b
	PEER_E_CLASSIFIER_TOO_LONG 0x8063_0201
	PEER_E_TOO_MANY_IDENTITIES 0x8063_0202
	PEER_E_NO_KEY_ACCESS 0x8063_0203
	PEER_E_GROUPS_EXIST 0x8063_0204
	PEER_E_RECORD_NOT_FOUND 0x8063_0301
	PEER_E_DATABASE_ACCESSDENIED 0x8063_0302
	PEER_E_DBINITIALIZATION_FAILED 0x8063_0303
	PEER_E_MAX_RECORD_SIZE_EXCEEDED 0x8063_0304
	PEER_E_DATABASE_ALREADY_PRESENT 0x8063_0305
	PEER_E_DATABASE_NOT_PRESENT 0x8063_0306
	PEER_E_IDENTITY_NOT_FOUND 0x8063_0401
	PEER_E_EVENT_HANDLE_NOT_FOUND 0x8063_0501
	PEER_E_INVALID_SEARCH 0x8063_0601
	PEER_E_INVALID_ATTRIBUTES 0x8063_0602
	PEER_E_INVITATION_NOT_TRUSTED 0x8063_0701
	PEER_E_CHAIN_TOO_LONG 0x8063_0703
	PEER_E_INVALID_TIME_PERIOD 0x8063_0705
	PEER_E_CIRCULAR_CHAIN_DETECTED 0x8063_0706
	PEER_E_CERT_STORE_CORRUPTED 0x8063_0801
	PEER_E_NO_CLOUD 0x8063_1001
	PEER_E_CLOUD_NAME_AMBIGUOUS 0x8063_1005
	PEER_E_INVALID_RECORD 0x8063_2010
	PEER_E_NOT_AUTHORIZED 0x8063_2020
	PEER_E_PASSWORD_DOES_NOT_MEET_POLICY 0x8063_2021
	PEER_E_DEFERRED_VALIDATION 0x8063_2030
	PEER_E_INVALID_GROUP_PROPERTIES 0x8063_2040
	PEER_E_INVALID_PEER_NAME 0x8063_2050
	PEER_E_INVALID_CLASSIFIER 0x8063_2060
	PEER_E_INVALID_FRIENDLY_NAME 0x8063_2070
	PEER_E_INVALID_ROLE_PROPERTY 0x8063_2071
	PEER_E_INVALID_CLASSIFIER_PROPERTY 0x8063_2072
	PEER_E_INVALID_RECORD_EXPIRATION 0x8063_2080
	PEER_E_INVALID_CREDENTIAL_INFO 0x8063_2081
	PEER_E_INVALID_CREDENTIAL 0x8063_2082
	PEER_E_INVALID_RECORD_SIZE 0x8063_2083
	PEER_E_UNSUPPORTED_VERSION 0x8063_2090
	PEER_E_GROUP_NOT_READY 0x8063_2091
	PEER_E_GROUP_IN_USE 0x8063_2092
	PEER_E_INVALID_GROUP 0x8063_2093
	PEER_E_NO_MEMBERS_FOUND 0x8063_2094
	PEER_E_NO_MEMBER_CONNECTIONS 0x8063_2095
	PEER_E_UNABLE_TO_LISTEN 0x8063_2096
	PEER_E_IDENTITY_DELETED 0x8063_20a0
	PEER_E_SERVICE_NOT_AVAILABLE 0x8063_20a1
	PEER_E_CONTACT_NOT_FOUND 0x8063_6001
	PEER_S_GRAPH_DATA_CREATED 0x0063_0001
	PEER_S_NO_EVENT_DATA 0x0063_0002
	PEER_S_ALREADY_CONNECTED 0x0063_2000
	PEER_S_SUBSCRIPTION_EXISTS 0x0063_6000
	PEER_S_NO_CONNECTIVITY 0x0063_0005
	PEER_S_ALREADY_A_MEMBER 0x0063_0006
	PEER_E_CANNOT_CONVERT_PEER_NAME 0x8063_4001
	PEER_E_INVALID_PEER_HOST_NAME 0x8063_4002
	PEER_E_NO_MORE 0x8063_4003
	PEER_E_PNRP_DUPLICATE_PEER_NAME 0x8063_4005
	PEER_E_INVITE_CANCELLED 0x8063_7000
	PEER_E_INVITE_RESPONSE_NOT_AVAILABLE 0x8063_7001
	PEER_E_NOT_SIGNED_IN 0x8063_7003
	PEER_E_PRIVACY_DECLINED 0x8063_7004
	PEER_E_TIMEOUT 0x8063_7005
	PEER_E_INVALID_ADDRESS 0x8063_7007
	PEER_E_FW_EXCEPTION_DISABLED 0x8063_7008
	PEER_E_FW_BLOCKED_BY_POLICY 0x8063_7009
	PEER_E_FW_BLOCKED_BY_SHIELDS_UP 0x8063_700a
	PEER_E_FW_DECLINED 0x8063_700b
	UI_E_CREATE_FAILED 0x802a_0001
	UI_E_SHUTDOWN_CALLED 0x802a_0002
	UI_E_ILLEGAL_REENTRANCY 0x802a_0003
	UI_E_OBJECT_SEALED 0x802a_0004
	UI_E_VALUE_NOT_SET 0x802a_0005
	UI_E_VALUE_NOT_DETERMINED 0x802a_0006
	UI_E_INVALID_OUTPUT 0x802a_0007
	UI_E_BOOLEAN_EXPECTED 0x802a_0008
	UI_E_DIFFERENT_OWNER 0x802a_0009
	UI_E_AMBIGUOUS_MATCH 0x802a_000a
	UI_E_FP_OVERFLOW 0x802a_000b
	UI_E_WRONG_THREAD 0x802a_000c
	UI_E_STORYBOARD_ACTIVE 0x802a_0101
	UI_E_STORYBOARD_NOT_PLAYING 0x802a_0102
	UI_E_START_KEYFRAME_AFTER_END 0x802a_0103
	UI_E_END_KEYFRAME_NOT_DETERMINED 0x802a_0104
	UI_E_LOOPS_OVERLAP 0x802a_0105
	UI_E_TRANSITION_ALREADY_USED 0x802a_0106
	UI_E_TRANSITION_NOT_IN_STORYBOARD 0x802a_0107
	UI_E_TRANSITION_ECLIPSED 0x802a_0108
	UI_E_TIME_BEFORE_LAST_UPDATE 0x802a_0109
	UI_E_TIMER_CLIENT_ALREADY_CONNECTED 0x802a_010a
	UI_E_INVALID_DIMENSION 0x802a_010b
	UI_E_PRIMITIVE_OUT_OF_BOUNDS 0x802a_010c
	UI_E_WINDOW_CLOSED 0x802a_0201
	E_BLUETOOTH_ATT_INVALID_HANDLE 0x8065_0001
	E_BLUETOOTH_ATT_READ_NOT_PERMITTED 0x8065_0002
	E_BLUETOOTH_ATT_WRITE_NOT_PERMITTED 0x8065_0003
	E_BLUETOOTH_ATT_INVALID_PDU 0x8065_0004
	E_BLUETOOTH_ATT_INSUFFICIENT_AUTHENTICATION 0x8065_0005
	E_BLUETOOTH_ATT_REQUEST_NOT_SUPPORTED 0x8065_0006
	E_BLUETOOTH_ATT_INVALID_OFFSET 0x8065_0007
	E_BLUETOOTH_ATT_INSUFFICIENT_AUTHORIZATION 0x8065_0008
	E_BLUETOOTH_ATT_PREPARE_QUEUE_FULL 0x8065_0009
	E_BLUETOOTH_ATT_ATTRIBUTE_NOT_FOUND 0x8065_000a
	E_BLUETOOTH_ATT_ATTRIBUTE_NOT_LONG 0x8065_000b
	E_BLUETOOTH_ATT_INSUFFICIENT_ENCRYPTION_KEY_SIZE 0x8065_000c
	E_BLUETOOTH_ATT_INVALID_ATTRIBUTE_VALUE_LENGTH 0x8065_000d
	E_BLUETOOTH_ATT_UNLIKELY 0x8065_000e
	E_BLUETOOTH_ATT_INSUFFICIENT_ENCRYPTION 0x8065_000f
	E_BLUETOOTH_ATT_UNSUPPORTED_GROUP_TYPE 0x8065_0010
	E_BLUETOOTH_ATT_INSUFFICIENT_RESOURCES 0x8065_0011
	E_BLUETOOTH_ATT_UNKNOWN_ERROR 0x8065_1000
	E_AUDIO_ENGINE_NODE_NOT_FOUND 0x8066_0001
	E_HDAUDIO_EMPTY_CONNECTION_LIST 0x8066_0002
	E_HDAUDIO_CONNECTION_LIST_NOT_SUPPORTED 0x8066_0003
	E_HDAUDIO_NO_LOGICAL_DEVICES_CREATED 0x8066_0004
	E_HDAUDIO_NULL_LINKED_LIST_ENTRY 0x8066_0005
	STATEREPOSITORY_E_CONCURRENCY_LOCKING_FAILURE 0x8067_0001
	STATEREPOSITORY_E_STATEMENT_INPROGRESS 0x8067_0002
	STATEREPOSITORY_E_CONFIGURATION_INVALID 0x8067_0003
	STATEREPOSITORY_E_UNKNOWN_SCHEMA_VERSION 0x8067_0004
	STATEREPOSITORY_ERROR_DICTIONARY_CORRUPTED 0x8067_0005
	STATEREPOSITORY_E_BLOCKED 0x8067_0006
	STATEREPOSITORY_E_BUSY_RETRY 0x8067_0007
	STATEREPOSITORY_E_BUSY_RECOVERY_RETRY 0x8067_0008
	STATEREPOSITORY_E_LOCKED_RETRY 0x8067_0009
	STATEREPOSITORY_E_LOCKED_SHAREDCACHE_RETRY 0x8067_000a
	STATEREPOSITORY_E_TRANSACTION_REQUIRED 0x8067_000b
	STATEREPOSITORY_E_BUSY_TIMEOUT_EXCEEDED 0x8067_000c
	STATEREPOSITORY_E_BUSY_RECOVERY_TIMEOUT_EXCEEDED 0x8067_000d
	STATEREPOSITORY_E_LOCKED_TIMEOUT_EXCEEDED 0x8067_000e
	STATEREPOSITORY_E_LOCKED_SHAREDCACHE_TIMEOUT_EXCEEDED 0x8067_000f
	STATEREPOSITORY_E_SERVICE_STOP_IN_PROGRESS 0x8067_0010
	STATEREPOSTORY_E_NESTED_TRANSACTION_NOT_SUPPORTED 0x8067_0011
	STATEREPOSITORY_ERROR_CACHE_CORRUPTED 0x8067_0012
	STATEREPOSITORY_TRANSACTION_CALLER_ID_CHANGED 0x0067_0013
	STATEREPOSITORY_TRANSACTION_IN_PROGRESS 0x0067_0014
	ERROR_SPACES_POOL_WAS_DELETED 0x00e7_0001
	ERROR_SPACES_FAULT_DOMAIN_TYPE_INVALID 0x80e7_0001
	ERROR_SPACES_INTERNAL_ERROR 0x80e7_0002
	ERROR_SPACES_RESILIENCY_TYPE_INVALID 0x80e7_0003
	ERROR_SPACES_DRIVE_SECTOR_SIZE_INVALID 0x80e7_0004
	ERROR_SPACES_DRIVE_REDUNDANCY_INVALID 0x80e7_0006
	ERROR_SPACES_NUMBER_OF_DATA_COPIES_INVALID 0x80e7_0007
	ERROR_SPACES_PARITY_LAYOUT_INVALID 0x80e7_0008
	ERROR_SPACES_INTERLEAVE_LENGTH_INVALID 0x80e7_0009
	ERROR_SPACES_NUMBER_OF_COLUMNS_INVALID 0x80e7_000a
	ERROR_SPACES_NOT_ENOUGH_DRIVES 0x80e7_000b
	ERROR_SPACES_EXTENDED_ERROR 0x80e7_000c
	ERROR_SPACES_PROVISIONING_TYPE_INVALID 0x80e7_000d
	ERROR_SPACES_ALLOCATION_SIZE_INVALID 0x80e7_000e
	ERROR_SPACES_ENCLOSURE_AWARE_INVALID 0x80e7_000f
	ERROR_SPACES_WRITE_CACHE_SIZE_INVALID 0x80e7_0010
	ERROR_SPACES_NUMBER_OF_GROUPS_INVALID 0x80e7_0011
	ERROR_SPACES_DRIVE_OPERATIONAL_STATE_INVALID 0x80e7_0012
	ERROR_SPACES_ENTRY_INCOMPLETE 0x80e7_0013
	ERROR_SPACES_ENTRY_INVALID 0x80e7_0014
	ERROR_VOLSNAP_BOOTFILE_NOT_VALID 0x8082_0001
	ERROR_VOLSNAP_ACTIVATION_TIMEOUT 0x8082_0002
	ERROR_TIERING_NOT_SUPPORTED_ON_VOLUME 0x8083_0001
	ERROR_TIERING_VOLUME_DISMOUNT_IN_PROGRESS 0x8083_0002
	ERROR_TIERING_STORAGE_TIER_NOT_FOUND 0x8083_0003
	ERROR_TIERING_INVALID_FILE_ID 0x8083_0004
	ERROR_TIERING_WRONG_CLUSTER_NODE 0x8083_0005
	ERROR_TIERING_ALREADY_PROCESSING 0x8083_0006
	ERROR_TIERING_CANNOT_PIN_OBJECT 0x8083_0007
	ERROR_TIERING_FILE_IS_NOT_PINNED 0x8083_0008
	ERROR_NOT_A_TIERED_VOLUME 0x8083_0009
	ERROR_ATTRIBUTE_NOT_PRESENT 0x8083_000a
	ERROR_SECCORE_INVALID_COMMAND 0xc0e8_0000
	ERROR_NO_APPLICABLE_APP_LICENSES_FOUND 0xc0ea_0001
	ERROR_CLIP_LICENSE_NOT_FOUND 0xc0ea_0002
	ERROR_CLIP_DEVICE_LICENSE_MISSING 0xc0ea_0003
	ERROR_CLIP_LICENSE_INVALID_SIGNATURE 0xc0ea_0004
	ERROR_CLIP_KEYHOLDER_LICENSE_MISSING_OR_INVALID 0xc0ea_0005
	ERROR_CLIP_LICENSE_EXPIRED 0xc0ea_0006
	ERROR_CLIP_LICENSE_SIGNED_BY_UNKNOWN_SOURCE 0xc0ea_0007
	ERROR_CLIP_LICENSE_NOT_SIGNED 0xc0ea_0008
	ERROR_CLIP_LICENSE_HARDWARE_ID_OUT_OF_TOLERANCE 0xc0ea_0009
	ERROR_CLIP_LICENSE_DEVICE_ID_MISMATCH 0xc0ea_000a
	D3D10_ERROR_TOO_MANY_UNIQUE_STATE_OBJECTS 0x8879_0001
	D3D10_ERROR_FILE_NOT_FOUND 0x8879_0002
	D3D11_ERROR_TOO_MANY_UNIQUE_STATE_OBJECTS 0x887c_0001
	D3D11_ERROR_FILE_NOT_FOUND 0x887c_0002
	D3D11_ERROR_TOO_MANY_UNIQUE_VIEW_OBJECTS 0x887c_0003
	D3D11_ERROR_DEFERRED_CONTEXT_MAP_WITHOUT_INITIAL_DISCARD 0x887c_0004
	D3D12_ERROR_ADAPTER_NOT_FOUND 0x887e_0001
	D3D12_ERROR_DRIVER_VERSION_MISMATCH 0x887e_0002
	D2DERR_WRONG_STATE 0x8899_0001
	D2DERR_NOT_INITIALIZED 0x8899_0002
	D2DERR_UNSUPPORTED_OPERATION 0x8899_0003
	D2DERR_SCANNER_FAILED 0x8899_0004
	D2DERR_SCREEN_ACCESS_DENIED 0x8899_0005
	D2DERR_DISPLAY_STATE_INVALID 0x8899_0006
	D2DERR_ZERO_VECTOR 0x8899_0007
	D2DERR_INTERNAL_ERROR 0x8899_0008
	D2DERR_DISPLAY_FORMAT_NOT_SUPPORTED 0x8899_0009
	D2DERR_INVALID_CALL 0x8899_000a
	D2DERR_NO_HARDWARE_DEVICE 0x8899_000b
	D2DERR_RECREATE_TARGET 0x8899_000c
	D2DERR_TOO_MANY_SHADER_ELEMENTS 0x8899_000d
	D2DERR_SHADER_COMPILE_FAILED 0x8899_000e
	D2DERR_MAX_TEXTURE_SIZE_EXCEEDED 0x8899_000f
	D2DERR_UNSUPPORTED_VERSION 0x8899_0010
	D2DERR_BAD_NUMBER 0x8899_0011
	D2DERR_WRONG_FACTORY 0x8899_0012
	D2DERR_LAYER_ALREADY_IN_USE 0x8899_0013
	D2DERR_POP_CALL_DID_NOT_MATCH_PUSH 0x8899_0014
	D2DERR_WRONG_RESOURCE_DOMAIN 0x8899_0015
	D2DERR_PUSH_POP_UNBALANCED 0x8899_0016
	D2DERR_RENDER_TARGET_HAS_LAYER_OR_CLIPRECT 0x8899_0017
	D2DERR_INCOMPATIBLE_BRUSH_TYPES 0x8899_0018
	D2DERR_WIN32_ERROR 0x8899_0019
	D2DERR_TARGET_NOT_GDI_COMPATIBLE 0x8899_001a
	D2DERR_TEXT_EFFECT_IS_WRONG_TYPE 0x8899_001b
	D2DERR_TEXT_RENDERER_NOT_RELEASED 0x8899_001c
	D2DERR_EXCEEDS_MAX_BITMAP_SIZE 0x8899_001d
	D2DERR_INVALID_GRAPH_CONFIGURATION 0x8899_001e
	D2DERR_INVALID_INTERNAL_GRAPH_CONFIGURATION 0x8899_001f
	D2DERR_CYCLIC_GRAPH 0x8899_0020
	D2DERR_BITMAP_CANNOT_DRAW 0x8899_0021
	D2DERR_OUTSTANDING_BITMAP_REFERENCES 0x8899_0022
	D2DERR_ORIGINAL_TARGET_NOT_BOUND 0x8899_0023
	D2DERR_INVALID_TARGET 0x8899_0024
	D2DERR_BITMAP_BOUND_AS_TARGET 0x8899_0025
	D2DERR_INSUFFICIENT_DEVICE_CAPABILITIES 0x8899_0026
	D2DERR_INTERMEDIATE_TOO_LARGE 0x8899_0027
	D2DERR_EFFECT_IS_NOT_REGISTERED 0x8899_0028
	D2DERR_INVALID_PROPERTY 0x8899_0029
	D2DERR_NO_SUBPROPERTIES 0x8899_002a
	D2DERR_PRINT_JOB_CLOSED 0x8899_002b
	D2DERR_PRINT_FORMAT_NOT_SUPPORTED 0x8899_002c
	D2DERR_TOO_MANY_TRANSFORM_INPUTS 0x8899_002d
	D2DERR_INVALID_GLYPH_IMAGE 0x8899_002e
	DWRITE_E_FILEFORMAT 0x8898_5000
	DWRITE_E_UNEXPECTED 0x8898_5001
	DWRITE_E_NOFONT 0x8898_5002
	DWRITE_E_FILENOTFOUND 0x8898_5003
	DWRITE_E_FILEACCESS 0x8898_5004
	DWRITE_E_FONTCOLLECTIONOBSOLETE 0x8898_5005
	DWRITE_E_ALREADYREGISTERED 0x8898_5006
	DWRITE_E_CACHEFORMAT 0x8898_5007
	DWRITE_E_CACHEVERSION 0x8898_5008
	DWRITE_E_UNSUPPORTEDOPERATION 0x8898_5009
	DWRITE_E_TEXTRENDERERINCOMPATIBLE 0x8898_500a
	DWRITE_E_FLOWDIRECTIONCONFLICTS 0x8898_500b
	DWRITE_E_NOCOLOR 0x8898_500c
	DWRITE_E_REMOTEFONT 0x8898_500d
	DWRITE_E_DOWNLOADCANCELLED 0x8898_500e
	DWRITE_E_DOWNLOADFAILED 0x8898_500f
	DWRITE_E_TOOMANYDOWNLOADS 0x8898_5010
	WINCODEC_ERR_WRONGSTATE 0x8898_2f04
	WINCODEC_ERR_VALUEOUTOFRANGE 0x8898_2f05
	WINCODEC_ERR_UNKNOWNIMAGEFORMAT 0x8898_2f07
	WINCODEC_ERR_UNSUPPORTEDVERSION 0x8898_2f0b
	WINCODEC_ERR_NOTINITIALIZED 0x8898_2f0c
	WINCODEC_ERR_ALREADYLOCKED 0x8898_2f0d
	WINCODEC_ERR_PROPERTYNOTFOUND 0x8898_2f40
	WINCODEC_ERR_PROPERTYNOTSUPPORTED 0x8898_2f41
	WINCODEC_ERR_PROPERTYSIZE 0x8898_2f42
	WINCODEC_ERR_CODECPRESENT 0x8898_2f43
	WINCODEC_ERR_CODECNOTHUMBNAIL 0x8898_2f44
	WINCODEC_ERR_PALETTEUNAVAILABLE 0x8898_2f45
	WINCODEC_ERR_CODECTOOMANYSCANLINES 0x8898_2f46
	WINCODEC_ERR_INTERNALERROR 0x8898_2f48
	WINCODEC_ERR_SOURCERECTDOESNOTMATCHDIMENSIONS 0x8898_2f49
	WINCODEC_ERR_COMPONENTNOTFOUND 0x8898_2f50
	WINCODEC_ERR_IMAGESIZEOUTOFRANGE 0x8898_2f51
	WINCODEC_ERR_TOOMUCHMETADATA 0x8898_2f52
	WINCODEC_ERR_BADIMAGE 0x8898_2f60
	WINCODEC_ERR_BADHEADER 0x8898_2f61
	WINCODEC_ERR_FRAMEMISSING 0x8898_2f62
	WINCODEC_ERR_BADMETADATAHEADER 0x8898_2f63
	WINCODEC_ERR_BADSTREAMDATA 0x8898_2f70
	WINCODEC_ERR_STREAMWRITE 0x8898_2f71
	WINCODEC_ERR_STREAMREAD 0x8898_2f72
	WINCODEC_ERR_STREAMNOTAVAILABLE 0x8898_2f73
	WINCODEC_ERR_UNSUPPORTEDPIXELFORMAT 0x8898_2f80
	WINCODEC_ERR_UNSUPPORTEDOPERATION 0x8898_2f81
	WINCODEC_ERR_INVALIDREGISTRATION 0x8898_2f8a
	WINCODEC_ERR_COMPONENTINITIALIZEFAILURE 0x8898_2f8b
	WINCODEC_ERR_INSUFFICIENTBUFFER 0x8898_2f8c
	WINCODEC_ERR_DUPLICATEMETADATAPRESENT 0x8898_2f8d
	WINCODEC_ERR_PROPERTYUNEXPECTEDTYPE 0x8898_2f8e
	WINCODEC_ERR_UNEXPECTEDSIZE 0x8898_2f8f
	WINCODEC_ERR_INVALIDQUERYREQUEST 0x8898_2f90
	WINCODEC_ERR_UNEXPECTEDMETADATATYPE 0x8898_2f91
	WINCODEC_ERR_REQUESTONLYVALIDATMETADATAROOT 0x8898_2f92
	WINCODEC_ERR_INVALIDQUERYCHARACTER 0x8898_2f93
	WINCODEC_ERR_WIN32ERROR 0x8898_2f94
	WINCODEC_ERR_INVALIDPROGRESSIVELEVEL 0x8898_2f95
	WINCODEC_ERR_INVALIDJPEGSCANINDEX 0x8898_2f96
	MILERR_OBJECTBUSY 0x8898_0001
	MILERR_INSUFFICIENTBUFFER 0x8898_0002
	MILERR_WIN32ERROR 0x8898_0003
	MILERR_SCANNER_FAILED 0x8898_0004
	MILERR_SCREENACCESSDENIED 0x8898_0005
	MILERR_DISPLAYSTATEINVALID 0x8898_0006
	MILERR_NONINVERTIBLEMATRIX 0x8898_0007
	MILERR_ZEROVECTOR 0x8898_0008
	MILERR_TERMINATED 0x8898_0009
	MILERR_BADNUMBER 0x8898_000a
	MILERR_INTERNALERROR 0x8898_0080
	MILERR_DISPLAYFORMATNOTSUPPORTED 0x8898_0084
	MILERR_INVALIDCALL 0x8898_0085
	MILERR_ALREADYLOCKED 0x8898_0086
	MILERR_NOTLOCKED 0x8898_0087
	MILERR_DEVICECANNOTRENDERTEXT 0x8898_0088
	MILERR_GLYPHBITMAPMISSED 0x8898_0089
	MILERR_MALFORMEDGLYPHCACHE 0x8898_008a
	MILERR_GENERIC_IGNORE 0x8898_008b
	MILERR_MALFORMED_GUIDELINE_DATA 0x8898_008c
	MILERR_NO_HARDWARE_DEVICE 0x8898_008d
	MILERR_NEED_RECREATE_AND_PRESENT 0x8898_008e
	MILERR_ALREADY_INITIALIZED 0x8898_008f
	MILERR_MISMATCHED_SIZE 0x8898_0090
	MILERR_NO_REDIRECTION_SURFACE_AVAILABLE 0x8898_0091
	MILERR_REMOTING_NOT_SUPPORTED 0x8898_0092
	MILERR_QUEUED_PRESENT_NOT_SUPPORTED 0x8898_0093
	MILERR_NOT_QUEUING_PRESENTS 0x8898_0094
	MILERR_NO_REDIRECTION_SURFACE_RETRY_LATER 0x8898_0095
	MILERR_TOOMANYSHADERELEMNTS 0x8898_0096
	MILERR_MROW_READLOCK_FAILED 0x8898_0097
	MILERR_MROW_UPDATE_FAILED 0x8898_0098
	MILERR_SHADER_COMPILE_FAILED 0x8898_0099
	MILERR_MAX_TEXTURE_SIZE_EXCEEDED 0x8898_009a
	MILERR_QPC_TIME_WENT_BACKWARD 0x8898_009b
	MILERR_DXGI_ENUMERATION_OUT_OF_SYNC 0x8898_009d
	MILERR_ADAPTER_NOT_FOUND 0x8898_009e
	MILERR_COLORSPACE_NOT_SUPPORTED 0x8898_009f
	MILERR_PREFILTER_NOT_SUPPORTED 0x8898_00a0
	MILERR_DISPLAYID_ACCESS_DENIED 0x8898_00a1
	UCEERR_INVALIDPACKETHEADER 0x8898_0400
	UCEERR_UNKNOWNPACKET 0x8898_0401
	UCEERR_ILLEGALPACKET 0x8898_0402
	UCEERR_MALFORMEDPACKET 0x8898_0403
	UCEERR_ILLEGALHANDLE 0x8898_0404
	UCEERR_HANDLELOOKUPFAILED 0x8898_0405
	UCEERR_RENDERTHREADFAILURE 0x8898_0406
	UCEERR_CTXSTACKFRSTTARGETNULL 0x8898_0407
	UCEERR_CONNECTIONIDLOOKUPFAILED 0x8898_0408
	UCEERR_BLOCKSFULL 0x8898_0409
	UCEERR_MEMORYFAILURE 0x8898_040a
	UCEERR_PACKETRECORDOUTOFRANGE 0x8898_040b
	UCEERR_ILLEGALRECORDTYPE 0x8898_040c
	UCEERR_OUTOFHANDLES 0x8898_040d
	UCEERR_UNCHANGABLE_UPDATE_ATTEMPTED 0x8898_040e
	UCEERR_NO_MULTIPLE_WORKER_THREADS 0x8898_040f
	UCEERR_REMOTINGNOTSUPPORTED 0x8898_0410
	UCEERR_MISSINGENDCOMMAND 0x8898_0411
	UCEERR_MISSINGBEGINCOMMAND 0x8898_0412
	UCEERR_CHANNELSYNCTIMEDOUT 0x8898_0413
	UCEERR_CHANNELSYNCABANDONED 0x8898_0414
	UCEERR_UNSUPPORTEDTRANSPORTVERSION 0x8898_0415
	UCEERR_TRANSPORTUNAVAILABLE 0x8898_0416
	UCEERR_FEEDBACK_UNSUPPORTED 0x8898_0417
	UCEERR_COMMANDTRANSPORTDENIED 0x8898_0418
	UCEERR_GRAPHICSSTREAMUNAVAILABLE 0x8898_0419
	UCEERR_GRAPHICSSTREAMALREADYOPEN 0x8898_0420
	UCEERR_TRANSPORTDISCONNECTED 0x8898_0421
	UCEERR_TRANSPORTOVERLOADED 0x8898_0422
	UCEERR_PARTITION_ZOMBIED 0x8898_0423
	MILAVERR_NOCLOCK 0x8898_0500
	MILAVERR_NOMEDIATYPE 0x8898_0501
	MILAVERR_NOVIDEOMIXER 0x8898_0502
	MILAVERR_NOVIDEOPRESENTER 0x8898_0503
	MILAVERR_NOREADYFRAMES 0x8898_0504
	MILAVERR_MODULENOTLOADED 0x8898_0505
	MILAVERR_WMPFACTORYNOTREGISTERED 0x8898_0506
	MILAVERR_INVALIDWMPVERSION 0x8898_0507
	MILAVERR_INSUFFICIENTVIDEORESOURCES 0x8898_0508
	MILAVERR_VIDEOACCELERATIONNOTAVAILABLE 0x8898_0509
	MILAVERR_REQUESTEDTEXTURETOOBIG 0x8898_050a
	MILAVERR_SEEKFAILED 0x8898_050b
	MILAVERR_UNEXPECTEDWMPFAILURE 0x8898_050c
	MILAVERR_MEDIAPLAYERCLOSED 0x8898_050d
	MILAVERR_UNKNOWNHARDWAREERROR 0x8898_050e
	MILEFFECTSERR_UNKNOWNPROPERTY 0x8898_060e
	MILEFFECTSERR_EFFECTNOTPARTOFGROUP 0x8898_060f
	MILEFFECTSERR_NOINPUTSOURCEATTACHED 0x8898_0610
	MILEFFECTSERR_CONNECTORNOTCONNECTED 0x8898_0611
	MILEFFECTSERR_CONNECTORNOTASSOCIATEDWITHEFFECT 0x8898_0612
	MILEFFECTSERR_RESERVED 0x8898_0613
	MILEFFECTSERR_CYCLEDETECTED 0x8898_0614
	MILEFFECTSERR_EFFECTINMORETHANONEGRAPH 0x8898_0615
	MILEFFECTSERR_EFFECTALREADYINAGRAPH 0x8898_0616
	MILEFFECTSERR_EFFECTHASNOCHILDREN 0x8898_0617
	MILEFFECTSERR_ALREADYATTACHEDTOLISTENER 0x8898_0618
	MILEFFECTSERR_NOTAFFINETRANSFORM 0x8898_0619
	MILEFFECTSERR_EMPTYBOUNDS 0x8898_061a
	MILEFFECTSERR_OUTPUTSIZETOOLARGE 0x8898_061b
	DWMERR_STATE_TRANSITION_FAILED 0x8898_0700
	DWMERR_THEME_FAILED 0x8898_0701
	DWMERR_CATASTROPHIC_FAILURE 0x8898_0702
	DCOMPOSITION_ERROR_WINDOW_ALREADY_COMPOSED 0x8898_0800
	DCOMPOSITION_ERROR_SURFACE_BEING_RENDERED 0x8898_0801
	DCOMPOSITION_ERROR_SURFACE_NOT_BEING_RENDERED 0x8898_0802
	ONL_E_INVALID_AUTHENTICATION_TARGET 0x8086_0001
	ONL_E_ACCESS_DENIED_BY_TOU 0x8086_0002
	ONL_E_INVALID_APPLICATION 0x8086_0003
	ONL_E_PASSWORD_UPDATE_REQUIRED 0x8086_0004
	ONL_E_ACCOUNT_UPDATE_REQUIRED 0x8086_0005
	ONL_E_FORCESIGNIN 0x8086_0006
	ONL_E_ACCOUNT_LOCKED 0x8086_0007
	ONL_E_PARENTAL_CONSENT_REQUIRED 0x8086_0008
	ONL_E_EMAIL_VERIFICATION_REQUIRED 0x8086_0009
	ONL_E_ACCOUNT_SUSPENDED_COMPROIMISE 0x8086_000a
	ONL_E_ACCOUNT_SUSPENDED_ABUSE 0x8086_000b
	ONL_E_ACTION_REQUIRED 0x8086_000c
	ONL_CONNECTION_COUNT_LIMIT 0x8086_000d
	ONL_E_CONNECTED_ACCOUNT_CAN_NOT_SIGNOUT 0x8086_000e
	ONL_E_USER_AUTHENTICATION_REQUIRED 0x8086_000f
	ONL_E_REQUEST_THROTTLED 0x8086_0010
	FA_E_MAX_PERSISTED_ITEMS_REACHED 0x8027_0220
	FA_E_HOMEGROUP_NOT_AVAILABLE 0x8027_0222
	E_MONITOR_RESOLUTION_TOO_LOW 0x8027_0250
	E_ELEVATED_ACTIVATION_NOT_SUPPORTED 0x8027_0251
	E_UAC_DISABLED 0x8027_0252
	E_FULL_ADMIN_NOT_SUPPORTED 0x8027_0253
	E_APPLICATION_NOT_REGISTERED 0x8027_0254
	E_MULTIPLE_EXTENSIONS_FOR_APPLICATION 0x8027_0255
	E_MULTIPLE_PACKAGES_FOR_FAMILY 0x8027_0256
	E_APPLICATION_MANAGER_NOT_RUNNING 0x8027_0257
	S_STORE_LAUNCHED_FOR_REMEDIATION 0x0027_0258
	S_APPLICATION_ACTIVATION_ERROR_HANDLED_BY_DIALOG 0x0027_0259
	E_APPLICATION_ACTIVATION_TIMED_OUT 0x8027_025a
	E_APPLICATION_ACTIVATION_EXEC_FAILURE 0x8027_025b
	E_APPLICATION_TEMPORARY_LICENSE_ERROR 0x8027_025c
	E_APPLICATION_TRIAL_LICENSE_EXPIRED 0x8027_025d
	E_SKYDRIVE_ROOT_TARGET_FILE_SYSTEM_NOT_SUPPORTED 0x8027_0260
	E_SKYDRIVE_ROOT_TARGET_OVERLAP 0x8027_0261
	E_SKYDRIVE_ROOT_TARGET_CANNOT_INDEX 0x8027_0262
	E_SKYDRIVE_FILE_NOT_UPLOADED 0x8027_0263
	E_SKYDRIVE_UPDATE_AVAILABILITY_FAIL 0x8027_0264
	E_SKYDRIVE_ROOT_TARGET_VOLUME_ROOT_NOT_SUPPORTED 0x8027_0265
	E_SYNCENGINE_FILE_SIZE_OVER_LIMIT 0x8802_b001
	E_SYNCENGINE_FILE_SIZE_EXCEEDS_REMAINING_QUOTA 0x8802_b002
	E_SYNCENGINE_UNSUPPORTED_FILE_NAME 0x8802_b003
	E_SYNCENGINE_FOLDER_ITEM_COUNT_LIMIT_EXCEEDED 0x8802_b004
	E_SYNCENGINE_FILE_SYNC_PARTNER_ERROR 0x8802_b005
	E_SYNCENGINE_SYNC_PAUSED_BY_SERVICE 0x8802_b006
	E_SYNCENGINE_FILE_IDENTIFIER_UNKNOWN 0x8802_c002
	E_SYNCENGINE_SERVICE_AUTHENTICATION_FAILED 0x8802_c003
	E_SYNCENGINE_UNKNOWN_SERVICE_ERROR 0x8802_c004
	E_SYNCENGINE_SERVICE_RETURNED_UNEXPECTED_SIZE 0x8802_c005
	E_SYNCENGINE_REQUEST_BLOCKED_BY_SERVICE 0x8802_c006
	E_SYNCENGINE_REQUEST_BLOCKED_DUE_TO_CLIENT_ERROR 0x8802_c007
	E_SYNCENGINE_FOLDER_INACCESSIBLE 0x8802_d001
	E_SYNCENGINE_UNSUPPORTED_FOLDER_NAME 0x8802_d002
	E_SYNCENGINE_UNSUPPORTED_MARKET 0x8802_d003
	E_SYNCENGINE_PATH_LENGTH_LIMIT_EXCEEDED 0x8802_d004
	E_SYNCENGINE_REMOTE_PATH_LENGTH_LIMIT_EXCEEDED 0x8802_d005
	E_SYNCENGINE_CLIENT_UPDATE_NEEDED 0x8802_d006
	E_SYNCENGINE_PROXY_AUTHENTICATION_REQUIRED 0x8802_d007
	E_SYNCENGINE_STORAGE_SERVICE_PROVISIONING_FAILED 0x8802_d008
	E_SYNCENGINE_UNSUPPORTED_REPARSE_POINT 0x8802_d009
	E_SYNCENGINE_STORAGE_SERVICE_BLOCKED 0x8802_d00a
	E_SYNCENGINE_FOLDER_IN_REDIRECTION 0x8802_d00b
	EAS_E_POLICY_NOT_MANAGED_BY_OS 0x8055_0001
	EAS_E_POLICY_COMPLIANT_WITH_ACTIONS 0x8055_0002
	EAS_E_REQUESTED_POLICY_NOT_ENFORCEABLE 0x8055_0003
	EAS_E_CURRENT_USER_HAS_BLANK_PASSWORD 0x8055_0004
	EAS_E_REQUESTED_POLICY_PASSWORD_EXPIRATION_INCOMPATIBLE 0x8055_0005
	EAS_E_USER_CANNOT_CHANGE_PASSWORD 0x8055_0006
	EAS_E_ADMINS_HAVE_BLANK_PASSWORD 0x8055_0007
	EAS_E_ADMINS_CANNOT_CHANGE_PASSWORD 0x8055_0008
	EAS_E_LOCAL_CONTROLLED_USERS_CANNOT_CHANGE_PASSWORD 0x8055_0009
	EAS_E_PASSWORD_POLICY_NOT_ENFORCEABLE_FOR_CONNECTED_ADMINS 0x8055_000a
	EAS_E_CONNECTED_ADMINS_NEED_TO_CHANGE_PASSWORD 0x8055_000b
	EAS_E_PASSWORD_POLICY_NOT_ENFORCEABLE_FOR_CURRENT_CONNECTED_USER 0x8055_000c
	EAS_E_CURRENT_CONNECTED_USER_NEED_TO_CHANGE_PASSWORD 0x8055_000d
	WEB_E_UNSUPPORTED_FORMAT 0x8375_0001
	WEB_E_INVALID_XML 0x8375_0002
	WEB_E_MISSING_REQUIRED_ELEMENT 0x8375_0003
	WEB_E_MISSING_REQUIRED_ATTRIBUTE 0x8375_0004
	WEB_E_UNEXPECTED_CONTENT 0x8375_0005
	WEB_E_RESOURCE_TOO_LARGE 0x8375_0006
	WEB_E_INVALID_JSON_STRING 0x8375_0007
	WEB_E_INVALID_JSON_NUMBER 0x8375_0008
	WEB_E_JSON_VALUE_NOT_FOUND 0x8375_0009
	HTTP_E_STATUS_UNEXPECTED 0x8019_0001
	HTTP_E_STATUS_UNEXPECTED_REDIRECTION 0x8019_0003
	HTTP_E_STATUS_UNEXPECTED_CLIENT_ERROR 0x8019_0004
	HTTP_E_STATUS_UNEXPECTED_SERVER_ERROR 0x8019_0005
	HTTP_E_STATUS_AMBIGUOUS 0x8019_012c
	HTTP_E_STATUS_MOVED 0x8019_012d
	HTTP_E_STATUS_REDIRECT 0x8019_012e
	HTTP_E_STATUS_REDIRECT_METHOD 0x8019_012f
	HTTP_E_STATUS_NOT_MODIFIED 0x8019_0130
	HTTP_E_STATUS_USE_PROXY 0x8019_0131
	HTTP_E_STATUS_REDIRECT_KEEP_VERB 0x8019_0133
	HTTP_E_STATUS_BAD_REQUEST 0x8019_0190
	HTTP_E_STATUS_DENIED 0x8019_0191
	HTTP_E_STATUS_PAYMENT_REQ 0x8019_0192
	HTTP_E_STATUS_FORBIDDEN 0x8019_0193
	HTTP_E_STATUS_NOT_FOUND 0x8019_0194
	HTTP_E_STATUS_BAD_METHOD 0x8019_0195
	HTTP_E_STATUS_NONE_ACCEPTABLE 0x8019_0196
	HTTP_E_STATUS_PROXY_AUTH_REQ 0x8019_0197
	HTTP_E_STATUS_REQUEST_TIMEOUT 0x8019_0198
	HTTP_E_STATUS_CONFLICT 0x8019_0199
	HTTP_E_STATUS_GONE 0x8019_019a
	HTTP_E_STATUS_LENGTH_REQUIRED 0x8019_019b
	HTTP_E_STATUS_PRECOND_FAILED 0x8019_019c
	HTTP_E_STATUS_REQUEST_TOO_LARGE 0x8019_019d
	HTTP_E_STATUS_URI_TOO_LONG 0x8019_019e
	HTTP_E_STATUS_UNSUPPORTED_MEDIA 0x8019_019f
	HTTP_E_STATUS_RANGE_NOT_SATISFIABLE 0x8019_01a0
	HTTP_E_STATUS_EXPECTATION_FAILED 0x8019_01a1
	HTTP_E_STATUS_SERVER_ERROR 0x8019_01f4
	HTTP_E_STATUS_NOT_SUPPORTED 0x8019_01f5
	HTTP_E_STATUS_BAD_GATEWAY 0x8019_01f6
	HTTP_E_STATUS_SERVICE_UNAVAIL 0x8019_01f7
	HTTP_E_STATUS_GATEWAY_TIMEOUT 0x8019_01f8
	HTTP_E_STATUS_VERSION_NOT_SUP 0x8019_01f9
	E_INVALID_PROTOCOL_OPERATION 0x8376_0001
	E_INVALID_PROTOCOL_FORMAT 0x8376_0002
	E_PROTOCOL_EXTENSIONS_NOT_SUPPORTED 0x8376_0003
	E_SUBPROTOCOL_NOT_SUPPORTED 0x8376_0004
	E_PROTOCOL_VERSION_NOT_SUPPORTED 0x8376_0005
	INPUT_E_OUT_OF_ORDER 0x8040_0000
	INPUT_E_REENTRANCY 0x8040_0001
	INPUT_E_MULTIMODAL 0x8040_0002
	INPUT_E_PACKET 0x8040_0003
	INPUT_E_FRAME 0x8040_0004
	INPUT_E_HISTORY 0x8040_0005
	INPUT_E_DEVICE_INFO 0x8040_0006
	INPUT_E_TRANSFORM 0x8040_0007
	INPUT_E_DEVICE_PROPERTY 0x8040_0008
	INET_E_INVALID_URL 0x800c_0002
	INET_E_NO_SESSION 0x800c_0003
	INET_E_CANNOT_CONNECT 0x800c_0004
	INET_E_RESOURCE_NOT_FOUND 0x800c_0005
	INET_E_OBJECT_NOT_FOUND 0x800c_0006
	INET_E_DATA_NOT_AVAILABLE 0x800c_0007
	INET_E_DOWNLOAD_FAILURE 0x800c_0008
	INET_E_AUTHENTICATION_REQUIRED 0x800c_0009
	INET_E_NO_VALID_MEDIA 0x800c_000a
	INET_E_CONNECTION_TIMEOUT 0x800c_000b
	INET_E_INVALID_REQUEST 0x800c_000c
	INET_E_UNKNOWN_PROTOCOL 0x800c_000d
	INET_E_SECURITY_PROBLEM 0x800c_000e
	INET_E_CANNOT_LOAD_DATA 0x800c_000f
	INET_E_CANNOT_INSTANTIATE_OBJECT 0x800c_0010
	INET_E_INVALID_CERTIFICATE 0x800c_0019
	INET_E_REDIRECT_FAILED 0x800c_0014
	INET_E_REDIRECT_TO_DIR 0x800c_0015
	ERROR_DBG_CREATE_PROCESS_FAILURE_LOCKDOWN 0x80b0_0001
	ERROR_DBG_ATTACH_PROCESS_FAILURE_LOCKDOWN 0x80b0_0002
	ERROR_DBG_CONNECT_SERVER_FAILURE_LOCKDOWN 0x80b0_0003
	ERROR_DBG_START_SERVER_FAILURE_LOCKDOWN 0x80b0_0004
	ERROR_IO_PREEMPTED 0x8901_0001
	JSCRIPT_E_CANTEXECUTE 0x8902_0001
	WEP_E_NOT_PROVISIONED_ON_ALL_VOLUMES 0x8801_0001
	WEP_E_FIXED_DATA_NOT_SUPPORTED 0x8801_0002
	WEP_E_HARDWARE_NOT_COMPLIANT 0x8801_0003
	WEP_E_LOCK_NOT_CONFIGURED 0x8801_0004
	WEP_E_PROTECTION_SUSPENDED 0x8801_0005
	WEP_E_NO_LICENSE 0x8801_0006
	WEP_E_OS_NOT_PROTECTED 0x8801_0007
	WEP_E_UNEXPECTED_FAIL 0x8801_0008
	WEP_E_BUFFER_TOO_LARGE 0x8801_0009
	ERROR_SVHDX_ERROR_STORED 0xc05c_0000
	ERROR_SVHDX_ERROR_NOT_AVAILABLE 0xc05c_ff00
	ERROR_SVHDX_UNIT_ATTENTION_AVAILABLE 0xc05c_ff01
	ERROR_SVHDX_UNIT_ATTENTION_CAPACITY_DATA_CHANGED 0xc05c_ff02
	ERROR_SVHDX_UNIT_ATTENTION_RESERVATIONS_PREEMPTED 0xc05c_ff03
	ERROR_SVHDX_UNIT_ATTENTION_RESERVATIONS_RELEASED 0xc05c_ff04
	ERROR_SVHDX_UNIT_ATTENTION_REGISTRATIONS_PREEMPTED 0xc05c_ff05
	ERROR_SVHDX_UNIT_ATTENTION_OPERATING_DEFINITION_CHANGED 0xc05c_ff06
	ERROR_SVHDX_RESERVATION_CONFLICT 0xc05c_ff07
	ERROR_SVHDX_WRONG_FILE_TYPE 0xc05c_ff08
	ERROR_SVHDX_VERSION_MISMATCH 0xc05c_ff09
	ERROR_VHD_SHARED 0xc05c_ff0a
	ERROR_SVHDX_NO_INITIATOR 0xc05c_ff0b
	ERROR_VHDSET_BACKING_STORAGE_NOT_FOUND 0xc05c_ff0c
	ERROR_SMB_NO_PREAUTH_INTEGRITY_HASH_OVERLAP 0xc05d_0000
	ERROR_SMB_BAD_CLUSTER_DIALECT 0xc05d_0001
	WININET_E_OUT_OF_HANDLES 0x8007_2ee1
	WININET_E_TIMEOUT 0x8007_2ee2
	WININET_E_EXTENDED_ERROR 0x8007_2ee3
	WININET_E_INTERNAL_ERROR 0x8007_2ee4
	WININET_E_INVALID_URL 0x8007_2ee5
	WININET_E_UNRECOGNIZED_SCHEME 0x8007_2ee6
	WININET_E_NAME_NOT_RESOLVED 0x8007_2ee7
	WININET_E_PROTOCOL_NOT_FOUND 0x8007_2ee8
	WININET_E_INVALID_OPTION 0x8007_2ee9
	WININET_E_BAD_OPTION_LENGTH 0x8007_2eea
	WININET_E_OPTION_NOT_SETTABLE 0x8007_2eeb
	WININET_E_SHUTDOWN 0x8007_2eec
	WININET_E_INCORRECT_USER_NAME 0x8007_2eed
	WININET_E_INCORRECT_PASSWORD 0x8007_2eee
	WININET_E_LOGIN_FAILURE 0x8007_2eef
	WININET_E_INVALID_OPERATION 0x8007_2ef0
	WININET_E_OPERATION_CANCELLED 0x8007_2ef1
	WININET_E_INCORRECT_HANDLE_TYPE 0x8007_2ef2
	WININET_E_INCORRECT_HANDLE_STATE 0x8007_2ef3
	WININET_E_NOT_PROXY_REQUEST 0x8007_2ef4
	WININET_E_REGISTRY_VALUE_NOT_FOUND 0x8007_2ef5
	WININET_E_BAD_REGISTRY_PARAMETER 0x8007_2ef6
	WININET_E_NO_DIRECT_ACCESS 0x8007_2ef7
	WININET_E_NO_CONTEXT 0x8007_2ef8
	WININET_E_NO_CALLBACK 0x8007_2ef9
	WININET_E_REQUEST_PENDING 0x8007_2efa
	WININET_E_INCORRECT_FORMAT 0x8007_2efb
	WININET_E_ITEM_NOT_FOUND 0x8007_2efc
	WININET_E_CANNOT_CONNECT 0x8007_2efd
	WININET_E_CONNECTION_ABORTED 0x8007_2efe
	WININET_E_CONNECTION_RESET 0x8007_2eff
	WININET_E_FORCE_RETRY 0x8007_2f00
	WININET_E_INVALID_PROXY_REQUEST 0x8007_2f01
	WININET_E_NEED_UI 0x8007_2f02
	WININET_E_HANDLE_EXISTS 0x8007_2f04
	WININET_E_SEC_CERT_DATE_INVALID 0x8007_2f05
	WININET_E_SEC_CERT_CN_INVALID 0x8007_2f06
	WININET_E_HTTP_TO_HTTPS_ON_REDIR 0x8007_2f07
	WININET_E_HTTPS_TO_HTTP_ON_REDIR 0x8007_2f08
	WININET_E_MIXED_SECURITY 0x8007_2f09
	WININET_E_CHG_POST_IS_NON_SECURE 0x8007_2f0a
	WININET_E_POST_IS_NON_SECURE 0x8007_2f0b
	WININET_E_CLIENT_AUTH_CERT_NEEDED 0x8007_2f0c
	WININET_E_INVALID_CA 0x8007_2f0d
	WININET_E_CLIENT_AUTH_NOT_SETUP 0x8007_2f0e
	WININET_E_ASYNC_THREAD_FAILED 0x8007_2f0f
	WININET_E_REDIRECT_SCHEME_CHANGE 0x8007_2f10
	WININET_E_DIALOG_PENDING 0x8007_2f11
	WININET_E_RETRY_DIALOG 0x8007_2f12
	WININET_E_NO_NEW_CONTAINERS 0x8007_2f13
	WININET_E_HTTPS_HTTP_SUBMIT_REDIR 0x8007_2f14
	WININET_E_SEC_CERT_ERRORS 0x8007_2f17
	WININET_E_SEC_CERT_REV_FAILED 0x8007_2f19
	WININET_E_HEADER_NOT_FOUND 0x8007_2f76
	WININET_E_DOWNLEVEL_SERVER 0x8007_2f77
	WININET_E_INVALID_SERVER_RESPONSE 0x8007_2f78
	WININET_E_INVALID_HEADER 0x8007_2f79
	WININET_E_INVALID_QUERY_REQUEST 0x8007_2f7a
	WININET_E_HEADER_ALREADY_EXISTS 0x8007_2f7b
	WININET_E_REDIRECT_FAILED 0x8007_2f7c
	WININET_E_SECURITY_CHANNEL_ERROR 0x8007_2f7d
	WININET_E_UNABLE_TO_CACHE_FILE 0x8007_2f7e
	WININET_E_TCPIP_NOT_INSTALLED 0x8007_2f7f
	WININET_E_DISCONNECTED 0x8007_2f83
	WININET_E_SERVER_UNREACHABLE 0x8007_2f84
	WININET_E_PROXY_SERVER_UNREACHABLE 0x8007_2f85
	WININET_E_BAD_AUTO_PROXY_SCRIPT 0x8007_2f86
	WININET_E_UNABLE_TO_DOWNLOAD_SCRIPT 0x8007_2f87
	WININET_E_SEC_INVALID_CERT 0x8007_2f89
	WININET_E_SEC_CERT_REVOKED 0x8007_2f8a
	WININET_E_FAILED_DUETOSECURITYCHECK 0x8007_2f8b
	WININET_E_NOT_INITIALIZED 0x8007_2f8c
	WININET_E_LOGIN_FAILURE_DISPLAY_ENTITY_BODY 0x8007_2f8e
	WININET_E_DECODING_FAILED 0x8007_2f8f
	WININET_E_NOT_REDIRECTED 0x8007_2f80
	WININET_E_COOKIE_NEEDS_CONFIRMATION 0x8007_2f81
	WININET_E_COOKIE_DECLINED 0x8007_2f82
	WININET_E_REDIRECT_NEEDS_CONFIRMATION 0x8007_2f88
	SQLITE_E_ERROR 0x87af_0001
	SQLITE_E_INTERNAL 0x87af_0002
	SQLITE_E_PERM 0x87af_0003
	SQLITE_E_ABORT 0x87af_0004
	SQLITE_E_BUSY 0x87af_0005
	SQLITE_E_LOCKED 0x87af_0006
	SQLITE_E_NOMEM 0x87af_0007
	SQLITE_E_READONLY 0x87af_0008
	SQLITE_E_INTERRUPT 0x87af_0009
	SQLITE_E_IOERR 0x87af_000a
	SQLITE_E_CORRUPT 0x87af_000b
	SQLITE_E_NOTFOUND 0x87af_000c
	SQLITE_E_FULL 0x87af_000d
	SQLITE_E_CANTOPEN 0x87af_000e
	SQLITE_E_PROTOCOL 0x87af_000f
	SQLITE_E_EMPTY 0x87af_0010
	SQLITE_E_SCHEMA 0x87af_0011
	SQLITE_E_TOOBIG 0x87af_0012
	SQLITE_E_CONSTRAINT 0x87af_0013
	SQLITE_E_MISMATCH 0x87af_0014
	SQLITE_E_MISUSE 0x87af_0015
	SQLITE_E_NOLFS 0x87af_0016
	SQLITE_E_AUTH 0x87af_0017
	SQLITE_E_FORMAT 0x87af_0018
	SQLITE_E_RANGE 0x87af_0019
	SQLITE_E_NOTADB 0x87af_001a
	SQLITE_E_NOTICE 0x87af_001b
	SQLITE_E_WARNING 0x87af_001c
	SQLITE_E_ROW 0x87af_0064
	SQLITE_E_DONE 0x87af_0065
	SQLITE_E_IOERR_READ 0x87af_010a
	SQLITE_E_IOERR_SHORT_READ 0x87af_020a
	SQLITE_E_IOERR_WRITE 0x87af_030a
	SQLITE_E_IOERR_FSYNC 0x87af_040a
	SQLITE_E_IOERR_DIR_FSYNC 0x87af_050a
	SQLITE_E_IOERR_TRUNCATE 0x87af_060a
	SQLITE_E_IOERR_FSTAT 0x87af_070a
	SQLITE_E_IOERR_UNLOCK 0x87af_080a
	SQLITE_E_IOERR_RDLOCK 0x87af_090a
	SQLITE_E_IOERR_DELETE 0x87af_0a0a
	SQLITE_E_IOERR_BLOCKED 0x87af_0b0a
	SQLITE_E_IOERR_NOMEM 0x87af_0c0a
	SQLITE_E_IOERR_ACCESS 0x87af_0d0a
	SQLITE_E_IOERR_CHECKRESERVEDLOCK 0x87af_0e0a
	SQLITE_E_IOERR_LOCK 0x87af_0f0a
	SQLITE_E_IOERR_CLOSE 0x87af_100a
	SQLITE_E_IOERR_DIR_CLOSE 0x87af_110a
	SQLITE_E_IOERR_SHMOPEN 0x87af_120a
	SQLITE_E_IOERR_SHMSIZE 0x87af_130a
	SQLITE_E_IOERR_SHMLOCK 0x87af_140a
	SQLITE_E_IOERR_SHMMAP 0x87af_150a
	SQLITE_E_IOERR_SEEK 0x87af_160a
	SQLITE_E_IOERR_DELETE_NOENT 0x87af_170a
	SQLITE_E_IOERR_MMAP 0x87af_180a
	SQLITE_E_IOERR_GETTEMPPATH 0x87af_190a
	SQLITE_E_IOERR_CONVPATH 0x87af_1a0a
	SQLITE_E_IOERR_VNODE 0x87af_1a02
	SQLITE_E_IOERR_AUTH 0x87af_1a03
	SQLITE_E_LOCKED_SHAREDCACHE 0x87af_0106
	SQLITE_E_BUSY_RECOVERY 0x87af_0105
	SQLITE_E_BUSY_SNAPSHOT 0x87af_0205
	SQLITE_E_CANTOPEN_NOTEMPDIR 0x87af_010e
	SQLITE_E_CANTOPEN_ISDIR 0x87af_020e
	SQLITE_E_CANTOPEN_FULLPATH 0x87af_030e
	SQLITE_E_CANTOPEN_CONVPATH 0x87af_040e
	SQLITE_E_CORRUPT_VTAB 0x87af_010b
	SQLITE_E_READONLY_RECOVERY 0x87af_0108
	SQLITE_E_READONLY_CANTLOCK 0x87af_0208
	SQLITE_E_READONLY_ROLLBACK 0x87af_0308
	SQLITE_E_READONLY_DBMOVED 0x87af_0408
	SQLITE_E_ABORT_ROLLBACK 0x87af_0204
	SQLITE_E_CONSTRAINT_CHECK 0x87af_0113
	SQLITE_E_CONSTRAINT_COMMITHOOK 0x87af_0213
	SQLITE_E_CONSTRAINT_FOREIGNKEY 0x87af_0313
	SQLITE_E_CONSTRAINT_FUNCTION 0x87af_0413
	SQLITE_E_CONSTRAINT_NOTNULL 0x87af_0513
	SQLITE_E_CONSTRAINT_PRIMARYKEY 0x87af_0613
	SQLITE_E_CONSTRAINT_TRIGGER 0x87af_0713
	SQLITE_E_CONSTRAINT_UNIQUE 0x87af_0813
	SQLITE_E_CONSTRAINT_VTAB 0x87af_0913
	SQLITE_E_CONSTRAINT_ROWID 0x87af_0a13
	SQLITE_E_NOTICE_RECOVER_WAL 0x87af_011b
	SQLITE_E_NOTICE_RECOVER_ROLLBACK 0x87af_021b
	SQLITE_E_WARNING_AUTOINDEX 0x87af_011c
	UTC_E_TOGGLE_TRACE_STARTED 0x87c5_1001
	UTC_E_ALTERNATIVE_TRACE_CANNOT_PREEMPT 0x87c5_1002
	UTC_E_AOT_NOT_RUNNING 0x87c5_1003
	UTC_E_SCRIPT_TYPE_INVALID 0x87c5_1004
	UTC_E_SCENARIODEF_NOT_FOUND 0x87c5_1005
	UTC_E_TRACEPROFILE_NOT_FOUND 0x87c5_1006
	UTC_E_FORWARDER_ALREADY_ENABLED 0x87c5_1007
	UTC_E_FORWARDER_ALREADY_DISABLED 0x87c5_1008
	UTC_E_EVENTLOG_ENTRY_MALFORMED 0x87c5_1009
	UTC_E_DIAGRULES_SCHEMAVERSION_MISMATCH 0x87c5_100a
	UTC_E_SCRIPT_TERMINATED 0x87c5_100b
	UTC_E_INVALID_CUSTOM_FILTER 0x87c5_100c
	UTC_E_TRACE_NOT_RUNNING 0x87c5_100d
	UTC_E_REESCALATED_TOO_QUICKLY 0x87c5_100e
	UTC_E_ESCALATION_ALREADY_RUNNING 0x87c5_100f
	UTC_E_PERFTRACK_ALREADY_TRACING 0x87c5_1010
	UTC_E_REACHED_MAX_ESCALATIONS 0x87c5_1011
	UTC_E_FORWARDER_PRODUCER_MISMATCH 0x87c5_1012
	UTC_E_INTENTIONAL_SCRIPT_FAILURE 0x87c5_1013
	UTC_E_SQM_INIT_FAILED 0x87c5_1014
	UTC_E_NO_WER_LOGGER_SUPPORTED 0x87c5_1015
	UTC_E_TRACERS_DONT_EXIST 0x87c5_1016
	UTC_E_WINRT_INIT_FAILED 0x87c5_1017
	UTC_E_SCENARIODEF_SCHEMAVERSION_MISMATCH 0x87c5_1018
	UTC_E_INVALID_FILTER 0x87c5_1019
	UTC_E_EXE_TERMINATED 0x87c5_101a
	UTC_E_ESCALATION_NOT_AUTHORIZED 0x87c5_101b
	UTC_E_SETUP_NOT_AUTHORIZED 0x87c5_101c
	UTC_E_CHILD_PROCESS_FAILED 0x87c5_101d
	UTC_E_COMMAND_LINE_NOT_AUTHORIZED 0x87c5_101e
	UTC_E_CANNOT_LOAD_SCENARIO_EDITOR_XML 0x87c5_101f
	UTC_E_ESCALATION_TIMED_OUT 0x87c5_1020
	UTC_E_SETUP_TIMED_OUT 0x87c5_1021
	UTC_E_TRIGGER_MISMATCH 0x87c5_1022
	UTC_E_TRIGGER_NOT_FOUND 0x87c5_1023
	UTC_E_SIF_NOT_SUPPORTED 0x87c5_1024
	UTC_E_DELAY_TERMINATED 0x87c5_1025
	UTC_E_DEVICE_TICKET_ERROR 0x87c5_1026
	UTC_E_TRACE_BUFFER_LIMIT_EXCEEDED 0x87c5_1027
	UTC_E_API_RESULT_UNAVAILABLE 0x87c5_1028
	UTC_E_RPC_TIMEOUT 0x87c5_1029
	UTC_E_RPC_WAIT_FAILED 0x87c5_102a
	UTC_E_API_BUSY 0x87c5_102b
	UTC_E_TRACE_MIN_DURATION_REQUIREMENT_NOT_MET 0x87c5_102c
	UTC_E_EXCLUSIVITY_NOT_AVAILABLE 0x87c5_102d
	UTC_E_GETFILE_FILE_PATH_NOT_APPROVED 0x87c5_102e
	UTC_E_ESCALATION_DIRECTORY_ALREADY_EXISTS 0x87c5_102f
	UTC_E_TIME_TRIGGER_ON_START_INVALID 0x87c5_1030
	UTC_E_TIME_TRIGGER_ONLY_VALID_ON_SINGLE_TRANSITION 0x87c5_1031
	UTC_E_TIME_TRIGGER_INVALID_TIME_RANGE 0x87c5_1032
	UTC_E_MULTIPLE_TIME_TRIGGER_ON_SINGLE_STATE 0x87c5_1033
	UTC_E_BINARY_MISSING 0x87c5_1034
	UTC_E_NETWORK_CAPTURE_NOT_ALLOWED 0x87c5_1035
	UTC_E_FAILED_TO_RESOLVE_CONTAINER_ID 0x87c5_1036
	UTC_E_UNABLE_TO_RESOLVE_SESSION 0x87c5_1037
	UTC_E_THROTTLED 0x87c5_1038
	UTC_E_UNAPPROVED_SCRIPT 0x87c5_1039
	UTC_E_SCRIPT_MISSING 0x87c5_103a
	UTC_E_SCENARIO_THROTTLED 0x87c5_103b
	UTC_E_API_NOT_SUPPORTED 0x87c5_103c
	UTC_E_GETFILE_EXTERNAL_PATH_NOT_APPROVED 0x87c5_103d
	UTC_E_TRY_GET_SCENARIO_TIMEOUT_EXCEEDED 0x87c5_103e
	UTC_E_CERT_REV_FAILED 0x87c5_103f
	UTC_E_FAILED_TO_START_NDISCAP 0x87c5_1040
	UTC_E_KERNELDUMP_LIMIT_REACHED 0x87c5_1041
	UTC_E_MISSING_AGGREGATE_EVENT_TAG 0x87c5_1042
	UTC_E_INVALID_AGGREGATION_STRUCT 0x87c5_1043
	UTC_E_ACTION_NOT_SUPPORTED_IN_DESTINATION 0x87c5_1044
	UTC_E_FILTER_MISSING_ATTRIBUTE 0x87c5_1045
	UTC_E_FILTER_INVALID_TYPE 0x87c5_1046
	UTC_E_FILTER_VARIABLE_NOT_FOUND 0x87c5_1047
	UTC_E_FILTER_FUNCTION_RESTRICTED 0x87c5_1048
	UTC_E_FILTER_VERSION_MISMATCH 0x87c5_1049
	UTC_E_FILTER_INVALID_FUNCTION 0x87c5_1050
	UTC_E_FILTER_INVALID_FUNCTION_PARAMS 0x87c5_1051
	UTC_E_FILTER_INVALID_COMMAND 0x87c5_1052
	UTC_E_FILTER_ILLEGAL_EVAL 0x87c5_1053
	UTC_E_TTTRACER_RETURNED_ERROR 0x87c5_1054
	UTC_E_AGENT_DIAGNOSTICS_TOO_LARGE 0x87c5_1055
	UTC_E_FAILED_TO_RECEIVE_AGENT_DIAGNOSTICS 0x87c5_1056
	UTC_E_SCENARIO_HAS_NO_ACTIONS 0x87c5_1057
	UTC_E_TTTRACER_STORAGE_FULL 0x87c5_1058
	UTC_E_INSUFFICIENT_SPACE_TO_START_TRACE 0x87c5_1059
	UTC_E_ESCALATION_CANCELLED_AT_SHUTDOWN 0x87c5_105a
	UTC_E_GETFILEINFOACTION_FILE_NOT_APPROVED 0x87c5_105b
	UTC_E_SETREGKEYACTION_TYPE_NOT_APPROVED 0x87c5_105c
	WINML_ERR_INVALID_DEVICE 0x8890_0001
	WINML_ERR_INVALID_BINDING 0x8890_0002
	WINML_ERR_VALUE_NOTFOUND 0x8890_0003
	WINML_ERR_SIZE_MISMATCH 0x8890_0004
}
