/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.monitor.fs;

import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.HashSet;
import java.util.Set;
import java.util.function.LongSupplier;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.env.NodeEnvironment;
import org.elasticsearch.monitor.NodeHealthService;
import org.elasticsearch.monitor.StatusInfo;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;

public final class FsHealthService
extends AbstractLifecycleComponent
implements NodeHealthService {
    private static final Logger logger = LogManager.getLogger(FsHealthService.class);
    private static final StatusInfo HEALTHY_DISABLED = new StatusInfo(StatusInfo.Status.HEALTHY, "health check disabled");
    private static final StatusInfo UNHEALTHY_BROKEN_NODE_LOCK = new StatusInfo(StatusInfo.Status.UNHEALTHY, "health check failed due to broken node lock");
    private static final StatusInfo HEALTHY_SUCCESS = new StatusInfo(StatusInfo.Status.HEALTHY, "health check passed");
    private final ThreadPool threadPool;
    private volatile boolean enabled;
    private volatile boolean brokenLock;
    private final TimeValue refreshInterval;
    private volatile TimeValue slowPathLoggingThreshold;
    private final NodeEnvironment nodeEnv;
    private final LongSupplier currentTimeMillisSupplier;
    private Scheduler.Cancellable scheduledFuture;
    @Nullable
    private volatile Set<Path> unhealthyPaths;
    public static final Setting<Boolean> ENABLED_SETTING = Setting.boolSetting("monitor.fs.health.enabled", true, Setting.Property.NodeScope, Setting.Property.Dynamic);
    public static final Setting<TimeValue> REFRESH_INTERVAL_SETTING = Setting.timeSetting("monitor.fs.health.refresh_interval", TimeValue.timeValueSeconds((long)120L), TimeValue.timeValueMillis((long)1L), Setting.Property.NodeScope);
    public static final Setting<TimeValue> SLOW_PATH_LOGGING_THRESHOLD_SETTING = Setting.timeSetting("monitor.fs.health.slow_path_logging_threshold", TimeValue.timeValueSeconds((long)5L), TimeValue.timeValueMillis((long)1L), Setting.Property.NodeScope, Setting.Property.Dynamic);

    public FsHealthService(Settings settings, ClusterSettings clusterSettings, ThreadPool threadPool, NodeEnvironment nodeEnv) {
        this.threadPool = threadPool;
        this.enabled = ENABLED_SETTING.get(settings);
        this.refreshInterval = REFRESH_INTERVAL_SETTING.get(settings);
        this.slowPathLoggingThreshold = SLOW_PATH_LOGGING_THRESHOLD_SETTING.get(settings);
        this.currentTimeMillisSupplier = threadPool.relativeTimeInMillisSupplier();
        this.nodeEnv = nodeEnv;
        clusterSettings.addSettingsUpdateConsumer(SLOW_PATH_LOGGING_THRESHOLD_SETTING, this::setSlowPathLoggingThreshold);
        clusterSettings.addSettingsUpdateConsumer(ENABLED_SETTING, this::setEnabled);
    }

    @Override
    protected void doStart() {
        this.scheduledFuture = this.threadPool.scheduleWithFixedDelay(new FsHealthMonitor(), this.refreshInterval, this.threadPool.generic());
    }

    @Override
    protected void doStop() {
        this.scheduledFuture.cancel();
    }

    @Override
    protected void doClose() {
    }

    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    public void setSlowPathLoggingThreshold(TimeValue slowPathLoggingThreshold) {
        this.slowPathLoggingThreshold = slowPathLoggingThreshold;
    }

    @Override
    public StatusInfo getHealth() {
        if (!this.enabled) {
            return HEALTHY_DISABLED;
        }
        if (this.brokenLock) {
            return UNHEALTHY_BROKEN_NODE_LOCK;
        }
        Set<Path> unhealthyPaths = this.unhealthyPaths;
        if (unhealthyPaths != null) {
            assert (!unhealthyPaths.isEmpty());
            return new StatusInfo(StatusInfo.Status.UNHEALTHY, "health check failed on [" + unhealthyPaths.stream().map(Path::toString).collect(Collectors.joining(",")) + "]");
        }
        return HEALTHY_SUCCESS;
    }

    class FsHealthMonitor
    extends AbstractRunnable {
        static final String TEMP_FILE_NAME = ".es_temp_file";
        private final byte[] bytesToWrite = UUIDs.randomBase64UUID().getBytes(StandardCharsets.UTF_8);

        FsHealthMonitor() {
        }

        @Override
        public void onFailure(Exception e) {
            logger.error("health check failed", (Throwable)e);
        }

        @Override
        public void onRejection(Exception e) {
            EsRejectedExecutionException esre;
            if (e instanceof EsRejectedExecutionException && (esre = (EsRejectedExecutionException)e).isExecutorShutdown()) {
                logger.debug("health check skipped (executor shut down)", (Throwable)e);
            } else {
                this.onFailure(e);
                assert (false) : e;
            }
        }

        @Override
        public void doRun() {
            if (FsHealthService.this.enabled) {
                this.monitorFSHealth();
                logger.debug("health check completed");
            }
        }

        private void monitorFSHealth() {
            Path[] paths;
            HashSet<Path> currentUnhealthyPaths = null;
            try {
                paths = FsHealthService.this.nodeEnv.nodeDataPaths();
            }
            catch (IllegalStateException e) {
                logger.error("health check failed", (Throwable)e);
                FsHealthService.this.brokenLock = true;
                return;
            }
            for (Path path : paths) {
                long executionStartTime = FsHealthService.this.currentTimeMillisSupplier.getAsLong();
                try {
                    if (!Files.exists(path, new LinkOption[0])) continue;
                    Path tempDataPath = path.resolve(TEMP_FILE_NAME);
                    Files.deleteIfExists(tempDataPath);
                    try (OutputStream os = Files.newOutputStream(tempDataPath, StandardOpenOption.CREATE_NEW);){
                        os.write(this.bytesToWrite);
                        IOUtils.fsync((Path)tempDataPath, (boolean)false);
                    }
                    Files.delete(tempDataPath);
                    long elapsedTime = FsHealthService.this.currentTimeMillisSupplier.getAsLong() - executionStartTime;
                    if (elapsedTime <= FsHealthService.this.slowPathLoggingThreshold.millis()) continue;
                    logger.warn("health check of [{}] took [{}ms] which is above the warn threshold of [{}]", (Object)path, (Object)elapsedTime, (Object)FsHealthService.this.slowPathLoggingThreshold);
                }
                catch (Exception ex) {
                    logger.error(() -> "health check of [" + String.valueOf(path) + "] failed", (Throwable)ex);
                    if (currentUnhealthyPaths == null) {
                        currentUnhealthyPaths = Sets.newHashSetWithExpectedSize(1);
                    }
                    currentUnhealthyPaths.add(path);
                }
            }
            FsHealthService.this.unhealthyPaths = currentUnhealthyPaths;
            FsHealthService.this.brokenLock = false;
        }
    }
}

