/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.events.core;

import org.apiguardian.api.API;
import org.opentest4j.reporting.events.api.Appendable;
import org.opentest4j.reporting.events.api.ChildElement;
import org.opentest4j.reporting.events.api.Context;
import org.opentest4j.reporting.schema.Namespace;
import org.opentest4j.reporting.schema.QualifiedName;

import java.io.File;
import java.util.Optional;

import static org.apiguardian.api.API.Status.EXPERIMENTAL;

/**
 * The {@code fileSource} element of the core namespace.
 *
 * @since 0.1.0
 */
@API(status = EXPERIMENTAL, since = "0.1.0")
public class FileSource extends ChildElement<Sources, FileSource> {

	private static final QualifiedName ELEMENT = QualifiedName.of(Namespace.REPORTING_CORE, "fileSource");
	private static final QualifiedName PATH = QualifiedName.of(Namespace.REPORTING_CORE, "path");

	FileSource(Context context) {
		super(context, FileSource.ELEMENT);
	}

	FileSource withPath(File file) {
		withAttribute(PATH, file.getPath());
		return this;
	}

	/**
	 * Add a {@code filePosition} child element to this element.
	 *
	 * @param line   the line number
	 * @param column the column number
	 * @return this element
	 */
	public Appendable<FileSource> addFilePosition(int line, Optional<Integer> column) {
		return append(CoreFactory.filePosition(line, column));
	}
}
