package cyclonedx

import (
	"testing"

	"github.com/CycloneDX/cyclonedx-go"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"github.com/anchore/grype/grype/presenter/models"
	"github.com/anchore/grype/grype/vulnerability"
)

func TestCvssVersionToMethod(t *testing.T) {
	testCases := []struct {
		desc     string
		input    string
		expected cyclonedx.ScoringMethod
		errors   bool
	}{
		{
			desc:     "invalid (not float)",
			input:    "",
			expected: "",
			errors:   true,
		},
		{
			desc:     "CVSS v2",
			input:    "2.0",
			expected: cyclonedx.ScoringMethodCVSSv2,
			errors:   false,
		},
		{
			desc:     "CVSS v31",
			input:    "3.1",
			expected: cyclonedx.ScoringMethodCVSSv31,
			errors:   false,
		},
		{
			desc:     "CVSS v3",
			input:    "3",
			expected: cyclonedx.ScoringMethodCVSSv3,
			errors:   false,
		},
		{
			desc:     "invalid (no match)",
			input:    "15.4",
			expected: cyclonedx.ScoringMethodOther,
			errors:   false,
		},
	}
	for _, tc := range testCases {
		t.Run(tc.desc, func(t *testing.T) {
			actual, err := cvssVersionToMethod(tc.input)
			if !tc.errors {
				assert.NoError(t, err)
			} else {
				assert.Error(t, err)
			}

			assert.Equal(t, tc.expected, actual)
		})
	}
}

type metadataProvider struct {
	severity string
	cvss     []vulnerability.Cvss
}

func (m metadataProvider) VulnerabilityMetadata(ref vulnerability.Reference) (*vulnerability.Metadata, error) {
	return &vulnerability.Metadata{
		ID:          ref.ID,
		DataSource:  "",
		Namespace:   ref.Namespace,
		Severity:    m.severity,
		URLs:        nil,
		Description: "",
		Cvss:        m.cvss,
	}, nil
}

func TestNewVulnerability_AlwaysIncludesSeverity(t *testing.T) {
	tests := []struct {
		name  string
		match models.Match
	}{
		{
			name: "populates severity with missing CVSS records",
			match: models.Match{
				Vulnerability: models.Vulnerability{
					VulnerabilityMetadata: models.VulnerabilityMetadata{
						Severity: "High",
					},
				},
				Artifact:     models.Package{},
				MatchDetails: nil,
			},
		},
		{
			name: "populates severity with all CVSS records",
			match: models.Match{
				Vulnerability: models.Vulnerability{
					VulnerabilityMetadata: models.VulnerabilityMetadata{
						Severity: "High",
						Cvss: []models.Cvss{
							{
								Version: "2.0",
								Metrics: models.CvssMetrics{
									BaseScore: 1.1,
								},
							},
							{
								Version: "3.0",
								Metrics: models.CvssMetrics{
									BaseScore: 2.1,
								},
							},
							{
								Version: "3.1",
								Metrics: models.CvssMetrics{
									BaseScore: 3.1,
								},
							},
						},
					},
				},
				Artifact:     models.Package{},
				MatchDetails: nil,
			},
		},
	}
	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			actual, err := NewVulnerability(test.match)
			require.NoError(t, err)
			require.NotNil(t, actual.Ratings, "cyclonedx document ratings should not be nil")
			require.NotEmpty(t, actual.Ratings)
			require.Equal(t, cdxSeverityFromGrypeSeverity(test.match.Vulnerability.Severity), (*actual.Ratings)[0].Severity)
			if len(test.match.Vulnerability.Cvss) > 0 {
				for i, rating := range *actual.Ratings {
					require.Equal(t, test.match.Vulnerability.Cvss[i].Metrics.BaseScore, *rating.Score)
				}
			}
		})
	}
}
