/*
** Copyright (C) 2001-2025 Zabbix SIA
**
** Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
** documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
** rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
** permit persons to whom the Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in all copies or substantial portions
** of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
** WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
** COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
** TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
** SOFTWARE.
**/

package zbxflag

import (
	"flag"
	"testing"

	"github.com/google/go-cmp/cmp"
)

func TestStringFlag_Register(t *testing.T) {
	t.Parallel()

	type fields struct {
		Flag    Flag
		Default string
		Dest    *string
		set     bool
	}

	tests := []struct {
		name      string
		fields    fields
		args      []string
		wantValue string
		wantErr   bool
	}{
		{
			"+valid",
			fields{
				Flag: Flag{
					Name:        "flag",
					Shorthand:   "f",
					Description: "test flagy",
				},
				Default: "default",
				Dest:    new(string),
			},
			[]string{"--flag", "value"},
			"value",
			false,
		},
		{
			"+default",
			fields{
				Flag: Flag{
					Name:        "flag",
					Shorthand:   "f",
					Description: "test flagy",
				},
				Default: "default",
				Dest:    new(string),
			},
			[]string{},
			"default",
			false,
		},
		{
			"+shorthand",
			fields{
				Flag: Flag{
					Name:        "flag",
					Shorthand:   "f",
					Description: "test flagy",
				},
				Default: "default",
				Dest:    new(string),
			},
			[]string{"-f", "value"},
			"value",
			false,
		},
		{
			"-duplicate",
			fields{
				Flag: Flag{
					Name:        "flag",
					Shorthand:   "f",
					Description: "test flagy",
				},
				Default: "default",
				Dest:    new(string),
			},
			[]string{"--flag", "value", "--flag", "value2"},
			"value",
			true,
		},
		{
			"-duplicate(shorthands)",
			fields{
				Flag: Flag{
					Name:        "flag",
					Shorthand:   "f",
					Description: "test flagy",
				},
				Default: "default",
				Dest:    new(string),
			},
			[]string{"-f", "value", "-f", "value2"},
			"value",
			true,
		},
		{
			"-duplicate(mixed)",
			fields{
				Flag: Flag{
					Name:        "flag",
					Shorthand:   "f",
					Description: "test flagy",
				},
				Default: "default",
				Dest:    new(string),
			},
			[]string{"-f", "value", "--flag", "value2"},
			"value",
			true,
		},
		{
			"-duplicate(mixed,reversed)",
			fields{
				Flag: Flag{
					Name:        "flag",
					Shorthand:   "f",
					Description: "test flagy",
				},
				Default: "default",
				Dest:    new(string),
			},
			[]string{"--flag", "value2", "-f", "value"},
			"value2",
			true,
		},
	}
	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			f := &StringFlag{
				Flag:    tt.fields.Flag,
				Default: tt.fields.Default,
				Dest:    tt.fields.Dest,
				set:     tt.fields.set,
			}

			fs := flag.NewFlagSet("", flag.ContinueOnError)

			f.Register(fs)

			err := fs.Parse(tt.args)
			if (err != nil) != tt.wantErr {
				t.Fatalf(
					"StringFlag.Register() fs.Parse() error = %v, wantErr %v",
					err, tt.wantErr,
				)
			}

			if diff := cmp.Diff(tt.wantValue, *f.Dest); diff != "" {
				t.Fatalf(
					"StringFlag.Register() mismatch (-want +got):\n%s",
					diff,
				)
			}
		})
	}
}
