// <copyright file="INavigation.cs" company="Selenium Committers">
// Licensed to the Software Freedom Conservancy (SFC) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The SFC licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.
// </copyright>

using System;
using System.Threading.Tasks;

namespace OpenQA.Selenium
{
    /// <summary>
    /// Defines an interface allowing the user to access the browser's history and to
    /// navigate to a given URL.
    /// </summary>
    public interface INavigation
    {
        /// <summary>
        /// Move back a single entry in the browser's history.
        /// </summary>
        void Back();

        /// <summary>
        /// Move back a single entry in the browser's history as an asynchronous task.
        /// </summary>
        /// <returns>A task object representing the asynchronous operation.</returns>
        Task BackAsync();

        /// <summary>
        /// Move a single "item" forward in the browser's history.
        /// </summary>
        /// <remarks>Does nothing if we are on the latest page viewed.</remarks>
        void Forward();

        /// <summary>
        /// Move a single "item" forward in the browser's history as an asynchronous task.
        /// </summary>
        /// <returns>A task object representing the asynchronous operation.</returns>
        Task ForwardAsync();

        /// <summary>
        ///  Load a new web page in the current browser window.
        /// </summary>
        /// <param name="url">The URL to load. It is best to use a fully qualified URL</param>
        /// <remarks>
        /// Calling the <see cref="GoToUrl(string)"/> method will load a new web page in the current browser window.
        /// This is done using an HTTP GET operation, and the method will block until the
        /// load is complete. This will follow redirects issued either by the server or
        /// as a meta-redirect from within the returned HTML. Should a meta-redirect "rest"
        /// for any duration of time, it is best to wait until this timeout is over, since
        /// should the underlying page change while your test is executing the results of
        /// future calls against this interface will be against the freshly loaded page.
        /// </remarks>
        /// <exception cref="ArgumentNullException">If <paramref name="url"/> is <see langword="null"/>.</exception>
        void GoToUrl(string url);

        /// <summary>
        /// Navigate to a url as an asynchronous task.
        /// </summary>
        /// <param name="url">String of where you want the browser to go.</param>
        /// <returns>A task object representing the asynchronous operation.</returns>
        /// <exception cref="ArgumentNullException">If <paramref name="url"/> is <see langword="null"/>.</exception>
        Task GoToUrlAsync(string url);

        /// <summary>
        ///  Load a new web page in the current browser window.
        /// </summary>
        /// <param name="url">The URL to load.</param>
        /// <remarks>
        /// Calling the <see cref="GoToUrl(System.Uri)"/> method will load a new web page in the current browser window.
        /// This is done using an HTTP GET operation, and the method will block until the
        /// load is complete. This will follow redirects issued either by the server or
        /// as a meta-redirect from within the returned HTML. Should a meta-redirect "rest"
        /// for any duration of time, it is best to wait until this timeout is over, since
        /// should the underlying page change while your test is executing the results of
        /// future calls against this interface will be against the freshly loaded page.
        /// </remarks>
        /// <exception cref="ArgumentNullException">If <paramref name="url"/> is <see langword="null"/>.</exception>
        void GoToUrl(Uri url);

        /// <summary>
        /// Navigate to a url as an asynchronous task.
        /// </summary>
        /// <param name="url">Uri object of where you want the browser to go.</param>
        /// <returns>A task object representing the asynchronous operation.</returns>
        /// <exception cref="ArgumentNullException">If <paramref name="url"/> is <see langword="null"/>.</exception>
        Task GoToUrlAsync(Uri url);

        /// <summary>
        /// Refreshes the current page.
        /// </summary>
        void Refresh();

        /// <summary>
        /// Reload the current page as an asynchronous task.
        /// </summary>
        /// <returns>A task object representing the asynchronous operation.</returns>
        Task RefreshAsync();
    }
}
