//
// Syd: rock-solid application kernel
// src/kernel/link.rs: link(2) and linkat(2) handlers
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::os::fd::AsFd;

use libseccomp::ScmpNotifResp;
use nix::{errno::Errno, fcntl::AtFlags, unistd::linkat, NixPath};

use crate::{
    fs::FsFlags,
    hook::{PathArgs, SysArg, SysFlags, UNotifyEventRequest},
    kernel::syscall_path_handler,
};

pub(crate) fn sys_link(request: UNotifyEventRequest) -> ScmpNotifResp {
    let argv = &[
        SysArg {
            path: Some(0),
            fsflags: FsFlags::MUST_PATH | FsFlags::NO_FOLLOW_LAST | FsFlags::WANT_BASE,
            ..Default::default()
        },
        SysArg {
            path: Some(1),
            dotlast: Some(Errno::ENOENT),
            fsflags: FsFlags::MISS_LAST | FsFlags::NO_FOLLOW_LAST,
            ..Default::default()
        },
    ];

    syscall_path_handler(
        request,
        "link",
        argv,
        |path_args: PathArgs, request, sandbox| {
            drop(sandbox); // release the read-lock.

            syscall_link_handler(request, path_args)
        },
    )
}

pub(crate) fn sys_linkat(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    #[allow(clippy::cast_possible_truncation)]
    let flags = req.data.args[4] as libc::c_int;

    let empty = flags & libc::AT_EMPTY_PATH != 0;

    let mut fsflags = FsFlags::MUST_PATH;
    if flags & libc::AT_SYMLINK_FOLLOW == 0 {
        fsflags |= FsFlags::NO_FOLLOW_LAST;
    }

    let mut flags = SysFlags::empty();
    if empty {
        flags |= SysFlags::EMPTY_PATH;
    } else {
        fsflags |= FsFlags::WANT_BASE;
    }

    let argv = &[
        SysArg {
            dirfd: Some(0),
            path: Some(1),
            flags,
            fsflags,
            ..Default::default()
        },
        SysArg {
            dirfd: Some(2),
            path: Some(3),
            dotlast: Some(Errno::ENOENT),
            fsflags: FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            ..Default::default()
        },
    ];

    syscall_path_handler(
        request,
        "linkat",
        argv,
        |path_args: PathArgs, request, sandbox| {
            drop(sandbox); // release the read-lock.

            syscall_link_handler(request, path_args)
        },
    )
}

/// A helper function to handle link{,at} syscalls.
fn syscall_link_handler(
    request: &UNotifyEventRequest,
    args: PathArgs,
) -> Result<ScmpNotifResp, Errno> {
    // SAFETY: SysArg has two elements.
    #[allow(clippy::disallowed_methods)]
    let old_path = args.0.as_ref().unwrap();
    #[allow(clippy::disallowed_methods)]
    let new_path = args.1.as_ref().unwrap();

    // SAFETY: linkat does not follow symbolic links in old path by
    // default unless AT_SYMLINK_FOLLOW flag is passed. As such,
    // AT_SYMLINK_NOFOLLOW is an invalid flag for linkat.
    linkat(
        old_path
            .dir
            .as_ref()
            .map(|fd| fd.as_fd())
            .ok_or(Errno::EBADF)?,
        old_path.base,
        new_path
            .dir
            .as_ref()
            .map(|fd| fd.as_fd())
            .ok_or(Errno::EBADF)?,
        new_path.base,
        if old_path.base.is_empty() {
            AtFlags::AT_EMPTY_PATH
        } else {
            AtFlags::empty()
        },
    )
    .map(|_| request.return_syscall(0))
}
