//
// Syd: rock-solid application kernel
// src/kernel/ioctl.rs: ioctl(2) handler
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use libseccomp::ScmpNotifResp;
use nix::errno::Errno;

use crate::{
    hook::{SysArg, SysFlags, UNotifyEventRequest},
    kernel::syscall_path_handler,
    log_enabled,
    sandbox::{Action, Capability},
    syslog::LogLevel,
    warn,
};

#[allow(clippy::cognitive_complexity)]
pub(crate) fn sys_ioctl(request: UNotifyEventRequest) -> ScmpNotifResp {
    // SAFETY: Deny if the ioctl request is denylisted.
    let req = request.scmpreq;
    let arg = req.data.args[1];

    let sandbox = request.get_sandbox();

    if let Some(deny) = sandbox.has_ioctl(&arg) {
        return if deny {
            // Request is denylisted.
            let cap = Capability::CAP_IOCTL;
            let action = sandbox.default_action(cap);
            let filter = action == Action::Filter;

            if !filter && action >= Action::Warn && log_enabled!(LogLevel::Warn) {
                let grp = cap.to_string().to_ascii_lowercase();
                if sandbox.verbose {
                    warn!("ctx": "access", "cap": cap, "act": action,
                        "sys": "ioctl", "ioctl": arg,
                        "tip": format!("configure `{grp}/allow+{arg:#x}'"),
                        "req": &request);
                } else {
                    warn!("ctx": "access", "cap": cap, "act": action,
                        "sys": "ioctl", "ioctl": arg,
                        "tip": format!("configure `{grp}/allow+{arg:#x}'"),
                        "pid": request.scmpreq.pid);
                }
            }

            match action {
                Action::Allow | Action::Warn => {
                    // SAFETY: ioctl is fd-only.
                    unsafe { request.continue_syscall() }
                }
                Action::Filter | Action::Deny => request.fail_syscall(Errno::EACCES),
                Action::Panic => panic!(),
                Action::Exit => std::process::exit(libc::EACCES),
                action => {
                    // Stop|Kill
                    let _ = request.kill(action);
                    request.fail_syscall(Errno::EACCES)
                }
            }
        } else {
            // Request is allowlisted.
            // SAFETY: ioctl is fd-only.
            unsafe { request.continue_syscall() }
        };
    }
    drop(sandbox); // release the read-lock.

    // SAFETY: ioctl is fd-only, so UNSAFE_CONT is ok.
    let argv = &[SysArg {
        dirfd: Some(0),
        flags: SysFlags::UNSAFE_CONT,
        ..Default::default()
    }];

    syscall_path_handler(request, "ioctl", argv, |_, request, sandbox| {
        drop(sandbox); // release the read-lock.

        // SAFETY: ioctl is fd-only.
        Ok(unsafe { request.continue_syscall() })
    })
}
