//
// Syd: rock-solid application kernel
// src/cookie.rs: Syscall argument cookies
//
// Copyright (c) 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::os::fd::{AsFd, AsRawFd, FromRawFd, OwnedFd, RawFd};

use nix::{
    errno::Errno,
    fcntl::{AtFlags, OpenHow},
    unistd::UnlinkatFlags,
    NixPath,
};
use once_cell::sync::Lazy;

use crate::{fs::fillrandom, kernel::rename::RenameFlags, path::XPath};

/// A platform‐sized secure cookie
///
/// 32 bits on 32-bit, 64 bits on 64-bit targets.
#[cfg(target_pointer_width = "32")]
pub(crate) type Cookie = u32;
#[cfg(target_pointer_width = "64")]
pub(crate) type Cookie = u64;

/// Generate a random `Cookie` using the OS's secure RNG.
///
/// This uses `syd::fs::fillrandom` under the hood to pull in
/// exactly the number of bytes needed for `Cookie`,
/// interprets them in little-endian order, and returns the result.
pub(crate) fn getcookie() -> Result<Cookie, Errno> {
    #[cfg(target_pointer_width = "32")]
    {
        const N: usize = 4;
        let mut buf = [0u8; N];
        fillrandom(&mut buf)?;
        Ok(Cookie::from_le_bytes(buf))
    }

    #[cfg(target_pointer_width = "64")]
    {
        const N: usize = 8;
        let mut buf = [0u8; N];
        fillrandom(&mut buf)?;
        Ok(Cookie::from_le_bytes(buf))
    }
}

// These cookies are confined by seccomp for use with openat2(2).
#[allow(clippy::disallowed_methods)]
pub(crate) static OPENAT2_COOKIE_ARG4: Lazy<Cookie> = Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static OPENAT2_COOKIE_ARG5: Lazy<Cookie> = Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static MEMFD_CREATE_COOKIE_ARG2: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static MEMFD_CREATE_COOKIE_ARG3: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static MEMFD_CREATE_COOKIE_ARG4: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static MEMFD_CREATE_COOKIE_ARG5: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
pub(crate) static RENAMEAT2_COOKIE_ARG5: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static UNLINKAT_COOKIE_ARG3: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static UNLINKAT_COOKIE_ARG4: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static UNLINKAT_COOKIE_ARG5: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));

/// Safe openat2 confined by syscall cookies.
#[inline(always)]
pub(crate) fn safe_openat2<Fd: AsFd>(
    dirfd: Fd,
    path: &XPath,
    mut how: OpenHow,
) -> Result<OwnedFd, Errno> {
    // SAFETY: In libc we trust.
    #[allow(clippy::cast_possible_truncation)]
    let fd = path.with_nix_path(|cstr| unsafe {
        libc::syscall(
            libc::SYS_openat2,
            dirfd.as_fd().as_raw_fd(),
            cstr.as_ptr(),
            std::ptr::addr_of_mut!(how),
            std::mem::size_of::<libc::open_how>(),
            *OPENAT2_COOKIE_ARG4,
            *OPENAT2_COOKIE_ARG5,
        )
    })? as RawFd;

    Errno::result(fd)?;

    // SAFETY:
    //
    // `openat2(2)` should return a valid owned fd on success
    Ok(unsafe { OwnedFd::from_raw_fd(fd) })
}

/// Safe memfd_create confined by syscall cookies.
#[inline(always)]
pub(crate) fn safe_memfd_create(name: &[u8], flags: libc::c_uint) -> Result<OwnedFd, Errno> {
    // SAFETY: In libc we trust.
    #[allow(clippy::cast_possible_truncation)]
    let fd = Errno::result(unsafe {
        libc::syscall(
            libc::SYS_memfd_create,
            name.as_ptr(),
            flags,
            *MEMFD_CREATE_COOKIE_ARG2,
            *MEMFD_CREATE_COOKIE_ARG3,
            *MEMFD_CREATE_COOKIE_ARG4,
            *MEMFD_CREATE_COOKIE_ARG5,
        )
    })? as RawFd;

    // SAFETY:
    //
    // `memfd_create(2)` should return a valid owned fd on success
    Ok(unsafe { OwnedFd::from_raw_fd(fd) })
}

/// Safe renameat2(2) confined by syscall cookies.
#[inline(always)]
pub(crate) fn safe_renameat2<Fd1: AsFd, Fd2: AsFd>(
    old_dirfd: Fd1,
    old_path: &XPath,
    new_dirfd: Fd2,
    new_path: &XPath,
    flags: RenameFlags,
) -> Result<(), Errno> {
    let res = old_path.with_nix_path(|old_cstr| {
        new_path.with_nix_path(|new_cstr| unsafe {
            libc::syscall(
                libc::SYS_renameat2,
                old_dirfd.as_fd().as_raw_fd(),
                old_cstr.as_ptr(),
                new_dirfd.as_fd().as_raw_fd(),
                new_cstr.as_ptr(),
                flags.bits(),
                *RENAMEAT2_COOKIE_ARG5,
            )
        })
    })??;

    Errno::result(res).map(drop)
}

/// Safe unlinkat(2) confined by syscall cookies.
#[inline(always)]
pub(crate) fn safe_unlinkat<Fd: AsFd>(
    dirfd: Fd,
    path: &XPath,
    flag: UnlinkatFlags,
) -> Result<(), Errno> {
    let atflag = match flag {
        UnlinkatFlags::RemoveDir => AtFlags::AT_REMOVEDIR,
        UnlinkatFlags::NoRemoveDir => AtFlags::empty(),
    };

    // SAFETY: In libc we trust.
    let res = path.with_nix_path(|cstr| unsafe {
        libc::syscall(
            libc::SYS_unlinkat,
            dirfd.as_fd().as_raw_fd(),
            cstr.as_ptr(),
            atflag.bits(),
            *UNLINKAT_COOKIE_ARG3,
            *UNLINKAT_COOKIE_ARG4,
            *UNLINKAT_COOKIE_ARG5,
        )
    })?;

    Errno::result(res).map(drop)
}
