//
// Syd: rock-solid application kernel
// src/hash.rs: Utilities for caching
//
// Copyright (c) 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

// SAFETY: This module has been liberated from unsafe code!
#![forbid(unsafe_code)]

use std::{
    collections::BTreeMap,
    sync::{Arc, Condvar, Mutex},
};

use libseccomp::ScmpSyscall;
use nix::{errno::Errno, unistd::Pid};
use serde::{ser::SerializeMap, Serializer};

use crate::{
    elf::ExecutableFile, fs::CanonicalPath, hook::RemoteProcess, ScmpNotifReq, SydArch,
    SydMemoryMap, SydSigSet,
};

/// Metadata on a blocking syscall invocation
#[derive(Copy, Clone, Debug)]
pub struct SysInterrupt {
    /// The thread group ID
    pub tgid: Pid,
    /// Syd handler thread ID
    pub handler: Pid,
    /// System call request
    pub request: ScmpNotifReq,
    /// True if `SA_RESTART` is ignored
    /// (e.g. due to a socket timeout).
    pub ignore_restart: bool,
}

/// This is the data type used to handle syscall interrupts.
#[derive(Debug)]
#[allow(clippy::type_complexity)]
pub struct SysInterruptMap {
    /// Map of blocking syscalls by request id.
    pub sys_block: Arc<(Mutex<BTreeMap<u64, SysInterrupt>>, Condvar)>,
    /// Map of restarting signals by TGID.
    /// Used for SA_RESTART tracking.
    pub sig_restart: Arc<Mutex<BTreeMap<Pid, SydSigSet>>>,
}

/// Represents an exec(3) check result
#[derive(Debug)]
pub struct ExecResult {
    pub(crate) file: ExecutableFile,
    pub(crate) arch: u32,
    pub(crate) ip: u64,
    pub(crate) sp: u64,
    pub(crate) args: [u64; 6],
    pub(crate) ip_mem: Option<[u8; 64]>,
    pub(crate) sp_mem: Option<[u8; 64]>,
    pub(crate) memmap: Option<Vec<SydMemoryMap>>,
}

/// Represents a sigreturn(2) check result
#[derive(Debug)]
pub struct SigreturnResult {
    pub(crate) is_realtime: bool,
    pub(crate) ip: u64,
    pub(crate) sp: u64,
    pub(crate) args: [u64; 6],
    pub(crate) ip_mem: Option<[u8; 64]>,
    pub(crate) sp_mem: Option<[u8; 64]>,
}

/// Results map for ptrace(2) hooks chdir, execve, sigaction and sigreturn.
#[derive(Debug)]
#[allow(clippy::type_complexity)]
pub struct SysResultMap<'a> {
    /// syscall-agnostic error map
    pub trace_error: Arc<Mutex<BTreeMap<RemoteProcess, Option<Errno>>>>,
    /// chdir(2) result map
    pub trace_chdir: Arc<Mutex<BTreeMap<RemoteProcess, CanonicalPath<'a>>>>,
    /// exec(3) result map
    pub trace_execv: Arc<Mutex<BTreeMap<RemoteProcess, ExecResult>>>,
    /// {rt_,}sigreturn(2) result map
    pub trace_sigret: Arc<Mutex<BTreeMap<RemoteProcess, SigreturnResult>>>,
}

/// Signal map, used by signal counting for SROP mitigation:
/// If a TGID is not in sig_handle_map at the entry of sigreturn(2),
/// we terminate the process because the sigreturn(2) is artificial.
#[derive(Debug)]
#[allow(clippy::type_complexity)]
pub struct SignalMap {
    /// Set of TGIDs that have received count signals for handled signals.
    pub sig_handle: Arc<Mutex<BTreeMap<Pid, u64>>>,
}

impl SysInterrupt {
    pub(crate) fn new(
        request: ScmpNotifReq,
        tgid: Pid,
        handler: Pid,
        ignore_restart: bool,
    ) -> Self {
        Self {
            tgid,
            handler,
            request,
            ignore_restart,
        }
    }
}

impl serde::Serialize for SysInterrupt {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        let mut map = serializer.serialize_map(Some(3))?;

        let data = &self.request.data;
        let syscall = ScmpSyscall::get_name_by_arch(data.syscall, data.arch)
            .unwrap_or_else(|_| format!("{}", i32::from(data.syscall)));
        let _ = map.serialize_entry("pid", &self.request.pid);
        let _ = map.serialize_entry("tgid", &self.tgid.as_raw());
        let _ = map.serialize_entry("sys", &syscall);
        let _ = map.serialize_entry("arch", &SydArch(data.arch));
        let _ = map.serialize_entry("args", &data.args);
        let _ = map.serialize_entry("handler", &self.handler.as_raw());
        let _ = map.serialize_entry("ignore_restart", &self.ignore_restart);

        map.end()
    }
}

/// Create a new SysInterruptMap.
pub fn sys_interrupt_map_new() -> SysInterruptMap {
    SysInterruptMap {
        sys_block: Arc::new((Mutex::new(BTreeMap::new()), Condvar::new())),
        sig_restart: Arc::new(Mutex::new(BTreeMap::new())),
    }
}

/// Create a new SysResultMap.
pub fn sys_result_map_new<'a>() -> SysResultMap<'a> {
    SysResultMap {
        trace_error: Arc::new(Mutex::new(BTreeMap::new())),
        trace_chdir: Arc::new(Mutex::new(BTreeMap::new())),
        trace_execv: Arc::new(Mutex::new(BTreeMap::new())),
        trace_sigret: Arc::new(Mutex::new(BTreeMap::new())),
    }
}

/// Create a new SignalMap.
pub fn signal_map_new() -> SignalMap {
    SignalMap {
        sig_handle: Arc::new(Mutex::new(BTreeMap::new())),
    }
}
