/**
 * @file
 *
 * @author G. Kaplan and A. Kovacs
 *
 *  SuperNOVAS implementations for the IAU2000 nutation series calculations, with varying
 *  trade-offs between computational cost and precision. It provides support for both the
 *  IAU 2000A and IAU2000B series as well as a NOVAS-specific truncated low-precision
 *  version we call NU2000K.
 *
 *  Based on the NOVAS C Edition, Version 3.1:
 *
 *  U. S. Naval Observatory<br>
 *  Astronomical Applications Dept.<br>
 *  Washington, DC<br>
 *  <a href="http://www.usno.navy.mil/USNO/astronomical-applications">
 *  http://www.usno.navy.mil/USNO/astronomical-applications</a>
 */

#include <stdint.h>
#include <errno.h>

/// \cond PRIVATE
#define __NOVAS_INTERNAL_API__    ///< Use definitions meant for internal use by SuperNOVAS only
#include "novas.h"

#define T0        NOVAS_JD_J2000
/// \endcond


/**
 * Returns the values for nutation in longitude and nutation in obliquity for a given TDB
 * Julian date.  The nutation model selected depends upon the input value of 'accuracy'. See
 * notes below for important details.
 *
 * This function selects the nutation model depending first upon the input value of 'accuracy'.
 * If 'accuracy' is NOVAS_FULL_ACCURACY (0), the IAU 2000A nutation model is used. Otherwise
 * the model set by set_nutation_lp_provider() is used, or else the default nu2000k().
 *
 * See the prologs of the nutation functions in file 'nutation.c' for details concerning the
 * models.
 *
 * NOTES:
 * <ol>
 * <li>As of version 1.4, this function applies the recommended rescaling of the IAU 2000 nutation
 * angles by the factors recommended by the P03rev2 (Capitaine et al. 2005; Coppola et al. 2009), to
 * match the model used by SOFA.</li>
 * </ol>
 *
 * REFERENCES:
 * <ol>
 * <li>Kaplan, G. (2005), US Naval Observatory Circular 179.</li>
 * <li>Capitaine, N., P.T. Wallace and J. Chapront (2005), “Improvement of the IAU 2000 precession
 *     model.” Astronomy &amp; Astrophysics, Vol. 432, pp. 355–67.</li>
 * <li>Coppola, V., Seago, G.H., &amp; Vallado, D.A. (2009), AAS 09-159</li>
 * </ol>
 *
 * @param t           [cy] TDB time in Julian centuries since J2000.0
 * @param accuracy    NOVAS_FULL_ACCURACY (0) or NOVAS_REDUCED_ACCURACY (1)
 * @param[out] dpsi   [arcsec] Nutation in longitude in arcseconds.
 * @param[out] deps   [arcsec] Nutation in obliquity in arcseconds.
 *
 * @return            0 if successful, or -1 if the output pointer arguments are NULL
 *
 * @sa set_nutation_lp_provider()
 * @sa nutation()
 * @sa iau2000b()
 * @sa nu2000k()
 * @sa cio_basis()
 * @sa NOVAS_CIRS
 * @sa NOVAS_JD_J2000
 */
int nutation_angles(double t, enum novas_accuracy accuracy, double *restrict dpsi, double *restrict deps) {
  static THREAD_LOCAL double last_t = NAN, last_dpsi, last_deps;
  static THREAD_LOCAL enum novas_accuracy last_acc = -1;

  // P03 scaling factor.
  static const double f = -2.7774e-6;

  if(!dpsi || !deps) {
    if(dpsi)
      *dpsi = NAN;
    if(deps)
      *deps = NAN;

    return novas_error(-1, EINVAL, "nutation_angles", "NULL output pointer: dspi=%p, deps=%p", dpsi, deps);
  }

  if(!(fabs(t - last_t) < 1e-12) || (accuracy != last_acc)) {
    novas_nutation_provider nutate_call = (accuracy == NOVAS_FULL_ACCURACY) ? iau2000a : get_nutation_lp_provider();
    nutate_call(JD_J2000, t * JULIAN_CENTURY_DAYS, &last_dpsi, &last_deps);

    // Apply P03 (Capitaine et al. 2005) rescaling to IAU 2006 model.
    // Convert output to arcseconds.
    last_dpsi *= (1.0000004697 + f) / ARCSEC;
    last_deps *= (1.0 + f) / ARCSEC;

    last_acc = accuracy;
    last_t = t;
  }

  *dpsi = last_dpsi;
  *deps = last_deps;

  return 0;
}




/**
 * Computes the IAU 2000A nutation high-precision series for the specified date. It is rather
 * expensive computationally.
 *
 * The IAU 2000A nutation model is MHB_2000 without the free core nutation and without the
 * corrections to Lieske precession.
 *
 * REFERENCES:
 * <ol>
 *  <li>IERS Conventions (2003), Chapter 5.</li>
 *  <li>Simon et al. (1994) Astronomy and Astrophysics 282, 663-683, esp. Sections 3.4-3.5.</li>
 * </ol>
 *
 * @param jd_tt_high  [day] High-order part of the Terrestrial Time (TT) based Julian date.
 *                    Typically it may be the integer part of a split date for the highest precision,
 *                    or the full date for normal (reduced) precision.
 * @param jd_tt_low   [day] Low-order part of the Terrestrial Time (TT) based Julian date. Typically
 *                    it may be the fractional part of a split date for the highest precision, or 0.0
 *                    for normal (reduced) precision.
 * @param[out] dpsi   [rad] &delta;&psi; Nutation (luni-solar + planetary) in longitude, in radians.
 *                    It may be NULL if not required.
 * @param[out] deps   [rad] &delta;&epsilon; Nutation (luni-solar + planetary) in obliquity, in
 *                    radians. It may be NULL if not required.
 * @return            0
 *
 * @sa iau2000b()
 * @sa nu2000k()
 * @sa nutation_angles()
 * @sa novas_nutation_provider
 *
 */
int iau2000a(double jd_tt_high, double jd_tt_low, double *restrict dpsi, double *restrict deps) {
  // @formatter:off

  // Luni-Solar argument multipliers:
  // L     L'    F     D     Om
  static const int8_t nals_t[678][5] = { //
          { 0, 0, 0, 0, 1 }, //
          { 0, 0, 2, -2, 2 }, //
          { 0, 0, 2, 0, 2 }, //
          { 0, 0, 0, 0, 2 }, //
          { 0, 1, 0, 0, 0 }, //
          { 0, 1, 2, -2, 2 }, //
          { 1, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 1 }, //
          { 1, 0, 2, 0, 2 }, //
          { 0, -1, 2, -2, 2 }, //
          { 0, 0, 2, -2, 1 }, //
          { -1, 0, 2, 0, 2 }, //
          { -1, 0, 0, 2, 0 }, //
          { 1, 0, 0, 0, 1 }, //
          { -1, 0, 0, 0, 1 }, //
          { -1, 0, 2, 2, 2 }, //
          { 1, 0, 2, 0, 1 }, //
          { -2, 0, 2, 0, 1 }, //
          { 0, 0, 0, 2, 0 }, //
          { 0, 0, 2, 2, 2 }, //
          { 0, -2, 2, -2, 2 }, //
          { -2, 0, 0, 2, 0 }, //
          { 2, 0, 2, 0, 2 }, //
          { 1, 0, 2, -2, 2 }, //
          { -1, 0, 2, 0, 1 }, //
          { 2, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 0 }, //
          { 0, 1, 0, 0, 1 }, //
          { -1, 0, 0, 2, 1 }, //
          { 0, 2, 2, -2, 2 }, //
          { 0, 0, -2, 2, 0 }, //
          { 1, 0, 0, -2, 1 }, //
          { 0, -1, 0, 0, 1 }, //
          { -1, 0, 2, 2, 1 }, //
          { 0, 2, 0, 0, 0 }, //
          { 1, 0, 2, 2, 2 }, //
          { -2, 0, 2, 0, 0 }, //
          { 0, 1, 2, 0, 2 }, //
          { 0, 0, 2, 2, 1 }, //
          { 0, -1, 2, 0, 2 }, //
          { 0, 0, 0, 2, 1 }, //
          { 1, 0, 2, -2, 1 }, //
          { 2, 0, 2, -2, 2 }, //
          { -2, 0, 0, 2, 1 }, //
          { 2, 0, 2, 0, 1 }, //
          { 0, -1, 2, -2, 1 }, //
          { 0, 0, 0, -2, 1 }, //
          { -1, -1, 0, 2, 0 }, //
          { 2, 0, 0, -2, 1 }, //
          { 1, 0, 0, 2, 0 }, //
          { 0, 1, 2, -2, 1 }, //
          { 1, -1, 0, 0, 0 }, //
          { -2, 0, 2, 0, 2 }, //
          { 3, 0, 2, 0, 2 }, //
          { 0, -1, 0, 2, 0 }, //
          { 1, -1, 2, 0, 2 }, //
          { 0, 0, 0, 1, 0 }, //
          { -1, -1, 2, 2, 2 }, //
          { -1, 0, 2, 0, 0 }, //
          { 0, -1, 2, 2, 2 }, //
          { -2, 0, 0, 0, 1 }, //
          { 1, 1, 2, 0, 2 }, //
          { 2, 0, 0, 0, 1 }, //
          { -1, 1, 0, 1, 0 }, //
          { 1, 1, 0, 0, 0 }, //
          { 1, 0, 2, 0, 0 }, //
          { -1, 0, 2, -2, 1 }, //
          { 1, 0, 0, 0, 2 }, //
          { -1, 0, 0, 1, 0 }, //
          { 0, 0, 2, 1, 2 }, //
          { -1, 0, 2, 4, 2 }, //
          { -1, 1, 0, 1, 1 }, //
          { 0, -2, 2, -2, 1 }, //
          { 1, 0, 2, 2, 1 }, //
          { -2, 0, 2, 2, 2 }, //
          { -1, 0, 0, 0, 2 }, //
          { 1, 1, 2, -2, 2 }, //
          { -2, 0, 2, 4, 2 }, //
          { -1, 0, 4, 0, 2 }, //
          { 2, 0, 2, -2, 1 }, //
          { 2, 0, 2, 2, 2 }, //
          { 1, 0, 0, 2, 1 }, //
          { 3, 0, 0, 0, 0 }, //
          { 3, 0, 2, -2, 2 }, //
          { 0, 0, 4, -2, 2 }, //
          { 0, 1, 2, 0, 1 }, //
          { 0, 0, -2, 2, 1 }, //
          { 0, 0, 2, -2, 3 }, //
          { -1, 0, 0, 4, 0 }, //
          { 2, 0, -2, 0, 1 }, //
          { -2, 0, 0, 4, 0 }, //
          { -1, -1, 0, 2, 1 }, //
          { -1, 0, 0, 1, 1 }, //
          { 0, 1, 0, 0, 2 }, //
          { 0, 0, -2, 0, 1 }, //
          { 0, -1, 2, 0, 1 }, //
          { 0, 0, 2, -1, 2 }, //
          { 0, 0, 2, 4, 2 }, //
          { -2, -1, 0, 2, 0 }, //
          { 1, 1, 0, -2, 1 }, //
          { -1, 1, 0, 2, 0 }, //
          { -1, 1, 0, 1, 2 }, //
          { 1, -1, 0, 0, 1 }, //
          { 1, -1, 2, 2, 2 }, //
          { -1, 1, 2, 2, 2 }, //
          { 3, 0, 2, 0, 1 }, //
          { 0, 1, -2, 2, 0 }, //
          { -1, 0, 0, -2, 1 }, //
          { 0, 1, 2, 2, 2 }, //
          { -1, -1, 2, 2, 1 }, //
          { 0, -1, 0, 0, 2 }, //
          { 1, 0, 2, -4, 1 }, //
          { -1, 0, -2, 2, 0 }, //
          { 0, -1, 2, 2, 1 }, //
          { 2, -1, 2, 0, 2 }, //
          { 0, 0, 0, 2, 2 }, //
          { 1, -1, 2, 0, 1 }, //
          { -1, 1, 2, 0, 2 }, //
          { 0, 1, 0, 2, 0 }, //
          { 0, -1, -2, 2, 0 }, //
          { 0, 3, 2, -2, 2 }, //
          { 0, 0, 0, 1, 1 }, //
          { -1, 0, 2, 2, 0 }, //
          { 2, 1, 2, 0, 2 }, //
          { 1, 1, 0, 0, 1 }, //
          { 1, 1, 2, 0, 1 }, //
          { 2, 0, 0, 2, 0 }, //
          { 1, 0, -2, 2, 0 }, //
          { -1, 0, 0, 2, 2 }, //
          { 0, 1, 0, 1, 0 }, //
          { 0, 1, 0, -2, 1 }, //
          { -1, 0, 2, -2, 2 }, //
          { 0, 0, 0, -1, 1 }, //
          { -1, 1, 0, 0, 1 }, //
          { 1, 0, 2, -1, 2 }, //
          { 1, -1, 0, 2, 0 }, //
          { 0, 0, 0, 4, 0 }, //
          { 1, 0, 2, 1, 2 }, //
          { 0, 0, 2, 1, 1 }, //
          { 1, 0, 0, -2, 2 }, //
          { -1, 0, 2, 4, 1 }, //
          { 1, 0, -2, 0, 1 }, //
          { 1, 1, 2, -2, 1 }, //
          { 0, 0, 2, 2, 0 }, //
          { -1, 0, 2, -1, 1 }, //
          { -2, 0, 2, 2, 1 }, //
          { 4, 0, 2, 0, 2 }, //
          { 2, -1, 0, 0, 0 }, //
          { 2, 1, 2, -2, 2 }, //
          { 0, 1, 2, 1, 2 }, //
          { 1, 0, 4, -2, 2 }, //
          { -1, -1, 0, 0, 1 }, //
          { 0, 1, 0, 2, 1 }, //
          { -2, 0, 2, 4, 1 }, //
          { 2, 0, 2, 0, 0 }, //
          { 1, 0, 0, 1, 0 }, //
          { -1, 0, 0, 4, 1 }, //
          { -1, 0, 4, 0, 1 }, //
          { 2, 0, 2, 2, 1 }, //
          { 0, 0, 2, -3, 2 }, //
          { -1, -2, 0, 2, 0 }, //
          { 2, 1, 0, 0, 0 }, //
          { 0, 0, 4, 0, 2 }, //
          { 0, 0, 0, 0, 3 }, //
          { 0, 3, 0, 0, 0 }, //
          { 0, 0, 2, -4, 1 }, //
          { 0, -1, 0, 2, 1 }, //
          { 0, 0, 0, 4, 1 }, //
          { -1, -1, 2, 4, 2 }, //
          { 1, 0, 2, 4, 2 }, //
          { -2, 2, 0, 2, 0 }, //
          { -2, -1, 2, 0, 1 }, //
          { -2, 0, 0, 2, 2 }, //
          { -1, -1, 2, 0, 2 }, //
          { 0, 0, 4, -2, 1 }, //
          { 3, 0, 2, -2, 1 }, //
          { -2, -1, 0, 2, 1 }, //
          { 1, 0, 0, -1, 1 }, //
          { 0, -2, 0, 2, 0 }, //
          { -2, 0, 0, 4, 1 }, //
          { -3, 0, 0, 0, 1 }, //
          { 1, 1, 2, 2, 2 }, //
          { 0, 0, 2, 4, 1 }, //
          { 3, 0, 2, 2, 2 }, //
          { -1, 1, 2, -2, 1 }, //
          { 2, 0, 0, -4, 1 }, //
          { 0, 0, 0, -2, 2 }, //
          { 2, 0, 2, -4, 1 }, //
          { -1, 1, 0, 2, 1 }, //
          { 0, 0, 2, -1, 1 }, //
          { 0, -2, 2, 2, 2 }, //
          { 2, 0, 0, 2, 1 }, //
          { 4, 0, 2, -2, 2 }, //
          { 2, 0, 0, -2, 2 }, //
          { 0, 2, 0, 0, 1 }, //
          { 1, 0, 0, -4, 1 }, //
          { 0, 2, 2, -2, 1 }, //
          { -3, 0, 0, 4, 0 }, //
          { -1, 1, 2, 0, 1 }, //
          { -1, -1, 0, 4, 0 }, //
          { -1, -2, 2, 2, 2 }, //
          { -2, -1, 2, 4, 2 }, //
          { 1, -1, 2, 2, 1 }, //
          { -2, 1, 0, 2, 0 }, //
          { -2, 1, 2, 0, 1 }, //
          { 2, 1, 0, -2, 1 }, //
          { -3, 0, 2, 0, 1 }, //
          { -2, 0, 2, -2, 1 }, //
          { -1, 1, 0, 2, 2 }, //
          { 0, -1, 2, -1, 2 }, //
          { -1, 0, 4, -2, 2 }, //
          { 0, -2, 2, 0, 2 }, //
          { -1, 0, 2, 1, 2 }, //
          { 2, 0, 0, 0, 2 }, //
          { 0, 0, 2, 0, 3 }, //
          { -2, 0, 4, 0, 2 }, //
          { -1, 0, -2, 0, 1 }, //
          { -1, 1, 2, 2, 1 }, //
          { 3, 0, 0, 0, 1 }, //
          { -1, 0, 2, 3, 2 }, //
          { 2, -1, 2, 0, 1 }, //
          { 0, 1, 2, 2, 1 }, //
          { 0, -1, 2, 4, 2 }, //
          { 2, -1, 2, 2, 2 }, //
          { 0, 2, -2, 2, 0 }, //
          { -1, -1, 2, -1, 1 }, //
          { 0, -2, 0, 0, 1 }, //
          { 1, 0, 2, -4, 2 }, //
          { 1, -1, 0, -2, 1 }, //
          { -1, -1, 2, 0, 1 }, //
          { 1, -1, 2, -2, 2 }, //
          { -2, -1, 0, 4, 0 }, //
          { -1, 0, 0, 3, 0 }, //
          { -2, -1, 2, 2, 2 }, //
          { 0, 2, 2, 0, 2 }, //
          { 1, 1, 0, 2, 0 }, //
          { 2, 0, 2, -1, 2 }, //
          { 1, 0, 2, 1, 1 }, //
          { 4, 0, 0, 0, 0 }, //
          { 2, 1, 2, 0, 1 }, //
          { 3, -1, 2, 0, 2 }, //
          { -2, 2, 0, 2, 1 }, //
          { 1, 0, 2, -3, 1 }, //
          { 1, 1, 2, -4, 1 }, //
          { -1, -1, 2, -2, 1 }, //
          { 0, -1, 0, -1, 1 }, //
          { 0, -1, 0, -2, 1 }, //
          { -2, 0, 0, 0, 2 }, //
          { -2, 0, -2, 2, 0 }, //
          { -1, 0, -2, 4, 0 }, //
          { 1, -2, 0, 0, 0 }, //
          { 0, 1, 0, 1, 1 }, //
          { -1, 2, 0, 2, 0 }, //
          { 1, -1, 2, -2, 1 }, //
          { 1, 2, 2, -2, 2 }, //
          { 2, -1, 2, -2, 2 }, //
          { 1, 0, 2, -1, 1 }, //
          { 2, 1, 2, -2, 1 }, //
          { -2, 0, 0, -2, 1 }, //
          { 1, -2, 2, 0, 2 }, //
          { 0, 1, 2, 1, 1 }, //
          { 1, 0, 4, -2, 1 }, //
          { -2, 0, 4, 2, 2 }, //
          { 1, 1, 2, 1, 2 }, //
          { 1, 0, 0, 4, 0 }, //
          { 1, 0, 2, 2, 0 }, //
          { 2, 0, 2, 1, 2 }, //
          { 3, 1, 2, 0, 2 }, //
          { 4, 0, 2, 0, 1 }, //
          { -2, -1, 2, 0, 0 }, //
          { 0, 1, -2, 2, 1 }, //
          { 1, 0, -2, 1, 0 }, //
          { 0, -1, -2, 2, 1 }, //
          { 2, -1, 0, -2, 1 }, //
          { -1, 0, 2, -1, 2 }, //
          { 1, 0, 2, -3, 2 }, //
          { 0, 1, 2, -2, 3 }, //
          { 0, 0, 2, -3, 1 }, //
          { -1, 0, -2, 2, 1 }, //
          { 0, 0, 2, -4, 2 }, //
          { -2, 1, 0, 0, 1 }, //
          { -1, 0, 0, -1, 1 }, //
          { 2, 0, 2, -4, 2 }, //
          { 0, 0, 4, -4, 4 }, //
          { 0, 0, 4, -4, 2 }, //
          { -1, -2, 0, 2, 1 }, //
          { -2, 0, 0, 3, 0 }, //
          { 1, 0, -2, 2, 1 }, //
          { -3, 0, 2, 2, 2 }, //
          { -3, 0, 2, 2, 1 }, //
          { -2, 0, 2, 2, 0 }, //
          { 2, -1, 0, 0, 1 }, //
          { -2, 1, 2, 2, 2 }, //
          { 1, 1, 0, 1, 0 }, //
          { 0, 1, 4, -2, 2 }, //
          { -1, 1, 0, -2, 1 }, //
          { 0, 0, 0, -4, 1 }, //
          { 1, -1, 0, 2, 1 }, //
          { 1, 1, 0, 2, 1 }, //
          { -1, 2, 2, 2, 2 }, //
          { 3, 1, 2, -2, 2 }, //
          { 0, -1, 0, 4, 0 }, //
          { 2, -1, 0, 2, 0 }, //
          { 0, 0, 4, 0, 1 }, //
          { 2, 0, 4, -2, 2 }, //
          { -1, -1, 2, 4, 1 }, //
          { 1, 0, 0, 4, 1 }, //
          { 1, -2, 2, 2, 2 }, //
          { 0, 0, 2, 3, 2 }, //
          { -1, 1, 2, 4, 2 }, //
          { 3, 0, 0, 2, 0 }, //
          { -1, 0, 4, 2, 2 }, //
          { 1, 1, 2, 2, 1 }, //
          { -2, 0, 2, 6, 2 }, //
          { 2, 1, 2, 2, 2 }, //
          { -1, 0, 2, 6, 2 }, //
          { 1, 0, 2, 4, 1 }, //
          { 2, 0, 2, 4, 2 }, //
          { 1, 1, -2, 1, 0 }, //
          { -3, 1, 2, 1, 2 }, //
          { 2, 0, -2, 0, 2 }, //
          { -1, 0, 0, 1, 2 }, //
          { -4, 0, 2, 2, 1 }, //
          { -1, -1, 0, 1, 0 }, //
          { 0, 0, -2, 2, 2 }, //
          { 1, 0, 0, -1, 2 }, //
          { 0, -1, 2, -2, 3 }, //
          { -2, 1, 2, 0, 0 }, //
          { 0, 0, 2, -2, 4 }, //
          { -2, -2, 0, 2, 0 }, //
          { -2, 0, -2, 4, 0 }, //
          { 0, -2, -2, 2, 0 }, //
          { 1, 2, 0, -2, 1 }, //
          { 3, 0, 0, -4, 1 }, //
          { -1, 1, 2, -2, 2 }, //
          { 1, -1, 2, -4, 1 }, //
          { 1, 1, 0, -2, 2 }, //
          { -3, 0, 2, 0, 0 }, //
          { -3, 0, 2, 0, 2 }, //
          { -2, 0, 0, 1, 0 }, //
          { 0, 0, -2, 1, 0 }, //
          { -3, 0, 0, 2, 1 }, //
          { -1, -1, -2, 2, 0 }, //
          { 0, 1, 2, -4, 1 }, //
          { 2, 1, 0, -4, 1 }, //
          { 0, 2, 0, -2, 1 }, //
          { 1, 0, 0, -3, 1 }, //
          { -2, 0, 2, -2, 2 }, //
          { -2, -1, 0, 0, 1 }, //
          { -4, 0, 0, 2, 0 }, //
          { 1, 1, 0, -4, 1 }, //
          { -1, 0, 2, -4, 1 }, //
          { 0, 0, 4, -4, 1 }, //
          { 0, 3, 2, -2, 2 }, //
          { -3, -1, 0, 4, 0 }, //
          { -3, 0, 0, 4, 1 }, //
          { 1, -1, -2, 2, 0 }, //
          { -1, -1, 0, 2, 2 }, //
          { 1, -2, 0, 0, 1 }, //
          { 1, -1, 0, 0, 2 }, //
          { 0, 0, 0, 1, 2 }, //
          { -1, -1, 2, 0, 0 }, //
          { 1, -2, 2, -2, 2 }, //
          { 0, -1, 2, -1, 1 }, //
          { -1, 0, 2, 0, 3 }, //
          { 1, 1, 0, 0, 2 }, //
          { -1, 1, 2, 0, 0 }, //
          { 1, 2, 0, 0, 0 }, //
          { -1, 2, 2, 0, 2 }, //
          { -1, 0, 4, -2, 1 }, //
          { 3, 0, 2, -4, 2 }, //
          { 1, 2, 2, -2, 1 }, //
          { 1, 0, 4, -4, 2 }, //
          { -2, -1, 0, 4, 1 }, //
          { 0, -1, 0, 2, 2 }, //
          { -2, 1, 0, 4, 0 }, //
          { -2, -1, 2, 2, 1 }, //
          { 2, 0, -2, 2, 0 }, //
          { 1, 0, 0, 1, 1 }, //
          { 0, 1, 0, 2, 2 }, //
          { 1, -1, 2, -1, 2 }, //
          { -2, 0, 4, 0, 1 }, //
          { 2, 1, 0, 0, 1 }, //
          { 0, 1, 2, 0, 0 }, //
          { 0, -1, 4, -2, 2 }, //
          { 0, 0, 4, -2, 4 }, //
          { 0, 2, 2, 0, 1 }, //
          { -3, 0, 0, 6, 0 }, //
          { -1, -1, 0, 4, 1 }, //
          { 1, -2, 0, 2, 0 }, //
          { -1, 0, 0, 4, 2 }, //
          { -1, -2, 2, 2, 1 }, //
          { -1, 0, 0, -2, 2 }, //
          { 1, 0, -2, -2, 1 }, //
          { 0, 0, -2, -2, 1 }, //
          { -2, 0, -2, 0, 1 }, //
          { 0, 0, 0, 3, 1 }, //
          { 0, 0, 0, 3, 0 }, //
          { -1, 1, 0, 4, 0 }, //
          { -1, -1, 2, 2, 0 }, //
          { -2, 0, 2, 3, 2 }, //
          { 1, 0, 0, 2, 2 }, //
          { 0, -1, 2, 1, 2 }, //
          { 3, -1, 0, 0, 0 }, //
          { 2, 0, 0, 1, 0 }, //
          { 1, -1, 2, 0, 0 }, //
          { 0, 0, 2, 1, 0 }, //
          { 1, 0, 2, 0, 3 }, //
          { 3, 1, 0, 0, 0 }, //
          { 3, -1, 2, -2, 2 }, //
          { 2, 0, 2, -1, 1 }, //
          { 1, 1, 2, 0, 0 }, //
          { 0, 0, 4, -1, 2 }, //
          { 1, 2, 2, 0, 2 }, //
          { -2, 0, 0, 6, 0 }, //
          { 0, -1, 0, 4, 1 }, //
          { -2, -1, 2, 4, 1 }, //
          { 0, -2, 2, 2, 1 }, //
          { 0, -1, 2, 2, 0 }, //
          { -1, 0, 2, 3, 1 }, //
          { -2, 1, 2, 4, 2 }, //
          { 2, 0, 0, 2, 2 }, //
          { 2, -2, 2, 0, 2 }, //
          { -1, 1, 2, 3, 2 }, //
          { 3, 0, 2, -1, 2 }, //
          { 4, 0, 2, -2, 1 }, //
          { -1, 0, 0, 6, 0 }, //
          { -1, -2, 2, 4, 2 }, //
          { -3, 0, 2, 6, 2 }, //
          { -1, 0, 2, 4, 0 }, //
          { 3, 0, 0, 2, 1 }, //
          { 3, -1, 2, 0, 1 }, //
          { 3, 0, 2, 0, 0 }, //
          { 1, 0, 4, 0, 2 }, //
          { 5, 0, 2, -2, 2 }, //
          { 0, -1, 2, 4, 1 }, //
          { 2, -1, 2, 2, 1 }, //
          { 0, 1, 2, 4, 2 }, //
          { 1, -1, 2, 4, 2 }, //
          { 3, -1, 2, 2, 2 }, //
          { 3, 0, 2, 2, 1 }, //
          { 5, 0, 2, 0, 2 }, //
          { 0, 0, 2, 6, 2 }, //
          { 4, 0, 2, 2, 2 }, //
          { 0, -1, 1, -1, 1 }, //
          { -1, 0, 1, 0, 3 }, //
          { 0, -2, 2, -2, 3 }, //
          { 1, 0, -1, 0, 1 }, //
          { 2, -2, 0, -2, 1 }, //
          { -1, 0, 1, 0, 2 }, //
          { -1, 0, 1, 0, 1 }, //
          { -1, -1, 2, -1, 2 }, //
          { -2, 2, 0, 2, 2 }, //
          { -1, 0, 1, 0, 0 }, //
          { -4, 1, 2, 2, 2 }, //
          { -3, 0, 2, 1, 1 }, //
          { -2, -1, 2, 0, 2 }, //
          { 1, 0, -2, 1, 1 }, //
          { 2, -1, -2, 0, 1 }, //
          { -4, 0, 2, 2, 0 }, //
          { -3, 1, 0, 3, 0 }, //
          { -1, 0, -1, 2, 0 }, //
          { 0, -2, 0, 0, 2 }, //
          { 0, -2, 0, 0, 2 }, //
          { -3, 0, 0, 3, 0 }, //
          { -2, -1, 0, 2, 2 }, //
          { -1, 0, -2, 3, 0 }, //
          { -4, 0, 0, 4, 0 }, //
          { 2, 1, -2, 0, 1 }, //
          { 2, -1, 0, -2, 2 }, //
          { 0, 0, 1, -1, 0 }, //
          { -1, 2, 0, 1, 0 }, //
          { -2, 1, 2, 0, 2 }, //
          { 1, 1, 0, -1, 1 }, //
          { 1, 0, 1, -2, 1 }, //
          { 0, 2, 0, 0, 2 }, //
          { 1, -1, 2, -3, 1 }, //
          { -1, 1, 2, -1, 1 }, //
          { -2, 0, 4, -2, 2 }, //
          { -2, 0, 4, -2, 1 }, //
          { -2, -2, 0, 2, 1 }, //
          { -2, 0, -2, 4, 0 }, //
          { 1, 2, 2, -4, 1 }, //
          { 1, 1, 2, -4, 2 }, //
          { -1, 2, 2, -2, 1 }, //
          { 2, 0, 0, -3, 1 }, //
          { -1, 2, 0, 0, 1 }, //
          { 0, 0, 0, -2, 0 }, //
          { -1, -1, 2, -2, 2 }, //
          { -1, 1, 0, 0, 2 }, //
          { 0, 0, 0, -1, 2 }, //
          { -2, 1, 0, 1, 0 }, //
          { 1, -2, 0, -2, 1 }, //
          { 1, 0, -2, 0, 2 }, //
          { -3, 1, 0, 2, 0 }, //
          { -1, 1, -2, 2, 0 }, //
          { -1, -1, 0, 0, 2 }, //
          { -3, 0, 0, 2, 0 }, //
          { -3, -1, 0, 2, 0 }, //
          { 2, 0, 2, -6, 1 }, //
          { 0, 1, 2, -4, 2 }, //
          { 2, 0, 0, -4, 2 }, //
          { -2, 1, 2, -2, 1 }, //
          { 0, -1, 2, -4, 1 }, //
          { 0, 1, 0, -2, 2 }, //
          { -1, 0, 0, -2, 0 }, //
          { 2, 0, -2, -2, 1 }, //
          { -4, 0, 2, 0, 1 }, //
          { -1, -1, 0, -1, 1 }, //
          { 0, 0, -2, 0, 2 }, //
          { -3, 0, 0, 1, 0 }, //
          { -1, 0, -2, 1, 0 }, //
          { -2, 0, -2, 2, 1 }, //
          { 0, 0, -4, 2, 0 }, //
          { -2, -1, -2, 2, 0 }, //
          { 1, 0, 2, -6, 1 }, //
          { -1, 0, 2, -4, 2 }, //
          { 1, 0, 0, -4, 2 }, //
          { 2, 1, 2, -4, 2 }, //
          { 2, 1, 2, -4, 1 }, //
          { 0, 1, 4, -4, 4 }, //
          { 0, 1, 4, -4, 2 }, //
          { -1, -1, -2, 4, 0 }, //
          { -1, -3, 0, 2, 0 }, //
          { -1, 0, -2, 4, 1 }, //
          { -2, -1, 0, 3, 0 }, //
          { 0, 0, -2, 3, 0 }, //
          { -2, 0, 0, 3, 1 }, //
          { 0, -1, 0, 1, 0 }, //
          { -3, 0, 2, 2, 0 }, //
          { 1, 1, -2, 2, 0 }, //
          { -1, 1, 0, 2, 2 }, //
          { 1, -2, 2, -2, 1 }, //
          { 0, 0, 1, 0, 2 }, //
          { 0, 0, 1, 0, 1 }, //
          { 0, 0, 1, 0, 0 }, //
          { -1, 2, 0, 2, 1 }, //
          { 0, 0, 2, 0, 2 }, //
          { -2, 0, 2, 0, 2 }, //
          { 2, 0, 0, -1, 1 }, //
          { 3, 0, 0, -2, 1 }, //
          { 1, 0, 2, -2, 3 }, //
          { 1, 2, 0, 0, 1 }, //
          { 2, 0, 2, -3, 2 }, //
          { -1, 1, 4, -2, 2 }, //
          { -2, -2, 0, 4, 0 }, //
          { 0, -3, 0, 2, 0 }, //
          { 0, 0, -2, 4, 0 }, //
          { -1, -1, 0, 3, 0 }, //
          { -2, 0, 0, 4, 2 }, //
          { -1, 0, 0, 3, 1 }, //
          { 2, -2, 0, 0, 0 }, //
          { 1, -1, 0, 1, 0 }, //
          { -1, 0, 0, 2, 0 }, //
          { 0, -2, 2, 0, 1 }, //
          { -1, 0, 1, 2, 1 }, //
          { -1, 1, 0, 3, 0 }, //
          { -1, -1, 2, 1, 2 }, //
          { 0, -1, 2, 0, 0 }, //
          { -2, 1, 2, 2, 1 }, //
          { 2, -2, 2, -2, 2 }, //
          { 1, 1, 0, 1, 1 }, //
          { 1, 0, 1, 0, 1 }, //
          { 1, 0, 1, 0, 0 }, //
          { 0, 2, 0, 2, 0 }, //
          { 2, -1, 2, -2, 1 }, //
          { 0, -1, 4, -2, 1 }, //
          { 0, 0, 4, -2, 3 }, //
          { 0, 1, 4, -2, 1 }, //
          { 4, 0, 2, -4, 2 }, //
          { 2, 2, 2, -2, 2 }, //
          { 2, 0, 4, -4, 2 }, //
          { -1, -2, 0, 4, 0 }, //
          { -1, -3, 2, 2, 2 }, //
          { -3, 0, 2, 4, 2 }, //
          { -3, 0, 2, -2, 1 }, //
          { -1, -1, 0, -2, 1 }, //
          { -3, 0, 0, 0, 2 }, //
          { -3, 0, -2, 2, 0 }, //
          { 0, 1, 0, -4, 1 }, //
          { -2, 1, 0, -2, 1 }, //
          { -4, 0, 0, 0, 1 }, //
          { -1, 0, 0, -4, 1 }, //
          { -3, 0, 0, -2, 1 }, //
          { 0, 0, 0, 3, 2 }, //
          { -1, 1, 0, 4, 1 }, //
          { 1, -2, 2, 0, 1 }, //
          { 0, 1, 0, 3, 0 }, //
          { -1, 0, 2, 2, 3 }, //
          { 0, 0, 2, 2, 2 }, //
          { -2, 0, 2, 2, 2 }, //
          { -1, 1, 2, 2, 0 }, //
          { 3, 0, 0, 0, 2 }, //
          { 2, 1, 0, 1, 0 }, //
          { 2, -1, 2, -1, 2 }, //
          { 0, 0, 2, 0, 1 }, //
          { 0, 0, 3, 0, 3 }, //
          { 0, 0, 3, 0, 2 }, //
          { -1, 2, 2, 2, 1 }, //
          { -1, 0, 4, 0, 0 }, //
          { 1, 2, 2, 0, 1 }, //
          { 3, 1, 2, -2, 1 }, //
          { 1, 1, 4, -2, 2 }, //
          { -2, -1, 0, 6, 0 }, //
          { 0, -2, 0, 4, 0 }, //
          { -2, 0, 0, 6, 1 }, //
          { -2, -2, 2, 4, 2 }, //
          { 0, -3, 2, 2, 2 }, //
          { 0, 0, 0, 4, 2 }, //
          { -1, -1, 2, 3, 2 }, //
          { -2, 0, 2, 4, 0 }, //
          { 2, -1, 0, 2, 1 }, //
          { 1, 0, 0, 3, 0 }, //
          { 0, 1, 0, 4, 1 }, //
          { 0, 1, 0, 4, 0 }, //
          { 1, -1, 2, 1, 2 }, //
          { 0, 0, 2, 2, 3 }, //
          { 1, 0, 2, 2, 2 }, //
          { -1, 0, 2, 2, 2 }, //
          { -2, 0, 4, 2, 1 }, //
          { 2, 1, 0, 2, 1 }, //
          { 2, 1, 0, 2, 0 }, //
          { 2, -1, 2, 0, 0 }, //
          { 1, 0, 2, 1, 0 }, //
          { 0, 1, 2, 2, 0 }, //
          { 2, 0, 2, 0, 3 }, //
          { 3, 0, 2, 0, 2 }, //
          { 1, 0, 2, 0, 2 }, //
          { 1, 0, 3, 0, 3 }, //
          { 1, 1, 2, 1, 1 }, //
          { 0, 2, 2, 2, 2 }, //
          { 2, 1, 2, 0, 0 }, //
          { 2, 0, 4, -2, 1 }, //
          { 4, 1, 2, -2, 2 }, //
          { -1, -1, 0, 6, 0 }, //
          { -3, -1, 2, 6, 2 }, //
          { -1, 0, 0, 6, 1 }, //
          { -3, 0, 2, 6, 1 }, //
          { 1, -1, 0, 4, 1 }, //
          { 1, -1, 0, 4, 0 }, //
          { -2, 0, 2, 5, 2 }, //
          { 1, -2, 2, 2, 1 }, //
          { 3, -1, 0, 2, 0 }, //
          { 1, -1, 2, 2, 0 }, //
          { 0, 0, 2, 3, 1 }, //
          { -1, 1, 2, 4, 1 }, //
          { 0, 1, 2, 3, 2 }, //
          { -1, 0, 4, 2, 1 }, //
          { 2, 0, 2, 1, 1 }, //
          { 5, 0, 0, 0, 0 }, //
          { 2, 1, 2, 1, 2 }, //
          { 1, 0, 4, 0, 1 }, //
          { 3, 1, 2, 0, 1 }, //
          { 3, 0, 4, -2, 2 }, //
          { -2, -1, 2, 6, 2 }, //
          { 0, 0, 0, 6, 0 }, //
          { 0, -2, 2, 4, 2 }, //
          { -2, 0, 2, 6, 1 }, //
          { 2, 0, 0, 4, 1 }, //
          { 2, 0, 0, 4, 0 }, //
          { 2, -2, 2, 2, 2 }, //
          { 0, 0, 2, 4, 0 }, //
          { 1, 0, 2, 3, 2 }, //
          { 4, 0, 0, 2, 0 }, //
          { 2, 0, 2, 2, 0 }, //
          { 0, 0, 4, 2, 2 }, //
          { 4, -1, 2, 0, 2 }, //
          { 3, 0, 2, 1, 2 }, //
          { 2, 1, 2, 2, 1 }, //
          { 4, 1, 2, 0, 2 }, //
          { -1, -1, 2, 6, 2 }, //
          { -1, 0, 2, 6, 1 }, //
          { 1, -1, 2, 4, 1 }, //
          { 1, 1, 2, 4, 2 }, //
          { 3, 1, 2, 2, 2 }, //
          { 5, 0, 2, 0, 1 }, //
          { 2, -1, 2, 4, 2 }, //
          { 2, 0, 2, 4, 1 } };

  // Luni-Solar nutation coefficients, unit 1e-7 arcsec:
  // longitude (sin, t*sin, cos), obliquity (cos, t*cos, sin)

  // Each row of coefficients in 'cls_t' belongs with the corresponding
  // row of fundamental-argument multipliers in 'nals_t'.
  static const int32_t cls_t[678][6] = { { -172064161, -174666, 33386, 92052331, 9086, 15377 }, //
          { -13170906, -1675, -13696, 5730336, -3015, -4587 }, //
          { -2276413, -234, 2796, 978459, -485, 1374 }, //
          { 2074554, 207, -698, -897492, 470, -291 }, //
          { 1475877, -3633, 11817, 73871, -184, -1924 }, //
          { -516821, 1226, -524, 224386, -677, -174 }, //
          { 711159, 73, -872, -6750, 0, 358 }, //
          { -387298, -367, 380, 200728, 18, 318 }, //
          { -301461, -36, 816, 129025, -63, 367 }, //
          { 215829, -494, 111, -95929, 299, 132 }, //
          { 128227, 137, 181, -68982, -9, 39 }, //
          { 123457, 11, 19, -53311, 32, -4 }, //
          { 156994, 10, -168, -1235, 0, 82 }, //
          { 63110, 63, 27, -33228, 0, -9 }, //
          { -57976, -63, -189, 31429, 0, -75 }, //
          { -59641, -11, 149, 25543, -11, 66 }, //
          { -51613, -42, 129, 26366, 0, 78 }, //
          { 45893, 50, 31, -24236, -10, 20 }, //
          { 63384, 11, -150, -1220, 0, 29 }, //
          { -38571, -1, 158, 16452, -11, 68 }, //
          { 32481, 0, 0, -13870, 0, 0 }, //
          { -47722, 0, -18, 477, 0, -25 }, //
          { -31046, -1, 131, 13238, -11, 59 }, //
          { 28593, 0, -1, -12338, 10, -3 }, //
          { 20441, 21, 10, -10758, 0, -3 }, //
          { 29243, 0, -74, -609, 0, 13 }, //
          { 25887, 0, -66, -550, 0, 11 }, //
          { -14053, -25, 79, 8551, -2, -45 }, //
          { 15164, 10, 11, -8001, 0, -1 }, //
          { -15794, 72, -16, 6850, -42, -5 }, //
          { 21783, 0, 13, -167, 0, 13 }, //
          { -12873, -10, -37, 6953, 0, -14 }, //
          { -12654, 11, 63, 6415, 0, 26 }, //
          { -10204, 0, 25, 5222, 0, 15 }, //
          { 16707, -85, -10, 168, -1, 10 }, //
          { -7691, 0, 44, 3268, 0, 19 }, //
          { -11024, 0, -14, 104, 0, 2 }, //
          { 7566, -21, -11, -3250, 0, -5 }, //
          { -6637, -11, 25, 3353, 0, 14 }, //
          { -7141, 21, 8, 3070, 0, 4 }, //
          { -6302, -11, 2, 3272, 0, 4 }, //
          { 5800, 10, 2, -3045, 0, -1 }, //
          { 6443, 0, -7, -2768, 0, -4 }, //
          { -5774, -11, -15, 3041, 0, -5 }, //
          { -5350, 0, 21, 2695, 0, 12 }, //
          { -4752, -11, -3, 2719, 0, -3 }, //
          { -4940, -11, -21, 2720, 0, -9 }, //
          { 7350, 0, -8, -51, 0, 4 }, //
          { 4065, 0, 6, -2206, 0, 1 }, //
          { 6579, 0, -24, -199, 0, 2 }, //
          { 3579, 0, 5, -1900, 0, 1 }, //
          { 4725, 0, -6, -41, 0, 3 }, //
          { -3075, 0, -2, 1313, 0, -1 }, //
          { -2904, 0, 15, 1233, 0, 7 }, //
          { 4348, 0, -10, -81, 0, 2 }, //
          { -2878, 0, 8, 1232, 0, 4 }, //
          { -4230, 0, 5, -20, 0, -2 }, //
          { -2819, 0, 7, 1207, 0, 3 }, //
          { -4056, 0, 5, 40, 0, -2 }, //
          { -2647, 0, 11, 1129, 0, 5 }, //
          { -2294, 0, -10, 1266, 0, -4 }, //
          { 2481, 0, -7, -1062, 0, -3 }, //
          { 2179, 0, -2, -1129, 0, -2 }, //
          { 3276, 0, 1, -9, 0, 0 }, //
          { -3389, 0, 5, 35, 0, -2 }, //
          { 3339, 0, -13, -107, 0, 1 }, //
          { -1987, 0, -6, 1073, 0, -2 }, //
          { -1981, 0, 0, 854, 0, 0 }, //
          { 4026, 0, -353, -553, 0, -139 }, //
          { 1660, 0, -5, -710, 0, -2 }, //
          { -1521, 0, 9, 647, 0, 4 }, //
          { 1314, 0, 0, -700, 0, 0 }, //
          { -1283, 0, 0, 672, 0, 0 }, //
          { -1331, 0, 8, 663, 0, 4 }, //
          { 1383, 0, -2, -594, 0, -2 }, //
          { 1405, 0, 4, -610, 0, 2 }, //
          { 1290, 0, 0, -556, 0, 0 }, //
          { -1214, 0, 5, 518, 0, 2 }, //
          { 1146, 0, -3, -490, 0, -1 }, //
          { 1019, 0, -1, -527, 0, -1 }, //
          { -1100, 0, 9, 465, 0, 4 }, //
          { -970, 0, 2, 496, 0, 1 }, //
          { 1575, 0, -6, -50, 0, 0 }, //
          { 934, 0, -3, -399, 0, -1 }, //
          { 922, 0, -1, -395, 0, -1 }, //
          { 815, 0, -1, -422, 0, -1 }, //
          { 834, 0, 2, -440, 0, 1 }, //
          { 1248, 0, 0, -170, 0, 1 }, //
          { 1338, 0, -5, -39, 0, 0 }, //
          { 716, 0, -2, -389, 0, -1 }, //
          { 1282, 0, -3, -23, 0, 1 }, //
          { 742, 0, 1, -391, 0, 0 }, //
          { 1020, 0, -25, -495, 0, -10 }, //
          { 715, 0, -4, -326, 0, 2 }, //
          { -666, 0, -3, 369, 0, -1 }, //
          { -667, 0, 1, 346, 0, 1 }, //
          { -704, 0, 0, 304, 0, 0 }, //
          { -694, 0, 5, 294, 0, 2 }, //
          { -1014, 0, -1, 4, 0, -1 }, //
          { -585, 0, -2, 316, 0, -1 }, //
          { -949, 0, 1, 8, 0, -1 }, //
          { -595, 0, 0, 258, 0, 0 }, //
          { 528, 0, 0, -279, 0, 0 }, //
          { -590, 0, 4, 252, 0, 2 }, //
          { 570, 0, -2, -244, 0, -1 }, //
          { -502, 0, 3, 250, 0, 2 }, //
          { -875, 0, 1, 29, 0, 0 }, //
          { -492, 0, -3, 275, 0, -1 }, //
          { 535, 0, -2, -228, 0, -1 }, //
          { -467, 0, 1, 240, 0, 1 }, //
          { 591, 0, 0, -253, 0, 0 }, //
          { -453, 0, -1, 244, 0, -1 }, //
          { 766, 0, 1, 9, 0, 0 }, //
          { -446, 0, 2, 225, 0, 1 }, //
          { -488, 0, 2, 207, 0, 1 }, //
          { -468, 0, 0, 201, 0, 0 }, //
          { -421, 0, 1, 216, 0, 1 }, //
          { 463, 0, 0, -200, 0, 0 }, //
          { -673, 0, 2, 14, 0, 0 }, //
          { 658, 0, 0, -2, 0, 0 }, //
          { -438, 0, 0, 188, 0, 0 }, //
          { -390, 0, 0, 205, 0, 0 }, //
          { 639, -11, -2, -19, 0, 0 }, //
          { 412, 0, -2, -176, 0, -1 }, //
          { -361, 0, 0, 189, 0, 0 }, //
          { 360, 0, -1, -185, 0, -1 }, //
          { 588, 0, -3, -24, 0, 0 }, //
          { -578, 0, 1, 5, 0, 0 }, //
          { -396, 0, 0, 171, 0, 0 }, //
          { 565, 0, -1, -6, 0, 0 }, //
          { -335, 0, -1, 184, 0, -1 }, //
          { 357, 0, 1, -154, 0, 0 }, //
          { 321, 0, 1, -174, 0, 0 }, //
          { -301, 0, -1, 162, 0, 0 }, //
          { -334, 0, 0, 144, 0, 0 }, //
          { 493, 0, -2, -15, 0, 0 }, //
          { 494, 0, -2, -19, 0, 0 }, //
          { 337, 0, -1, -143, 0, -1 }, //
          { 280, 0, -1, -144, 0, 0 }, //
          { 309, 0, 1, -134, 0, 0 }, //
          { -263, 0, 2, 131, 0, 1 }, //
          { 253, 0, 1, -138, 0, 0 }, //
          { 245, 0, 0, -128, 0, 0 }, //
          { 416, 0, -2, -17, 0, 0 }, //
          { -229, 0, 0, 128, 0, 0 }, //
          { 231, 0, 0, -120, 0, 0 }, //
          { -259, 0, 2, 109, 0, 1 }, //
          { 375, 0, -1, -8, 0, 0 }, //
          { 252, 0, 0, -108, 0, 0 }, //
          { -245, 0, 1, 104, 0, 0 }, //
          { 243, 0, -1, -104, 0, 0 }, //
          { 208, 0, 1, -112, 0, 0 }, //
          { 199, 0, 0, -102, 0, 0 }, //
          { -208, 0, 1, 105, 0, 0 }, //
          { 335, 0, -2, -14, 0, 0 }, //
          { -325, 0, 1, 7, 0, 0 }, //
          { -187, 0, 0, 96, 0, 0 }, //
          { 197, 0, -1, -100, 0, 0 }, //
          { -192, 0, 2, 94, 0, 1 }, //
          { -188, 0, 0, 83, 0, 0 }, //
          { 276, 0, 0, -2, 0, 0 }, //
          { -286, 0, 1, 6, 0, 0 }, //
          { 186, 0, -1, -79, 0, 0 }, //
          { -219, 0, 0, 43, 0, 0 }, //
          { 276, 0, 0, 2, 0, 0 }, //
          { -153, 0, -1, 84, 0, 0 }, //
          { -156, 0, 0, 81, 0, 0 }, //
          { -154, 0, 1, 78, 0, 0 }, //
          { -174, 0, 1, 75, 0, 0 }, //
          { -163, 0, 2, 69, 0, 1 }, //
          { -228, 0, 0, 1, 0, 0 }, //
          { 91, 0, -4, -54, 0, -2 }, //
          { 175, 0, 0, -75, 0, 0 }, //
          { -159, 0, 0, 69, 0, 0 }, //
          { 141, 0, 0, -72, 0, 0 }, //
          { 147, 0, 0, -75, 0, 0 }, //
          { -132, 0, 0, 69, 0, 0 }, //
          { 159, 0, -28, -54, 0, 11 }, //
          { 213, 0, 0, -4, 0, 0 }, //
          { 123, 0, 0, -64, 0, 0 }, //
          { -118, 0, -1, 66, 0, 0 }, //
          { 144, 0, -1, -61, 0, 0 }, //
          { -121, 0, 1, 60, 0, 0 }, //
          { -134, 0, 1, 56, 0, 1 }, //
          { -105, 0, 0, 57, 0, 0 }, //
          { -102, 0, 0, 56, 0, 0 }, //
          { 120, 0, 0, -52, 0, 0 }, //
          { 101, 0, 0, -54, 0, 0 }, //
          { -113, 0, 0, 59, 0, 0 }, //
          { -106, 0, 0, 61, 0, 0 }, //
          { -129, 0, 1, 55, 0, 0 }, //
          { -114, 0, 0, 57, 0, 0 }, //
          { 113, 0, -1, -49, 0, 0 }, //
          { -102, 0, 0, 44, 0, 0 }, //
          { -94, 0, 0, 51, 0, 0 }, //
          { -100, 0, -1, 56, 0, 0 }, //
          { 87, 0, 0, -47, 0, 0 }, //
          { 161, 0, 0, -1, 0, 0 }, //
          { 96, 0, 0, -50, 0, 0 }, //
          { 151, 0, -1, -5, 0, 0 }, //
          { -104, 0, 0, 44, 0, 0 }, //
          { -110, 0, 0, 48, 0, 0 }, //
          { -100, 0, 1, 50, 0, 0 }, //
          { 92, 0, -5, 12, 0, -2 }, //
          { 82, 0, 0, -45, 0, 0 }, //
          { 82, 0, 0, -45, 0, 0 }, //
          { -78, 0, 0, 41, 0, 0 }, //
          { -77, 0, 0, 43, 0, 0 }, //
          { 2, 0, 0, 54, 0, 0 }, //
          { 94, 0, 0, -40, 0, 0 }, //
          { -93, 0, 0, 40, 0, 0 }, //
          { -83, 0, 10, 40, 0, -2 }, //
          { 83, 0, 0, -36, 0, 0 }, //
          { -91, 0, 0, 39, 0, 0 }, //
          { 128, 0, 0, -1, 0, 0 }, //
          { -79, 0, 0, 34, 0, 0 }, //
          { -83, 0, 0, 47, 0, 0 }, //
          { 84, 0, 0, -44, 0, 0 }, //
          { 83, 0, 0, -43, 0, 0 }, //
          { 91, 0, 0, -39, 0, 0 }, //
          { -77, 0, 0, 39, 0, 0 }, //
          { 84, 0, 0, -43, 0, 0 }, //
          { -92, 0, 1, 39, 0, 0 }, //
          { -92, 0, 1, 39, 0, 0 }, //
          { -94, 0, 0, 0, 0, 0 }, //
          { 68, 0, 0, -36, 0, 0 }, //
          { -61, 0, 0, 32, 0, 0 }, //
          { 71, 0, 0, -31, 0, 0 }, //
          { 62, 0, 0, -34, 0, 0 }, //
          { -63, 0, 0, 33, 0, 0 }, //
          { -73, 0, 0, 32, 0, 0 }, //
          { 115, 0, 0, -2, 0, 0 }, //
          { -103, 0, 0, 2, 0, 0 }, //
          { 63, 0, 0, -28, 0, 0 }, //
          { 74, 0, 0, -32, 0, 0 }, //
          { -103, 0, -3, 3, 0, -1 }, //
          { -69, 0, 0, 30, 0, 0 }, //
          { 57, 0, 0, -29, 0, 0 }, //
          { 94, 0, 0, -4, 0, 0 }, //
          { 64, 0, 0, -33, 0, 0 }, //
          { -63, 0, 0, 26, 0, 0 }, //
          { -38, 0, 0, 20, 0, 0 }, //
          { -43, 0, 0, 24, 0, 0 }, //
          { -45, 0, 0, 23, 0, 0 }, //
          { 47, 0, 0, -24, 0, 0 }, //
          { -48, 0, 0, 25, 0, 0 }, //
          { 45, 0, 0, -26, 0, 0 }, //
          { 56, 0, 0, -25, 0, 0 }, //
          { 88, 0, 0, 2, 0, 0 }, //
          { -75, 0, 0, 0, 0, 0 }, //
          { 85, 0, 0, 0, 0, 0 }, //
          { 49, 0, 0, -26, 0, 0 }, //
          { -74, 0, -3, -1, 0, -1 }, //
          { -39, 0, 0, 21, 0, 0 }, //
          { 45, 0, 0, -20, 0, 0 }, //
          { 51, 0, 0, -22, 0, 0 }, //
          { -40, 0, 0, 21, 0, 0 }, //
          { 41, 0, 0, -21, 0, 0 }, //
          { -42, 0, 0, 24, 0, 0 }, //
          { -51, 0, 0, 22, 0, 0 }, //
          { -42, 0, 0, 22, 0, 0 }, //
          { 39, 0, 0, -21, 0, 0 }, //
          { 46, 0, 0, -18, 0, 0 }, //
          { -53, 0, 0, 22, 0, 0 }, //
          { 82, 0, 0, -4, 0, 0 }, //
          { 81, 0, -1, -4, 0, 0 }, //
          { 47, 0, 0, -19, 0, 0 }, //
          { 53, 0, 0, -23, 0, 0 }, //
          { -45, 0, 0, 22, 0, 0 }, //
          { -44, 0, 0, -2, 0, 0 }, //
          { -33, 0, 0, 16, 0, 0 }, //
          { -61, 0, 0, 1, 0, 0 }, //
          { 28, 0, 0, -15, 0, 0 }, //
          { -38, 0, 0, 19, 0, 0 }, //
          { -33, 0, 0, 21, 0, 0 }, //
          { -60, 0, 0, 0, 0, 0 }, //
          { 48, 0, 0, -10, 0, 0 }, //
          { 27, 0, 0, -14, 0, 0 }, //
          { 38, 0, 0, -20, 0, 0 }, //
          { 31, 0, 0, -13, 0, 0 }, //
          { -29, 0, 0, 15, 0, 0 }, //
          { 28, 0, 0, -15, 0, 0 }, //
          { -32, 0, 0, 15, 0, 0 }, //
          { 45, 0, 0, -8, 0, 0 }, //
          { -44, 0, 0, 19, 0, 0 }, //
          { 28, 0, 0, -15, 0, 0 }, //
          { -51, 0, 0, 0, 0, 0 }, //
          { -36, 0, 0, 20, 0, 0 }, //
          { 44, 0, 0, -19, 0, 0 }, //
          { 26, 0, 0, -14, 0, 0 }, //
          { -60, 0, 0, 2, 0, 0 }, //
          { 35, 0, 0, -18, 0, 0 }, //
          { -27, 0, 0, 11, 0, 0 }, //
          { 47, 0, 0, -1, 0, 0 }, //
          { 36, 0, 0, -15, 0, 0 }, //
          { -36, 0, 0, 20, 0, 0 }, //
          { -35, 0, 0, 19, 0, 0 }, //
          { -37, 0, 0, 19, 0, 0 }, //
          { 32, 0, 0, -16, 0, 0 }, //
          { 35, 0, 0, -14, 0, 0 }, //
          { 32, 0, 0, -13, 0, 0 }, //
          { 65, 0, 0, -2, 0, 0 }, //
          { 47, 0, 0, -1, 0, 0 }, //
          { 32, 0, 0, -16, 0, 0 }, //
          { 37, 0, 0, -16, 0, 0 }, //
          { -30, 0, 0, 15, 0, 0 }, //
          { -32, 0, 0, 16, 0, 0 }, //
          { -31, 0, 0, 13, 0, 0 }, //
          { 37, 0, 0, -16, 0, 0 }, //
          { 31, 0, 0, -13, 0, 0 }, //
          { 49, 0, 0, -2, 0, 0 }, //
          { 32, 0, 0, -13, 0, 0 }, //
          { 23, 0, 0, -12, 0, 0 }, //
          { -43, 0, 0, 18, 0, 0 }, //
          { 26, 0, 0, -11, 0, 0 }, //
          { -32, 0, 0, 14, 0, 0 }, //
          { -29, 0, 0, 14, 0, 0 }, //
          { -27, 0, 0, 12, 0, 0 }, //
          { 30, 0, 0, 0, 0, 0 }, //
          { -11, 0, 0, 5, 0, 0 }, //
          { -21, 0, 0, 10, 0, 0 }, //
          { -34, 0, 0, 15, 0, 0 }, //
          { -10, 0, 0, 6, 0, 0 }, //
          { -36, 0, 0, 0, 0, 0 }, //
          { -9, 0, 0, 4, 0, 0 }, //
          { -12, 0, 0, 5, 0, 0 }, //
          { -21, 0, 0, 5, 0, 0 }, //
          { -29, 0, 0, -1, 0, 0 }, //
          { -15, 0, 0, 3, 0, 0 }, //
          { -20, 0, 0, 0, 0, 0 }, //
          { 28, 0, 0, 0, 0, -2 }, //
          { 17, 0, 0, 0, 0, 0 }, //
          { -22, 0, 0, 12, 0, 0 }, //
          { -14, 0, 0, 7, 0, 0 }, //
          { 24, 0, 0, -11, 0, 0 }, //
          { 11, 0, 0, -6, 0, 0 }, //
          { 14, 0, 0, -6, 0, 0 }, //
          { 24, 0, 0, 0, 0, 0 }, //
          { 18, 0, 0, -8, 0, 0 }, //
          { -38, 0, 0, 0, 0, 0 }, //
          { -31, 0, 0, 0, 0, 0 }, //
          { -16, 0, 0, 8, 0, 0 }, //
          { 29, 0, 0, 0, 0, 0 }, //
          { -18, 0, 0, 10, 0, 0 }, //
          { -10, 0, 0, 5, 0, 0 }, //
          { -17, 0, 0, 10, 0, 0 }, //
          { 9, 0, 0, -4, 0, 0 }, //
          { 16, 0, 0, -6, 0, 0 }, //
          { 22, 0, 0, -12, 0, 0 }, //
          { 20, 0, 0, 0, 0, 0 }, //
          { -13, 0, 0, 6, 0, 0 }, //
          { -17, 0, 0, 9, 0, 0 }, //
          { -14, 0, 0, 8, 0, 0 }, //
          { 0, 0, 0, -7, 0, 0 }, //
          { 14, 0, 0, 0, 0, 0 }, //
          { 19, 0, 0, -10, 0, 0 }, //
          { -34, 0, 0, 0, 0, 0 }, //
          { -20, 0, 0, 8, 0, 0 }, //
          { 9, 0, 0, -5, 0, 0 }, //
          { -18, 0, 0, 7, 0, 0 }, //
          { 13, 0, 0, -6, 0, 0 }, //
          { 17, 0, 0, 0, 0, 0 }, //
          { -12, 0, 0, 5, 0, 0 }, //
          { 15, 0, 0, -8, 0, 0 }, //
          { -11, 0, 0, 3, 0, 0 }, //
          { 13, 0, 0, -5, 0, 0 }, //
          { -18, 0, 0, 0, 0, 0 }, //
          { -35, 0, 0, 0, 0, 0 }, //
          { 9, 0, 0, -4, 0, 0 }, //
          { -19, 0, 0, 10, 0, 0 }, //
          { -26, 0, 0, 11, 0, 0 }, //
          { 8, 0, 0, -4, 0, 0 }, //
          { -10, 0, 0, 4, 0, 0 }, //
          { 10, 0, 0, -6, 0, 0 }, //
          { -21, 0, 0, 9, 0, 0 }, //
          { -15, 0, 0, 0, 0, 0 }, //
          { 9, 0, 0, -5, 0, 0 }, //
          { -29, 0, 0, 0, 0, 0 }, //
          { -19, 0, 0, 10, 0, 0 }, //
          { 12, 0, 0, -5, 0, 0 }, //
          { 22, 0, 0, -9, 0, 0 }, //
          { -10, 0, 0, 5, 0, 0 }, //
          { -20, 0, 0, 11, 0, 0 }, //
          { -20, 0, 0, 0, 0, 0 }, //
          { -17, 0, 0, 7, 0, 0 }, //
          { 15, 0, 0, -3, 0, 0 }, //
          { 8, 0, 0, -4, 0, 0 }, //
          { 14, 0, 0, 0, 0, 0 }, //
          { -12, 0, 0, 6, 0, 0 }, //
          { 25, 0, 0, 0, 0, 0 }, //
          { -13, 0, 0, 6, 0, 0 }, //
          { -14, 0, 0, 8, 0, 0 }, //
          { 13, 0, 0, -5, 0, 0 }, //
          { -17, 0, 0, 9, 0, 0 }, //
          { -12, 0, 0, 6, 0, 0 }, //
          { -10, 0, 0, 5, 0, 0 }, //
          { 10, 0, 0, -6, 0, 0 }, //
          { -15, 0, 0, 0, 0, 0 }, //
          { -22, 0, 0, 0, 0, 0 }, //
          { 28, 0, 0, -1, 0, 0 }, //
          { 15, 0, 0, -7, 0, 0 }, //
          { 23, 0, 0, -10, 0, 0 }, //
          { 12, 0, 0, -5, 0, 0 }, //
          { 29, 0, 0, -1, 0, 0 }, //
          { -25, 0, 0, 1, 0, 0 }, //
          { 22, 0, 0, 0, 0, 0 }, //
          { -18, 0, 0, 0, 0, 0 }, //
          { 15, 0, 0, 3, 0, 0 }, //
          { -23, 0, 0, 0, 0, 0 }, //
          { 12, 0, 0, -5, 0, 0 }, //
          { -8, 0, 0, 4, 0, 0 }, //
          { -19, 0, 0, 0, 0, 0 }, //
          { -10, 0, 0, 4, 0, 0 }, //
          { 21, 0, 0, -9, 0, 0 }, //
          { 23, 0, 0, -1, 0, 0 }, //
          { -16, 0, 0, 8, 0, 0 }, //
          { -19, 0, 0, 9, 0, 0 }, //
          { -22, 0, 0, 10, 0, 0 }, //
          { 27, 0, 0, -1, 0, 0 }, //
          { 16, 0, 0, -8, 0, 0 }, //
          { 19, 0, 0, -8, 0, 0 }, //
          { 9, 0, 0, -4, 0, 0 }, //
          { -9, 0, 0, 4, 0, 0 }, //
          { -9, 0, 0, 4, 0, 0 }, //
          { -8, 0, 0, 4, 0, 0 }, //
          { 18, 0, 0, -9, 0, 0 }, //
          { 16, 0, 0, -1, 0, 0 }, //
          { -10, 0, 0, 4, 0, 0 }, //
          { -23, 0, 0, 9, 0, 0 }, //
          { 16, 0, 0, -1, 0, 0 }, //
          { -12, 0, 0, 6, 0, 0 }, //
          { -8, 0, 0, 4, 0, 0 }, //
          { 30, 0, 0, -2, 0, 0 }, //
          { 24, 0, 0, -10, 0, 0 }, //
          { 10, 0, 0, -4, 0, 0 }, //
          { -16, 0, 0, 7, 0, 0 }, //
          { -16, 0, 0, 7, 0, 0 }, //
          { 17, 0, 0, -7, 0, 0 }, //
          { -24, 0, 0, 10, 0, 0 }, //
          { -12, 0, 0, 5, 0, 0 }, //
          { -24, 0, 0, 11, 0, 0 }, //
          { -23, 0, 0, 9, 0, 0 }, //
          { -13, 0, 0, 5, 0, 0 }, //
          { -15, 0, 0, 7, 0, 0 }, //
          { 0, 0, -1988, 0, 0, -1679 }, //
          { 0, 0, -63, 0, 0, -27 }, //
          { -4, 0, 0, 0, 0, 0 }, //
          { 0, 0, 5, 0, 0, 4 }, //
          { 5, 0, 0, -3, 0, 0 }, //
          { 0, 0, 364, 0, 0, 176 }, //
          { 0, 0, -1044, 0, 0, -891 }, //
          { -3, 0, 0, 1, 0, 0 }, //
          { 4, 0, 0, -2, 0, 0 }, //
          { 0, 0, 330, 0, 0, 0 }, //
          { 5, 0, 0, -2, 0, 0 }, //
          { 3, 0, 0, -2, 0, 0 }, //
          { -3, 0, 0, 1, 0, 0 }, //
          { -5, 0, 0, 2, 0, 0 }, //
          { 3, 0, 0, -1, 0, 0 }, //
          { 3, 0, 0, 0, 0, 0 }, //
          { 3, 0, 0, 0, 0, 0 }, //
          { 0, 0, 5, 0, 0, 0 }, //
          { 0, 0, 0, 1, 0, 0 }, //
          { 4, 0, 0, -2, 0, 0 }, //
          { 6, 0, 0, 0, 0, 0 }, //
          { 5, 0, 0, -2, 0, 0 }, //
          { -7, 0, 0, 0, 0, 0 }, //
          { -12, 0, 0, 0, 0, 0 }, //
          { 5, 0, 0, -3, 0, 0 }, //
          { 3, 0, 0, -1, 0, 0 }, //
          { -5, 0, 0, 0, 0, 0 }, //
          { 3, 0, 0, 0, 0, 0 }, //
          { -7, 0, 0, 3, 0, 0 }, //
          { 7, 0, 0, -4, 0, 0 }, //
          { 0, 0, -12, 0, 0, -10 }, //
          { 4, 0, 0, -2, 0, 0 }, //
          { 3, 0, 0, -2, 0, 0 }, //
          { -3, 0, 0, 2, 0, 0 }, //
          { -7, 0, 0, 3, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { -3, 0, 0, 1, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0 }, //
          { -3, 0, 0, 1, 0, 0 }, //
          { 7, 0, 0, -3, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { 4, 0, 0, -2, 0, 0 }, //
          { -5, 0, 0, 3, 0, 0 }, //
          { 5, 0, 0, 0, 0, 0 }, //
          { -5, 0, 0, 2, 0, 0 }, //
          { 5, 0, 0, -2, 0, 0 }, //
          { -8, 0, 0, 3, 0, 0 }, //
          { 9, 0, 0, 0, 0, 0 }, //
          { 6, 0, 0, -3, 0, 0 }, //
          { -5, 0, 0, 2, 0, 0 }, //
          { 3, 0, 0, 0, 0, 0 }, //
          { -7, 0, 0, 0, 0, 0 }, //
          { -3, 0, 0, 1, 0, 0 }, //
          { 5, 0, 0, 0, 0, 0 }, //
          { 3, 0, 0, 0, 0, 0 }, //
          { -3, 0, 0, 2, 0, 0 }, //
          { 4, 0, 0, -2, 0, 0 }, //
          { 3, 0, 0, -1, 0, 0 }, //
          { -5, 0, 0, 2, 0, 0 }, //
          { 4, 0, 0, -2, 0, 0 }, //
          { 9, 0, 0, -3, 0, 0 }, //
          { 4, 0, 0, 0, 0, 0 }, //
          { 4, 0, 0, -2, 0, 0 }, //
          { -3, 0, 0, 2, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { 9, 0, 0, -3, 0, 0 }, //
          { -4, 0, 0, 0, 0, 0 }, //
          { -4, 0, 0, 0, 0, 0 }, //
          { 3, 0, 0, -2, 0, 0 }, //
          { 8, 0, 0, 0, 0, 0 }, //
          { 3, 0, 0, 0, 0, 0 }, //
          { -3, 0, 0, 2, 0, 0 }, //
          { 3, 0, 0, -1, 0, 0 }, //
          { 3, 0, 0, -1, 0, 0 }, //
          { -3, 0, 0, 1, 0, 0 }, //
          { 6, 0, 0, -3, 0, 0 }, //
          { 3, 0, 0, 0, 0, 0 }, //
          { -3, 0, 0, 1, 0, 0 }, //
          { -7, 0, 0, 0, 0, 0 }, //
          { 9, 0, 0, 0, 0, 0 }, //
          { -3, 0, 0, 2, 0, 0 }, //
          { -3, 0, 0, 0, 0, 0 }, //
          { -4, 0, 0, 0, 0, 0 }, //
          { -5, 0, 0, 3, 0, 0 }, //
          { -13, 0, 0, 0, 0, 0 }, //
          { -7, 0, 0, 0, 0, 0 }, //
          { 10, 0, 0, 0, 0, 0 }, //
          { 3, 0, 0, -1, 0, 0 }, //
          { 10, 0, 13, 6, 0, -5 }, //
          { 0, 0, 30, 0, 0, 14 }, //
          { 0, 0, -162, 0, 0, -138 }, //
          { 0, 0, 75, 0, 0, 0 }, //
          { -7, 0, 0, 4, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { 4, 0, 0, -2, 0, 0 }, //
          { 5, 0, 0, -2, 0, 0 }, //
          { 5, 0, 0, -3, 0, 0 }, //
          { -3, 0, 0, 0, 0, 0 }, //
          { -3, 0, 0, 2, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { -5, 0, 0, 2, 0, 0 }, //
          { 6, 0, 0, 0, 0, 0 }, //
          { 9, 0, 0, 0, 0, 0 }, //
          { 5, 0, 0, 0, 0, 0 }, //
          { -7, 0, 0, 0, 0, 0 }, //
          { -3, 0, 0, 1, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { 7, 0, 0, 0, 0, 0 }, //
          { -4, 0, 0, 0, 0, 0 }, //
          { 4, 0, 0, 0, 0, 0 }, //
          { -6, 0, -3, 3, 0, 1 }, //
          { 0, 0, -3, 0, 0, -2 }, //
          { 11, 0, 0, 0, 0, 0 }, //
          { 3, 0, 0, -1, 0, 0 }, //
          { 11, 0, 0, 0, 0, 0 }, //
          { -3, 0, 0, 2, 0, 0 }, //
          { -1, 0, 3, 3, 0, -1 }, //
          { 4, 0, 0, -2, 0, 0 }, //
          { 0, 0, -13, 0, 0, -11 }, //
          { 3, 0, 6, 0, 0, 0 }, //
          { -7, 0, 0, 0, 0, 0 }, //
          { 5, 0, 0, -3, 0, 0 }, //
          { -3, 0, 0, 1, 0, 0 }, //
          { 3, 0, 0, 0, 0, 0 }, //
          { 5, 0, 0, -3, 0, 0 }, //
          { -7, 0, 0, 3, 0, 0 }, //
          { 8, 0, 0, -3, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { 11, 0, 0, 0, 0, 0 }, //
          { -3, 0, 0, 1, 0, 0 }, //
          { 3, 0, 0, -1, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { 8, 0, 0, -4, 0, 0 }, //
          { 3, 0, 0, -1, 0, 0 }, //
          { 11, 0, 0, 0, 0, 0 }, //
          { -6, 0, 0, 3, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { -8, 0, 0, 4, 0, 0 }, //
          { -7, 0, 0, 3, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { 3, 0, 0, -1, 0, 0 }, //
          { 6, 0, 0, -3, 0, 0 }, //
          { -6, 0, 0, 3, 0, 0 }, //
          { 6, 0, 0, 0, 0, 0 }, //
          { 6, 0, 0, -1, 0, 0 }, //
          { 5, 0, 0, -2, 0, 0 }, //
          { -5, 0, 0, 2, 0, 0 }, //
          { -4, 0, 0, 0, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { 4, 0, 0, 0, 0, 0 }, //
          { 6, 0, 0, -3, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { 0, 0, -26, 0, 0, -11 }, //
          { 0, 0, -10, 0, 0, -5 }, //
          { 5, 0, 0, -3, 0, 0 }, //
          { -13, 0, 0, 0, 0, 0 }, //
          { 3, 0, 0, -2, 0, 0 }, //
          { 4, 0, 0, -2, 0, 0 }, //
          { 7, 0, 0, -3, 0, 0 }, //
          { 4, 0, 0, 0, 0, 0 }, //
          { 5, 0, 0, 0, 0, 0 }, //
          { -3, 0, 0, 2, 0, 0 }, //
          { -6, 0, 0, 2, 0, 0 }, //
          { -5, 0, 0, 2, 0, 0 }, //
          { -7, 0, 0, 3, 0, 0 }, //
          { 5, 0, 0, -2, 0, 0 }, //
          { 13, 0, 0, 0, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { -3, 0, 0, 0, 0, 0 }, //
          { 5, 0, 0, -2, 0, 0 }, //
          { -11, 0, 0, 0, 0, 0 }, //
          { 5, 0, 0, -2, 0, 0 }, //
          { 4, 0, 0, 0, 0, 0 }, //
          { 4, 0, 0, -2, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { 6, 0, 0, -3, 0, 0 }, //
          { 3, 0, 0, -2, 0, 0 }, //
          { -12, 0, 0, 0, 0, 0 }, //
          { 4, 0, 0, 0, 0, 0 }, //
          { -3, 0, 0, 0, 0, 0 }, //
          { -4, 0, 0, 0, 0, 0 }, //
          { 3, 0, 0, 0, 0, 0 }, //
          { 3, 0, 0, -1, 0, 0 }, //
          { -3, 0, 0, 1, 0, 0 }, //
          { 0, 0, -5, 0, 0, -2 }, //
          { -7, 0, 0, 4, 0, 0 }, //
          { 6, 0, 0, -3, 0, 0 }, //
          { -3, 0, 0, 0, 0, 0 }, //
          { 5, 0, 0, -3, 0, 0 }, //
          { 3, 0, 0, -1, 0, 0 }, //
          { 3, 0, 0, 0, 0, 0 }, //
          { -3, 0, 0, 1, 0, 0 }, //
          { -5, 0, 0, 3, 0, 0 }, //
          { -3, 0, 0, 2, 0, 0 }, //
          { -3, 0, 0, 2, 0, 0 }, //
          { 12, 0, 0, 0, 0, 0 }, //
          { 3, 0, 0, -1, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { 4, 0, 0, 0, 0, 0 }, //
          { 6, 0, 0, 0, 0, 0 }, //
          { 5, 0, 0, -3, 0, 0 }, //
          { 4, 0, 0, -2, 0, 0 }, //
          { -6, 0, 0, 3, 0, 0 }, //
          { 4, 0, 0, -2, 0, 0 }, //
          { 6, 0, 0, -3, 0, 0 }, //
          { 6, 0, 0, 0, 0, 0 }, //
          { -6, 0, 0, 3, 0, 0 }, //
          { 3, 0, 0, -2, 0, 0 }, //
          { 7, 0, 0, -4, 0, 0 }, //
          { 4, 0, 0, -2, 0, 0 }, //
          { -5, 0, 0, 2, 0, 0 }, //
          { 5, 0, 0, 0, 0, 0 }, //
          { -6, 0, 0, 3, 0, 0 }, //
          { -6, 0, 0, 3, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { 10, 0, 0, 0, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { 7, 0, 0, 0, 0, 0 }, //
          { 7, 0, 0, -3, 0, 0 }, //
          { 4, 0, 0, 0, 0, 0 }, //
          { 11, 0, 0, 0, 0, 0 }, //
          { 5, 0, 0, -2, 0, 0 }, //
          { -6, 0, 0, 2, 0, 0 }, //
          { 4, 0, 0, -2, 0, 0 }, //
          { 3, 0, 0, -2, 0, 0 }, //
          { 5, 0, 0, -2, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { -4, 0, 0, 2, 0, 0 }, //
          { -3, 0, 0, 2, 0, 0 }, //
          { 4, 0, 0, -2, 0, 0 }, //
          { 3, 0, 0, -1, 0, 0 }, //
          { -3, 0, 0, 1, 0, 0 }, //
          { -3, 0, 0, 1, 0, 0 }, //
          { -3, 0, 0, 2, 0, 0 } };

  //Planetary argument multipliers:
  // L   L'  F   D   Om  Me  Ve  E  Ma  Ju  Sa  Ur  Ne  pre
  static const int8_t napl_t[687][14] = { //
          { 0, 0, 0, 0, 0, 0, 0, 8, -16, 4, 5, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -8, 16, -4, -5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 8, -16, 4, 5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -1, 2, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -4, 8, -1, -5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, -8, 3, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, 3, -8, 3, 0, 0, 0, 0 }, //
          { -1, 0, 0, 0, 0, 0, 10, -3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, -2, 6, -3, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, -8, 3, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -5, 8, -3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -4, 8, -3, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, -8, 1, 5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -5, 6, 4, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, -5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, -5, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 2, -5, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, -5, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, -2, 5, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, -2, 5, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, -2, 5, 0, 0, 2 }, //
          { 2, 0, -1, -1, 0, 0, 0, 3, -7, 0, 0, 0, 0, 0 }, //
          { 1, 0, 0, -2, 0, 0, 19, -21, 3, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 2, -4, 0, -3, 0, 0, 0, 0 }, //
          { 1, 0, 0, -1, 1, 0, 0, -1, 0, 2, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, -4, 10, 0, 0, 0 }, //
          { -2, 0, 0, 2, 1, 0, 0, 2, 0, 0, -5, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 3, -7, 4, 0, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 0, 1, 0, 1, -1, 0, 0, 0 }, //
          { -2, 0, 0, 2, 1, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { -1, 0, 0, 0, 0, 0, 18, -16, 0, 0, 0, 0, 0, 0 }, //
          { -2, 0, 1, 1, 2, 0, 0, 1, 0, -2, 0, 0, 0, 0 }, //
          { -1, 0, 1, -1, 1, 0, 18, -17, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 0, 1, 1, 0, 0, 2, -2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -8, 13, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 2, -2, 2, 0, -8, 11, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -8, 13, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, -8, 12, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 8, -13, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 8, -14, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 8, -13, 0, 0, 0, 0, 0, 1 }, //
          { -2, 0, 0, 2, 1, 0, 0, 2, 0, -4, 5, 0, 0, 0 }, //
          { -2, 0, 0, 2, 2, 0, 3, -3, 0, 0, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 0, 2, 0, -3, 1, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 3, -5, 0, 2, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 0, 2, 0, -4, 3, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, -1, 2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 2, 0, 0, -2, 2, 0, 0, 0, 0, 0 }, //
          { -1, 0, 1, 0, 1, 0, 3, -5, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 0, 1, 0, 0, 3, -4, 0, 0, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 0, 2, 0, -2, -2, 0, 0, 0 }, //
          { -2, 0, 2, 0, 2, 0, 0, -5, 9, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 0, 0, -1, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 0, 0, 0, 2, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 2 }, //
          { -1, 0, 0, 1, 0, 0, 0, 3, -4, 0, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 0, 1, 0, 0, 2, 0, 0, 0 }, //
          { 0, 0, 1, -1, 2, 0, 0, -1, 0, 0, 2, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, -9, 17, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 2, 0, -3, 5, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, -1, 2, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, -2, 0, 0, 0 }, //
          { 1, 0, 0, -2, 0, 0, 17, -16, 0, -2, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 1, -3, 0, 0, 0 }, //
          { -2, 0, 0, 2, 1, 0, 0, 5, -6, 0, 0, 0, 0, 0 }, //
          { 0, 0, -2, 2, 0, 0, 0, 9, -13, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 2, 0, 0, -1, 0, 0, 1, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0 }, //
          { 0, 0, -2, 2, 0, 0, 5, -6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 1, 0, 5, -7, 0, 0, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 6, -8, 0, 0, 0, 0, 0, 0 }, //
          { 2, 0, 1, -3, 1, 0, -6, 7, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 2, 0, 0, 0, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 0, 0, 2, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -8, 15, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -8, 15, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, -9, 15, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 8, -15, 0, 0, 0, 0, 0 }, //
          { 1, 0, -1, -1, 0, 0, 0, 8, -15, 0, 0, 0, 0, 0 }, //
          { 2, 0, 0, -2, 0, 0, 2, -5, 0, 0, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 0, 2, 0, -5, 5, 0, 0, 0 }, //
          { 2, 0, 0, -2, 1, 0, 0, -6, 8, 0, 0, 0, 0, 0 }, //
          { 2, 0, 0, -2, 1, 0, 0, -2, 0, 3, 0, 0, 0, 0 }, //
          { -2, 0, 1, 1, 0, 0, 0, 1, 0, -3, 0, 0, 0, 0 }, //
          { -2, 0, 1, 1, 1, 0, 0, 1, 0, -3, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 0, 2, 0, -3, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 0, 6, -8, 0, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 0, 2, 0, -1, -5, 0, 0, 0 }, //
          { -1, 0, 0, 1, 0, 0, 0, 1, 0, -1, 0, 0, 0, 0 }, //
          { -1, 0, 1, 1, 1, 0, -20, 20, 0, 0, 0, 0, 0, 0 }, //
          { 1, 0, 0, -2, 0, 0, 20, -21, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 8, -15, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, 0, -10, 15, 0, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 2, 0, 0, -1, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, -2, 4, 0, 0, 0 }, //
          { 2, 0, 0, -2, 1, 0, -6, 8, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, -2, 2, 1, 0, 5, -6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 0, -1, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 0, 1, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 2 }, //
          { 0, 0, 2, -2, 1, 0, 0, -9, 13, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 7, -13, 0, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 0, 5, -6, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 9, -17, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -9, 17, 0, 0, 0, 0, 2 }, //
          { 1, 0, 0, -1, 1, 0, 0, -3, 4, 0, 0, 0, 0, 0 }, //
          { 1, 0, 0, -1, 1, 0, -3, 4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 2, 0, 0, -1, 2, 0, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 1, 0, 0, 0, 2, 0, 0, 0, 0, 0 }, //
          { 0, 0, -2, 2, 0, 1, 0, -2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 3, -5, 0, 2, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 1, 0, 0, 2, 0, -3, 1, 0, 0, 0 }, //
          { -2, 0, 0, 2, 1, 0, 3, -3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 8, -13, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 8, -12, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, -8, 11, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 0, 1, 0, 0, 0, 2, -2, 0, 0, 0, 0, 0 }, //
          { -1, 0, 0, 0, 1, 0, 18, -16, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, -1, 1, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 3, -7, 4, 0, 0, 0, 0, 0 }, //
          { -2, 0, 1, 1, 1, 0, 0, -3, 7, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 2, 0, 0, -1, 0, -2, 5, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 0, 0, -2, 5, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, -4, 8, -3, 0, 0, 0, 0 }, //
          { 1, 0, 0, 0, 1, 0, -10, 3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, 0, -2, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 0, 0, 1, 0, 10, -3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 4, -8, 3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 0, 0, 2, -5, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 0, 1, 0, 2, -5, 0, 0, 0 }, //
          { 2, 0, -1, -1, 1, 0, 0, 3, -7, 0, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 0, 2, 0, 0, -5, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, -3, 7, -4, 0, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { 1, 0, 0, 0, 1, 0, -18, 16, 0, 0, 0, 0, 0, 0 }, //
          { -2, 0, 1, 1, 1, 0, 0, 1, 0, -2, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 2, 0, -8, 12, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, -8, 13, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, -2, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, 0, -2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, -2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -2, 2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 2, 0, 0, 0, 0, 1 }, //
          { -1, 0, 0, 1, 1, 0, 3, -4, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 0, 1, 1, 0, 0, 3, -4, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 0, -2, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 0, 2, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 2 }, //
          { 0, 0, 1, -1, 0, 0, 3, -6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, -3, 5, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 2, 0, -3, 4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, -2, 4, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, -5, 6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 5, -7, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 5, -8, 0, 0, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 1, 0, 6, -8, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, -8, 15, 0, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 1, 0, 0, 2, 0, -3, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 1, 0, 0, 6, -8, 0, 0, 0, 0, 0 }, //
          { 1, 0, 0, -1, 1, 0, 0, -1, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, -5, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, -1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 2 }, //
          { 0, 0, 1, -1, 2, 0, 0, -1, 0, 0, -1, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, -1, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 0, 1, 0, 0, -1, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -7, 13, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 7, -13, 0, 0, 0, 0, 0 }, //
          { 2, 0, 0, -2, 1, 0, 0, -5, 6, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, 0, -8, 11, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, -1, 0, 2, 0, 0, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 0, 4, -4, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, -2, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 0, 3, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 2 }, //
          { -2, 0, 0, 2, 0, 0, 3, -3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 2, 0, 0, -4, 8, -3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 2, 0, 0, 4, -8, 3, 0, 0, 0, 0 }, //
          { 2, 0, 0, -2, 1, 0, 0, -2, 0, 2, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 2, 0, 0, -1, 0, 2, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 2, 0, 0, 0, -2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 1, -2, 0, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 0, 2, -2, 0, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 0, 1, 0, 0, -2, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, 0, -2, 0, 0, 2, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 3, -6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 3, -5, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 3, -5, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, -3, 4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -3, 5, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, -3, 5, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 2, -2, 2, 0, -3, 3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -3, 5, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, -4, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, 1, -4, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, -4, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -2, 4, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, -3, 4, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -2, 4, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, -2, 4, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -5, 8, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 2, -2, 2, 0, -5, 6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -5, 8, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -5, 8, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, -5, 7, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -5, 8, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 5, -8, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 2, 0, 0, -1, 0, -1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 0, 0, -1, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 0, 1, 0, -1, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, 0, -2, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -6, 11, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, -11, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, -1, 0, 4, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 1, 0, -4, 0, 0, 0, 0, 0, 0 }, //
          { 2, 0, 0, -2, 1, 0, -3, 3, 0, 0, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 0, 2, 0, 0, -2, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, 0, -7, 9, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, -5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 2, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 2 }, //
          { 0, 0, 2, -2, 2, 0, 0, -2, 0, 2, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 1, 0, 3, -5, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 3, -4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, -3, 3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 2, -4, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, 0, -4, 4, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 2, 0, -5, 7, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, -6, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -3, 6, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, -4, 6, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -3, 6, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, -3, 6, 0, 0, 0, 0, 2 }, //
          { 0, 0, -1, 1, 0, 0, 2, -2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 2, -3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -5, 9, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -5, 9, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 5, -9, 0, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 0, 1, 0, -2, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, 0, -2, 0, 2, 0, 0, 0, 0 }, //
          { -2, 0, 1, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, -2, 2, 0, 0, 3, -3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -6, 10, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, -6, 10, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -2, 3, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -2, 3, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, -2, 2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 2, -3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 2, -3, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, -8, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -4, 8, 0, 0, 0, 0, 2 }, //
          { 0, 0, -2, 2, 0, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -4, 7, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -4, 7, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, -7, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, -2, 3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, 0, -2, 0, 3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -5, 10, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 1, 0, -1, 2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -3, 5, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -3, 5, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, -5, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 1, -2, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 1, -3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 1, -2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -1, 2, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, -1, 2, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -7, 11, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -7, 11, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, -2, 2, 0, 0, 4, -4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, -3, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, -4, 4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 4, -5, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, -1, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -4, 7, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, -4, 6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -4, 7, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -4, 6, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -4, 6, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, -4, 5, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -4, 6, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 4, -6, 0, 0, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 2, -2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 1, -1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 0, 5, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, -3, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 3, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -7, 12, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -1, 1, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -1, 1, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, -1, 0, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 1, -1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 1, -1, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 1, -2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -2, 5, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 0, 4, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, -4, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, -1, 1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -6, 10, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -6, 10, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, 0, -3, 0, 3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -3, 7, 0, 0, 0, 0, 2 }, //
          { -2, 0, 0, 2, 0, 0, 4, -4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -5, 8, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 5, -8, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 0, 3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 0, 3, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, -3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 2, -4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -2, 4, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, -2, 3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -2, 4, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -6, 9, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -6, 9, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 6, -9, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 1, 0, -2, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, -2, 2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -4, 6, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, -6, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 3, -4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 0, 2, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, -2, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 1, 0, -1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -5, 9, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, -4, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -3, 4, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -3, 4, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 3, -4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 3, -4, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 1, 0, 0, 2, -2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, -1, 0, 2, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, -3, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, -5, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 0, 1, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, -1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, -1, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, -3, 5, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, -3, 4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, -2, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, -2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, -1, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, -1, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, -2, 2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -8, 14, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, -5, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 5, -8, 3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 5, -8, 3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, -8, 3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -3, 8, -3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, -2, 5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -8, 12, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -8, 12, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, -2, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 2, 0, 0, 2 }, //
          { 0, 0, 2, -2, 1, 0, -5, 5, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 3, -6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -3, 6, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, -3, 6, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 4, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -5, 7, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -5, 7, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, -5, 6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 5, -7, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, 0, -1, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, -1, 0, 3, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -2, 6, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 1, 0, 2, -2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -6, 9, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, -9, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -2, 2, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, -2, 1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 2, -2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 2, -2, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -5, 7, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 5, -7, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, -2, 2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, -5, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 1, -3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -1, 3, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, -1, 2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -1, 3, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -7, 10, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -7, 10, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, -3, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -4, 8, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -4, 5, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -4, 5, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 4, -5, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -2, 0, 5, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -9, 13, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 5, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -2, 0, 4, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, -4, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -2, 7, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, -3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -2, 5, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, -2, 5, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -6, 8, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -6, 8, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 6, -8, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -3, 9, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 5, -6, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 5, -6, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, -2, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, -2, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -5, 10, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, -4, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, -4, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -3, 3, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 3, -3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 3, -3, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 3, -3, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, -3, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -5, 13, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, -1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, -1, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, -2, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, -2, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, -2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, -2, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, -1, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -6, 15, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -8, 15, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -3, 9, -4, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 2, -5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -2, 8, -1, -5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, -8, 3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -6, 16, -4, -5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -2, 8, -3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -2, 8, -3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, -8, 1, 5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, -2, 5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 3, -5, 4, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -8, 11, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -8, 11, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, -8, 11, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 11, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 1, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 3, -3, 0, 2, 0, 0, 0, 2 }, //
          { 0, 0, 2, -2, 1, 0, 0, 4, -8, 3, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, 0, -4, 8, -3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 1, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -3, 7, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -5, 6, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -5, 6, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 5, -6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 5, -6, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 2, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 6, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 7, -9, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 2, -1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 2, -1, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, -7, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 5, -5, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -1, 4, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, -1, 4, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -7, 9, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -7, 9, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, -3, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, -1, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -4, 4, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 4, -4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 4, -4, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 4, -4, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -3, 0, 5, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -9, 12, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, 0, -4, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, 1, -1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 7, -8, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, 0, -3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, 0, -3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -2, 6, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -6, 7, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 6, -7, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, -6, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, 0, -2, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, 0, -2, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 5, -4, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 3, -2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 3, -2, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, 0, -1, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, 0, -1, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, -2, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, -2, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, -1, 0, 0, 2 }, //
          { 0, 0, 2, -2, 1, 0, 0, 1, 0, -1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -8, 16, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, 0, 2, -5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 7, -8, 3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -5, 16, -4, -5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 8, -3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -8, 10, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -8, 10, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, -8, 10, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 2, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, 0, 1, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -3, 8, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -5, 5, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 5, -5, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 5, -5, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 5, -5, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 7, -7, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 7, -7, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, -5, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 7, -8, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 5, -3, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 4, -3, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -9, 11, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -9, 11, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, 0, -4, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, 0, -3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -6, 6, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 6, -6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 6, -6, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, 0, -2, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, -4, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 3, -1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 3, -1, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 3, -1, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, 0, -1, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, -2, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 5, -2, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 8, -9, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 5, -4, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, -7, 7, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 7, -7, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 4, -2, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 4, -2, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 4, -2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 4, -2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 5, 0, -4, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 5, 0, -3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 5, 0, -2, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -8, 8, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 8, -8, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 5, -3, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 5, -3, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -9, 9, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, -9, 9, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, -9, 9, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 9, -9, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 6, -4, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2 }, //
          { 1, 0, 0, -2, 0, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { 1, 0, 0, -2, 0, 0, 2, -2, 0, 0, 0, 0, 0, 0 }, //
          { 1, 0, 0, -2, 0, 0, 0, 1, 0, -1, 0, 0, 0, 0 }, //
          { 1, 0, 0, -2, 0, 0, 1, -1, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 0, 0, 0, 0, 3, -3, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 0, 0, 0, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { -1, 0, 0, 2, 0, 0, 0, 4, -8, 3, 0, 0, 0, 0 }, //
          { 1, 0, 0, -2, 0, 0, 0, 4, -8, 3, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 0, 4, -8, 3, 0, 0, 0, 0 }, //
          { -1, 0, 0, 0, 0, 0, 0, 2, 0, -3, 0, 0, 0, 0 }, //
          { -1, 0, 0, 0, 0, 0, 0, 1, 0, -1, 0, 0, 0, 0 }, //
          { -1, 0, 0, 0, 0, 0, 1, -1, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 0, 2, 0, 0, 2, -2, 0, 0, 0, 0, 0, 0 }, //
          { 1, 0, -1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 0, 2, 0, 0, 0, 2, 0, -3, 0, 0, 0, 0 }, //
          { -2, 0, 0, 0, 0, 0, 0, 2, 0, -3, 0, 0, 0, 0 }, //
          { 1, 0, 0, 0, 0, 0, 0, 4, -8, 3, 0, 0, 0, 0 }, //
          { -1, 0, 1, -1, 1, 0, 0, -1, 0, 0, 0, 0, 0, 0 }, //
          { 1, 0, 1, -1, 1, 0, 0, -1, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 0, 0, 0, 0, 0, 4, -8, 3, 0, 0, 0, 0 }, //
          { -1, 0, 0, 2, 1, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { -1, 0, 0, 2, 0, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { -1, 0, 0, 2, 0, 0, 3, -3, 0, 0, 0, 0, 0, 0 }, //
          { 1, 0, 0, -2, 1, 0, 0, -2, 0, 2, 0, 0, 0, 0 }, //
          { 1, 0, 2, -2, 2, 0, -3, 3, 0, 0, 0, 0, 0, 0 }, //
          { 1, 0, 2, -2, 2, 0, 0, -2, 0, 2, 0, 0, 0, 0 }, //
          { 1, 0, 0, 0, 0, 0, 1, -1, 0, 0, 0, 0, 0, 0 }, //
          { 1, 0, 0, 0, 0, 0, 0, 1, 0, -1, 0, 0, 0, 0 }, //
          { 0, 0, 0, -2, 0, 0, 2, -2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, -2, 0, 0, 0, 1, 0, -1, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 2, 0, -2, 2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 2, 0, 0, -1, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 2, 0, -1, 1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 2, 0, -2, 3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 2, 0, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { 0, 0, 1, 1, 2, 0, 0, 1, 0, 0, 0, 0, 0, 0 }, //
          { 1, 0, 2, 0, 2, 0, 0, 1, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 2, 0, 2, 0, 10, -3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0 }, //
          { 1, 0, 2, 0, 2, 0, 0, 1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 2, 0, 0, 4, -8, 3, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 2, 0, 0, -4, 8, -3, 0, 0, 0, 0 }, //
          { -1, 0, 2, 0, 2, 0, 0, -4, 8, -3, 0, 0, 0, 0 }, //
          { 2, 0, 2, -2, 2, 0, 0, -2, 0, 3, 0, 0, 0, 0 }, //
          { 1, 0, 2, 0, 1, 0, 0, -2, 0, 3, 0, 0, 0, 0 }, //
          { 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 2, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0 }, //
          { -2, 0, 2, 2, 2, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 2, 0, 2, -3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 2, 0, 1, -1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 2, 0, 0, 1, 0, -1, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 2, 0, 2, -2, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 2, 2, 2, 0, 0, -1, 0, 1, 0, 0, 0, 0 }, //
          { 1, 0, 2, 0, 2, 0, -1, 1, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 2, 2, 2, 0, 0, 2, 0, -3, 0, 0, 0, 0 }, //
          { 2, 0, 2, 0, 2, 0, 0, 2, 0, -3, 0, 0, 0, 0 }, //
          { 1, 0, 2, 0, 2, 0, 0, -4, 8, -3, 0, 0, 0, 0 }, //
          { 1, 0, 2, 0, 2, 0, 0, 4, -8, 3, 0, 0, 0, 0 }, //
          { 1, 0, 1, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 2, 0, 0, 1, 0, 0, 0, 0, 0, 0 }, //
          { 2, 0, 2, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 2, 2, 2, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { -1, 0, 2, 2, 2, 0, 3, -3, 0, 0, 0, 0, 0, 0 }, //
          { 1, 0, 2, 0, 2, 0, 1, -1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, 2, 2, 0, 0, 2, 0, -2, 0, 0, 0, 0 } };

  // Planetary nutation coefficients, unit 1e-7 arcsec:
  // longitude (sin, cos), obliquity (sin, cos)

  // Each row of coefficients in 'cpl_t' belongs with the corresponding
  // row of fundamental-argument multipliers in 'napl_t'.
  static const int16_t cpl_t[687][4] = {
          { 1440, 0, 0, 0 }, //
          { 56, -117, -42, -40 }, //
          { 125, -43, 0, -54 }, //
          { 0, 5, 0, 0 }, //
          { 3, -7, -3, 0 }, //
          { 3, 0, 0, -2 }, //
          { -114, 0, 0, 61 }, //
          { -219, 89, 0, 0 }, //
          { -3, 0, 0, 0 }, //
          { -462, 1604, 0, 0 }, //
          { 99, 0, 0, -53 }, //
          { -3, 0, 0, 2 }, //
          { 0, 6, 2, 0 }, //
          { 3, 0, 0, 0 }, //
          { -12, 0, 0, 0 }, //
          { 14, -218, 117, 8 }, //
          { 31, -481, -257, -17 }, //
          { -491, 128, 0, 0 }, //
          { -3084, 5123, 2735, 1647 }, //
          { -1444, 2409, -1286, -771 }, //
          { 11, -24, -11, -9 }, //
          { 26, -9, 0, 0 }, //
          { 103, -60, 0, 0 }, //
          { 0, -13, -7, 0 }, //
          { -26, -29, -16, 14 }, //
          { 9, -27, -14, -5 }, //
          { 12, 0, 0, -6 }, //
          { -7, 0, 0, 0 }, //
          { 0, 24, 0, 0 }, //
          { 284, 0, 0, -151 }, //
          { 226, 101, 0, 0 }, //
          { 0, -8, -2, 0 }, //
          { 0, -6, -3, 0 }, //
          { 5, 0, 0, -3 }, //
          { -41, 175, 76, 17 }, //
          { 0, 15, 6, 0 }, //
          { 425, 212, -133, 269 }, //
          { 1200, 598, 319, -641 }, //
          { 235, 334, 0, 0 }, //
          { 11, -12, -7, -6 }, //
          { 5, -6, 3, 3 }, //
          { -5, 0, 0, 3 }, //
          { 6, 0, 0, -3 }, //
          { 15, 0, 0, 0 }, //
          { 13, 0, 0, -7 }, //
          { -6, -9, 0, 0 }, //
          { 266, -78, 0, 0 }, //
          { -460, -435, -232, 246 }, //
          { 0, 15, 7, 0 }, //
          { -3, 0, 0, 2 }, //
          { 0, 131, 0, 0 }, //
          { 4, 0, 0, 0 }, //
          { 0, 3, 0, 0 }, //
          { 0, 4, 2, 0 }, //
          { 0, 3, 0, 0 }, //
          { -17, -19, -10, 9 }, //
          { -9, -11, 6, -5 }, //
          { -6, 0, 0, 3 }, //
          { -16, 8, 0, 0 }, //
          { 0, 3, 0, 0 }, //
          { 11, 24, 11, -5 }, //
          { -3, -4, -2, 1 }, //
          { 3, 0, 0, -1 }, //
          { 0, -8, -4, 0 }, //
          { 0, 3, 0, 0 }, //
          { 0, 5, 0, 0 }, //
          { 0, 3, 2, 0 }, //
          { -6, 4, 2, 3 }, //
          { -3, -5, 0, 0 }, //
          { -5, 0, 0, 2 }, //
          { 4, 24, 13, -2 }, //
          { -42, 20, 0, 0 }, //
          { -10, 233, 0, 0 }, //
          { -3, 0, 0, 1 }, //
          { 78, -18, 0, 0 }, //
          { 0, 3, 1, 0 }, //
          { 0, -3, -1, 0 }, //
          { 0, -4, -2, 1 }, //
          { 0, -8, -4, -1 }, //
          { 0, -5, 3, 0 }, //
          { -7, 0, 0, 3 }, //
          { -14, 8, 3, 6 }, //
          { 0, 8, -4, 0 }, //
          { 0, 19, 10, 0 }, //
          { 45, -22, 0, 0 }, //
          { -3, 0, 0, 0 }, //
          { 0, -3, 0, 0 }, //
          { 0, 3, 0, 0 }, //
          { 3, 5, 3, -2 }, //
          { 89, -16, -9, -48 }, //
          { 0, 3, 0, 0 }, //
          { -3, 7, 4, 2 }, //
          { -349, -62, 0, 0 }, //
          { -15, 22, 0, 0 }, //
          { -3, 0, 0, 0 }, //
          { -53, 0, 0, 0 }, //
          { 5, 0, 0, -3 }, //
          { 0, -8, 0, 0 }, //
          { 15, -7, -4, -8 }, //
          { -3, 0, 0, 1 }, //
          { -21, -78, 0, 0 }, //
          { 20, -70, -37, -11 }, //
          { 0, 6, 3, 0 }, //
          { 5, 3, 2, -2 }, //
          { -17, -4, -2, 9 }, //
          { 0, 6, 3, 0 }, //
          { 32, 15, -8, 17 }, //
          { 174, 84, 45, -93 }, //
          { 11, 56, 0, 0 }, //
          { -66, -12, -6, 35 }, //
          { 47, 8, 4, -25 }, //
          { 0, 8, 4, 0 }, //
          { 10, -22, -12, -5 }, //
          { -3, 0, 0, 2 }, //
          { -24, 12, 0, 0 }, //
          { 5, -6, 0, 0 }, //
          { 3, 0, 0, -2 }, //
          { 4, 3, 1, -2 }, //
          { 0, 29, 15, 0 }, //
          { -5, -4, -2, 2 }, //
          { 8, -3, -1, -5 }, //
          { 0, -3, 0, 0 }, //
          { 10, 0, 0, 0 }, //
          { 3, 0, 0, -2 }, //
          { -5, 0, 0, 3 }, //
          { 46, 66, 35, -25 }, //
          { -14, 7, 0, 0 }, //
          { 0, 3, 2, 0 }, //
          { -5, 0, 0, 0 }, //
          { -68, -34, -18, 36 }, //
          { 0, 14, 7, 0 }, //
          { 10, -6, -3, -5 }, //
          { -5, -4, -2, 3 }, //
          { -3, 5, 2, 1 }, //
          { 76, 17, 9, -41 }, //
          { 84, 298, 159, -45 }, //
          { 3, 0, 0, -1 }, //
          { -3, 0, 0, 2 }, //
          { -3, 0, 0, 1 }, //
          { -82, 292, 156, 44 }, //
          { -73, 17, 9, 39 }, //
          { -9, -16, 0, 0 }, //
          { 3, 0, -1, -2 }, //
          { -3, 0, 0, 0 }, //
          { -9, -5, -3, 5 }, //
          { -439, 0, 0, 0 }, //
          { 57, -28, -15, -30 }, //
          { 0, -6, -3, 0 }, //
          { -4, 0, 0, 2 }, //
          { -40, 57, 30, 21 }, //
          { 23, 7, 3, -13 }, //
          { 273, 80, 43, -146 }, //
          { -449, 430, 0, 0 }, //
          { -8, -47, -25, 4 }, //
          { 6, 47, 25, -3 }, //
          { 0, 23, 13, 0 }, //
          { -3, 0, 0, 2 }, //
          { 3, -4, -2, -2 }, //
          { -48, -110, -59, 26 }, //
          { 51, 114, 61, -27 }, //
          { -133, 0, 0, 57 }, //
          { 0, 4, 0, 0 }, //
          { -21, -6, -3, 11 }, //
          { 0, -3, -1, 0 }, //
          { -11, -21, -11, 6 }, //
          { -18, -436, -233, 9 }, //
          { 35, -7, 0, 0 }, //
          { 0, 5, 3, 0 }, //
          { 11, -3, -1, -6 }, //
          { -5, -3, -1, 3 }, //
          { -53, -9, -5, 28 }, //
          { 0, 3, 2, 1 }, //
          { 4, 0, 0, -2 }, //
          { 0, -4, 0, 0 }, //
          { -50, 194, 103, 27 }, //
          { -13, 52, 28, 7 }, //
          { -91, 248, 0, 0 }, //
          { 6, 49, 26, -3 }, //
          { -6, -47, -25, 3 }, //
          { 0, 5, 3, 0 }, //
          { 52, 23, 10, -23 }, //
          { -3, 0, 0, 1 }, //
          { 0, 5, 3, 0 }, //
          { -4, 0, 0, 0 }, //
          { -4, 8, 3, 2 }, //
          { 10, 0, 0, 0 }, //
          { 3, 0, 0, -2 }, //
          { 0, 8, 4, 0 }, //
          { 0, 8, 4, 1 }, //
          { -4, 0, 0, 0 }, //
          { -4, 0, 0, 0 }, //
          { -8, 4, 2, 4 }, //
          { 8, -4, -2, -4 }, //
          { 0, 15, 7, 0 }, //
          { -138, 0, 0, 0 }, //
          { 0, -7, -3, 0 }, //
          { 0, -7, -3, 0 }, //
          { 54, 0, 0, -29 }, //
          { 0, 10, 4, 0 }, //
          { -7, 0, 0, 3 }, //
          { -37, 35, 19, 20 }, //
          { 0, 4, 0, 0 }, //
          { -4, 9, 0, 0 }, //
          { 8, 0, 0, -4 }, //
          { -9, -14, -8, 5 }, //
          { -3, -9, -5, 3 }, //
          { -145, 47, 0, 0 }, //
          { -10, 40, 21, 5 }, //
          { 11, -49, -26, -7 }, //
          { -2150, 0, 0, 932 }, //
          { -12, 0, 0, 5 }, //
          { 85, 0, 0, -37 }, //
          { 4, 0, 0, -2 }, //
          { 3, 0, 0, -2 }, //
          { -86, 153, 0, 0 }, //
          { -6, 9, 5, 3 }, //
          { 9, -13, -7, -5 }, //
          { -8, 12, 6, 4 }, //
          { -51, 0, 0, 22 }, //
          { -11, -268, -116, 5 }, //
          { 0, 12, 5, 0 }, //
          { 0, 7, 3, 0 }, //
          { 31, 6, 3, -17 }, //
          { 140, 27, 14, -75 }, //
          { 57, 11, 6, -30 }, //
          { -14, -39, 0, 0 }, //
          { 0, -6, -2, 0 }, //
          { 4, 15, 8, -2 }, //
          { 0, 4, 0, 0 }, //
          { -3, 0, 0, 1 }, //
          { 0, 11, 5, 0 }, //
          { 9, 6, 0, 0 }, //
          { -4, 10, 4, 2 }, //
          { 5, 3, 0, 0 }, //
          { 16, 0, 0, -9 }, //
          { -3, 0, 0, 0 }, //
          { 0, 3, 2, -1 }, //
          { 7, 0, 0, -3 }, //
          { -25, 22, 0, 0 }, //
          { 42, 223, 119, -22 }, //
          { -27, -143, -77, 14 }, //
          { 9, 49, 26, -5 }, //
          { -1166, 0, 0, 505 }, //
          { -5, 0, 0, 2 }, //
          { -6, 0, 0, 3 }, //
          { -8, 0, 1, 4 }, //
          { 0, -4, 0, 0 }, //
          { 117, 0, 0, -63 }, //
          { -4, 8, 4, 2 }, //
          { 3, 0, 0, -2 }, //
          { -5, 0, 0, 2 }, //
          { 0, 31, 0, 0 }, //
          { -5, 0, 1, 3 }, //
          { 4, 0, 0, -2 }, //
          { -4, 0, 0, 2 }, //
          { -24, -13, -6, 10 }, //
          { 3, 0, 0, 0 }, //
          { 0, -32, -17, 0 }, //
          { 8, 12, 5, -3 }, //
          { 3, 0, 0, -1 }, //
          { 7, 13, 0, 0 }, //
          { -3, 16, 0, 0 }, //
          { 50, 0, 0, -27 }, //
          { 0, -5, -3, 0 }, //
          { 13, 0, 0, 0 }, //
          { 0, 5, 3, 1 }, //
          { 24, 5, 2, -11 }, //
          { 5, -11, -5, -2 }, //
          { 30, -3, -2, -16 }, //
          { 18, 0, 0, -9 }, //
          { 8, 614, 0, 0 }, //
          { 3, -3, -1, -2 }, //
          { 6, 17, 9, -3 }, //
          { -3, -9, -5, 2 }, //
          { 0, 6, 3, -1 }, //
          { -127, 21, 9, 55 }, //
          { 3, 5, 0, 0 }, //
          { -6, -10, -4, 3 }, //
          { 5, 0, 0, 0 }, //
          { 16, 9, 4, -7 }, //
          { 3, 0, 0, -2 }, //
          { 0, 22, 0, 0 }, //
          { 0, 19, 10, 0 }, //
          { 7, 0, 0, -4 }, //
          { 0, -5, -2, 0 }, //
          { 0, 3, 1, 0 }, //
          { -9, 3, 1, 4 }, //
          { 17, 0, 0, -7 }, //
          { 0, -3, -2, -1 }, //
          { -20, 34, 0, 0 }, //
          { -10, 0, 1, 5 }, //
          { -4, 0, 0, 2 }, //
          { 22, -87, 0, 0 }, //
          { -4, 0, 0, 2 }, //
          { -3, -6, -2, 1 }, //
          { -16, -3, -1, 7 }, //
          { 0, -3, -2, 0 }, //
          { 4, 0, 0, 0 }, //
          { -68, 39, 0, 0 }, //
          { 27, 0, 0, -14 }, //
          { 0, -4, 0, 0 }, //
          { -25, 0, 0, 0 }, //
          { -12, -3, -2, 6 }, //
          { 3, 0, 0, -1 }, //
          { 3, 66, 29, -1 }, //
          { 490, 0, 0, -213 }, //
          { -22, 93, 49, 12 }, //
          { -7, 28, 15, 4 }, //
          { -3, 13, 7, 2 }, //
          { -46, 14, 0, 0 }, //
          { -5, 0, 0, 0 }, //
          { 2, 1, 0, 0 }, //
          { 0, -3, 0, 0 }, //
          { -28, 0, 0, 15 }, //
          { 5, 0, 0, -2 }, //
          { 0, 3, 0, 0 }, //
          { -11, 0, 0, 5 }, //
          { 0, 3, 1, 0 }, //
          { -3, 0, 0, 1 }, //
          { 25, 106, 57, -13 }, //
          { 5, 21, 11, -3 }, //
          { 1485, 0, 0, 0 }, //
          { -7, -32, -17, 4 }, //
          { 0, 5, 3, 0 }, //
          { -6, -3, -2, 3 }, //
          { 30, -6, -2, -13 }, //
          { -4, 4, 0, 0 }, //
          { -19, 0, 0, 10 }, //
          { 0, 4, 2, -1 }, //
          { 0, 3, 0, 0 }, //
          { 4, 0, 0, -2 }, //
          { 0, -3, -1, 0 }, //
          { -3, 0, 0, 0 }, //
          { 5, 3, 1, -2 }, //
          { 0, 11, 0, 0 }, //
          { 118, 0, 0, -52 }, //
          { 0, -5, -3, 0 }, //
          { -28, 36, 0, 0 }, //
          { 5, -5, 0, 0 }, //
          { 14, -59, -31, -8 }, //
          { 0, 9, 5, 1 }, //
          { -458, 0, 0, 198 }, //
          { 0, -45, -20, 0 }, //
          { 9, 0, 0, -5 }, //
          { 0, -3, 0, 0 }, //
          { 0, -4, -2, -1 }, //
          { 11, 0, 0, -6 }, //
          { 6, 0, 0, -2 }, //
          { -16, 23, 0, 0 }, //
          { 0, -4, -2, 0 }, //
          { -5, 0, 0, 2 }, //
          { -166, 269, 0, 0 }, //
          { 15, 0, 0, -8 }, //
          { 10, 0, 0, -4 }, //
          { -78, 45, 0, 0 }, //
          { 0, -5, -2, 0 }, //
          { 7, 0, 0, -4 }, //
          { -5, 328, 0, 0 }, //
          { 3, 0, 0, -2 }, //
          { 5, 0, 0, -2 }, //
          { 0, 3, 1, 0 }, //
          { -3, 0, 0, 0 }, //
          { -3, 0, 0, 0 }, //
          { 0, -4, -2, 0 }, //
          { -1223, -26, 0, 0 }, //
          { 0, 7, 3, 0 }, //
          { 3, 0, 0, 0 }, //
          { 0, 3, 2, 0 }, //
          { -6, 20, 0, 0 }, //
          { -368, 0, 0, 0 }, //
          { -75, 0, 0, 0 }, //
          { 11, 0, 0, -6 }, //
          { 3, 0, 0, -2 }, //
          { -3, 0, 0, 1 }, //
          { -13, -30, 0, 0 }, //
          { 21, 3, 0, 0 }, //
          { -3, 0, 0, 1 }, //
          { -4, 0, 0, 2 }, //
          { 8, -27, 0, 0 }, //
          { -19, -11, 0, 0 }, //
          { -4, 0, 0, 2 }, //
          { 0, 5, 2, 0 }, //
          { -6, 0, 0, 2 }, //
          { -8, 0, 0, 0 }, //
          { -1, 0, 0, 0 }, //
          { -14, 0, 0, 6 }, //
          { 6, 0, 0, 0 }, //
          { -74, 0, 0, 32 }, //
          { 0, -3, -1, 0 }, //
          { 4, 0, 0, -2 }, //
          { 8, 11, 0, 0 }, //
          { 0, 3, 2, 0 }, //
          { -262, 0, 0, 114 }, //
          { 0, -4, 0, 0 }, //
          { -7, 0, 0, 4 }, //
          { 0, -27, -12, 0 }, //
          { -19, -8, -4, 8 }, //
          { 202, 0, 0, -87 }, //
          { -8, 35, 19, 5 }, //
          { 0, 4, 2, 0 }, //
          { 16, -5, 0, 0 }, //
          { 5, 0, 0, -3 }, //
          { 0, -3, 0, 0 }, //
          { 1, 0, 0, 0 }, //
          { -35, -48, -21, 15 }, //
          { -3, -5, -2, 1 }, //
          { 6, 0, 0, -3 }, //
          { 3, 0, 0, -1 }, //
          { 0, -5, 0, 0 }, //
          { 12, 55, 29, -6 }, //
          { 0, 5, 3, 0 }, //
          { -598, 0, 0, 0 }, //
          { -3, -13, -7, 1 }, //
          { -5, -7, -3, 2 }, //
          { 3, 0, 0, -1 }, //
          { 5, -7, 0, 0 }, //
          { 4, 0, 0, -2 }, //
          { 16, -6, 0, 0 }, //
          { 8, -3, 0, 0 }, //
          { 8, -31, -16, -4 }, //
          { 0, 3, 1, 0 }, //
          { 113, 0, 0, -49 }, //
          { 0, -24, -10, 0 }, //
          { 4, 0, 0, -2 }, //
          { 27, 0, 0, 0 }, //
          { -3, 0, 0, 1 }, //
          { 0, -4, -2, 0 }, //
          { 5, 0, 0, -2 }, //
          { 0, -3, 0, 0 }, //
          { -13, 0, 0, 6 }, //
          { 5, 0, 0, -2 }, //
          { -18, -10, -4, 8 }, //
          { -4, -28, 0, 0 }, //
          { -5, 6, 3, 2 }, //
          { -3, 0, 0, 1 }, //
          { -5, -9, -4, 2 }, //
          { 17, 0, 0, -7 }, //
          { 11, 4, 0, 0 }, //
          { 0, -6, -2, 0 }, //
          { 83, 15, 0, 0 }, //
          { -4, 0, 0, 2 }, //
          { 0, -114, -49, 0 }, //
          { 117, 0, 0, -51 }, //
          { -5, 19, 10, 2 }, //
          { -3, 0, 0, 0 }, //
          { -3, 0, 0, 2 }, //
          { 0, -3, -1, 0 }, //
          { 3, 0, 0, 0 }, //
          { 0, -6, -2, 0 }, //
          { 393, 3, 0, 0 }, //
          { -4, 21, 11, 2 }, //
          { -6, 0, -1, 3 }, //
          { -3, 8, 4, 1 }, //
          { 8, 0, 0, 0 }, //
          { 18, -29, -13, -8 }, //
          { 8, 34, 18, -4 }, //
          { 89, 0, 0, 0 }, //
          { 3, 12, 6, -1 }, //
          { 54, -15, -7, -24 }, //
          { 0, 3, 0, 0 }, //
          { 3, 0, 0, -1 }, //
          { 0, 35, 0, 0 }, //
          { -154, -30, -13, 67 }, //
          { 15, 0, 0, 0 }, //
          { 0, 4, 2, 0 }, //
          { 0, 9, 0, 0 }, //
          { 80, -71, -31, -35 }, //
          { 0, -20, -9, 0 }, //
          { 11, 5, 2, -5 }, //
          { 61, -96, -42, -27 }, //
          { 14, 9, 4, -6 }, //
          { -11, -6, -3, 5 }, //
          { 0, -3, -1, 0 }, //
          { 123, -415, -180, -53 }, //
          { 0, 0, 0, -35 }, //
          { -5, 0, 0, 0 }, //
          { 7, -32, -17, -4 }, //
          { 0, -9, -5, 0 }, //
          { 0, -4, 2, 0 }, //
          { -89, 0, 0, 38 }, //
          { 0, -86, -19, -6 }, //
          { 0, 0, -19, 6 }, //
          { -123, -416, -180, 53 }, //
          { 0, -3, -1, 0 }, //
          { 12, -6, -3, -5 }, //
          { -13, 9, 4, 6 }, //
          { 0, -15, -7, 0 }, //
          { 3, 0, 0, -1 }, //
          { -62, -97, -42, 27 }, //
          { -11, 5, 2, 5 }, //
          { 0, -19, -8, 0 }, //
          { -3, 0, 0, 1 }, //
          { 0, 4, 2, 0 }, //
          { 0, 3, 0, 0 }, //
          { 0, 4, 2, 0 }, //
          { -85, -70, -31, 37 }, //
          { 163, -12, -5, -72 }, //
          { -63, -16, -7, 28 }, //
          { -21, -32, -14, 9 }, //
          { 0, -3, -1, 0 }, //
          { 3, 0, 0, -2 }, //
          { 0, 8, 0, 0 }, //
          { 3, 10, 4, -1 }, //
          { 3, 0, 0, -1 }, //
          { 0, -7, -3, 0 }, //
          { 0, -4, -2, 0 }, //
          { 6, 19, 0, 0 }, //
          { 5, -173, -75, -2 }, //
          { 0, -7, -3, 0 }, //
          { 7, -12, -5, -3 }, //
          { -3, 0, 0, 2 }, //
          { 3, -4, -2, -1 }, //
          { 74, 0, 0, -32 }, //
          { -3, 12, 6, 2 }, //
          { 26, -14, -6, -11 }, //
          { 19, 0, 0, -8 }, //
          { 6, 24, 13, -3 }, //
          { 83, 0, 0, 0 }, //
          { 0, -10, -5, 0 }, //
          { 11, -3, -1, -5 }, //
          { 3, 0, 1, -1 }, //
          { 3, 0, 0, -1 }, //
          { -4, 0, 0, 0 }, //
          { 5, -23, -12, -3 }, //
          { -339, 0, 0, 147 }, //
          { 0, -10, -5, 0 }, //
          { 5, 0, 0, 0 }, //
          { 3, 0, 0, -1 }, //
          { 0, -4, -2, 0 }, //
          { 18, -3, 0, 0 }, //
          { 9, -11, -5, -4 }, //
          { -8, 0, 0, 4 }, //
          { 3, 0, 0, -1 }, //
          { 0, 9, 0, 0 }, //
          { 6, -9, -4, -2 }, //
          { -4, -12, 0, 0 }, //
          { 67, -91, -39, -29 }, //
          { 30, -18, -8, -13 }, //
          { 0, 0, 0, 0 }, //
          { 0, -114, -50, 0 }, //
          { 0, 0, 0, 23 }, //
          { 517, 16, 7, -224 }, //
          { 0, -7, -3, 0 }, //
          { 143, -3, -1, -62 }, //
          { 29, 0, 0, -13 }, //
          { -4, 0, 0, 2 }, //
          { -6, 0, 0, 3 }, //
          { 5, 12, 5, -2 }, //
          { -25, 0, 0, 11 }, //
          { -3, 0, 0, 1 }, //
          { 0, 4, 2, 0 }, //
          { -22, 12, 5, 10 }, //
          { 50, 0, 0, -22 }, //
          { 0, 7, 4, 0 }, //
          { 0, 3, 1, 0 }, //
          { -4, 4, 2, 2 }, //
          { -5, -11, -5, 2 }, //
          { 0, 4, 2, 0 }, //
          { 4, 17, 9, -2 }, //
          { 59, 0, 0, 0 }, //
          { 0, -4, -2, 0 }, //
          { -8, 0, 0, 4 }, //
          { -3, 0, 0, 0 }, //
          { 4, -15, -8, -2 }, //
          { 370, -8, 0, -160 }, //
          { 0, 0, -3, 0 }, //
          { 0, 3, 1, 0 }, //
          { -6, 3, 1, 3 }, //
          { 0, 6, 0, 0 }, //
          { -10, 0, 0, 4 }, //
          { 0, 9, 4, 0 }, //
          { 4, 17, 7, -2 }, //
          { 34, 0, 0, -15 }, //
          { 0, 5, 3, 0 }, //
          { -5, 0, 0, 2 }, //
          { -37, -7, -3, 16 }, //
          { 3, 13, 7, -2 }, //
          { 40, 0, 0, 0 }, //
          { 0, -3, -2, 0 }, //
          { -184, -3, -1, 80 }, //
          { -3, 0, 0, 1 }, //
          { -3, 0, 0, 0 }, //
          { 0, -10, -6, -1 }, //
          { 31, -6, 0, -13 }, //
          { -3, -32, -14, 1 }, //
          { -7, 0, 0, 3 }, //
          { 0, -8, -4, 0 }, //
          { 3, -4, 0, 0 }, //
          { 0, 4, 0, 0 }, //
          { 0, 3, 1, 0 }, //
          { 19, -23, -10, 2 }, //
          { 0, 0, 0, -10 }, //
          { 0, 3, 2, 0 }, //
          { 0, 9, 5, -1 }, //
          { 28, 0, 0, 0 }, //
          { 0, -7, -4, 0 }, //
          { 8, -4, 0, -4 }, //
          { 0, 0, -2, 0 }, //
          { 0, 3, 0, 0 }, //
          { -3, 0, 0, 1 }, //
          { -9, 0, 1, 4 }, //
          { 3, 12, 5, -1 }, //
          { 17, -3, -1, 0 }, //
          { 0, 7, 4, 0 }, //
          { 19, 0, 0, 0 }, //
          { 0, -5, -3, 0 }, //
          { 14, -3, 0, -1 }, //
          { 0, 0, -1, 0 }, //
          { 0, 0, 0, -5 }, //
          { 0, 5, 3, 0 }, //
          { 13, 0, 0, 0 }, //
          { 0, -3, -2, 0 }, //
          { 2, 9, 4, 3 }, //
          { 0, 0, 0, -4 }, //
          { 8, 0, 0, 0 }, //
          { 0, 4, 2, 0 }, //
          { 6, 0, 0, -3 }, //
          { 6, 0, 0, 0 }, //
          { 0, 3, 1, 0 }, //
          { 5, 0, 0, -2 }, //
          { 3, 0, 0, -1 }, //
          { -3, 0, 0, 0 }, //
          { 6, 0, 0, 0 }, //
          { 7, 0, 0, 0 }, //
          { -4, 0, 0, 0 }, //
          { 4, 0, 0, 0 }, //
          { 6, 0, 0, 0 }, //
          { 0, -4, 0, 0 }, //
          { 0, -4, 0, 0 }, //
          { 5, 0, 0, 0 }, //
          { -3, 0, 0, 0 }, //
          { 4, 0, 0, 0 }, //
          { -5, 0, 0, 0 }, //
          { 4, 0, 0, 0 }, //
          { 0, 3, 0, 0 }, //
          { 13, 0, 0, 0 }, //
          { 21, 11, 0, 0 }, //
          { 0, -5, 0, 0 }, //
          { 0, -5, -2, 0 }, //
          { 0, 5, 3, 0 }, //
          { 0, -5, 0, 0 }, //
          { -3, 0, 0, 2 }, //
          { 20, 10, 0, 0 }, //
          { -34, 0, 0, 0 }, //
          { -19, 0, 0, 0 }, //
          { 3, 0, 0, -2 }, //
          { -3, 0, 0, 1 }, //
          { -6, 0, 0, 3 }, //
          { -4, 0, 0, 0 }, //
          { 3, 0, 0, 0 }, //
          { 3, 0, 0, 0 }, //
          { 4, 0, 0, 0 }, //
          { 3, 0, 0, -1 }, //
          { 6, 0, 0, -3 }, //
          { -8, 0, 0, 3 }, //
          { 0, 3, 1, 0 }, //
          { -3, 0, 0, 0 }, //
          { 0, -3, -2, 0 }, //
          { 126, -63, -27, -55 }, //
          { -5, 0, 1, 2 }, //
          { -3, 28, 15, 2 }, //
          { 5, 0, 1, -2 }, //
          { 0, 9, 4, 1 }, //
          { 0, 9, 4, -1 }, //
          { -126, -63, -27, 55 }, //
          { 3, 0, 0, -1 }, //
          { 21, -11, -6, -11 }, //
          { 0, -4, 0, 0 }, //
          { -21, -11, -6, 11 }, //
          { -3, 0, 0, 1 }, //
          { 0, 3, 1, 0 }, //
          { 8, 0, 0, -4 }, //
          { -6, 0, 0, 3 }, //
          { -3, 0, 0, 1 }, //
          { 3, 0, 0, -1 }, //
          { -3, 0, 0, 1 }, //
          { -5, 0, 0, 2 }, //
          { 24, -12, -5, -11 }, //
          { 0, 3, 1, 0 }, //
          { 0, 3, 1, 0 }, //
          { 0, 3, 2, 0 }, //
          { -24, -12, -5, 10 }, //
          { 4, 0, -1, -2 }, //
          { 13, 0, 0, -6 }, //
          { 7, 0, 0, -3 }, //
          { 3, 0, 0, -1 }, //
          { 3, 0, 0, -1 } };

    // @formatter:on

  // Convert from 0.1 microarcsec units to radians.
  const double factor = 1.0e-7 * ASEC2RAD;

  // Interval between fundamental epoch J2000.0 and given date.
  const double t = ((jd_tt_high - T0) + jd_tt_low) / JULIAN_CENTURY_DAYS;

  // Planetary longitudes, Mercury through Neptune, wrt mean dynamical
  // ecliptic and equinox of J2000
  // (Simon et al. 1994, 5.8.1-5.8.8).
  const double alme = planet_lon(t, NOVAS_MERCURY);
  const double alve = planet_lon(t, NOVAS_VENUS);
  const double alea = planet_lon(t, NOVAS_EARTH);
  const double alma = planet_lon(t, NOVAS_MARS);
  const double alju = planet_lon(t, NOVAS_JUPITER);
  const double alsa = planet_lon(t, NOVAS_SATURN);
  const double alur = planet_lon(t, NOVAS_URANUS);
  const double alne = planet_lon(t, NOVAS_NEPTUNE);

  // General precession in longitude (Simon et al. 1994), equivalent
  // to 5028.8200 arcsec/cy at J2000.
  const double apa = accum_prec(t);

  novas_delaunay_args a;
  double dpsils = 0.0, depsls = 0.0, dpsipl = 0.0, depspl = 0.0;
  int i;

  // Compute fundamental arguments from Simon et al. (1994), in radians.
  fund_args(t, &a);

  // ** Luni-solar nutation. **
  // Summation of luni-solar nutation series (in reverse order).
  for(i = 678; --i >= 0;) {
    const int8_t *n = &nals_t[i][0];
    const int32_t *c = &cls_t[i][0];

    // Argument and functions.
    double arg = 0.0, sarg, carg;

    if(n[0])
      arg += n[0] * a.l;
    if(n[1])
      arg += n[1] * a.l1;
    if(n[2])
      arg += n[2] * a.F;
    if(n[3])
      arg += n[3] * a.D;
    if(n[4])
      arg += n[4] * a.Omega;

    sarg = sin(arg);
    carg = cos(arg);

    // Term.
    dpsils += (c[0] + c[1] * t) * sarg + c[2] * carg;
    depsls += (c[3] + c[4] * t) * carg + c[5] * sarg;
  }

  // ** Planetary nutation. **

  // Summation of planetary nutation series (in reverse order).
  for(i = 687; --i >= 0;) {
    const int8_t *n = &napl_t[i][0];
    const int16_t *c = &cpl_t[i][0];

    // Argument and functions.
    double arg = 0.0, sarg, carg;

    if(n[0])
      arg += n[0] * a.l;
    /* The series does not contain terms for l1
    if(n[1])
      arg += n[1] * a.l1;
    */
    if(n[2])
      arg += n[2] * a.F;
    if(n[3])
      arg += n[3] * a.D;
    if(n[4])
      arg += n[4] * a.Omega;
    if(n[5])
      arg += n[5] * alme;
    if(n[6])
      arg += n[6] * alve;
    if(n[7])
      arg += n[7] * alea;
    if(n[8])
      arg += n[8] * alma;
    if(n[9])
      arg += n[9] * alju;
    if(n[10])
      arg += n[10] * alsa;
    if(n[11])
      arg += n[11] * alur;
    if(n[12])
      arg += n[12] * alne;
    if(n[13])
      arg += n[13] * apa;

    sarg = sin(arg);
    carg = cos(arg);

    // Term.
    dpsipl += c[0] * sarg + c[1] * carg;
    depspl += c[2] * sarg + c[3] * carg;
  }

  // Total: Add planetary and luni-solar components.
  if(dpsi)
    *dpsi = (dpsils + dpsipl) * factor;
  if(deps)
    *deps = (depsls + depspl) * factor;

  return 0;
}

/**
 * Compute the forced nutation of the non-rigid Earth based on the IAU 2000B precession /
 * nutation model.
 *
 * IAU 2000B reproduces the IAU 2000A model to a precision of 1 milliarcsecond in the
 * interval 1995-2020.
 *
 * REFERENCES:
 * <ol>
 * <li>McCarthy, D. and Luzum, B. (2003). "An Abridged Model of the
 *     Precession &amp; Nutation of the Celestial Pole," Celestial
 *     Mechanics and Dynamical Astronomy, Volume 85, Issue 1,
 *     Jan. 2003, p. 37. (IAU 2000B) IERS Conventions (2003), Chapter 5.</li>
 * </ol>
 *
 * @param jd_tt_high  [day] High-order part of the Terrestrial Time (TT) based Julian date.
 *                    Typically it may be the integer part of a split date for the highest
 *                    precision, or the full date for normal (reduced) precision.
 * @param jd_tt_low   [day] Low-order part of the Terrestrial Time (TT) based Julian date.
 *                    Typically it may be the fractional part of a split date for the highest
 *                    precision, or 0.0 for normal (reduced) precision.
 * @param[out] dpsi   [rad] &delta;&psi; Nutation (luni-solar + planetary) in longitude, in
 *                    radians. It may be NULL if not required.
 * @param[out] deps   [rad] &delta;&epsilon; Nutation (luni-solar + planetary) in obliquity,
 *                    in radians. It may be NULL if not required.
 * @return            0
 *
 * @sa iau2000a()
 * @sa nu2000k()
 * @sa set_nutation_lp_provider()
 * @sa nutation_angles()
 *
 */
int iau2000b(double jd_tt_high, double jd_tt_low, double *restrict dpsi, double *restrict deps) {
  int16_t i;

  // @formatter:off

  // Planetary nutation (arcsec).  These fixed terms account for the
  // omission of the long-period planetary terms in the truncated model.

  // Luni-Solar argument multipliers:
  // L     L'    F     D     Om
  static const int8_t nals_t[77][5] = { //
          { 0, 0, 0, 0, 1 }, //
          { 0, 0, 2, -2, 2 }, //
          { 0, 0, 2, 0, 2 }, //
          { 0, 0, 0, 0, 2 }, //
          { 0, 1, 0, 0, 0 }, //
          { 0, 1, 2, -2, 2 }, //
          { 1, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 1 }, //
          { 1, 0, 2, 0, 2 }, //
          { 0, -1, 2, -2, 2 }, //
          { 0, 0, 2, -2, 1 }, //
          { -1, 0, 2, 0, 2 }, //
          { -1, 0, 0, 2, 0 }, //
          { 1, 0, 0, 0, 1 }, //
          { -1, 0, 0, 0, 1 }, //
          { -1, 0, 2, 2, 2 }, //
          { 1, 0, 2, 0, 1 }, //
          { -2, 0, 2, 0, 1 }, //
          { 0, 0, 0, 2, 0 }, //
          { 0, 0, 2, 2, 2 }, //
          { 0, -2, 2, -2, 2 }, //
          { -2, 0, 0, 2, 0 }, //
          { 2, 0, 2, 0, 2 }, //
          { 1, 0, 2, -2, 2 }, //
          { -1, 0, 2, 0, 1 }, //
          { 2, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 0 }, //
          { 0, 1, 0, 0, 1 }, //
          { -1, 0, 0, 2, 1 }, //
          { 0, 2, 2, -2, 2 }, //
          { 0, 0, -2, 2, 0 }, //
          { 1, 0, 0, -2, 1 }, //
          { 0, -1, 0, 0, 1 }, //
          { -1, 0, 2, 2, 1 }, //
          { 0, 2, 0, 0, 0 }, //
          { 1, 0, 2, 2, 2 }, //
          { -2, 0, 2, 0, 0 }, //
          { 0, 1, 2, 0, 2 }, //
          { 0, 0, 2, 2, 1 }, //
          { 0, -1, 2, 0, 2 }, //
          { 0, 0, 0, 2, 1 }, //
          { 1, 0, 2, -2, 1 }, //
          { 2, 0, 2, -2, 2 }, //
          { -2, 0, 0, 2, 1 }, //
          { 2, 0, 2, 0, 1 }, //
          { 0, -1, 2, -2, 1 }, //
          { 0, 0, 0, -2, 1 }, //
          { -1, -1, 0, 2, 0 }, //
          { 2, 0, 0, -2, 1 }, //
          { 1, 0, 0, 2, 0 }, //
          { 0, 1, 2, -2, 1 }, //
          { 1, -1, 0, 0, 0 }, //
          { -2, 0, 2, 0, 2 }, //
          { 3, 0, 2, 0, 2 }, //
          { 0, -1, 0, 2, 0 }, //
          { 1, -1, 2, 0, 2 }, //
          { 0, 0, 0, 1, 0 }, //
          { -1, -1, 2, 2, 2 }, //
          { -1, 0, 2, 0, 0 }, //
          { 0, -1, 2, 2, 2 }, //
          { -2, 0, 0, 0, 1 }, //
          { 1, 1, 2, 0, 2 }, //
          { 2, 0, 0, 0, 1 }, //
          { -1, 1, 0, 1, 0 }, //
          { 1, 1, 0, 0, 0 }, //
          { 1, 0, 2, 0, 0 }, //
          { -1, 0, 2, -2, 1 }, //
          { 1, 0, 0, 0, 2 }, //
          { -1, 0, 0, 1, 0 }, //
          { 0, 0, 2, 1, 2 }, //
          { -1, 0, 2, 4, 2 }, //
          { -1, 1, 0, 1, 1 }, //
          { 0, -2, 2, -2, 1 }, //
          { 1, 0, 2, 2, 1 }, //
          { -2, 0, 2, 2, 2 }, //
          { -1, 0, 0, 0, 2 }, //
          { 1, 1, 2, -2, 2 } };

  // Luni-Solar nutation coefficients, unit 1e-7 arcsec:
  // longitude (sin, t*sin, cos), obliquity (cos, t*cos, sin)

  // Each row of coefficients in 'cls_t' belongs with the corresponding
  // row of fundamental-argument multipliers in 'nals_t'.
  static const int32_t cls_t[77][6] = { //
          { -172064161, -174666, 33386, 92052331, 9086, 15377 }, //
          { -13170906, -1675, -13696, 5730336, -3015, -4587 }, //
          { -2276413, -234, 2796, 978459, -485, 1374 }, //
          { 2074554, 207, -698, -897492, 470, -291 }, //
          { 1475877, -3633, 11817, 73871, -184, -1924 }, //
          { -516821, 1226, -524, 224386, -677, -174 }, //
          { 711159, 73, -872, -6750, 0, 358 }, //
          { -387298, -367, 380, 200728, 18, 318 }, //
          { -301461, -36, 816, 129025, -63, 367 }, //
          { 215829, -494, 111, -95929, 299, 132 }, //
          { 128227, 137, 181, -68982, -9, 39 }, //
          { 123457, 11, 19, -53311, 32, -4 }, //
          { 156994, 10, -168, -1235, 0, 82 }, //
          { 63110, 63, 27, -33228, 0, -9 }, //
          { -57976, -63, -189, 31429, 0, -75 }, //
          { -59641, -11, 149, 25543, -11, 66 }, //
          { -51613, -42, 129, 26366, 0, 78 }, //
          { 45893, 50, 31, -24236, -10, 20 }, //
          { 63384, 11, -150, -1220, 0, 29 }, //
          { -38571, -1, 158, 16452, -11, 68 }, //
          { 32481, 0, 0, -13870, 0, 0 }, //
          { -47722, 0, -18, 477, 0, -25 }, //
          { -31046, -1, 131, 13238, -11, 59 }, //
          { 28593, 0, -1, -12338, 10, -3 }, //
          { 20441, 21, 10, -10758, 0, -3 }, //
          { 29243, 0, -74, -609, 0, 13 }, //
          { 25887, 0, -66, -550, 0, 11 }, //
          { -14053, -25, 79, 8551, -2, -45 }, //
          { 15164, 10, 11, -8001, 0, -1 }, //
          { -15794, 72, -16, 6850, -42, -5 }, //
          { 21783, 0, 13, -167, 0, 13 }, //
          { -12873, -10, -37, 6953, 0, -14 }, //
          { -12654, 11, 63, 6415, 0, 26 }, //
          { -10204, 0, 25, 5222, 0, 15 }, //
          { 16707, -85, -10, 168, -1, 10 }, //
          { -7691, 0, 44, 3268, 0, 19 }, //
          { -11024, 0, -14, 104, 0, 2 }, //
          { 7566, -21, -11, -3250, 0, -5 }, //
          { -6637, -11, 25, 3353, 0, 14 }, //
          { -7141, 21, 8, 3070, 0, 4 }, //
          { -6302, -11, 2, 3272, 0, 4 }, //
          { 5800, 10, 2, -3045, 0, -1 }, //
          { 6443, 0, -7, -2768, 0, -4 }, //
          { -5774, -11, -15, 3041, 0, -5 }, //
          { -5350, 0, 21, 2695, 0, 12 }, //
          { -4752, -11, -3, 2719, 0, -3 }, //
          { -4940, -11, -21, 2720, 0, -9 }, //
          { 7350, 0, -8, -51, 0, 4 }, //
          { 4065, 0, 6, -2206, 0, 1 }, //
          { 6579, 0, -24, -199, 0, 2 }, //
          { 3579, 0, 5, -1900, 0, 1 }, //
          { 4725, 0, -6, -41, 0, 3 }, //
          { -3075, 0, -2, 1313, 0, -1 }, //
          { -2904, 0, 15, 1233, 0, 7 }, //
          { 4348, 0, -10, -81, 0, 2 }, //
          { -2878, 0, 8, 1232, 0, 4 }, //
          { -4230, 0, 5, -20, 0, -2 }, //
          { -2819, 0, 7, 1207, 0, 3 }, //
          { -4056, 0, 5, 40, 0, -2 }, //
          { -2647, 0, 11, 1129, 0, 5 }, //
          { -2294, 0, -10, 1266, 0, -4 }, //
          { 2481, 0, -7, -1062, 0, -3 }, //
          { 2179, 0, -2, -1129, 0, -2 }, //
          { 3276, 0, 1, -9, 0, 0 }, //
          { -3389, 0, 5, 35, 0, -2 }, //
          { 3339, 0, -13, -107, 0, 1 }, //
          { -1987, 0, -6, 1073, 0, -2 }, //
          { -1981, 0, 0, 854, 0, 0 }, //
          { 4026, 0, -353, -553, 0, -139 }, //
          { 1660, 0, -5, -710, 0, -2 }, //
          { -1521, 0, 9, 647, 0, 4 }, //
          { 1314, 0, 0, -700, 0, 0 }, //
          { -1283, 0, 0, 672, 0, 0 }, //
          { -1331, 0, 8, 663, 0, 4 }, //
          { 1383, 0, -2, -594, 0, -2 }, //
          { 1405, 0, 4, -610, 0, 2 }, //
          { 1290, 0, 0, -556, 0, 0 } };

    // @formatter:on

  // Interval between fundamental epoch J2000.0 and given date.
  const double t = ((jd_tt_high - T0) + jd_tt_low) / JULIAN_CENTURY_DAYS;

  double dpsils = 0.0, depsls = 0.0;
  novas_delaunay_args a;

  // ** Luni-solar nutation. **
  fund_args(t, &a);

  // Summation of luni-solar nutation series (in reverse order)
  for(i = 77; --i >= 0;) {
    const int8_t *n = &nals_t[i][0];
    const int32_t *c = &cls_t[i][0];

    // Argument and functions.
    double arg = 0.0, sarg, carg;

    if(n[0])
      arg += n[0] * a.l;
    if(n[1])
      arg += n[1] * a.l1;
    if(n[2])
      arg += n[2] * a.F;
    if(n[3])
      arg += n[3] * a.D;
    if(n[4])
      arg += n[4] * a.Omega;

    sarg = sin(arg);
    carg = cos(arg);

    // Term.
    dpsils += (c[0] + c[1] * t) * sarg + c[2] * carg;
    depsls += (c[3] + c[4] * t) * carg + c[5] * sarg;
  }

  // Total: Add planetary and luni-solar components.
  if(dpsi)
    *dpsi = (1e-7 * dpsils - 0.000135) * ARCSEC;
  if(deps)
    *deps = (1e-7 * depsls + 0.000388) * ARCSEC;

  return 0;
}

/**
 * Computes the forced nutation of the non-rigid Earth: Model NU2000K.  This model is a
 * modified version of IAU 2000A, which has been truncated for speed of execution, and uses
 * Simon et al. (1994) fundamental arguments throughout.  NU2000K agrees with IAU 2000A at
 * the 0.1 milliarcsecond level from 1700 to 2300. It has the most modest computational cost
 * among the implementations provided in the NOVAS library.
 *
 * NU2000K was compared to IAU 2000A over six centuries (1700-2300). The average error in
 * d&psi; is 20 microarcseconds, with 98% of the errors < 60 microarcseconds;  the average
 * error in d&epsilon;is 8 microarcseconds, with 100% of the errors < 60 microarcseconds.
 *
 * NU2000K was developed by G. Kaplan (USNO) in March 2004
 *
 * REFERENCES:
 * <ol>
 * <li>IERS Conventions (2003), Chapter 5.</li>
 * <li>Simon et al. (1994) Astronomy and Astrophysics 282, 663-683, esp. Sections 3.4-3.5.</li>
 * </ol>
 *
 * @param jd_tt_high  [day] High-order part of the Terrestrial Time (TT) based Julian date.
 *                    Typically it may be the integer part of a split date for the highest
 *                    precision, or the full date for normal (reduced) precision.
 * @param jd_tt_low   [day] Low-order part of the Terrestrial Time (TT) based Julian date.
 *                    Typically it may be the fractional part of a split date for the highest
 *                    precision, or 0.0 for normal (reduced) precision.
 * @param[out] dpsi   [rad] &delta;&psi; Nutation (luni-solar + planetary) in longitude, in
 *                    radians. It may be NULL if not required.
 * @param[out] deps   [rad] &delta;&epsilon; Nutation (luni-solar + planetary) in obliquity,
 *                    in radians. It may be NULL if not required.
 * @return            0
 *
 * @sa iau2000a()
 * @sa iau2000b()
 * @sa nutation_angles()
 * @sa novas_nutation_provider
 *
 */
int nu2000k(double jd_tt_high, double jd_tt_low, double *restrict dpsi, double *restrict deps) {

  // @formatter:off

  // Luni-Solar argument multipliers:
  // L     L'    F     D     Om
  static const int8_t nals_t[323][5] = { //
          { 0, 0, 0, 0, 1 }, //
          { 0, 0, 2, -2, 2 }, //
          { 0, 0, 2, 0, 2 }, //
          { 0, 0, 0, 0, 2 }, //
          { 0, 1, 0, 0, 0 }, //
          { 0, 1, 2, -2, 2 }, //
          { 1, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 1 }, //
          { 1, 0, 2, 0, 2 }, //
          { 0, -1, 2, -2, 2 }, //
          { 0, 0, 2, -2, 1 }, //
          { -1, 0, 2, 0, 2 }, //
          { -1, 0, 0, 2, 0 }, //
          { 1, 0, 0, 0, 1 }, //
          { -1, 0, 0, 0, 1 }, //
          { -1, 0, 2, 2, 2 }, //
          { 1, 0, 2, 0, 1 }, //
          { -2, 0, 2, 0, 1 }, //
          { 0, 0, 0, 2, 0 }, //
          { 0, 0, 2, 2, 2 }, //
          { 0, -2, 2, -2, 2 }, //
          { -2, 0, 0, 2, 0 }, //
          { 2, 0, 2, 0, 2 }, //
          { 1, 0, 2, -2, 2 }, //
          { -1, 0, 2, 0, 1 }, //
          { 2, 0, 0, 0, 0 }, //
          { 0, 0, 2, 0, 0 }, //
          { 0, 1, 0, 0, 1 }, //
          { -1, 0, 0, 2, 1 }, //
          { 0, 2, 2, -2, 2 }, //
          { 0, 0, -2, 2, 0 }, //
          { 1, 0, 0, -2, 1 }, //
          { 0, -1, 0, 0, 1 }, //
          { -1, 0, 2, 2, 1 }, //
          { 0, 2, 0, 0, 0 }, //
          { 1, 0, 2, 2, 2 }, //
          { -2, 0, 2, 0, 0 }, //
          { 0, 1, 2, 0, 2 }, //
          { 0, 0, 2, 2, 1 }, //
          { 0, -1, 2, 0, 2 }, //
          { 0, 0, 0, 2, 1 }, //
          { 1, 0, 2, -2, 1 }, //
          { 2, 0, 2, -2, 2 }, //
          { -2, 0, 0, 2, 1 }, //
          { 2, 0, 2, 0, 1 }, //
          { 0, -1, 2, -2, 1 }, //
          { 0, 0, 0, -2, 1 }, //
          { -1, -1, 0, 2, 0 }, //
          { 2, 0, 0, -2, 1 }, //
          { 1, 0, 0, 2, 0 }, //
          { 0, 1, 2, -2, 1 }, //
          { 1, -1, 0, 0, 0 }, //
          { -2, 0, 2, 0, 2 }, //
          { 3, 0, 2, 0, 2 }, //
          { 0, -1, 0, 2, 0 }, //
          { 1, -1, 2, 0, 2 }, //
          { 0, 0, 0, 1, 0 }, //
          { -1, -1, 2, 2, 2 }, //
          { -1, 0, 2, 0, 0 }, //
          { 0, -1, 2, 2, 2 }, //
          { -2, 0, 0, 0, 1 }, //
          { 1, 1, 2, 0, 2 }, //
          { 2, 0, 0, 0, 1 }, //
          { -1, 1, 0, 1, 0 }, //
          { 1, 1, 0, 0, 0 }, //
          { 1, 0, 2, 0, 0 }, //
          { -1, 0, 2, -2, 1 }, //
          { 1, 0, 0, 0, 2 }, //
          { -1, 0, 0, 1, 0 }, //
          { 0, 0, 2, 1, 2 }, //
          { -1, 0, 2, 4, 2 }, //
          { -1, 1, 0, 1, 1 }, //
          { 0, -2, 2, -2, 1 }, //
          { 1, 0, 2, 2, 1 }, //
          { -2, 0, 2, 2, 2 }, //
          { -1, 0, 0, 0, 2 }, //
          { 1, 1, 2, -2, 2 }, //
          { -2, 0, 2, 4, 2 }, //
          { -1, 0, 4, 0, 2 }, //
          { 2, 0, 2, -2, 1 }, //
          { 2, 0, 2, 2, 2 }, //
          { 1, 0, 0, 2, 1 }, //
          { 3, 0, 0, 0, 0 }, //
          { 3, 0, 2, -2, 2 }, //
          { 0, 0, 4, -2, 2 }, //
          { 0, 1, 2, 0, 1 }, //
          { 0, 0, -2, 2, 1 }, //
          { 0, 0, 2, -2, 3 }, //
          { -1, 0, 0, 4, 0 }, //
          { 2, 0, -2, 0, 1 }, //
          { -2, 0, 0, 4, 0 }, //
          { -1, -1, 0, 2, 1 }, //
          { -1, 0, 0, 1, 1 }, //
          { 0, 1, 0, 0, 2 }, //
          { 0, 0, -2, 0, 1 }, //
          { 0, -1, 2, 0, 1 }, //
          { 0, 0, 2, -1, 2 }, //
          { 0, 0, 2, 4, 2 }, //
          { -2, -1, 0, 2, 0 }, //
          { 1, 1, 0, -2, 1 }, //
          { -1, 1, 0, 2, 0 }, //
          { -1, 1, 0, 1, 2 }, //
          { 1, -1, 0, 0, 1 }, //
          { 1, -1, 2, 2, 2 }, //
          { -1, 1, 2, 2, 2 }, //
          { 3, 0, 2, 0, 1 }, //
          { 0, 1, -2, 2, 0 }, //
          { -1, 0, 0, -2, 1 }, //
          { 0, 1, 2, 2, 2 }, //
          { -1, -1, 2, 2, 1 }, //
          { 0, -1, 0, 0, 2 }, //
          { 1, 0, 2, -4, 1 }, //
          { -1, 0, -2, 2, 0 }, //
          { 0, -1, 2, 2, 1 }, //
          { 2, -1, 2, 0, 2 }, //
          { 0, 0, 0, 2, 2 }, //
          { 1, -1, 2, 0, 1 }, //
          { -1, 1, 2, 0, 2 }, //
          { 0, 1, 0, 2, 0 }, //
          { 0, -1, -2, 2, 0 }, //
          { 0, 3, 2, -2, 2 }, //
          { 0, 0, 0, 1, 1 }, //
          { -1, 0, 2, 2, 0 }, //
          { 2, 1, 2, 0, 2 }, //
          { 1, 1, 0, 0, 1 }, //
          { 1, 1, 2, 0, 1 }, //
          { 2, 0, 0, 2, 0 }, //
          { 1, 0, -2, 2, 0 }, //
          { -1, 0, 0, 2, 2 }, //
          { 0, 1, 0, 1, 0 }, //
          { 0, 1, 0, -2, 1 }, //
          { -1, 0, 2, -2, 2 }, //
          { 0, 0, 0, -1, 1 }, //
          { -1, 1, 0, 0, 1 }, //
          { 1, 0, 2, -1, 2 }, //
          { 1, -1, 0, 2, 0 }, //
          { 0, 0, 0, 4, 0 }, //
          { 1, 0, 2, 1, 2 }, //
          { 0, 0, 2, 1, 1 }, //
          { 1, 0, 0, -2, 2 }, //
          { -1, 0, 2, 4, 1 }, //
          { 1, 0, -2, 0, 1 }, //
          { 1, 1, 2, -2, 1 }, //
          { 0, 0, 2, 2, 0 }, //
          { -1, 0, 2, -1, 1 }, //
          { -2, 0, 2, 2, 1 }, //
          { 4, 0, 2, 0, 2 }, //
          { 2, -1, 0, 0, 0 }, //
          { 2, 1, 2, -2, 2 }, //
          { 0, 1, 2, 1, 2 }, //
          { 1, 0, 4, -2, 2 }, //
          { -1, -1, 0, 0, 1 }, //
          { 0, 1, 0, 2, 1 }, //
          { -2, 0, 2, 4, 1 }, //
          { 2, 0, 2, 0, 0 }, //
          { 1, 0, 0, 1, 0 }, //
          { -1, 0, 0, 4, 1 }, //
          { -1, 0, 4, 0, 1 }, //
          { 2, 0, 2, 2, 1 }, //
          { 0, 0, 2, -3, 2 }, //
          { -1, -2, 0, 2, 0 }, //
          { 2, 1, 0, 0, 0 }, //
          { 0, 0, 4, 0, 2 }, //
          { 0, 0, 0, 0, 3 }, //
          { 0, 3, 0, 0, 0 }, //
          { 0, 0, 2, -4, 1 }, //
          { 0, -1, 0, 2, 1 }, //
          { 0, 0, 0, 4, 1 }, //
          { -1, -1, 2, 4, 2 }, //
          { 1, 0, 2, 4, 2 }, //
          { -2, 2, 0, 2, 0 }, //
          { -2, -1, 2, 0, 1 }, //
          { -2, 0, 0, 2, 2 }, //
          { -1, -1, 2, 0, 2 }, //
          { 0, 0, 4, -2, 1 }, //
          { 3, 0, 2, -2, 1 }, //
          { -2, -1, 0, 2, 1 }, //
          { 1, 0, 0, -1, 1 }, //
          { 0, -2, 0, 2, 0 }, //
          { -2, 0, 0, 4, 1 }, //
          { -3, 0, 0, 0, 1 }, //
          { 1, 1, 2, 2, 2 }, //
          { 0, 0, 2, 4, 1 }, //
          { 3, 0, 2, 2, 2 }, //
          { -1, 1, 2, -2, 1 }, //
          { 2, 0, 0, -4, 1 }, //
          { 0, 0, 0, -2, 2 }, //
          { 2, 0, 2, -4, 1 }, //
          { -1, 1, 0, 2, 1 }, //
          { 0, 0, 2, -1, 1 }, //
          { 0, -2, 2, 2, 2 }, //
          { 2, 0, 0, 2, 1 }, //
          { 4, 0, 2, -2, 2 }, //
          { 2, 0, 0, -2, 2 }, //
          { 0, 2, 0, 0, 1 }, //
          { 1, 0, 0, -4, 1 }, //
          { 0, 2, 2, -2, 1 }, //
          { -3, 0, 0, 4, 0 }, //
          { -1, 1, 2, 0, 1 }, //
          { -1, -1, 0, 4, 0 }, //
          { -1, -2, 2, 2, 2 }, //
          { -2, -1, 2, 4, 2 }, //
          { 1, -1, 2, 2, 1 }, //
          { -2, 1, 0, 2, 0 }, //
          { -2, 1, 2, 0, 1 }, //
          { 2, 1, 0, -2, 1 }, //
          { -3, 0, 2, 0, 1 }, //
          { -2, 0, 2, -2, 1 }, //
          { -1, 1, 0, 2, 2 }, //
          { 0, -1, 2, -1, 2 }, //
          { -1, 0, 4, -2, 2 }, //
          { 0, -2, 2, 0, 2 }, //
          { -1, 0, 2, 1, 2 }, //
          { 2, 0, 0, 0, 2 }, //
          { 0, 0, 2, 0, 3 }, //
          { -2, 0, 4, 0, 2 }, //
          { -1, 0, -2, 0, 1 }, //
          { -1, 1, 2, 2, 1 }, //
          { 3, 0, 0, 0, 1 }, //
          { -1, 0, 2, 3, 2 }, //
          { 2, -1, 2, 0, 1 }, //
          { 0, 1, 2, 2, 1 }, //
          { 0, -1, 2, 4, 2 }, //
          { 2, -1, 2, 2, 2 }, //
          { 0, 2, -2, 2, 0 }, //
          { -1, -1, 2, -1, 1 }, //
          { 0, -2, 0, 0, 1 }, //
          { 1, 0, 2, -4, 2 }, //
          { 1, -1, 0, -2, 1 }, //
          { -1, -1, 2, 0, 1 }, //
          { 1, -1, 2, -2, 2 }, //
          { -2, -1, 0, 4, 0 }, //
          { -1, 0, 0, 3, 0 }, //
          { -2, -1, 2, 2, 2 }, //
          { 0, 2, 2, 0, 2 }, //
          { 1, 1, 0, 2, 0 }, //
          { 2, 0, 2, -1, 2 }, //
          { 1, 0, 2, 1, 1 }, //
          { 4, 0, 0, 0, 0 }, //
          { 2, 1, 2, 0, 1 }, //
          { 3, -1, 2, 0, 2 }, //
          { -2, 2, 0, 2, 1 }, //
          { 1, 0, 2, -3, 1 }, //
          { 1, 1, 2, -4, 1 }, //
          { -1, -1, 2, -2, 1 }, //
          { 0, -1, 0, -1, 1 }, //
          { 0, -1, 0, -2, 1 }, //
          { -2, 0, 0, 0, 2 }, //
          { -2, 0, -2, 2, 0 }, //
          { -1, 0, -2, 4, 0 }, //
          { 1, -2, 0, 0, 0 }, //
          { 0, 1, 0, 1, 1 }, //
          { -1, 2, 0, 2, 0 }, //
          { 1, -1, 2, -2, 1 }, //
          { 1, 2, 2, -2, 2 }, //
          { 2, -1, 2, -2, 2 }, //
          { 1, 0, 2, -1, 1 }, //
          { 2, 1, 2, -2, 1 }, //
          { -2, 0, 0, -2, 1 }, //
          { 1, -2, 2, 0, 2 }, //
          { 0, 1, 2, 1, 1 }, //
          { 1, 0, 4, -2, 1 }, //
          { -2, 0, 4, 2, 2 }, //
          { 1, 1, 2, 1, 2 }, //
          { 1, 0, 0, 4, 0 }, //
          { 1, 0, 2, 2, 0 }, //
          { 2, 0, 2, 1, 2 }, //
          { 3, 1, 2, 0, 2 }, //
          { 4, 0, 2, 0, 1 }, //
          { -2, -1, 2, 0, 0 }, //
          { 0, 1, -2, 2, 1 }, //
          { 1, 0, -2, 1, 0 }, //
          { 2, -1, 0, -2, 1 }, //
          { -1, 0, 2, -1, 2 }, //
          { 1, 0, 2, -3, 2 }, //
          { 0, 1, 2, -2, 3 }, //
          { -1, 0, -2, 2, 1 }, //
          { 0, 0, 2, -4, 2 }, //
          { 2, 0, 2, -4, 2 }, //
          { 0, 0, 4, -4, 4 }, //
          { 0, 0, 4, -4, 2 }, //
          { -2, 0, 0, 3, 0 }, //
          { 1, 0, -2, 2, 1 }, //
          { -3, 0, 2, 2, 2 }, //
          { -2, 0, 2, 2, 0 }, //
          { 2, -1, 0, 0, 1 }, //
          { 1, 1, 0, 1, 0 }, //
          { 0, 1, 4, -2, 2 }, //
          { -1, 1, 0, -2, 1 }, //
          { 0, 0, 0, -4, 1 }, //
          { 1, -1, 0, 2, 1 }, //
          { 1, 1, 0, 2, 1 }, //
          { -1, 2, 2, 2, 2 }, //
          { 3, 1, 2, -2, 2 }, //
          { 0, -1, 0, 4, 0 }, //
          { 2, -1, 0, 2, 0 }, //
          { 0, 0, 4, 0, 1 }, //
          { 2, 0, 4, -2, 2 }, //
          { -1, -1, 2, 4, 1 }, //
          { 1, 0, 0, 4, 1 }, //
          { 1, -2, 2, 2, 2 }, //
          { 0, 0, 2, 3, 2 }, //
          { -1, 1, 2, 4, 2 }, //
          { 3, 0, 0, 2, 0 }, //
          { -1, 0, 4, 2, 2 }, //
          { -2, 0, 2, 6, 2 }, //
          { -1, 0, 2, 6, 2 }, //
          { 1, 1, -2, 1, 0 }, //
          { -1, 0, 0, 1, 2 }, //
          { -1, -1, 0, 1, 0 }, //
          { -2, 0, 0, 1, 0 }, //
          { 0, 0, -2, 1, 0 }, //
          { 1, -1, -2, 2, 0 }, //
          { 1, 2, 0, 0, 0 }, //
          { 3, 0, 2, 0, 0 }, //
          { 0, -1, 1, -1, 1 }, //
          { -1, 0, 1, 0, 3 }, //
          { -1, 0, 1, 0, 2 }, //
          { -1, 0, 1, 0, 1 }, //
          { -1, 0, 1, 0, 0 }, //
          { 0, 0, 1, 0, 2 }, //
          { 0, 0, 1, 0, 1 }, //
          { 0, 0, 1, 0, 0 } };

  // Luni-Solar nutation coefficients, unit 1e-7 arcsec:
  // longitude (sin, t*sin, cos), obliquity (cos, t*cos, sin)

  // Each row of coefficients in 'cls_t' belongs with the corresponding
  // row of fundamental-argument multipliers in 'nals_t'.
  static const int32_t cls_t[323][6] = { //
          { -172064161, -174666, 33386, 92052331, 9086, 15377 }, //
          { -13170906, -1675, -13696, 5730336, -3015, -4587 }, //
          { -2276413, -234, 2796, 978459, -485, 1374 }, //
          { 2074554, 207, -698, -897492, 470, -291 }, //
          { 1475877, -3633, 11817, 73871, -184, -1924 }, //
          { -516821, 1226, -524, 224386, -677, -174 }, //
          { 711159, 73, -872, -6750, 0, 358 }, //
          { -387298, -367, 380, 200728, 18, 318 }, //
          { -301461, -36, 816, 129025, -63, 367 }, //
          { 215829, -494, 111, -95929, 299, 132 }, //
          { 128227, 137, 181, -68982, -9, 39 }, //
          { 123457, 11, 19, -53311, 32, -4 }, //
          { 156994, 10, -168, -1235, 0, 82 }, //
          { 63110, 63, 27, -33228, 0, -9 }, //
          { -57976, -63, -189, 31429, 0, -75 }, //
          { -59641, -11, 149, 25543, -11, 66 }, //
          { -51613, -42, 129, 26366, 0, 78 }, //
          { 45893, 50, 31, -24236, -10, 20 }, //
          { 63384, 11, -150, -1220, 0, 29 }, //
          { -38571, -1, 158, 16452, -11, 68 }, //
          { 32481, 0, 0, -13870, 0, 0 }, //
          { -47722, 0, -18, 477, 0, -25 }, //
          { -31046, -1, 131, 13238, -11, 59 }, //
          { 28593, 0, -1, -12338, 10, -3 }, //
          { 20441, 21, 10, -10758, 0, -3 }, //
          { 29243, 0, -74, -609, 0, 13 }, //
          { 25887, 0, -66, -550, 0, 11 }, //
          { -14053, -25, 79, 8551, -2, -45 }, //
          { 15164, 10, 11, -8001, 0, -1 }, //
          { -15794, 72, -16, 6850, -42, -5 }, //
          { 21783, 0, 13, -167, 0, 13 }, //
          { -12873, -10, -37, 6953, 0, -14 }, //
          { -12654, 11, 63, 6415, 0, 26 }, //
          { -10204, 0, 25, 5222, 0, 15 }, //
          { 16707, -85, -10, 168, -1, 10 }, //
          { -7691, 0, 44, 3268, 0, 19 }, //
          { -11024, 0, -14, 104, 0, 2 }, //
          { 7566, -21, -11, -3250, 0, -5 }, //
          { -6637, -11, 25, 3353, 0, 14 }, //
          { -7141, 21, 8, 3070, 0, 4 }, //
          { -6302, -11, 2, 3272, 0, 4 }, //
          { 5800, 10, 2, -3045, 0, -1 }, //
          { 6443, 0, -7, -2768, 0, -4 }, //
          { -5774, -11, -15, 3041, 0, -5 }, //
          { -5350, 0, 21, 2695, 0, 12 }, //
          { -4752, -11, -3, 2719, 0, -3 }, //
          { -4940, -11, -21, 2720, 0, -9 }, //
          { 7350, 0, -8, -51, 0, 4 }, //
          { 4065, 0, 6, -2206, 0, 1 }, //
          { 6579, 0, -24, -199, 0, 2 }, //
          { 3579, 0, 5, -1900, 0, 1 }, //
          { 4725, 0, -6, -41, 0, 3 }, //
          { -3075, 0, -2, 1313, 0, -1 }, //
          { -2904, 0, 15, 1233, 0, 7 }, //
          { 4348, 0, -10, -81, 0, 2 }, //
          { -2878, 0, 8, 1232, 0, 4 }, //
          { -4230, 0, 5, -20, 0, -2 }, //
          { -2819, 0, 7, 1207, 0, 3 }, //
          { -4056, 0, 5, 40, 0, -2 }, //
          { -2647, 0, 11, 1129, 0, 5 }, //
          { -2294, 0, -10, 1266, 0, -4 }, //
          { 2481, 0, -7, -1062, 0, -3 }, //
          { 2179, 0, -2, -1129, 0, -2 }, //
          { 3276, 0, 1, -9, 0, 0 }, //
          { -3389, 0, 5, 35, 0, -2 }, //
          { 3339, 0, -13, -107, 0, 1 }, //
          { -1987, 0, -6, 1073, 0, -2 }, //
          { -1981, 0, 0, 854, 0, 0 }, //
          { 4026, 0, -353, -553, 0, -139 }, //
          { 1660, 0, -5, -710, 0, -2 }, //
          { -1521, 0, 9, 647, 0, 4 }, //
          { 1314, 0, 0, -700, 0, 0 }, //
          { -1283, 0, 0, 672, 0, 0 }, //
          { -1331, 0, 8, 663, 0, 4 }, //
          { 1383, 0, -2, -594, 0, -2 }, //
          { 1405, 0, 4, -610, 0, 2 }, //
          { 1290, 0, 0, -556, 0, 0 }, //
          { -1214, 0, 5, 518, 0, 2 }, //
          { 1146, 0, -3, -490, 0, -1 }, //
          { 1019, 0, -1, -527, 0, -1 }, //
          { -1100, 0, 9, 465, 0, 4 }, //
          { -970, 0, 2, 496, 0, 1 }, //
          { 1575, 0, -6, -50, 0, 0 }, //
          { 934, 0, -3, -399, 0, -1 }, //
          { 922, 0, -1, -395, 0, -1 }, //
          { 815, 0, -1, -422, 0, -1 }, //
          { 834, 0, 2, -440, 0, 1 }, //
          { 1248, 0, 0, -170, 0, 1 }, //
          { 1338, 0, -5, -39, 0, 0 }, //
          { 716, 0, -2, -389, 0, -1 }, //
          { 1282, 0, -3, -23, 0, 1 }, //
          { 742, 0, 1, -391, 0, 0 }, //
          { 1020, 0, -25, -495, 0, -10 }, //
          { 715, 0, -4, -326, 0, 2 }, //
          { -666, 0, -3, 369, 0, -1 }, //
          { -667, 0, 1, 346, 0, 1 }, //
          { -704, 0, 0, 304, 0, 0 }, //
          { -694, 0, 5, 294, 0, 2 }, //
          { -1014, 0, -1, 4, 0, -1 }, //
          { -585, 0, -2, 316, 0, -1 }, //
          { -949, 0, 1, 8, 0, -1 }, //
          { -595, 0, 0, 258, 0, 0 }, //
          { 528, 0, 0, -279, 0, 0 }, //
          { -590, 0, 4, 252, 0, 2 }, //
          { 570, 0, -2, -244, 0, -1 }, //
          { -502, 0, 3, 250, 0, 2 }, //
          { -875, 0, 1, 29, 0, 0 }, //
          { -492, 0, -3, 275, 0, -1 }, //
          { 535, 0, -2, -228, 0, -1 }, //
          { -467, 0, 1, 240, 0, 1 }, //
          { 591, 0, 0, -253, 0, 0 }, //
          { -453, 0, -1, 244, 0, -1 }, //
          { 766, 0, 1, 9, 0, 0 }, //
          { -446, 0, 2, 225, 0, 1 }, //
          { -488, 0, 2, 207, 0, 1 }, //
          { -468, 0, 0, 201, 0, 0 }, //
          { -421, 0, 1, 216, 0, 1 }, //
          { 463, 0, 0, -200, 0, 0 }, //
          { -673, 0, 2, 14, 0, 0 }, //
          { 658, 0, 0, -2, 0, 0 }, //
          { -438, 0, 0, 188, 0, 0 }, //
          { -390, 0, 0, 205, 0, 0 }, //
          { 639, -11, -2, -19, 0, 0 }, //
          { 412, 0, -2, -176, 0, -1 }, //
          { -361, 0, 0, 189, 0, 0 }, //
          { 360, 0, -1, -185, 0, -1 }, //
          { 588, 0, -3, -24, 0, 0 }, //
          { -578, 0, 1, 5, 0, 0 }, //
          { -396, 0, 0, 171, 0, 0 }, //
          { 565, 0, -1, -6, 0, 0 }, //
          { -335, 0, -1, 184, 0, -1 }, //
          { 357, 0, 1, -154, 0, 0 }, //
          { 321, 0, 1, -174, 0, 0 }, //
          { -301, 0, -1, 162, 0, 0 }, //
          { -334, 0, 0, 144, 0, 0 }, //
          { 493, 0, -2, -15, 0, 0 }, //
          { 494, 0, -2, -19, 0, 0 }, //
          { 337, 0, -1, -143, 0, -1 }, //
          { 280, 0, -1, -144, 0, 0 }, //
          { 309, 0, 1, -134, 0, 0 }, //
          { -263, 0, 2, 131, 0, 1 }, //
          { 253, 0, 1, -138, 0, 0 }, //
          { 245, 0, 0, -128, 0, 0 }, //
          { 416, 0, -2, -17, 0, 0 }, //
          { -229, 0, 0, 128, 0, 0 }, //
          { 231, 0, 0, -120, 0, 0 }, //
          { -259, 0, 2, 109, 0, 1 }, //
          { 375, 0, -1, -8, 0, 0 }, //
          { 252, 0, 0, -108, 0, 0 }, //
          { -245, 0, 1, 104, 0, 0 }, //
          { 243, 0, -1, -104, 0, 0 }, //
          { 208, 0, 1, -112, 0, 0 }, //
          { 199, 0, 0, -102, 0, 0 }, //
          { -208, 0, 1, 105, 0, 0 }, //
          { 335, 0, -2, -14, 0, 0 }, //
          { -325, 0, 1, 7, 0, 0 }, //
          { -187, 0, 0, 96, 0, 0 }, //
          { 197, 0, -1, -100, 0, 0 }, //
          { -192, 0, 2, 94, 0, 1 }, //
          { -188, 0, 0, 83, 0, 0 }, //
          { 276, 0, 0, -2, 0, 0 }, //
          { -286, 0, 1, 6, 0, 0 }, //
          { 186, 0, -1, -79, 0, 0 }, //
          { -219, 0, 0, 43, 0, 0 }, //
          { 276, 0, 0, 2, 0, 0 }, //
          { -153, 0, -1, 84, 0, 0 }, //
          { -156, 0, 0, 81, 0, 0 }, //
          { -154, 0, 1, 78, 0, 0 }, //
          { -174, 0, 1, 75, 0, 0 }, //
          { -163, 0, 2, 69, 0, 1 }, //
          { -228, 0, 0, 1, 0, 0 }, //
          { 91, 0, -4, -54, 0, -2 }, //
          { 175, 0, 0, -75, 0, 0 }, //
          { -159, 0, 0, 69, 0, 0 }, //
          { 141, 0, 0, -72, 0, 0 }, //
          { 147, 0, 0, -75, 0, 0 }, //
          { -132, 0, 0, 69, 0, 0 }, //
          { 159, 0, -28, -54, 0, 11 }, //
          { 213, 0, 0, -4, 0, 0 }, //
          { 123, 0, 0, -64, 0, 0 }, //
          { -118, 0, -1, 66, 0, 0 }, //
          { 144, 0, -1, -61, 0, 0 }, //
          { -121, 0, 1, 60, 0, 0 }, //
          { -134, 0, 1, 56, 0, 1 }, //
          { -105, 0, 0, 57, 0, 0 }, //
          { -102, 0, 0, 56, 0, 0 }, //
          { 120, 0, 0, -52, 0, 0 }, //
          { 101, 0, 0, -54, 0, 0 }, //
          { -113, 0, 0, 59, 0, 0 }, //
          { -106, 0, 0, 61, 0, 0 }, //
          { -129, 0, 1, 55, 0, 0 }, //
          { -114, 0, 0, 57, 0, 0 }, //
          { 113, 0, -1, -49, 0, 0 }, //
          { -102, 0, 0, 44, 0, 0 }, //
          { -94, 0, 0, 51, 0, 0 }, //
          { -100, 0, -1, 56, 0, 0 }, //
          { 87, 0, 0, -47, 0, 0 }, //
          { 161, 0, 0, -1, 0, 0 }, //
          { 96, 0, 0, -50, 0, 0 }, //
          { 151, 0, -1, -5, 0, 0 }, //
          { -104, 0, 0, 44, 0, 0 }, //
          { -110, 0, 0, 48, 0, 0 }, //
          { -100, 0, 1, 50, 0, 0 }, //
          { 92, 0, -5, 12, 0, -2 }, //
          { 82, 0, 0, -45, 0, 0 }, //
          { 82, 0, 0, -45, 0, 0 }, //
          { -78, 0, 0, 41, 0, 0 }, //
          { -77, 0, 0, 43, 0, 0 }, //
          { 2, 0, 0, 54, 0, 0 }, //
          { 94, 0, 0, -40, 0, 0 }, //
          { -93, 0, 0, 40, 0, 0 }, //
          { -83, 0, 10, 40, 0, -2 }, //
          { 83, 0, 0, -36, 0, 0 }, //
          { -91, 0, 0, 39, 0, 0 }, //
          { 128, 0, 0, -1, 0, 0 }, //
          { -79, 0, 0, 34, 0, 0 }, //
          { -83, 0, 0, 47, 0, 0 }, //
          { 84, 0, 0, -44, 0, 0 }, //
          { 83, 0, 0, -43, 0, 0 }, //
          { 91, 0, 0, -39, 0, 0 }, //
          { -77, 0, 0, 39, 0, 0 }, //
          { 84, 0, 0, -43, 0, 0 }, //
          { -92, 0, 1, 39, 0, 0 }, //
          { -92, 0, 1, 39, 0, 0 }, //
          { -94, 0, 0, 0, 0, 0 }, //
          { 68, 0, 0, -36, 0, 0 }, //
          { -61, 0, 0, 32, 0, 0 }, //
          { 71, 0, 0, -31, 0, 0 }, //
          { 62, 0, 0, -34, 0, 0 }, //
          { -63, 0, 0, 33, 0, 0 }, //
          { -73, 0, 0, 32, 0, 0 }, //
          { 115, 0, 0, -2, 0, 0 }, //
          { -103, 0, 0, 2, 0, 0 }, //
          { 63, 0, 0, -28, 0, 0 }, //
          { 74, 0, 0, -32, 0, 0 }, //
          { -103, 0, -3, 3, 0, -1 }, //
          { -69, 0, 0, 30, 0, 0 }, //
          { 57, 0, 0, -29, 0, 0 }, //
          { 94, 0, 0, -4, 0, 0 }, //
          { 64, 0, 0, -33, 0, 0 }, //
          { -63, 0, 0, 26, 0, 0 }, //
          { -38, 0, 0, 20, 0, 0 }, //
          { -43, 0, 0, 24, 0, 0 }, //
          { -45, 0, 0, 23, 0, 0 }, //
          { 47, 0, 0, -24, 0, 0 }, //
          { -48, 0, 0, 25, 0, 0 }, //
          { 45, 0, 0, -26, 0, 0 }, //
          { 56, 0, 0, -25, 0, 0 }, //
          { 88, 0, 0, 2, 0, 0 }, //
          { -75, 0, 0, 0, 0, 0 }, //
          { 85, 0, 0, 0, 0, 0 }, //
          { 49, 0, 0, -26, 0, 0 }, //
          { -74, 0, -3, -1, 0, -1 }, //
          { -39, 0, 0, 21, 0, 0 }, //
          { 45, 0, 0, -20, 0, 0 }, //
          { 51, 0, 0, -22, 0, 0 }, //
          { -40, 0, 0, 21, 0, 0 }, //
          { 41, 0, 0, -21, 0, 0 }, //
          { -42, 0, 0, 24, 0, 0 }, //
          { -51, 0, 0, 22, 0, 0 }, //
          { -42, 0, 0, 22, 0, 0 }, //
          { 39, 0, 0, -21, 0, 0 }, //
          { 46, 0, 0, -18, 0, 0 }, //
          { -53, 0, 0, 22, 0, 0 }, //
          { 82, 0, 0, -4, 0, 0 }, //
          { 81, 0, -1, -4, 0, 0 }, //
          { 47, 0, 0, -19, 0, 0 }, //
          { 53, 0, 0, -23, 0, 0 }, //
          { -45, 0, 0, 22, 0, 0 }, //
          { -44, 0, 0, -2, 0, 0 }, //
          { -33, 0, 0, 16, 0, 0 }, //
          { -61, 0, 0, 1, 0, 0 }, //
          { -38, 0, 0, 19, 0, 0 }, //
          { -33, 0, 0, 21, 0, 0 }, //
          { -60, 0, 0, 0, 0, 0 }, //
          { 48, 0, 0, -10, 0, 0 }, //
          { 38, 0, 0, -20, 0, 0 }, //
          { 31, 0, 0, -13, 0, 0 }, //
          { -32, 0, 0, 15, 0, 0 }, //
          { 45, 0, 0, -8, 0, 0 }, //
          { -44, 0, 0, 19, 0, 0 }, //
          { -51, 0, 0, 0, 0, 0 }, //
          { -36, 0, 0, 20, 0, 0 }, //
          { 44, 0, 0, -19, 0, 0 }, //
          { -60, 0, 0, 2, 0, 0 }, //
          { 35, 0, 0, -18, 0, 0 }, //
          { 47, 0, 0, -1, 0, 0 }, //
          { 36, 0, 0, -15, 0, 0 }, //
          { -36, 0, 0, 20, 0, 0 }, //
          { -35, 0, 0, 19, 0, 0 }, //
          { -37, 0, 0, 19, 0, 0 }, //
          { 32, 0, 0, -16, 0, 0 }, //
          { 35, 0, 0, -14, 0, 0 }, //
          { 32, 0, 0, -13, 0, 0 }, //
          { 65, 0, 0, -2, 0, 0 }, //
          { 47, 0, 0, -1, 0, 0 }, //
          { 32, 0, 0, -16, 0, 0 }, //
          { 37, 0, 0, -16, 0, 0 }, //
          { -30, 0, 0, 15, 0, 0 }, //
          { -32, 0, 0, 16, 0, 0 }, //
          { -31, 0, 0, 13, 0, 0 }, //
          { 37, 0, 0, -16, 0, 0 }, //
          { 31, 0, 0, -13, 0, 0 }, //
          { 49, 0, 0, -2, 0, 0 }, //
          { 32, 0, 0, -13, 0, 0 }, //
          { -43, 0, 0, 18, 0, 0 }, //
          { -32, 0, 0, 14, 0, 0 }, //
          { 30, 0, 0, 0, 0, 0 }, //
          { -34, 0, 0, 15, 0, 0 }, //
          { -36, 0, 0, 0, 0, 0 }, //
          { -38, 0, 0, 0, 0, 0 }, //
          { -31, 0, 0, 0, 0, 0 }, //
          { -34, 0, 0, 0, 0, 0 }, //
          { -35, 0, 0, 0, 0, 0 }, //
          { 30, 0, 0, -2, 0, 0 }, //
          { 0, 0, -1988, 0, 0, -1679 }, //
          { 0, 0, -63, 0, 0, -27 }, //
          { 0, 0, 364, 0, 0, 176 }, //
          { 0, 0, -1044, 0, 0, -891 }, //
          { 0, 0, 330, 0, 0, 0 }, //
          { 0, 0, 30, 0, 0, 14 }, //
          { 0, 0, -162, 0, 0, -138 }, //
          { 0, 0, 75, 0, 0, 0 } };

  // Planetary argument multipliers:
  // L   L'  F   D   Om  Me  Ve  E  Ma  Ju  Sa  Ur  Ne  pre
  static const int8_t napl_t[165][14] = { //
          { 0, 0, 0, 0, 0, 0, 0, 8, -16, 4, 5, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -8, 16, -4, -5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 8, -16, 4, 5, 0, 0, 2 }, //
          { 0, 0, 1, -1, 1, 0, 0, 3, -8, 3, 0, 0, 0, 0 }, //
          { -1, 0, 0, 0, 0, 0, 10, -3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, -8, 3, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -5, 8, -3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, -5, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 2, -5, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, -5, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, -2, 5, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, -2, 5, 0, 0, 1 }, //
          { 1, 0, 0, -2, 0, 0, 19, -21, 3, 0, 0, 0, 0, 0 }, //
          { 1, 0, 0, -1, 1, 0, 0, -1, 0, 2, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 1, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { -1, 0, 0, 0, 0, 0, 18, -16, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -8, 13, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -8, 13, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, -8, 12, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 8, -13, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, -1, 2, 0, 0, 0, 0, 0 }, //
          { -1, 0, 0, 1, 0, 0, 3, -4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0 }, //
          { 0, 0, -2, 2, 0, 0, 5, -6, 0, 0, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 6, -8, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 8, -15, 0, 0, 0, 0, 0 }, //
          { 2, 0, 0, -2, 1, 0, 0, -2, 0, 3, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 0, 2, 0, -3, 0, 0, 0, 0 }, //
          { -1, 0, 0, 1, 0, 0, 0, 1, 0, -1, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 0, -1, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 0, 1, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1 }, //
          { 0, 0, 0, 0, 1, 0, 8, -13, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 0, 0, 1, 0, 18, -16, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 0, 0, -2, 5, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, -4, 8, -3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 4, -8, 3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 0, 0, 2, -5, 0, 0, 0 }, //
          { -2, 0, 0, 2, 0, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { 1, 0, 0, 0, 1, 0, -18, 16, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, -8, 13, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, 0, -2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, -2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -2, 2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 2, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 0, 2, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 2 }, //
          { 0, 0, 2, -2, 1, 0, -5, 6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, -1, 1, 0, 0, 5, -7, 0, 0, 0, 0, 0, 0 }, //
          { -2, 0, 0, 2, 1, 0, 0, 2, 0, -3, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, -1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 2 }, //
          { -2, 0, 0, 2, 0, 0, 3, -3, 0, 0, 0, 0, 0, 0 }, //
          { 2, 0, 0, -2, 1, 0, 0, -2, 0, 2, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 0, 1, -2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 3, -5, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 1, -1, 1, 0, -3, 4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -3, 5, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, -3, 5, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -3, 5, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, -4, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, -2, 4, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -5, 8, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -5, 8, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, -5, 7, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -5, 8, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 5, -8, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 1 }, //
          { 0, 0, 1, -1, 1, 0, 0, -1, 0, 2, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 2 }, //
          { 0, 0, 2, -2, 1, 0, -3, 3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, -6, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 1, 0, 2, -3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 2, -2, 1, 0, 0, -2, 0, 2, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -2, 3, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 2, -3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, -5, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 1, -2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, -3, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -4, 7, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -4, 6, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -4, 6, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 4, -6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -1, 1, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 1, -1, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 1, -1, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 0, 4, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -1, 0, 3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, -3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -2, 4, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, -2, 4, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -6, 9, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, -2, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, -4, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 3, -4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, -1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, -2, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, -1, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, -5, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -5, 7, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -5, 7, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -2, 2, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 2, -2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -1, 3, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, -1, 3, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, -3, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, -2, 5, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -6, 8, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, -4, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -3, 3, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 3, -3, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 3, -3, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, -1, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, -1, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, -2, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -8, 15, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 6, -8, 3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 1 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -6, 16, -4, -5, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, -2, 8, -3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -8, 11, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 2, 0, 1, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -3, 7, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 2, -1, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -7, 9, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 4, -4, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, 0, -2, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 5, -4, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 3, -2, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 3, 0, -1, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, -2, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -8, 10, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 5, -5, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, -9, 11, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, 0, -3, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 6, -6, 0, 0, 0, 0, 0, 0 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, 0, -2, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 3, -1, 0, 0, 0, 0, 0, 2 }, //
          { 0, 0, 0, 0, 0, 0, 0, 4, 0, -1, 0, 0, 0, 2 }, //
          { -1, 0, 0, 2, 0, 0, 0, 2, 0, -2, 0, 0, 0, 0 }, //
          { 1, 0, 2, 0, 2, 0, 0, 1, 0, 0, 0, 0, 0, 0 }, //
          { -1, 0, 2, 0, 2, 0, 0, -4, 8, -3, 0, 0, 0, 0 } };

  // Planetary nutation coefficients, unit 1e-7 arcsec:
  // longitude (sin, cos), obliquity (sin, cos)

  // Each row of coefficients in 'cpl_t' belongs with the corresponding
  // row of fundamental-argument multipliers in 'napl_t'.
  static const int16_t cpl_t[165][4] = { //
          { 1440, 0, 0, 0 }, //
          { 56, -117, -42, -40 }, //
          { 125, -43, 0, -54 }, //
          { -114, 0, 0, 61 }, //
          { -219, 89, 0, 0 }, //
          { -462, 1604, 0, 0 }, //
          { 99, 0, 0, -53 }, //
          { 14, -218, 117, 8 }, //
          { 31, -481, -257, -17 }, //
          { -491, 128, 0, 0 }, //
          { -3084, 5123, 2735, 1647 }, //
          { -1444, 2409, -1286, -771 }, //
          { 103, -60, 0, 0 }, //
          { -26, -29, -16, 14 }, //
          { 284, 0, 0, -151 }, //
          { 226, 101, 0, 0 }, //
          { -41, 175, 76, 17 }, //
          { 425, 212, -133, 269 }, //
          { 1200, 598, 319, -641 }, //
          { 235, 334, 0, 0 }, //
          { 266, -78, 0, 0 }, //
          { -460, -435, -232, 246 }, //
          { 0, 131, 0, 0 }, //
          { -42, 20, 0, 0 }, //
          { -10, 233, 0, 0 }, //
          { 78, -18, 0, 0 }, //
          { 45, -22, 0, 0 }, //
          { 89, -16, -9, -48 }, //
          { -349, -62, 0, 0 }, //
          { -53, 0, 0, 0 }, //
          { -21, -78, 0, 0 }, //
          { 20, -70, -37, -11 }, //
          { 32, 15, -8, 17 }, //
          { 174, 84, 45, -93 }, //
          { 11, 56, 0, 0 }, //
          { -66, -12, -6, 35 }, //
          { 47, 8, 4, -25 }, //
          { 46, 66, 35, -25 }, //
          { -68, -34, -18, 36 }, //
          { 76, 17, 9, -41 }, //
          { 84, 298, 159, -45 }, //
          { -82, 292, 156, 44 }, //
          { -73, 17, 9, 39 }, //
          { -439, 0, 0, 0 }, //
          { 57, -28, -15, -30 }, //
          { -40, 57, 30, 21 }, //
          { 273, 80, 43, -146 }, //
          { -449, 430, 0, 0 }, //
          { -8, -47, -25, 4 }, //
          { 6, 47, 25, -3 }, //
          { -48, -110, -59, 26 }, //
          { 51, 114, 61, -27 }, //
          { -133, 0, 0, 57 }, //
          { -18, -436, -233, 9 }, //
          { 35, -7, 0, 0 }, //
          { -53, -9, -5, 28 }, //
          { -50, 194, 103, 27 }, //
          { -13, 52, 28, 7 }, //
          { -91, 248, 0, 0 }, //
          { 6, 49, 26, -3 }, //
          { -6, -47, -25, 3 }, //
          { 52, 23, 10, -23 }, //
          { -138, 0, 0, 0 }, //
          { 54, 0, 0, -29 }, //
          { -37, 35, 19, 20 }, //
          { -145, 47, 0, 0 }, //
          { -10, 40, 21, 5 }, //
          { 11, -49, -26, -7 }, //
          { -2150, 0, 0, 932 }, //
          { 85, 0, 0, -37 }, //
          { -86, 153, 0, 0 }, //
          { -51, 0, 0, 22 }, //
          { -11, -268, -116, 5 }, //
          { 31, 6, 3, -17 }, //
          { 140, 27, 14, -75 }, //
          { 57, 11, 6, -30 }, //
          { -14, -39, 0, 0 }, //
          { -25, 22, 0, 0 }, //
          { 42, 223, 119, -22 }, //
          { -27, -143, -77, 14 }, //
          { 9, 49, 26, -5 }, //
          { -1166, 0, 0, 505 }, //
          { 117, 0, 0, -63 }, //
          { 0, 31, 0, 0 }, //
          { 0, -32, -17, 0 }, //
          { 50, 0, 0, -27 }, //
          { 30, -3, -2, -16 }, //
          { 8, 614, 0, 0 }, //
          { -127, 21, 9, 55 }, //
          { -20, 34, 0, 0 }, //
          { 22, -87, 0, 0 }, //
          { -68, 39, 0, 0 }, //
          { 3, 66, 29, -1 }, //
          { 490, 0, 0, -213 }, //
          { -22, 93, 49, 12 }, //
          { -46, 14, 0, 0 }, //
          { 25, 106, 57, -13 }, //
          { 1485, 0, 0, 0 }, //
          { -7, -32, -17, 4 }, //
          { 30, -6, -2, -13 }, //
          { 118, 0, 0, -52 }, //
          { -28, 36, 0, 0 }, //
          { 14, -59, -31, -8 }, //
          { -458, 0, 0, 198 }, //
          { 0, -45, -20, 0 }, //
          { -166, 269, 0, 0 }, //
          { -78, 45, 0, 0 }, //
          { -5, 328, 0, 0 }, //
          { -1223, -26, 0, 0 }, //
          { -368, 0, 0, 0 }, //
          { -75, 0, 0, 0 }, //
          { -13, -30, 0, 0 }, //
          { -74, 0, 0, 32 }, //
          { -262, 0, 0, 114 }, //
          { 202, 0, 0, -87 }, //
          { -8, 35, 19, 5 }, //
          { -35, -48, -21, 15 }, //
          { 12, 55, 29, -6 }, //
          { -598, 0, 0, 0 }, //
          { 8, -31, -16, -4 }, //
          { 113, 0, 0, -49 }, //
          { 83, 15, 0, 0 }, //
          { 0, -114, -49, 0 }, //
          { 117, 0, 0, -51 }, //
          { 393, 3, 0, 0 }, //
          { 18, -29, -13, -8 }, //
          { 8, 34, 18, -4 }, //
          { 89, 0, 0, 0 }, //
          { 54, -15, -7, -24 }, //
          { 0, 35, 0, 0 }, //
          { -154, -30, -13, 67 }, //
          { 80, -71, -31, -35 }, //
          { 61, -96, -42, -27 }, //
          { 123, -415, -180, -53 }, //
          { 0, 0, 0, -35 }, //
          { 7, -32, -17, -4 }, //
          { -89, 0, 0, 38 }, //
          { 0, -86, -19, -6 }, //
          { -123, -416, -180, 53 }, //
          { -62, -97, -42, 27 }, //
          { -85, -70, -31, 37 }, //
          { 163, -12, -5, -72 }, //
          { -63, -16, -7, 28 }, //
          { -21, -32, -14, 9 }, //
          { 5, -173, -75, -2 }, //
          { 74, 0, 0, -32 }, //
          { 83, 0, 0, 0 }, //
          { -339, 0, 0, 147 }, //
          { 67, -91, -39, -29 }, //
          { 30, -18, -8, -13 }, //
          { 0, -114, -50, 0 }, //
          { 517, 16, 7, -224 }, //
          { 143, -3, -1, -62 }, //
          { 50, 0, 0, -22 }, //
          { 59, 0, 0, 0 }, //
          { 370, -8, 0, -160 }, //
          { 34, 0, 0, -15 }, //
          { -37, -7, -3, 16 }, //
          { 40, 0, 0, 0 }, //
          { -184, -3, -1, 80 }, //
          { 31, -6, 0, -13 }, //
          { -3, -32, -14, 1 }, //
          { -34, 0, 0, 0 }, //
          { 126, -63, -27, -55 }, //
          { -126, -63, -27, 55 } };

    // @formatter:on

  // Interval between fundamental epoch J2000.0 and given date.
  const double t = ((jd_tt_high - T0) + jd_tt_low) / JULIAN_CENTURY_DAYS;

  // Planetary longitudes, Mercury through Neptune, wrt mean dynamical
  // ecliptic and equinox of J2000, with high order terms omitted
  // (Simon et al. 1994, 5.8.1-5.8.8).
  //const double alme = planet_lon(t, NOVAS_MERCURY);
  const double alve = planet_lon(t, NOVAS_VENUS);
  const double alea = planet_lon(t, NOVAS_EARTH);
  const double alma = planet_lon(t, NOVAS_MARS);
  const double alju = planet_lon(t, NOVAS_JUPITER);
  const double alsa = planet_lon(t, NOVAS_SATURN);
  //const double alur = planet_lon(t, NOVAS_URANUS);
  //const double alne = planet_lon(t, NOVAS_NEPTUNE);

  // General precession in longitude (Simon et al. 1994), equivalent
  // to 5028.8200 arcsec/cy at J2000.
  const double apa = accum_prec(t);

  novas_delaunay_args a;

  double dpsils = 0.0, depsls = 0.0, dpsipl = 0.0, depspl = 0.0;

  int i;

  // Compute fundamental arguments from Simon et al. (1994), in radians.
  fund_args(t, &a);

  // ** Luni-solar nutation. **
  // Summation of luni-solar nutation series (in reverse order).
  for(i = 323; --i >= 0;) {
    const int8_t *n = &nals_t[i][0];
    const int32_t *c = &cls_t[i][0];

    double arg = 0.0, sarg, carg;

    if(n[0])
      arg += n[0] * a.l;
    if(n[1])
      arg += n[1] * a.l1;
    if(n[2])
      arg += n[2] * a.F;
    if(n[3])
      arg += n[3] * a.D;
    if(n[4])
      arg += n[4] * a.Omega;

    sarg = sin(arg);
    carg = cos(arg);

    // Term.
    dpsils += (c[0] + c[1] * t) * sarg + c[2] * carg;
    depsls += (c[3] + c[4] * t) * carg + c[5] * sarg;
  }

  // ** Planetary nutation. **

  // Summation of planetary nutation series (in reverse order).
  for(i = 165; --i >= 0;) {
    const int8_t *n = &napl_t[i][0];
    const int16_t *c = &cpl_t[i][0];

    // Argument and functions.
    double arg = 0.0, sarg, carg;

    if(n[0])
      arg += n[0] * a.l;
    /* This version of Nutation does not contain terms for l1
    if(n[1])
      arg += n[1] * a.l1;
    */
    if(n[2])
      arg += n[2] * a.F;
    if(n[3])
      arg += n[3] * a.D;
    if(n[4])
      arg += n[4] * a.Omega;
    /* This version of Nutation does not contain terms for Mercury
    if(n[5])
      arg += n[5] * alme;
    */
    if(n[6])
      arg += n[6] * alve;
    if(n[7])
      arg += n[7] * alea;
    if(n[8])
      arg += n[8] * alma;
    if(n[9])
      arg += n[9] * alju;
    if(n[10])
      arg += n[10] * alsa;
    /* This version of Nutation does not contain terms for Uranus and Neptune
    if(n[11])
      arg += n[11] * alur;
    if(n[12])
      arg += n[12] * alne;
    */
    if(n[13])
      arg += n[13] * apa;

    sarg = sin(arg);
    carg = cos(arg);

    // Term.
    dpsipl += c[0] * sarg + c[1] * carg;
    depspl += c[2] * sarg + c[3] * carg;
  }

  // Total: Add planetary and luni-solar components.
  if(dpsi)
    *dpsi = 1e-7 * (dpsipl + dpsils) * ARCSEC;
  if(deps)
    *deps = 1e-7 * (depspl + depsls) * ARCSEC;

  return 0;
}
