package CXC::DB::DDL::Util;

use v5.26;
use strict;
use warnings;
use experimental 'signatures', 'postderef';

our $VERSION = '0.13';

use List::Util qw( pairs mesh );

use CXC::Exporter::Util ':all';
use CXC::DB::DDL::Constants -all;
my %SQL_TYPE_MAP;
use CXC::DB::DDL::Constants {
    into => \%SQL_TYPE_MAP,
    as   => sub { substr( $_, 4 ) },
  },
  -sql_type_constants;

use namespace::clean;

use base 'Exporter::Tiny';









use constant SQL_TYPE_NAMES => keys %SQL_TYPE_MAP;

# remove leading 'SQL_'
install_EXPORTS( {
    schema_funcs => [qw( xFIELDS xCHECK xTYPE  )],
    type_funcs   => [ keys %SQL_TYPE_MAP ],
    misc         => [ 'SQL_TYPE_NAMES', 'sqlt_producer_map' ],
} );

sub _exporter_expand_sub ( $class, $name, @rest ) {

    if ( defined( my $sub = $SQL_TYPE_MAP{$name} )
        && !$class->can( $name ) )
    {
        ## no critic (TestingAndDebugging::ProhibitNoStrict)
        no strict 'refs';
        my $constant = $sub->();
        *${ \"${class}::$name" }
          = sub ( %attr ) { _mk_field( $constant, \%attr ); };
    }
    $class->SUPER::_exporter_expand_sub( $name, @rest );
}

sub _mk_field ( $type, $attr ) {

    # do this here and use require to prevent import loop from
    # anything which uses CXC::DB::DDL::Field, which uses this module
    require CXC::DB::DDL::Field;

    sub ( $field ) {
        CXC::DB::DDL::Field->new( {
            name        => $field,
            data_type   => $type,
            is_nullable => 0,
            $attr->%*,
        } );
    };

}





























































sub xTYPE ( $type, %attr ) {
    _mk_field( $type, \%attr );
}


















sub xFIELDS ( @fields ) {
    return fields => [ map { $_->value->( $_->key ) } pairs( @fields ) ];
}











sub xCHECK ( $field, @values ) {
    ( check => sprintf( "$field in ( %s )", join( ', ', map { qq("$_") } @values ) ), )
}










sub sqlt_producer_map ( $dbd ) {
    state %map = ( Pg => 'PostgreSQL' );
    return $map{$dbd} // $dbd;
}


1;

#
# This file is part of CXC-DB-DDL
#
# This software is Copyright (c) 2022 by Smithsonian Astrophysical Observatory.
#
# This is free software, licensed under:
#
#   The GNU General Public License, Version 3, June 2007
#

__END__

=pod

=for :stopwords Diab Jerius Smithsonian Astrophysical Observatory TYPENAME VARCHAR xCHECK
xFIELDS xTYPE

=head1 NAME

CXC::DB::DDL::Util

=head1 VERSION

version 0.13

=head1 DESCRIPTION

C<CXC::DB::DDL::Util> provides generators to ease creation of,
amongst, others, L<CXC::DB::DDL::FIELD> objects.  It uses
L<Exporter::Tiny> as its base exporter, allowing renaming of exported
symbols and other things.

=head1 SUBROUTINES

=head2 SQL_TYPE_NAMES

  @typenames = SQL_TYPE_NAMES;

returns all of the DBI supported types (without the C<SQL_> prefix)

=head2 I<TYPENAME>

I<TYPENAME> is one of the SQL types recognized by L<DBI>.
See L<CXC::DB::DDL::Util/sql_type_constants> for more information.

Called as, e.g.

   INTEGER( %attr )

these are generators which return subroutines with the following signature:

   sub ( $field_name )

which return a L<CXC::DB::DDL::Field> object with the specified SQL
datatype (in this example C<INTEGER>), field name (C<$field_name>)
and attributes (C<%attr>).

These are available for individual export or in entirety via the
C<type_funcs> tag.

They are typically used in conjunction with the L</xFIELDS>
subroutine, e.g.

  xFIELDS(
      segment        => INTEGER,
      obsid          => INTEGER( is_primary_key => 1 ),
      target_type    => VARCHAR( is_nullable => 1 ),
  )

L</xFIELDS> essentially turns this into:

  fields => [
      INTEGER('segment'),
      INTEGER( 'obsid', is_primary_key => 1 ),
      VARCHAR( 'target_type' is_nullable => 1 ),
  ]

which is more painful to write and look at.

=head2 xTYPE

  xTYPE ( $type, %attr )

A generic form of e.g., L</INTEGER>. Type is a numeric C<DATA_TYPE> from
C<$DBI::EXPORT_TAGS{sql_types}>;

=head2 xFIELDS

   @field_spec = xFIELDS( array of  Tuple[ NonEmptyStr, CodeRef ] );

returns a list of

   fields => \@spec,

where C<@spec> generated by running

     CodeRef->(NonEmptyStr)

for each tuple.

=head2 xCHECK

    $string = xCHECK( $field, @values )

generates a check constraint as a string which looks like

  $field in ( $value[0], $value[1], ...  )

=head2 sqlt_producer_map

  $sqlt_producer = sqlt_producer_map( $dbd );

Map C<$dbd> (typically from C<$dbh->{Driver}{NAME}>) to what L<SQL::Translator> wants.
Actually, this just checks for known (to me) deviations; everything else is returned as.

=head1 SYNTAX

  use CXC::DB::DDL::Util -all;

  # import xFIELDS, xCHECK, xTYPE
  use CXC::DB::DDL::Util -schema_funcs;

  # import type function generators (e.g. INTEGER, DOUBLE )
  use CXC::DB::DDL::Util -type_funcs;

  # import type function generators (e.g. INTEGER, DOUBLE )
  use CXC::DB::DDL::Util { prefix => 'SQL_' }, -type_funcs;

  @CXC_DB_DDL_FIELD_objects = xFIELDS(
      segment        => INTEGER,
      obsid          => INTEGER( is_primary_key => 1 ),
      target_type    => VARCHAR( is_nullable => 1 ),
  );

=head1 SUPPORT

=head2 Bugs

Please report any bugs or feature requests to bug-cxc-db-ddl@rt.cpan.org  or through the web interface at: L<https://rt.cpan.org/Public/Dist/Display.html?Name=CXC-DB-DDL>

=head2 Source

Source is available at

  https://gitlab.com/djerius/cxc-db-ddl

and may be cloned from

  https://gitlab.com/djerius/cxc-db-ddl.git

=head1 SEE ALSO

Please see those modules/websites for more information related to this module.

=over 4

=item *

L<CXC::DB::DDL|CXC::DB::DDL>

=back

=head1 AUTHOR

Diab Jerius <djerius@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2022 by Smithsonian Astrophysical Observatory.

This is free software, licensed under:

  The GNU General Public License, Version 3, June 2007

=cut
