package Dancer::Core::Session;
{
    $Dancer::Core::Session::VERSION = '2.0000_01';
}

#ABSTRACT: class to represent any session object


use strict;
use warnings;
use Moo;
use Dancer::Core::Types;


has id => (
    is       => 'rw',
    isa      => Str,
    required => 1,
);


has data => (
    is      => 'rw',
    lazy    => 1,
    default => sub { {} },
);


has expires => (
    is     => 'rw',
    isa    => Str,
    coerce => sub {
        my $value = shift;
        $value += time;
    },
);


sub read {
    my ($self, $key) = @_;
    return $self->data->{$key};
}


sub write {
    my ($self, $key, $value) = @_;
    $self->data->{$key} = $value;
}


sub delete {
    my ($self, $key, $value) = @_;
    delete $self->data->{$key};
}

1;

__END__

=pod

=head1 NAME

Dancer::Core::Session - class to represent any session object

=head1 VERSION

version 2.0000_01

=head1 DESCRIPTION

A session object encapsulates anything related to a specific session: its ID,
its data, and its expiration.

It is completely agnostic of how it will be stored, this is the role of
a factory that consumes L<Dancer::Core::Role::SessionFactory> to know about that.

Generally, session objects should not be created directly.  The correct way to
get a new session object is to call the C<create()> method on a session engine
that implements the SessionFactory role.  This is done automatically by the
context object if a session engine is defined.

=head1 ATTRIBUTES

=head2 id

The identifier of the session object. Required. By default,
L<Dancer::Core::Role::SessionFactory> sets this to a randomly-generated,
guaranteed-unique string.

=head2 data

Contains the data of the session (Hash).

=head2 expires

Number of seconds for the expiry of the session cookie. Don't add the current
timestamp to it, will be done automatically.

Default is no expiry (session cookie will leave for the whole browser's
session).

For a lifetime of one hour:

  expires => 3600

=head1 METHODS

=head2 read

Reader on the session data

    my $value = $session->read('something');

Returns C<undef> if the key does not exist in the session.

=head2 write

Writer on the session data

  $session->write('something', $value);

Returns C<$value>.

=head2 delete

Deletes a key from session data

  $session->delete('something');

Returns the value deleted from the session.

=head1 AUTHOR

Dancer Core Developers

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2012 by Alexis Sukrieh.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
