use Test::More tests => 27;
## test some CWB::CL utility functions (without corpus access)

use strict;
use warnings;
use utf8;

use Encode;
use CWB::CL;


our $fs1a = CWB::CL::make_set("aaa Ba 0 BAa a", 'split'); # -- feature set manipulation
our $fs1b = CWB::CL::make_set("|0|a|BAa|Ba|aaa|");
our $fs1c = CWB::CL::make_set("0|a|BAa|Ba|aaa"); # CoNLL style accepted by CWB v3.4.29 and newer
our $fs1  = "|0|BAa|Ba|a|aaa|"; # this is the normalised form that should be generated by make_set()

our $input_latin1 = encode("latin1", "aaa äää ßa 0");
our $fs2a = CWB::CL::make_set($input_latin1, 'split');
our $fs2  = encode("latin1", "|ßa|äää|0|aaa|"); # expected result: signed char ordering for latin1

is($fs1a, $fs1, "feature set normalisation"); # T1
is($fs1b, $fs1, "feature set normalisation");
is($fs2a, $fs2, "normalisation of feature set with non-ASCII characters");
our $fs_lenient = defined $fs1c; # since we can't test for CWB version directly

SKIP: {
  skip "simple feature set notation (CoNLL) only supported by CWB v3.4.29 and newer", 4 unless $fs_lenient;
  is($fs1c, $fs1, "simple feature set notation accepted (CoNLL format)");
  is(CWB::CL::make_set(""), "|", "empty feature set from empty string");
  is(CWB::CL::make_set("_"), "|", "empty feature set from underscore");
  is(CWB::CL::make_set("_val"), "|_val|", "but features can start with underscore");
}

our $fs = CWB::CL::set_intersection($fs1, $fs2);
is($fs, "|0|aaa|", "intersection of feature sets"); # T4
diag("");
diag("- unify($fs1, $fs2) = $fs");
is(CWB::CL::set_size($fs), 2, "cardinality of feature set");

our $hash1 = CWB::CL::set2hash($fs1);
isa_ok($hash1, "HASH", "convert feature set to hash"); # T6
our $expected = { map { $_ => 1 } qw(0 BAa Ba a aaa) };
is_deeply($hash1, $expected, "convert feature set to hash");
our $fs1c = CWB::CL::make_set($hash1);
is($fs1c, $fs1, "convert hash to feature set");

$CWB::CL::Registry = "data/registry"; # string normalisation needs corpus handle (for charset)
our $C = new CWB::CL::Corpus "HOLMES-LATIN1";   # encoded in ISO-8859-1
isa_ok($C, "CWB::CL::Corpus", "corpus object for HOLMES-LATIN1 corpus") # T9
  or BAIL_OUT("failed to access test corpus HOLMES-LATIN1");
is($C->charset, "latin1", "identify character encoding of HOLMES-LATIN1 corpus");

our @orig = as_latin1(qw(Baum baum Bäume BÄÜME déjà DÉJÀ OÛ));
our @case_folded = as_latin1(qw(baum baum bäume bäüme déjà déjà oû));
our @diac_folded = as_latin1(qw(Baum baum Baume BAUME deja DEJA OU));
our @both_folded = as_latin1(qw(baum baum baume baume deja deja ou));

is_deeply([$C->normalize("c", @orig)], \@case_folded, "case-folding of strings"); # T11
is_deeply([$C->normalize("d", @orig)], \@diac_folded, "accent-folding of strings");
is_deeply([$C->normalize("cd", @orig)], \@both_folded, "case+accent-folding of strings");

our $C2 = new CWB::CL::Corpus "HOLMES-DE"; # HOLMES-DE is encoded in UTF-8
isa_ok($C2, "CWB::CL::Corpus", "corpus object for HOLMES-DE corpus") # T14
  or BAIL_OUT("failed to access test corpus HOLMES-DE");
is($C2->charset, "utf8", "identify character encoding of HOLMES-DE corpus");

@orig = as_utf8(qw(Baum baum Bäume BÄÜME déjà DÉJÀ OÛ));
@case_folded = as_utf8(qw(baum baum bäume bäüme déjà déjà oû));
@diac_folded = as_utf8(qw(Baum baum Baume BAUME deja DEJA OU));
@both_folded = as_utf8(qw(baum baum baume baume deja deja ou));

is_deeply([$C2->normalize("c", @orig)], \@case_folded, "case-folding of strings"); # T16
is_deeply([$C2->normalize("d", @orig)], \@diac_folded, "accent-folding of strings");
is_deeply([$C2->normalize("cd", @orig)], \@both_folded, "case+accent-folding of strings");

my @p_att = qw(word pos lemma no_data);
my @s_att = qw(story story_num story_title story_author story_year chapter chapter_num p s);
my @a_att = qw(vss);

my $C3 = new CWB::CL::Corpus "VSS";
isa_ok($C3, "CWB::CL::Corpus", "corpus object for VSS corpus") # T19
  or BAIL_OUT("failed to access test corpus VSS");

my @res = $C3->list_attributes("p");
is_deeply(\@res, \@p_att, "list all p-attributes");
@res = $C3->list_attributes("s");
is_deeply(\@res, \@s_att, "list all p-attributes");
@res = $C3->list_attributes("a");
is_deeply(\@res, \@a_att, "list all p-attributes");
@res = $C3->list_attributes();
is_deeply(\@res, [@p_att, @s_att, @a_att], "list all attributes"); # ordering in the registry file

# total: 27 tests

sub as_utf8 {
  map {encode("utf8", $_)} @_;
}

sub as_latin1 {
  map {encode("latin1", $_)} @_;
}
