package LCD::MatrixOrbital;

require 5.005;
$VERSION = "0.91";

###############################################################################
#
# LCD::MatrixOrbital
# Version 0.91
# Date: 12/13/1999
# Copyright (c) 1997 Devon Jones (soulcatcher@evilsoft.org).
# All rights reserved. This program is free software;
# you can redistribute it and/or modify it under the same terms as Perl itself. 
#
# Description: LDC::MatrixOrbital is a class designed to allow a programmer
# access the display based functions of a Matrix Orbital Serial LCD.  It
# should work with a Matrix Orbital VFD as well (although not supporting one
# or two functions that are VFD specific - this is because I do not yet own
# a VFD.) There is currently no support for keypads on LCDs or VFDs. (Also
# because I Currently do not own one.)
#
# Using this module: In order to use this module, you must do the following:
# 
# use LCD::MatrixOrbital
# $LCD = new LCD::MatrixOrbital('/dev/cua0', 20, 4, 0);
#
# Strictly speaking the only paramtere you need to send to new is the name
# of the device file for the serial port you are using, but I find it's better
# to be explicit.
#
# Following that, you can use any of the methods for the devicem but only
# certain ones are recommended:
#
# new (when creating)
# ClearAll
# CreateCharacter
# InitGraphHoriz
# InitGraphVert
# InitLargeDigit
# PrintGraphHoriz
# PrintGraphVert
# PrintLargeDigit
# PrintText
# ResetState
# SetBacklight
# SetContrast
# SetCursorBlink
# SetCursorDisplay
# SetCursorLeft
# SetCursorRight
# SetLineWrap
# SetPosition
# SetPositionTop
# SetScroll
# StateBacklight
# StateContrast
# StateCursorBlink
# StateCursorDisplay
# StateLineWrap
# StatePosition
# StateScroll
#
# All others should be considered 'Private' methods.
###############################################################################

sub new
###############################################################################
#          Name: LCD::MatrixOrbital->new (Public)
#    Parameters: Device - Should be the device file of your serial port that
#                         the LCD is attached to.  Typically /dev/cua0 or
#                         /dev/cua1. Required
#                     X - Integer of the width (in characters) of the LCD
#                         defaults to 20 (typically 20 or 40)
#                     Y - Integer of the height (in characters) of the LCD
#                         defaults to 4 (typically 2 or 4)
#                 Debug - If debug is turned to on, the system will print
#                         everything it sends to the lcd to STDOUT.  Could
#                         also be useful if you want to redirect the output
#                         to another place.  can be 0 (off) or 1 (on) defaults
#                         to 0
#       Returns: An LCD::MatrixOrbital object.
#  Precondition: Need to create a new LCD::MatrixOrbital object
# Postcondition: The object is created, and is initialized to default values.
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my ($this, $device, $x, $y, $debug) = @_;
  my $class = $this;
  my $self = {};
  bless $self, $class;
  $self->init($device,$x, $y, $debug);
  return $self;
}

sub init
###############################################################################
#          Name: LCD::MatrixOrbital->init (Private)
#    Parameters: Device - Should be the device file of your serial port that
#                         the LCD is attached to.  Typically /dev/cua0 or
#                         /dev/cua1. Required
#                     X - Integer of the width (in characters) of the LCD
#                         defaults to 20 (typically 20 or 40)
#                     Y - Integer of the height (in characters) of the LCD
#                         defaults to 4 (typically 2 or 4)
#                 Debug - If debug is turned to on, the system will print
#                         everything it sends to the lcd to STDOUT.  Could
#                         also be useful if you want to redirect the output
#                         to another place.  can be 0 (off) or 1 (on) defaults
#                         to 0
#       Returns: Nothing
#  Precondition: The LCD::MatrixOrbital object is created, but uninitialized
# Postcondition: The LCD::MatrixOrbital object has been initialized
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self, $device, $x, $y, $debug) = @_;
  open COMPORT, '>>/dev/cua0';
  if ($debug == "" || $debug == 0) { $self{'Debug'} = 0; }
  else { $self{'Debug'} = 1; }
  if ($x == "" || $x == 0) { $x = 20; }
  if ($y == "" || $y == 0) { $y = 4; }
  $self{'CMaxX'} = $x;
  $self{'CMaxY'} = $y;
  $self{'SBacklight'} = 0;
  $self->SetBacklight('off');
  $self{'SContrast'} = 0;
  $self->SetContrast(128);
  $self{'SCursorBlink'} = 0;
  $self->SetCursorBlink('off');
  $self{'SCursorDisplay'} = 0;
  $self->SetCursorDisplay('off');
  $self{'SLineWrap'} = 0;
  $self->SetLineWrap('off');
  $self{'SPositionX'} = 0;
  $self{'SPositionY'} = 0;
  $self->SetPositionTop;
  $self{'SScroll'} = 0;
  $self->SetScroll('off');
}


sub ClearAll
###############################################################################
#          Name: LCD::MatrixOrbital->ClearAll (Public)
#    Parameters: None
#       Returns: Nothing
#  Precondition: Programmer wants to clear screen
# Postcondition: LCD Screen is cleared, and cursor is at position 0,0
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift;
  $self->PrintCommandString;
  print COMPORT 'X';
  if ($self{'Debug'} == 1) { print STDOUT 'X'; }
  $self{'SPositionX'} = 0;
  $self{'SPositionY'} = 0;
}

sub CreateCharacter
###############################################################################
#          Name: LCD::MatrixOrbital->CreateCharacter (Public)
#          Note: A character is a 5 x 8 grid, with a 1 standing for on, and a 0
#                standing for off.  Another note - all lines are required, and
#                all require the programmer to give five ones or zeros
#    Parameters: Position - can be 0-7. this stands for the hexidecimal position
#                           of the character.  In order to print this character
#                           in the future, you must print the ascii
#                           representation of 0x00 - 0x07 (depending on where
#                           you place the character.) Required
#                  Line 1 - string of five ones and zeros (such as 01100 or
#                           00010) standing for whether the pixel is on or off
#                           for positions 1-5 of line 1. Required
#            Line 2 ... 7 - Same as above but appropriate for each line.
#                  Line 8 - Same as above, but the whole line will be on or off
#                           based on the first number.  So 01111 and 00000 would
#                           both print an empty line vs. 10010 and 11101 would
#                           print the full line. Required
#       Returns: Nothing
#  Precondition: Programmer wishes to create a special character
# Postcondition: The special character exists in one of 0x00 - 0x07
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self, $position, @line) = @_;
  my(@ASClines) = "";
  my($ASCposition) = "";

  if ($position <= 7 && $position >= 0)
  {
    $ASCposition = pack "H2", "0$position";
  }
  else { die "Position must be between 0 and 7"; }
  
  my ($i) = 0;
  foreach $BINline (@line)
  {
    $i++;
    my ($len) = length $BINline;
    if ($len < 5 || $len > 5)
      { die "A line in a new character can only be 5 1's or 0's.  No More, No Less"; }
    $ASClines[$i] = pack "B8", "000$BINline";
  }
  $self->PrintCommandString;
  print COMPORT 'N' . $ascposition;
  if ($self{'Debug'} == 1) { print STDOUT 'N' . $ascposition; }
  foreach $ASCline (@ASClines)
  {
    print COMPORT $ASCline;
    if ($self{'Debug'} == 1) { print STDOUT $ASCline; }
  }
}

sub InitGraphHoriz
###############################################################################
#          Name: LCD::MatrixOrbital->InitGraphHoriz (Public)
#    Parameters: None
#       Returns: Nothing
#  Precondition: Programmer wishes to use a horizontal graph
# Postcondition: All custom character slots are filled with horizontal graph
#                Characters.
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift; 
  $self->PrintCommandString;
  print COMPORT 'h';
  if ($self{'Debug'} == 1) { print STDOUT 'h'; }
}

sub InitGraphVert
###############################################################################
#          Name: LCD::MatrixOrbital->InitGraphVert (Public)
#    Parameters: Width - 'thick' or 'thin'. Defaults to thick
#       Returns: Nothing
#  Precondition: Programmer wishes to use a vertical graph
# Postcondition: All custom character slots are filled with vertical graph
#                Characters.
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift;
  my($width) = shift;

  if ($width = "")
  {
    $width = "thick";
  }
  if ($cmd =~ /thick/i)
  {
    $self->PrintCommandString;
    print COMPORT 'v';
    if ($self{'Debug'} == 1) { print STDOUT 'v'; }
  }
  elsif ($cmd =~ /thin/i)
  {
    $self->PrintCommandString;
    print COMPORT 's';
    if ($self{'Debug'} == 1) { print STDOUT 's'; }
  }
  else { die "Width must be either 'thin' or 'thick', not $width"; }
  return 1;
}

sub InitLargeDigit
###############################################################################
#          Name: LCD::MatrixOrbital->InitLargeDigit (Public)
#    Parameters: None
#       Returns: Nothing
#  Precondition: Programmer wishes to use large digits
# Postcondition: All custom character slots are filled with large digit
#                Characters.
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift; 
  $self->PrintCommandString;
  print COMPORT 'n';
  if ($self{'Debug'} == 1) { print STDOUT 'n'; }
}

sub PrintCommandString
###############################################################################
#          Name: LCD::MatrixOrbital->PrintCommandString (Private)
#    Parameters: None
#       Returns: Nothing
#  Precondition: Programmer wants to send a command to the LCD
# Postcondition: The LCD has recieved the command character, and will accept
#                the next character as a command.
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift;
  my($cmd) = pack "H2", "fe";
  print COMPORT $cmd;
  if ($self{'Debug'} == 1) { print STDOUT $cmd; }
}

sub PrintGraphHoriz
###############################################################################
#          Name: LCD::MatrixOrbital->PrintGraphHoriz (Public)
#    Parameters:   Cnumber - Column number the graph line will be printed in.
#                            Required.  Must be 1 through LCD max X
#                  Rnumber - Row number the graph line will be printed in.
#                            Required.  Must be 1 through LCD max Y
#                direction - 1 (Left to Right) or 0 (Right to Left) Required
#                   length - Length of the graph.  From 1 to 5 * LCD max X
#                            Required
#       Returns: Nothing
#  Precondition: Programmer wants to print a graph line, and InitGraphVert has been run
# Postcondition: A line will be printed on the LCD
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift;
  my($cnumber) = shift;
  my($rnumber) = shift;
  my($direction) = shift;
  my($len) = shift;

  if ($cnumber <= 0 || $cnumber > $self{'CMaxX'})
  { die "Column number must be between 1 and $self{'CMaxX'}.  $cnumber is unaccaptable."; }
  if ($rnumber <= 0 || $rnumber > $self{'CMaxY'})
  { die "Row number must be between 1 and $self{'CMaxY'}.  $rnumber is unaccaptable."; }
  if ($direction != 0 && $direction != 1)
  { die "Direction must be either 1 (left to right), or 0 (right to left)"; }
  my ($MaxLen) = $self{'CMaxX'} * 5;
  if ($len < 0 || $len > $MaxLen)
  { die "Column length must be between 0 and $MaxLen. $len is unacceptble"; }
  
  $ASCcnumber = pack "i", $cnumber;
  @ASCcnumbers = split //, $ASCcnumber;

  $ASCrnumber = pack "i", $rnumber;
  @ASCrnumbers = split //, $ASCrnumber;
  
  $ASCdirection = pack "i", $direction;
  @ASCdirection = split //, $ASCdirection;
  
  $ASClen = pack "i", $cnumber;
  @ASClens = split //, $ASClen;

  $self->PrintCommandString;
  print COMPORT '|' . $ASCcnumbers[0] . $ASCrnumbers[0] . $ASCdirection[0] . $ASClens[0];
  if ($self{'Debug'} == 1) { print STDOUT '|' . $ASCcnumbers[0] . $ASCrnumbers[0] . $ASCdirection[0] . $ASClens[0]; }
}

sub PrintGraphVert
###############################################################################
#          Name: LCD::MatrixOrbital->PrintGraphVert (Public)
#    Parameters:   Cnumber - Column number the graph line will be printed in.
#                            Required.  Must be 1 through LCD max X
#                   length - Length of the graph.  From 1 to 8 * LCD max Y
#                            Required
#       Returns: Nothing
#  Precondition: Programmer wants to print a graph line, and InitGraphVert has been run
# Postcondition: A line will be printed on the LCD
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift;
  my($cnumber) = shift;
  my($len) = shift;
  
  if ($cnumber <= 0 || $cnumber > $self{'CMaxX'})
  { die "Column number must be between 1 and $self{'CMaxX'}.  $cnumber is unaccaptable."; }
  my ($MaxLen) = $self{'CMaxY'} * 8;
  if ($len < 0 || $len > $MaxLen)
  { die "Column length must be between 0 and $MaxLen. $len is unacceptble"; }
  
  $ASCcnumber = pack "i", $cnumber;
  @ASCcnumbers = split //, $ASCcnumber;
  
  $ASClen = pack "i", $cnumber;
  @ASClens = split //, $ASClen;
  
  $self->PrintCommandString;
  print COMPORT '=' . $ASCcnumbers[0] . $ASClens[0];
  if ($self{'Debug'} == 1) { print STDOUT '=' . $ASCcnumbers[0] . $ASClens[0]; }
}

sub PrintLargeDigit
###############################################################################
#          Name: LCD::MatrixOrbital->PrintLargeDigit (Public)
#    Parameters:   Cnumber - Column number the graph line will be printed in.
#                            Required.  Must be 1 through LCD max X - 2
#                    Digit - 0 - 9. Required
#       Returns: Nothing
#  Precondition: User wantes to print a large digit, and has run InitLargeDigit
# Postcondition: A large digit has been printed
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift;
  my($cnumber) = shift;
  my($digit) = shift;
  
  my ($MaxColumn) = $self{'CMaxX'} - 2;
  if ($cnumber <= 0 || $cnumber > $MaxColumn)
  { die "Column number must be between 1 and $MaxColumn.  $cnumber is unaccaptable."; }
  if ($digit < 0 || $digit > 9)
  { die "Row number must be between 1 and $self{'CMaxY'}.  $rnumber is unaccaptable."; }
  
  $ASCcnumber = pack "i", $cnumber;
  @ASCcnumbers = split //, $ASCcnumber;
  
  $ASCdigit = pack "i", $digit;
  @ASCdigit = split //, $ASCdigit;
  
  $self->PrintCommandString;
  print COMPORT '#' . $ASCcnumbers[0] . $ASCdigit[0];
  if ($self{'Debug'} == 1) { print STDOUT '#' . $ASCcnumbers[0] . $ASCdigit[0]; }
}

sub PrintText
###############################################################################
#          Name: LCD::MatrixOrbital->PrintText (Public)
#    Parameters: Text - The test you want to print.  It will start from the
#                       current location, and print from there.  If LineWrap is
#                       on, it will wrap.
#       Returns: Nothing
#  Precondition: Programmer wants to print some text
# Postcondition: Text is printed
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift; 
  my($text) = shift;
  print COMPORT $text;
  if ($self{'Debug'} == 1) { print STDOUT $text; }
}

sub ResetState
###############################################################################
#          Name: LCD::MatrixOrbital->ResetState (Public)
#    Parameters: None
#       Returns: Nothing
#  Precondition: Programmer wants to reset the state to as it was at instantiation
# Postcondition: Backlight is off, blink os off, Cursor is off, Line Wrap is off
#                Position is 0,0 and Scroll is off
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift;
  $self->SetBacklight('off');
  $self->SetContrast(128);
  $self->SetCursorBlink('off');
  $self->SetCursorDisplay('off');
  $self->SetLineWrap('off');
  $self->SetPositionTop;
  $self->SetScroll('off');
}

sub SetBacklight
###############################################################################
#          Name: LCD::MatrixOrbital->Backlight (Public)
#    Parameters: cmd - 'on'/1 or 'off'/0 defaults to on
#                min - 0 - 180.  This is the number of min that backlight will
#                      remain on.  0 stands for forever.
#       Returns: Nothing
#  Precondition: Programmer wishes to change the state of the Backlight
# Postcondition: Backlight is now on or off as wished.
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift;
  my($cmd) = shift;
  my($min) = shift;
  if ($cmd eq "")
  {
    if ($self{'SBacklight'}) { $cmd = 'off'; }
    else { $cmd = 'on'; }
  }
  if ($min = "")
  {
    $min = 0;
  }
  if ($val <= 180 && $val >= 0)
  {
    if ($cmd =~ /on/i || $cmd eq 1)
    {
      $ASCmin = pack "i", $min;
      @ASCmins = split //, $ASCmin;
      $self->PrintCommandString;
      print COMPORT 'B' . $ASCmins[0];
      if ($self{'Debug'} == 1) { print STDOUT 'B' . $ASCmins[0]; }
      $self{'SBacklignt'} = 1;
    }
    elsif ($cmd =~ /off/i || $cmd eq 0)
    {
      $self->PrintCommandString;
      print COMPORT 'F';
      if ($self{'Debug'} == 1) { print STDOUT 'F'; }
      $self{'SBacklight'} = 0;
    }
    else { die "Backlight can only be (on or off) or (1 or 0), not '$cmd'"; }
  }
  else { die "Backlight on time must be between 0 and 180"; }
  
  return 1;
}

sub SetContrast
###############################################################################
#          Name: LCD::MatrixOrbital->SetContrast (public)
#    Parameters: Contrast - 0 - 256. Required
#       Returns: Nothing
#  Precondition: Programmer wishes to change the screen contrast.
# Postcondition: Contrast is set
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift;
  my($val) = shift;
  if ($val <= 256 && $val >= 0)
  {
    $ASCval = pack "i", $val;
    @ASCvals = split //, $ASCval;
    $self->PrintCommandString;
    print COMPORT 'P' . $ASCvals[0];
    if ($self{'Debug'} == 1) { print STDOUT 'P' . $ASCvals[0]; }
    $self{'SContrast'} = $val;
    return 1;
  }
  else { die "Contrast must be between 0 and 256, not '$val'"; }
}

sub SetCursorBlink
###############################################################################
#          Name: LCD::MatrixOrbital->SetCursorBlink (public)
#    Parameters: cmd - 'on'/1 or 'off'/0  Defaults to on
#       Returns: Nothing
#  Precondition: Programmer wishes to turn cursor blinking on/off
# Postcondition: cursor is now blinking/not blinking
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift;
  my($cmd) = shift;
  if ($cmd == "")
  {
    if ($self{'SCursorBlink'}) { $cmd = 'off'; }
    else { $cmd = 'on'; }
  }
  if ($cmd =~ /on/i || $cmd eq 1)
  {
    $self->PrintCommandString;
    print COMPORT 'S';
    if ($self{'Debug'} == 1) { print STDOUT 'S'; }
    $self{'SCursorBlink'} = 1;
  }
  elsif ($cmd =~ /off/i || $cmd eq 0)
  {
    $self->PrintCommandString;
    print COMPORT 'T';
    if ($self{'Debug'} == 1) { print STDOUT 'T'; }
    $self{'SCursorBlink'} = 0;
  }
  else { die "Blink can only be (on or off) or (1 or 0), not '$cmd'"; }
  return 1;
}

sub SetCursorDisplay
###############################################################################
#          Name: LCD::MatrixOrbital->SetCursorDisplay (public)
#    Parameters: cmd - 'on'/1 or 'off'/0  Defaults to on
#       Returns: nothing
#  Precondition: Programmer wishes to turn cursor display on/off
# Postcondition: cursor is now displayed/not displayed
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift;
  my($cmd) = shift;
  if ($cmd == "")
  {
    if ($self{'SCursorDisplay'}) { $cmd = 'off'; }
    else { $cmd = 'on'; }
  }
  if ($cmd =~ /on/i || $cmd eq 1)
  {
    $self->PrintCommandString;
    print COMPORT 'J';
    if ($self{'Debug'} == 1) { print STDOUT 'J'; }
    $self{'SCursorDisplay'} = 1;
  }
  elsif ($cmd =~ /off/i || $cmd eq 0)
  {
    $self->PrintCommandString;
    print COMPORT 'K';
    if ($self{'Debug'} == 1) { print STDOUT 'K'; }
    $self{'SCursorDisplay'} = 0;
  }
  else { die "Cursor can only be (on or off) or (1 or 0), not '$cmd'"; }
  return 1;
}

sub SetCursorLeft
###############################################################################
#          Name: LCD::MatrixOrbital->SetCursorLeft (public)
#    Parameters: None
#       Returns: Nothing
#  Precondition: Programmer wishes to move the cursor one spot to the left
# Postcondition: cursor has moved on spot to the left
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift; 
  $self->PrintCommandString;
  print COMPORT 'L';
  if ($self{'Debug'} == 1) { print STDOUT 'L'; }
}

sub SetCursorRight
###############################################################################
#          Name: LCD::MatrixOrbital->
#    Parameters: None
#       Returns: Nothing
#  Precondition: Programmer wishes to move the cursor one spot to the right
# Postcondition: cursor has moved on spot to the right
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift; 
  $self->PrintCommandString;
  print COMPORT 'M';
  if ($self{'Debug'} == 1) { print STDOUT 'M'; }
}

sub SetLineWrap
###############################################################################
#          Name: LCD::MatrixOrbital->SetLineWrap (public)
#    Parameters: cmd - 'on'/1 or 'off'/0  Defaults to on
#       Returns: nothing
#  Precondition: Programmer wants to turn line wrapping on/off
# Postcondition: Line Wrapping is on/off
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift;
  my($cmd) = shift;
  if ($cmd == "")
  {
    if ($self{'SLineWrap'}) { $cmd = 'off'; }
    else { $cmd = 'on'; }
  }
  if ($cmd =~ /on/i || $cmd eq 1)
  {
    $self->PrintCommandString;
    print COMPORT 'C';
    if ($self{'Debug'} == 1) { print STDOUT 'C'; }
    $self{'SLineWrap'} = 1;
  }
  elsif ($cmd =~ /off/i || $cmd eq 0)
  {
    $self->PrintCommandString;
    print COMPORT 'D';
    if ($self{'Debug'} == 1) { print STDOUT 'D'; }
    $self{'SLineWrap'} = 0;
  }
  else { die "Line Wrap can only be (on or off) or (1 or 0), not '$cmd'"; }
  return 1;
}

sub SetPosition
###############################################################################
#          Name: LCD::MatrixOrbital->SetPosition (public)
#    Parameters: x - Required. The x position of the cursor
#                y - Required. The y position of the cursor
#       Returns: Nothing
#  Precondition: Programmer wants to change the X,Y postion of the cursor
# Postcondition: the X,Y position has changed
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift;
  my($x) = shift;
  my($y) = shift;
  
  if ($x >= 0 && $x <= ($self{'CMaxX'} - 1))
  {
    if ($y >= 0 && $y <= ($self{'CMaxY'} - 1))
    {
      $ASCx = pack "i", $x;
      $ASCy = pack "i", $y;
      @ASCxa = split //, $ASCx;
      @ASCya = split //, $ASCy;
      $self->PrintCommandString;
      print COMPORT 'G'. $ASCxa[0] . $ASCya[0];
      if ($self{'Debug'} == 1) { print STDOUT 'G' . $ASCxa[0] . $ASCya[0]; }
      $self{'SLineWrap'} = 0;
    }
    else { die "'$y' value for Y is outside range of 1 to $self{'CMaxY'}"; }
  }
  else { die "'$x' value for X is outside range of 1 to $self{'CMaxX'}"; }
}

sub SetPositionTop
###############################################################################
#          Name: LCD::MatrixOrbital->SetPositionTop (public)
#    Parameters: None
#       Returns: Nothing
#  Precondition: Programmer wants the cursor to return to 0,0
# Postcondition: The cursor is at 0,0
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift; 
  $self->PrintCommandString;
  print COMPORT 'H';
  if ($self{'Debug'} == 1) { print STDOUT 'H'; }
}

sub SetScroll
###############################################################################
#          Name: LCD::MatrixOrbital->SetScroll (public)
#    Parameters: cmd - 'on'/1 or 'off'/0  Defaults to on
#       Returns: Nothing
#  Precondition: Programmer wants to turn scrolling on/off
# Postcondition: scrolling is on/off
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{
  my($self) = shift;
  my($cmd) = shift;
  if ($cmd == "")
  {
    if ($self{'SScroll'}) { $cmd = 'off'; }
    else { $cmd = 'on'; }
  }
  if ($cmd =~ /on/i || $cmd eq 1)
  {
    $self->PrintCommandString;
    print COMPORT 'Q';
    if ($self{'Debug'} == 1) { print STDOUT 'Q'; }
    $self{'SScroll'} = 1;
  }
  elsif ($cmd =~ /off/i || $cmd eq 0)
  {
    $self->PrintCommandString;
    print COMPORT 'R';
    if ($self{'Debug'} == 1) { print STDOUT 'R'; }
    $self{'SScroll'} = 0;
  }
  else { die "Scroll can only be (on or off) or (1 or 0), not '$cmd'"; }
  return 1;
}

sub StateBacklight
###############################################################################
#          Name: LCD::MatrixOrbital->StateBacklight (public)
#    Parameters: None
#       Returns: Backlight State - 1 or 0
#  Precondition: Programmer wants to know the state of the backlight
# Postcondition: The backlight state is returned
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{ my($self) = shift; return $self{'SBacklight'}; }

sub StateContrast
###############################################################################
#          Name: LCD::MatrixOrbital->Statecontrast (public)
#    Parameters: None
#       Returns: contrast State - 0 - 256
#  Precondition: Programmer wants to know the state of LCD contrast
# Postcondition: The contrast state is returned
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{ my($self) = shift; return $self{'SContrast'}; }

sub StateCursorBlink
###############################################################################
#          Name: LCD::MatrixOrbital->StateCursorBlink (public)
#    Parameters: None
#       Returns: cursor blinking State - 1 or 0
#  Precondition: Programmer wants to know the state of the corsor's blinking
# Postcondition: The cursor blinking state is returned
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{ my($self) = shift; return $self{'SCursorBlink'}; }

sub StateCursorDisplay
###############################################################################
#          Name: LCD::MatrixOrbital->StateCursorDisplay (public)
#    Parameters: None
#       Returns: Cursor Display State - 1 or 0
#  Precondition: Programmer wants to know the state of the cursor display
# Postcondition: The cursor display state is returned
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{ my($self) = shift; return $self{'SCursorDisplay'}; }

sub StateLineWrap
###############################################################################
#          Name: LCD::MatrixOrbital->StateLineWrap (public)
#    Parameters: None
#       Returns: Line Wrap State - 1 or 0
#  Precondition: Programmer wants to know the state of Line Wrapping
# Postcondition: The line wrapping state is returned
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{ my($self) = shift; return $self{'SLineWrap'}; }

sub StatePosition
###############################################################################
#          Name: LCD::MatrixOrbital->StatePosition (public)
#    Parameters: None
#       Returns: x - x position of the cursor
#                y - y position of the cursor
#  Precondition: Programmer wants to know the position of the cursor
# Postcondition: The cursor position is returned
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{ my($self) = shift; return $self{'SPositionX'}, $self{'SPositionY'}; }

sub StateScroll
###############################################################################
#          Name: LCD::MatrixOrbital->StateScroll (public)
#    Parameters: None
#       Returns: Scroll State - 1 or 0
#  Precondition: Programmer wants to know the state of scrolling
# Postcondition: The scolling state is returned
#        Author: Devon Jones (soulcatcher@evilsoft.org)
#          Date: 12/13/1999
###############################################################################
{ my($self) = shift; return $self{'SScroll'}; }
