# window.py
#
# Copyright 2025 Jamie Gravendeel
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-or-later

import logging
from pathlib import Path
from typing import Any

from gi.repository import Adw, Gio, GLib, GObject, Gtk

from morphosis import supported_formats
from morphosis.document import MorphosisDocument
from morphosis.font_style_dialog import MorphosisFontStyleDialog
from morphosis.help_dialog import MorphosisHelpDialog


@Gtk.Template(resource_path="/garden/jamie/Morphosis/window.ui")
class MorphosisWindow(Adw.ApplicationWindow):
    """The main application window."""

    __gtype_name__ = "MorphosisWindow"

    toast_overlay: Adw.ToastOverlay = Gtk.Template.Child()
    navigation_view: Adw.NavigationView = Gtk.Template.Child()

    format_row: Adw.ComboRow = Gtk.Template.Child()
    font_style_row: Adw.ComboRow = Gtk.Template.Child()

    has_font_options = GObject.Property(type=bool, default=False)

    schema = Gio.Settings.new("garden.jamie.Morphosis")
    document = GObject.Property(type=MorphosisDocument, default=MorphosisDocument())

    def __init__(self, **kwargs: Any) -> None:
        super().__init__(**kwargs)

        # TODO: Check whether the bindings can be in Blueprint
        self.schema.bind(
            "export-format",
            self.format_row,
            "selected",
            Gio.SettingsBindFlags.DEFAULT,
        )

        self.schema.bind(
            "font-style",
            self.font_style_row,
            "selected",
            Gio.SettingsBindFlags.DEFAULT,
        )

        self.document.connect("convert-finished", self._on_convert_finished)
        self.document.connect(
            "invalid-content-type",
            lambda *_: self.toast_overlay.add_toast(
                Adw.Toast.new(_("Unsupported file format"))
            ),
        )

    @Gtk.Template.Callback()
    def _open_document(self, *_args: Any) -> None:
        """Open a file dialog to select a document."""

        def open_document_cb(dialog: Gtk.FileDialog, res: Gio.AsyncResult) -> None:
            try:
                if not (file := dialog.open_finish(res)):
                    logging.error("Cannot get document")
                    return

            except GLib.Error as error:
                logging.error("Cannot get document: %s", error)
                return

            if not self.document.set_from_file(file):
                return

            # TODO: Find another way to show/hide the font options
            self._on_format_selected(self.format_row)
            self.navigation_view.push_by_tag("document")

        Gtk.FileDialog(
            title=_("Select Document"),
            default_filter=Gtk.FileFilter(
                name=_("Document"),
                mime_types=tuple(
                    mime_type for mime_type in supported_formats.MIME_TYPES
                ),
                suffixes=tuple(suffix for suffix in supported_formats.EXTENSIONS),
            ),
        ).open(self, callback=open_document_cb)

    @Gtk.Template.Callback()
    def _save_document(self, *_args: Any) -> None:
        """Open a file dialog to select a save location."""

        def save_document_cb(dialog: Gtk.FileDialog, res: Gio.AsyncResult) -> None:
            try:
                if not (output_file := dialog.save_finish(res)):
                    logging.error("Cannot get save location")
                    return

            except GLib.Error as error:
                logging.error("Cannot get save location: %s", error)
                return

            self.navigation_view.push_by_tag("converting")

            GLib.Thread.new(
                None,
                self.document.convert,
                output_file,
                self.format_row.get_selected_item().get_string(),
                self.font_style_row.get_selected_item().get_string(),
                self.has_font_options,
            )

        dialog = Gtk.FileDialog(
            initial_name=f"{Path(self.document.name).stem}.{supported_formats.FORMATS[self.format_row.get_selected_item().get_string()].ext}"
        )

        dialog.save(self, callback=save_document_cb)

    def _on_convert_finished(
        self,
        _obj: Any,
        output_file: Gio.File | None = None,
    ) -> None:
        self.navigation_view.pop_to_tag("start")

        toast = Adw.Toast(
            title=(_("Document converted") if output_file else _("Conversion failed")),
            button_label=(_("Open") if output_file else _("Help")),
        )

        toast.connect(
            "button-clicked",
            lambda *_: Gtk.FileLauncher.new(output_file).launch(self)
            if output_file
            else MorphosisHelpDialog().present(self),
        )

        self.toast_overlay.dismiss_all()
        self.toast_overlay.add_toast(toast)

    @Gtk.Template.Callback()
    def _on_format_selected(self, row: Adw.ComboRow, *_args: Any) -> None:
        self.has_font_options = row.get_selected_item().get_string() in ("HTML", "PDF")

    @Gtk.Template.Callback()
    def _show_font_style_dialog(self, *_args: Any) -> None:
        MorphosisFontStyleDialog().present(self)
