"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineRoutes = defineRoutes;
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */

// TODO: consider to extract entity CRUD operations and put it into a client class
function defineRoutes(router, dataSourceEnabled) {
  const internalUserSchema = _configSchema.schema.object({
    description: _configSchema.schema.maybe(_configSchema.schema.string()),
    password: _configSchema.schema.maybe(_configSchema.schema.string()),
    backend_roles: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    attributes: _configSchema.schema.any({
      defaultValue: {}
    })
  });
  const actionGroupSchema = _configSchema.schema.object({
    description: _configSchema.schema.maybe(_configSchema.schema.string()),
    allowed_actions: _configSchema.schema.arrayOf(_configSchema.schema.string())
    // type field is not supported in legacy implementation, comment it out for now.
    // type: schema.oneOf([
    //   schema.literal('cluster'),
    //   schema.literal('index'),
    //   schema.literal('opensearch_dashboards'),
    // ]),
  });

  const roleMappingSchema = _configSchema.schema.object({
    description: _configSchema.schema.maybe(_configSchema.schema.string()),
    backend_roles: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    hosts: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    users: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  });
  const roleSchema = _configSchema.schema.object({
    description: _configSchema.schema.maybe(_configSchema.schema.string()),
    cluster_permissions: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    tenant_permissions: _configSchema.schema.arrayOf(_configSchema.schema.any(), {
      defaultValue: []
    }),
    index_permissions: _configSchema.schema.arrayOf(_configSchema.schema.any(), {
      defaultValue: []
    })
  });
  const tenantSchema = _configSchema.schema.object({
    description: _configSchema.schema.string()
  });
  const accountSchema = _configSchema.schema.object({
    password: _configSchema.schema.string(),
    current_password: _configSchema.schema.string()
  });
  const schemaMap = {
    internalusers: internalUserSchema,
    actiongroups: actionGroupSchema,
    rolesmapping: roleMappingSchema,
    roles: roleSchema,
    tenants: tenantSchema,
    account: accountSchema
  };
  function validateRequestBody(resourceName, requestBody) {
    const inputSchema = schemaMap[resourceName];
    if (!inputSchema) {
      throw new Error(`Unknown resource ${resourceName}`);
    }
    inputSchema.validate(requestBody); // throws error if validation fail
  }

  function validateEntityId(resourceName) {
    if (!(0, _common.isValidResourceName)(resourceName)) {
      return 'Invalid entity name or id.';
    }
  }

  /**
   * Lists resources by resource name.
   *
   * The response format is:
   * {
   *   "total": <total_entity_count>,
   *   "data": {
   *     "entity_id_1": { <entity_structure> },
   *     "entity_id_2": { <entity_structure> },
   *     ...
   *   }
   * }
   *
   * e.g. when listing internal users, response may look like:
   * {
   *   "total": 2,
   *   "data": {
   *     "api_test_user2": {
   *       "hash": "",
   *       "reserved": false,
   *       "hidden": false,
   *       "backend_roles": [],
   *       "attributes": {},
   *       "description": "",
   *       "static": false
   *     },
   *     "api_test_user1": {
   *       "hash": "",
   *       "reserved": false,
   *       "hidden": false,
   *       "backend_roles": [],
   *       "attributes": {},
   *       "static": false
   *     }
   * }
   *
   * when listing action groups, response will look like:
   * {
   *   "total": 2,
   *   "data": {
   *     "read": {
   *       "reserved": true,
   *       "hidden": false,
   *       "allowed_actions": ["indices:data/read*", "indices:admin/mappings/fields/get*"],
   *       "type": "index",
   *       "description": "Allow all read operations",
   *       "static": false
   *     },
   *     "cluster_all": {
   *       "reserved": true,
   *       "hidden": false,
   *       "allowed_actions": ["cluster:*"],
   *       "type": "cluster",
   *       "description": "Allow everything on cluster level",
   *       "static": false
   *     }
   * }
   *
   * role:
   * {
   *   "total": 2,
   *   "data": {
   *     "opensearch_dashboards_user": {
   *       "reserved": true,
   *       "hidden": false,
   *       "description": "Provide the minimum permissions for a opensearch_dashboards user",
   *       "cluster_permissions": ["cluster_composite_ops"],
   *       "index_permissions": [{
   *         "index_patterns": [".opensearch_dashboards", ".opensearch_dashboards-6", ".opensearch_dashboards_*"],
   *         "fls": [],
   *         "masked_fields": [],
   *         "allowed_actions": ["read", "delete", "manage", "index"]
   *       }, {
   *         "index_patterns": [".tasks", ".management-beats"],
   *         "fls": [],
   *         "masked_fields": [],
   *         "allowed_actions": ["indices_all"]
   *       }],
   *       "tenant_permissions": [],
   *       "static": false
   *     },
   *     "all_access": {
   *       "reserved": true,
   *       "hidden": false,
   *       "description": "Allow full access to all indices and all cluster APIs",
   *       "cluster_permissions": ["*"],
   *       "index_permissions": [{
   *         "index_patterns": ["*"],
   *         "fls": [],
   *         "masked_fields": [],
   *         "allowed_actions": ["*"]
   *       }],
   *       "tenant_permissions": [{
   *         "tenant_patterns": ["*"],
   *         "allowed_actions": ["opensearch_dashboards_all_write"]
   *       }],
   *       "static": false
   *     }
   *   }
   * }
   *
   * rolesmapping:
   * {
   *   "total": 2,
   *   "data": {
   *     "security_manager": {
   *       "reserved": false,
   *       "hidden": false,
   *       "backend_roles": [],
   *       "hosts": [],
   *       "users": ["zengyan", "admin"],
   *       "and_backend_roles": []
   *     },
   *     "all_access": {
   *       "reserved": false,
   *       "hidden": false,
   *       "backend_roles": [],
   *       "hosts": [],
   *       "users": ["zengyan", "admin", "indextest"],
   *       "and_backend_roles": []
   *     }
   *   }
   * }
   *
   * tenants:
   * {
   *   "total": 2,
   *   "data": {
   *     "global_tenant": {
   *       "reserved": true,
   *       "hidden": false,
   *       "description": "Global tenant",
   *       "static": false
   *     },
   *     "test tenant": {
   *       "reserved": false,
   *       "hidden": false,
   *       "description": "tenant description",
   *       "static": false
   *     }
   *   }
   * }
   */
  router.get({
    path: `${_common.API_PREFIX}/${_common.CONFIGURATION_API_PREFIX}/{resourceName}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const esResp = await wrapRouteWithDataSource(dataSourceEnabled, context, request, 'opensearch_security.listResource', {
        resourceName: request.params.resourceName
      });
      return response.ok({
        body: {
          total: Object.keys(esResp).length,
          data: esResp
        }
      });
    } catch (error) {
      console.log(JSON.stringify(error));
      return errorResponse(response, error);
    }
  });

  /**
   * Gets entity by id.
   *
   * the response format differs from different resource types. e.g.
   *
   * for internal user, response will look like:
   * {
   *   "hash": "",
   *   "reserved": false,
   *   "hidden": false,
   *   "backend_roles": [],
   *   "attributes": {},
   *   "static": false
   * }
   *
   * for role, response will look like:
   * {
   *   "reserved": true,
   *   "hidden": false,
   *   "description": "Allow full access to all indices and all cluster APIs",
   *   "cluster_permissions": ["*"],
   *   "index_permissions": [{
   *     "index_patterns": ["*"],
   *     "fls": [],
   *     "masked_fields": [],
   *     "allowed_actions": ["*"]
   *   }],
   *   "tenant_permissions": [{
   *     "tenant_patterns": ["*"],
   *     "allowed_actions": ["opensearch_dashboards_all_write"]
   *   }],
   *   "static": false
   * }
   *
   * for roles mapping, response will look like:
   * {
   *   "reserved": true,
   *   "hidden": false,
   *   "description": "Allow full access to all indices and all cluster APIs",
   *   "cluster_permissions": ["*"],
   *   "index_permissions": [{
   *     "index_patterns": ["*"],
   *     "fls": [],
   *     "masked_fields": [],
   *     "allowed_actions": ["*"]
   *   }],
   *   "tenant_permissions": [{
   *     "tenant_patterns": ["*"],
   *     "allowed_actions": ["opensearch_dashboards_all_write"]
   *   }],
   *   "static": false
   * }
   *
   * for action groups, response will look like:
   * {
   *   "reserved": true,
   *   "hidden": false,
   *   "allowed_actions": ["indices:data/read*", "indices:admin/mappings/fields/get*"],
   *   "type": "index",
   *   "description": "Allow all read operations",
   *   "static": false
   * }
   *
   * for tenant, response will look like:
   * {
   *   "reserved": true,
   *   "hidden": false,
   *   "description": "Global tenant",
   *   "static": false
   * },
   */
  router.get({
    path: `${_common.API_PREFIX}/${_common.CONFIGURATION_API_PREFIX}/{resourceName}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string(),
        id: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const esResp = await wrapRouteWithDataSource(dataSourceEnabled, context, request, 'opensearch_security.getResource', {
        resourceName: request.params.resourceName,
        id: request.params.id
      });
      return response.ok({
        body: esResp[request.params.id]
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Deletes an entity by id.
   */
  router.delete({
    path: `${_common.API_PREFIX}/${_common.CONFIGURATION_API_PREFIX}/{resourceName}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string(),
        id: _configSchema.schema.string({
          minLength: 1
        })
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const esResp = await wrapRouteWithDataSource(dataSourceEnabled, context, request, 'opensearch_security.deleteResource', {
        resourceName: request.params.resourceName,
        id: request.params.id
      });
      return response.ok({
        body: {
          message: esResp.message
        }
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Update object with out Id. Resource identification is expected to computed from headers. Eg: auth headers
   *
   * Request sample:
   * /configuration/account
   * {
   *   "password": "new-password",
   *   "current_password": "old-password"
   * }
   */
  router.post({
    path: `${_common.API_PREFIX}/${_common.CONFIGURATION_API_PREFIX}/{resourceName}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      validateRequestBody(request.params.resourceName, request.body);
    } catch (error) {
      return response.badRequest({
        body: error
      });
    }
    try {
      const esResp = await wrapRouteWithDataSource(dataSourceEnabled, context, request, 'opensearch_security.saveResourceWithoutId', {
        resourceName: request.params.resourceName,
        body: request.body
      });
      return response.ok({
        body: {
          message: esResp.message
        }
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Update entity by Id.
   */
  router.post({
    path: `${_common.API_PREFIX}/${_common.CONFIGURATION_API_PREFIX}/{resourceName}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string(),
        id: _configSchema.schema.string({
          validate: validateEntityId
        })
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      validateRequestBody(request.params.resourceName, request.body);
    } catch (error) {
      return response.badRequest({
        body: error
      });
    }
    try {
      const esResp = await wrapRouteWithDataSource(dataSourceEnabled, context, request, 'opensearch_security.saveResource', {
        resourceName: request.params.resourceName,
        id: request.params.id,
        body: request.body
      });
      return response.ok({
        body: {
          message: esResp.message
        }
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Gets authentication info of the user.
   *
   * The response looks like:
   * {
   *   "user": "User [name=admin, roles=[], requestedTenant=__user__]",
   *   "user_name": "admin",
   *   "user_requested_tenant": "__user__",
   *   "remote_address": "127.0.0.1:35044",
   *   "backend_roles": [],
   *   "custom_attribute_names": [],
   *   "roles": ["all_access", "security_manager"],
   *   "tenants": {
   *     "another_tenant": true,
   *     "admin": true,
   *     "global_tenant": true,
   *     "aaaaa": true,
   *     "test tenant": true
   *   },
   *   "principal": null,
   *   "peer_certificates": "0",
   *   "sso_logout_url": null
   * }
   */
  router.get({
    path: `${_common.API_PREFIX}/auth/authinfo`,
    validate: false
  }, async (context, request, response) => {
    const client = context.security_plugin.esClient.asScoped(request);
    let esResp;
    try {
      esResp = await client.callAsCurrentUser('opensearch_security.authinfo');
      return response.ok({
        body: esResp
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });
  router.get({
    path: `${_common.API_PREFIX}/auth/dashboardsinfo`,
    validate: false
  }, async (context, request, response) => {
    const client = context.security_plugin.esClient.asScoped(request);
    let esResp;
    try {
      esResp = await client.callAsCurrentUser('opensearch_security.dashboardsinfo');
      return response.ok({
        body: esResp
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Gets audit log configuration。
   *
   * Sample payload:
   * {
   *   "enabled":true,
   *   "audit":{
   *     "enable_rest":false,
   *     "disabled_rest_categories":[
   *       "FAILED_LOGIN",
   *       "AUTHENTICATED"
   *     ],
   *     "enable_transport":true,
   *     "disabled_transport_categories":[
   *       "GRANTED_PRIVILEGES"
   *     ],
   *     "resolve_bulk_requests":true,
   *     "log_request_body":false,
   *     "resolve_indices":true,
   *     "exclude_sensitive_headers":true,
   *     "ignore_users":[
   *       "admin",
   *     ],
   *     "ignore_requests":[
   *       "SearchRequest",
   *       "indices:data/read/*"
   *     ]
   *   },
   *   "compliance":{
   *     "enabled":true,
   *     "internal_config":false,
   *     "external_config":false,
   *     "read_metadata_only":false,
   *     "read_watched_fields":{
   *       "indexName1":[
   *         "field1",
   *         "fields-*"
   *       ]
   *     },
   *     "read_ignore_users":[
   *       "opensearchdashboardsserver",
   *       "operator/*"
   *     ],
   *     "write_metadata_only":false,
   *     "write_log_diffs":false,
   *     "write_watched_indices":[
   *       "indexName2",
   *       "indexPatterns-*"
   *     ],
   *     "write_ignore_users":[
   *       "admin"
   *     ]
   *   }
   * }
   */
  router.get({
    path: `${_common.API_PREFIX}/configuration/audit`,
    validate: {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const esResp = await wrapRouteWithDataSource(dataSourceEnabled, context, request, 'opensearch_security.getAudit');
      return response.ok({
        body: esResp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode,
        body: parseEsErrorResponse(error)
      });
    }
  });

  /**
   * Update audit log configuration。
   *
   * Sample payload:
   * {
   *   "enabled":true,
   *   "audit":{
   *     "enable_rest":false,
   *     "disabled_rest_categories":[
   *       "FAILED_LOGIN",
   *       "AUTHENTICATED"
   *     ],
   *     "enable_transport":true,
   *     "disabled_transport_categories":[
   *       "GRANTED_PRIVILEGES"
   *     ],
   *     "resolve_bulk_requests":true,
   *     "log_request_body":false,
   *     "resolve_indices":true,
   *     "exclude_sensitive_headers":true,
   *     "ignore_users":[
   *       "admin",
   *     ],
   *     "ignore_requests":[
   *       "SearchRequest",
   *       "indices:data/read/*"
   *     ]
   *   },
   *   "compliance":{
   *     "enabled":true,
   *     "internal_config":false,
   *     "external_config":false,
   *     "read_metadata_only":false,
   *     "read_watched_fields":{
   *       "indexName1":[
   *         "field1",
   *         "fields-*"
   *       ]
   *     },
   *     "read_ignore_users":[
   *       "kibanaserver",
   *       "operator/*"
   *     ],
   *     "write_metadata_only":false,
   *     "write_log_diffs":false,
   *     "write_watched_indices":[
   *       "indexName2",
   *       "indexPatterns-*"
   *     ],
   *     "write_ignore_users":[
   *       "admin"
   *     ]
   *   }
   * }
   */
  router.post({
    path: `${_common.API_PREFIX}/configuration/audit/config`,
    validate: {
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const esResp = await wrapRouteWithDataSource(dataSourceEnabled, context, request, 'opensearch_security.saveAudit', {
        body: request.body
      });
      return response.ok({
        body: {
          message: esResp.message
        }
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Deletes cache.
   *
   * Sample response: {"message":"Cache flushed successfully."}
   */
  router.delete({
    path: `${_common.API_PREFIX}/configuration/cache`,
    validate: {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const esResp = await wrapRouteWithDataSource(dataSourceEnabled, context, request, 'opensearch_security.clearCache');
      return response.ok({
        body: {
          message: esResp.message
        }
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Gets permission info of current user.
   *
   * Sample response:
   * {
   *   "user": "User [name=admin, roles=[], requestedTenant=__user__]",
   *   "user_name": "admin",
   *   "has_api_access": true,
   *   "disabled_endpoints": {}
   * }
   */
  router.get({
    path: `${_common.API_PREFIX}/restapiinfo`,
    validate: false
  }, async (context, request, response) => {
    const client = context.security_plugin.esClient.asScoped(request);
    try {
      const esResponse = await client.callAsCurrentUser('opensearch_security.restapiinfo');
      return response.ok({
        body: esResponse
      });
    } catch (error) {
      return response.badRequest({
        body: error
      });
    }
  });

  /**
   * Validates DLS (document level security) query.
   *
   * Request payload is an ES query.
   */
  router.post({
    path: `${_common.API_PREFIX}/${_common.CONFIGURATION_API_PREFIX}/validatedls/{indexName}`,
    validate: {
      params: _configSchema.schema.object({
        // in legacy plugin implmentation, indexName is not used when calling ES API.
        indexName: _configSchema.schema.maybe(_configSchema.schema.string())
      }),
      body: _configSchema.schema.any()
    }
  }, async (context, request, response) => {
    const client = context.security_plugin.esClient.asScoped(request);
    try {
      const esResponse = await client.callAsCurrentUser('opensearch_security.validateDls', {
        body: request.body
      });
      return response.ok({
        body: esResponse
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Gets index mapping.
   *
   * Calling ES _mapping API under the hood. see
   * https://www.elastic.co/guide/en/elasticsearch/reference/current/indices-get-mapping.html
   */
  router.post({
    path: `${_common.API_PREFIX}/${_common.CONFIGURATION_API_PREFIX}/index_mappings`,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const client = context.security_plugin.esClient.asScoped(request);
    try {
      const esResponse = await client.callAsCurrentUser('opensearch_security.getIndexMappings', {
        index: request.body.index.join(','),
        ignore_unavailable: true,
        allow_no_indices: true
      });
      return response.ok({
        body: esResponse
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Gets all indices, and field mappings.
   *
   * Calls ES API '/_all/_mapping/field/*' under the hood. see
   * https://www.elastic.co/guide/en/elasticsearch/reference/current/indices-get-mapping.html
   */
  router.get({
    path: `${_common.API_PREFIX}/${_common.CONFIGURATION_API_PREFIX}/indices`,
    validate: false
  }, async (context, request, response) => {
    const client = context.security_plugin.esClient.asScoped(request);
    try {
      const esResponse = await client.callAsCurrentUser('opensearch_security.indices');
      return response.ok({
        body: esResponse
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });
}

/**
 * A helper function to wrap API calls with the appropriate client. If the multiple datasources feature is disabled, it will use
 * the existing client provided by the security plugin. Otherwise, it will use the one provided by the datasource plugin based on the id
 * that we extract via the UI.
 */
const wrapRouteWithDataSource = async (dataSourceEnabled, context, request, endpoint, body) => {
  var _request$query;
  if (!dataSourceEnabled || !((_request$query = request.query) !== null && _request$query !== void 0 && _request$query.dataSourceId)) {
    const client = context.security_plugin.esClient.asScoped(request);
    return await client.callAsCurrentUser(endpoint, body);
  } else {
    var _request$query2;
    const client = context.dataSource.opensearch.legacy.getClient((_request$query2 = request.query) === null || _request$query2 === void 0 ? void 0 : _request$query2.dataSourceId);
    return await client.callAPI(endpoint, body);
  }
};
function parseEsErrorResponse(error) {
  if (error.response) {
    try {
      const esErrorResponse = JSON.parse(error.response);
      return esErrorResponse.reason || error.response;
    } catch (parsingError) {
      return error.response;
    }
  }
  return error.message;
}
function errorResponse(response, error) {
  return response.custom({
    statusCode: error.statusCode,
    body: parseEsErrorResponse(error)
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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