"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.metaData = exports.getSelectedFields = exports.getOpenSearchData = exports.convertToExcel = exports.convertToCSV = exports.buildRequestBody = void 0;
var _elasticBuilder = _interopRequireDefault(require("elastic-builder"));
var _json2Csv = _interopRequireDefault(require("json-2-csv"));
var _lodash = _interopRequireDefault(require("lodash"));
var _momentTimezone = _interopRequireDefault(require("moment-timezone"));
var _common = require("../../../../../src/plugins/data/common");
var _excelBuilder = require("./excelBuilder");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

var metaData = exports.metaData = {
  saved_search_id: null,
  report_format: null,
  start: null,
  end: null,
  fields: null,
  type: null,
  timeFieldName: null,
  sorting: null,
  fields_exist: false,
  selectedFields: [],
  paternName: null,
  searchSourceJSON: [],
  dateFields: []
};

// Get the selected columns by the user.
const getSelectedFields = async columns => {
  const selectedFields = [];
  let fields_exist = false;
  for (let column of columns) {
    if (column !== '_source') {
      fields_exist = true;
      selectedFields.push(column);
    } else {
      fields_exist = false;
      selectedFields.push('_source');
    }
  }
  metaData.fields_exist = fields_exist;
  metaData.selectedFields = selectedFields;
};

// Build the OpenSearch query from the meta data
// is_count is set to 1 if we building the count query but 0 if we building the fetch data query
exports.getSelectedFields = getSelectedFields;
const buildRequestBody = (report, allowLeadingWildcards, is_count) => {
  let esbBoolQuery = _elasticBuilder.default.boolQuery();
  const searchSourceJSON = report._source.searchSourceJSON;
  const savedObjectQuery = JSON.parse(searchSourceJSON).query;
  const savedObjectFilter = JSON.parse(searchSourceJSON).filter;
  const savedObjectConfig = {
    allowLeadingWildcards: allowLeadingWildcards,
    queryStringOptions: {},
    ignoreFilterIfFieldNotInIndex: false
  };
  const QueryFromSavedObject = (0, _common.buildOpenSearchQuery)(undefined, savedObjectQuery, savedObjectFilter, savedObjectConfig);
  // Add time range
  if (report._source.timeFieldName && report._source.timeFieldName.length > 0) {
    esbBoolQuery.must(_elasticBuilder.default.rangeQuery(report._source.timeFieldName).format('epoch_millis').gte(report._source.start - 1).lte(report._source.end + 1));
  }
  if (is_count) {
    return _elasticBuilder.default.requestBodySearch().query(esbBoolQuery);
  }

  // Add sorting to the query
  let esbSearchQuery = _elasticBuilder.default.requestBodySearch().query(esbBoolQuery).version(true);
  let sorting = report._source.sorting;

  // We expect a list of [field, order] pairs for sorting. In some migration paths, though it's not
  // clear why, this list can get unnested in the case of one sort, [["field", "asc"]] becomes
  // ["field", "asc"]. The true root cause remains a mystery, so we work around it.
  // See: https://github.com/opensearch-project/dashboards-reporting/issues/371
  if (sorting.length > 0 && typeof sorting[0] === "string") {
    sorting = [sorting];
  }
  if (sorting.length > 0) {
    const sorts = sorting.map(element => {
      return _elasticBuilder.default.sort(element[0], element[1]);
    });
    esbSearchQuery.sorts(sorts);
  }

  // add selected fields to query
  if (report._source.fields_exist) {
    esbSearchQuery.source({
      includes: report._source.selectedFields
    });
  }
  // Add a customizer to merge queries to generate request body
  let requestBody = _lodash.default.mergeWith({
    query: QueryFromSavedObject
  }, esbSearchQuery.toJSON(), (objValue, srcValue) => {
    if (_lodash.default.isArray(objValue)) {
      return objValue.concat(srcValue);
    }
  });
  requestBody = addDocValueFields(report, requestBody);
  return requestBody;
};

// Fetch the data from OpenSearch
exports.buildRequestBody = buildRequestBody;
const getOpenSearchData = (arrayHits, report, params, dateFormat, timezone) => {
  let hits = [];
  for (let valueRes of arrayHits) {
    for (let data of valueRes.hits) {
      const fields = data.fields;
      // get all the fields of type date and format them to excel format
      let tempKeyElement = [];
      for (let dateField of report._source.dateFields) {
        let keys;
        keys = dateField.split('.');
        const dateValue = data._source[dateField];
        const fieldDateValue = fields !== undefined ? fields[dateField] : undefined;
        const isDateFieldPresent = isKeyPresent(data._source, dateField);
        if (isDateFieldPresent) {
          // if its not a nested date field
          if (keys.length === 1) {
            // if conditions to determine if the date field's value is an array or a string
            if (typeof dateValue === 'string') {
              data._source[keys] = _momentTimezone.default.utc(dateValue).tz(timezone).format(dateFormat);
            } else if (dateValue.length !== 0 && dateValue instanceof Array && fieldDateValue !== undefined) {
              fieldDateValue.forEach((element, index) => {
                data._source[keys][index] = _momentTimezone.default.utc(element).tz(timezone).format(dateFormat);
              });
            } else {
              data._source[keys] = [];
            }
            // else to cover cases with nested date fields
          } else {
            let keyElement = keys.shift();
            // if conditions to determine if the date field's value is an array or a string
            if (fieldDateValue !== undefined && typeof fieldDateValue === 'string') {
              keys.push(_momentTimezone.default.utc(fieldDateValue).tz(timezone).format(dateFormat));
            } else if (dateValue.length !== 0 && dateValue instanceof Array && fieldDateValue !== undefined) {
              let tempArray = [];
              fieldDateValue.forEach(index => {
                tempArray.push(_momentTimezone.default.utc(index).tz(timezone).format(dateFormat));
              });
              keys.push(tempArray);
            } else {
              keys.push([]);
            }
            const nestedJSON = arrayToNestedJSON(keys);
            let keyLength = Object.keys(data._source);
            // to check if the nested field have anyother keys apart from date field
            if (tempKeyElement.includes(keyElement) || keyLength.length > 1) {
              data._source[keyElement] = {
                ...data._source[keyElement],
                ...nestedJSON
              };
            } else {
              data._source[keyElement] = nestedJSON;
              tempKeyElement.push(keyElement);
            }
          }
        }
      }
      delete data['fields'];
      if (report._source.fields_exist === true) {
        let result = traverse(data, report._source.selectedFields);
        hits.push(params.excel ? sanitize(result) : result);
      } else {
        hits.push(params.excel ? sanitize(data) : data);
      }
      // Truncate to expected limit size
      if (hits.length >= params.limit) {
        return hits;
      }
    }
  }
  return hits;
};

// Convert the data to Csv format
exports.getOpenSearchData = getOpenSearchData;
const convertToCSV = async (dataset, csvSeparator) => {
  let convertedData = [];
  const options = {
    delimiter: {
      field: csvSeparator,
      eol: '\n'
    },
    emptyFieldValue: ' '
  };
  await _json2Csv.default.json2csvAsync(dataset[0], options).then(csv => {
    convertedData = csv;
  });
  return convertedData;
};
exports.convertToCSV = convertToCSV;
function flattenHits(hits, result = {}, prefix = '') {
  Object.entries(hits).forEach(([key, value]) => {
    if (value !== null && typeof value === 'object' && !Array.isArray(value) && Object.keys(value).length > 0) {
      flattenHits(value, result, `${prefix}${key}.`);
    } else {
      result[`${prefix.replace(/^_source\./, '')}${key}`] = value;
    }
  });
  return result;
}
function flattenObject(obj = {}, parentKey = '', result = {}) {
  for (const [key, value] of Object.entries(obj)) {
    const newKey = parentKey ? `${parentKey}.${key}` : key;
    if (typeof value == 'object' && value !== null && !Array.isArray(value) && Object.keys(value).length > 0) {
      flattenObject(value, newKey, result);
    } else if (Array.isArray(value)) {
      result[newKey] = JSON.stringify(value);
    } else {
      result[newKey] = value;
    }
  }
  return result;
}
function flattenArray(array = []) {
  return array.map(item => flattenObject(item));
}
const convertToExcel = async dataset => {
  const flatDataset = flattenArray(dataset[0]);
  const excelBuilder = new _excelBuilder.ExcelBuilder();
  const base64 = await excelBuilder.addHeaders(flatDataset).addRows(flatDataset).updateColumnWidths().applyHeaderStyles().applyAutoFilter().applyFreeze().getWorkbookAsBase64();
  return 'data:application/vnd.openxmlformats-officedocument.spreadsheetml.sheet;base64,' + base64;
};

//Return only the selected fields
exports.convertToExcel = convertToExcel;
function traverse(data, keys, result = {}) {
  // Flatten the data if necessary (ensure all nested fields are at the top level)
  data = flattenHits(data);
  keys.forEach(key => {
    const value = _lodash.default.get(data, key, undefined);
    if (value !== undefined) {
      result[key] = value;
    } else {
      const flattenedValues = {};
      Object.keys(data).forEach(dataKey => {
        if (dataKey.startsWith(key + '.')) {
          result[dataKey] = data[dataKey];
        }
        const arrayValue = data[dataKey];
        if (Array.isArray(arrayValue)) {
          arrayValue.forEach(item => {
            if (typeof item === 'object' && item !== null) {
              Object.keys(item).forEach(subKey => {
                const newKey = `${dataKey}.${subKey}`;
                if (!flattenedValues[newKey]) {
                  flattenedValues[newKey] = [];
                }
                flattenedValues[newKey].push(item[subKey]);
              });
            }
          });
        }
      });
      Object.keys(flattenedValues).forEach(newKey => {
        result[newKey] = flattenedValues[newKey];
      });
    }
  });
  return result;
}

/**
 * Escape special characters if field value prefixed with.
 * This is intend to avoid CSV injection in Microsoft Excel.
 * @param doc   document
 */
function sanitize(doc) {
  for (const field in doc) {
    if (doc[field] == null) continue;
    if (doc[field].toString().startsWith('+') || doc[field].toString().startsWith('-') && typeof doc[field] !== 'number' || doc[field].toString().startsWith('=') || doc[field].toString().startsWith('@')) {
      doc[field] = "'" + doc[field];
    }
  }
  return doc;
}
function arrayToNestedJSON(arr) {
  if (arr.length === 0) {
    return null;
  } else if (arr.length === 1) {
    return arr[0];
  } else {
    const key = arr[0];
    const rest = arr.slice(1);
    return {
      [key]: arrayToNestedJSON(rest)
    };
  }
}
function isKeyPresent(data, key) {
  if (typeof data === 'object' && data !== null) {
    if (key in data) {
      return true;
    }
    for (const value of Object.values(data)) {
      if (isKeyPresent(value, key)) {
        return true;
      }
    }
  }
  return false;
}
const addDocValueFields = (report, requestBody) => {
  const docValues = [];
  for (const dateType of report._source.dateFields) {
    docValues.push({
      field: dateType,
      format: 'date_hour_minute_second_fraction'
    });
  }
  // elastic-builder doesn't provide function to build docvalue_fields with format,
  // this is a workaround which appends docvalues field to the request body.
  requestBody = {
    ...requestBody,
    docvalue_fields: docValues
  };
  return requestBody;
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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