// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { needs, KeyringTraceFlag, immutableClass, readOnlyProperty, unwrapDataKey, } from '@aws-crypto/material-management';
import { KMS_PROVIDER_ID, generateDataKey, encrypt, decrypt, kmsResponseToEncryptedDataKey, } from './helpers';
import { validAwsKmsIdentifier, parseAwsKmsKeyArn } from './arn_parsing';
export function KmsKeyringClass(BaseKeyring) {
    class KmsKeyring extends BaseKeyring {
        constructor({ clientProvider, generatorKeyId, keyIds = [], grantTokens, discovery, discoveryFilter, }) {
            super();
            /* Precondition: This is an abstract class. (But TypeScript does not have a clean way to model this) */
            needs(this.constructor !== KmsKeyring, 'new KmsKeyring is not allowed');
            /* Precondition: A noop KmsKeyring is not allowed. */
            needs(!(!discovery && !generatorKeyId && !keyIds.length), 'Noop keyring is not allowed: Set a keyId or discovery');
            /* Precondition: A keyring can be either a Discovery or have keyIds configured. */
            needs(!(discovery && (generatorKeyId || keyIds.length)), 'A keyring can be either a Discovery or have keyIds configured.');
            /* Precondition: Discovery filter can only be configured in discovery mode. */
            needs(discovery || (!discovery && !discoveryFilter), 'Account and partition decrypt filtering are only supported when discovery === true');
            /* Precondition: A Discovery filter *must* be able to match something.
             * I am not going to wait to tell you
             * that no CMK can match an empty account list.
             * e.g. [], [''], '' are not valid.
             */
            needs(!discoveryFilter ||
                (discoveryFilter.accountIDs &&
                    discoveryFilter.accountIDs.length &&
                    !!discoveryFilter.partition &&
                    discoveryFilter.accountIDs.every((a) => typeof a === 'string' && !!a)), 'A discovery filter must be able to match something.');
            /* Precondition: All KMS key identifiers must be valid. */
            needs(!generatorKeyId || validAwsKmsIdentifier(generatorKeyId), 'Malformed arn.');
            needs(keyIds.every((keyArn) => validAwsKmsIdentifier(keyArn)), 'Malformed arn.');
            /* Precondition: clientProvider needs to be a callable function. */
            needs(typeof clientProvider === 'function', 'Missing clientProvider');
            readOnlyProperty(this, 'clientProvider', clientProvider);
            readOnlyProperty(this, 'keyIds', Object.freeze(keyIds.slice()));
            readOnlyProperty(this, 'generatorKeyId', generatorKeyId);
            readOnlyProperty(this, 'grantTokens', grantTokens);
            readOnlyProperty(this, 'isDiscovery', !!discovery);
            readOnlyProperty(this, 'discoveryFilter', discoveryFilter
                ? Object.freeze({
                    ...discoveryFilter,
                    accountIDs: Object.freeze(discoveryFilter.accountIDs),
                })
                : discoveryFilter);
        }
        /* Keyrings *must* preserve the order of EDK's.  The generatorKeyId is the first on this list. */
        async _onEncrypt(material) {
            /* Check for early return (Postcondition): Discovery Keyrings do not encrypt. */
            if (this.isDiscovery)
                return material;
            const keyIds = this.keyIds.slice();
            const { clientProvider, generatorKeyId, grantTokens } = this;
            if (generatorKeyId && !material.hasUnencryptedDataKey) {
                const dataKey = await generateDataKey(clientProvider, material.suite.keyLengthBytes, generatorKeyId, material.encryptionContext, grantTokens);
                /* Precondition: A generatorKeyId must generate if we do not have an unencrypted data key.
                 * Client supplier is allowed to return undefined if, for example, user wants to exclude particular
                 * regions. But if we are here it means that user configured keyring with a KMS key that was
                 * incompatible with the client supplier in use.
                 */
                if (!dataKey)
                    throw new Error('Generator KMS key did not generate a data key');
                const flags = KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY |
                    KeyringTraceFlag.WRAPPING_KEY_SIGNED_ENC_CTX |
                    KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY;
                const trace = {
                    keyNamespace: KMS_PROVIDER_ID,
                    keyName: dataKey.KeyId,
                    flags,
                };
                material
                    /* Postcondition: The generated unencryptedDataKey length must match the algorithm specification.
                     * See cryptographic_materials as setUnencryptedDataKey will throw in this case.
                     */
                    .setUnencryptedDataKey(dataKey.Plaintext, trace)
                    .addEncryptedDataKey(kmsResponseToEncryptedDataKey(dataKey), KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY |
                    KeyringTraceFlag.WRAPPING_KEY_SIGNED_ENC_CTX);
            }
            else if (generatorKeyId) {
                keyIds.unshift(generatorKeyId);
            }
            /* Precondition: If a generator does not exist, an unencryptedDataKey *must* already exist.
             * Furthermore *only* CMK's explicitly designated as generators can generate data keys.
             * See cryptographic_materials as getUnencryptedDataKey will throw in this case.
             */
            const unencryptedDataKey = unwrapDataKey(material.getUnencryptedDataKey());
            const flags = KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY |
                KeyringTraceFlag.WRAPPING_KEY_SIGNED_ENC_CTX;
            for (const kmsKey of keyIds) {
                const kmsEDK = await encrypt(clientProvider, unencryptedDataKey, kmsKey, material.encryptionContext, grantTokens);
                /* clientProvider may not return a client, in this case there is not an EDK to add */
                if (kmsEDK)
                    material.addEncryptedDataKey(kmsResponseToEncryptedDataKey(kmsEDK), flags);
            }
            return material;
        }
        async _onDecrypt(material, encryptedDataKeys) {
            const keyIds = this.keyIds.slice();
            const { clientProvider, generatorKeyId, grantTokens } = this;
            if (generatorKeyId)
                keyIds.unshift(generatorKeyId);
            /* If there are no key IDs in the list, keyring is in "discovery" mode and will attempt KMS calls with
             * every ARN it comes across in the message. If there are key IDs in the list, it will cross check the
             * ARN it reads with that list before attempting KMS calls. Note that if caller provided key IDs in
             * anything other than a CMK ARN format, the Encryption SDK will not attempt to decrypt those data keys, because
             * the EDK data format always specifies the CMK with the full (non-alias) ARN.
             */
            const decryptableEDKs = encryptedDataKeys.filter(filterEDKs(keyIds, this));
            const cmkErrors = [];
            for (const edk of decryptableEDKs) {
                let dataKey = false;
                try {
                    dataKey = await decrypt(clientProvider, edk, material.encryptionContext, grantTokens);
                }
                catch (e) {
                    /* Failures onDecrypt should not short-circuit the process
                     * If the caller does not have access they may have access
                     * through another Keyring.
                     */
                    cmkErrors.push({ errPlus: e });
                }
                /* Check for early return (Postcondition): clientProvider may not return a client. */
                if (!dataKey)
                    continue;
                /* Postcondition: The KeyId from KMS must match the encoded KeyID. */
                needs(dataKey.KeyId === edk.providerInfo, 'KMS Decryption key does not match the requested key id.');
                const flags = KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY |
                    KeyringTraceFlag.WRAPPING_KEY_VERIFIED_ENC_CTX;
                const trace = {
                    keyNamespace: KMS_PROVIDER_ID,
                    keyName: dataKey.KeyId,
                    flags,
                };
                /* Postcondition: The decrypted unencryptedDataKey length must match the algorithm specification.
                 * See cryptographic_materials as setUnencryptedDataKey will throw in this case.
                 */
                material.setUnencryptedDataKey(dataKey.Plaintext, trace);
                return material;
            }
            /* Postcondition: A CMK must provide a valid data key or KMS must not have raised any errors.
             * If I have a data key,
             * decrypt errors can be ignored.
             * However, if I was unable to decrypt a data key AND I have errors,
             * these errors should bubble up.
             * Otherwise, the only error customers will see is that
             * the material does not have an unencrypted data key.
             * So I return a concatenated Error message
             */
            needs(material.hasValidKey() ||
                (!material.hasValidKey() && !cmkErrors.length), cmkErrors.reduce((m, e, i) => `${m} Error #${i + 1} \n ${e.errPlus.stack} \n`, 'Unable to decrypt data key and one or more KMS CMKs had an error. \n '));
            return material;
        }
    }
    immutableClass(KmsKeyring);
    return KmsKeyring;
}
function filterEDKs(keyIds, { isDiscovery, discoveryFilter }) {
    return function filter({ providerId, providerInfo }) {
        /* Check for early return (Postcondition): Only AWS KMS EDK should be attempted. */
        if (providerId !== KMS_PROVIDER_ID)
            return false;
        /* Discovery keyrings can not have keyIds configured,
         * and non-discovery keyrings must have keyIds configured.
         */
        if (isDiscovery) {
            /* Check for early return (Postcondition): There is no discoveryFilter to further condition discovery. */
            if (!discoveryFilter)
                return true;
            const parsedArn = parseAwsKmsKeyArn(providerInfo);
            /* Postcondition: Provider info is a well formed AWS KMS ARN. */
            needs(parsedArn, 'Malformed arn in provider info.');
            /* If the providerInfo is an invalid ARN this will throw.
             * But, this function is also used to extract regions
             * from an CMK to generate a regional client.
             * This means it will NOT throw for something
             * that looks like a bare alias or key id.
             * However, these constructions will not have an account or partition.
             */
            const { AccountId, Partition } = parsedArn;
            /* Postcondition: The account and partition *must* match the discovery filter.
             * Since we are offering a runtime discovery of CMKs
             * it is best to have some form of filter on this.
             * The providerInfo is a CMK ARN and will have the account and partition.
             * By offering these levers customers can easily bound
             * the CMKs to ones they control without knowing the CMKs
             * when the AWS KMS Keyring is instantiated.
             */
            return (discoveryFilter.partition === Partition &&
                discoveryFilter.accountIDs.some((a) => a === AccountId));
        }
        else {
            /* Postcondition: The EDK CMK (providerInfo) *must* match a configured CMK. */
            return keyIds.includes(providerInfo);
        }
    };
}
//# sourceMappingURL=data:application/json;base64,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