"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenIdAuthentication = void 0;
var fs = _interopRequireWildcard(require("fs"));
var _wreck = _interopRequireDefault(require("@hapi/wreck"));
var _proxyAgent = require("proxy-agent");
var _security_cookie = require("../../../session/security_cookie");
var _routes = require("./routes");
var _authentication_type = require("../authentication_type");
var _helper = require("./helper");
var _object_properties_defined = require("../../../utils/object_properties_defined");
var _common = require("../../../../common");
var _cookie_splitter = require("../../../session/cookie_splitter");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class OpenIdAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, core, logger) {
    var _this$config$openid, _this$config$openid2;
    super(config, sessionStorageFactory, router, esClient, core, logger);
    _defineProperty(this, "type", _common.AuthType.OPEN_ID);
    _defineProperty(this, "openIdAuthConfig", void 0);
    _defineProperty(this, "authHeaderName", void 0);
    _defineProperty(this, "openIdConnectUrl", void 0);
    _defineProperty(this, "wreckClient", void 0);
    _defineProperty(this, "wreckHttpsOption", {});
    _defineProperty(this, "redirectOIDCCapture", (request, toolkit) => {
      const nextUrl = this.generateNextUrl(request);
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return toolkit.redirected({
        location: `${this.coreSetup.http.basePath.serverBasePath}/auth/openid/captureUrlFragment?nextUrl=${nextUrl}`,
        'set-cookie': clearOldVersionCookie
      });
    });
    this.wreckClient = this.createWreckClient();
    this.openIdAuthConfig = {};
    this.authHeaderName = ((_this$config$openid = this.config.openid) === null || _this$config$openid === void 0 ? void 0 : _this$config$openid.header) || '';
    this.openIdAuthConfig.authHeaderName = this.authHeaderName;
    this.openIdConnectUrl = ((_this$config$openid2 = this.config.openid) === null || _this$config$openid2 === void 0 ? void 0 : _this$config$openid2.connect_url) || '';
    let scope = this.config.openid.scope;
    if (scope.indexOf('openid') < 0) {
      scope = `openid ${scope}`;
    }
    this.openIdAuthConfig.scope = scope;
  }
  async init() {
    try {
      const response = await this.wreckClient.get(this.openIdConnectUrl);
      const payload = JSON.parse(response.payload);
      this.openIdAuthConfig.authorizationEndpoint = payload.authorization_endpoint;
      this.openIdAuthConfig.tokenEndpoint = payload.token_endpoint;
      this.openIdAuthConfig.endSessionEndpoint = payload.end_session_endpoint || undefined;
      this.createExtraStorage();
      const routes = new _routes.OpenIdAuthRoutes(this.router, this.config, this.sessionStorageFactory, this.openIdAuthConfig, this.securityClient, this.coreSetup, this.wreckClient);
      routes.setupRoutes();
    } catch (error) {
      this.logger.error(error); // TODO: log more info
      throw new Error('Failed when trying to obtain the endpoints from your IdP');
    }
  }
  generateNextUrl(request) {
    const path = this.coreSetup.http.basePath.serverBasePath + (request.url.pathname || '/app/opensearch-dashboards');
    return escape(path);
  }
  createWreckClient() {
    var _this$config$openid3, _this$config$openid4, _this$config$openid5, _this$config$openid6, _this$config$openid7, _this$config$openid9;
    if ((_this$config$openid3 = this.config.openid) !== null && _this$config$openid3 !== void 0 && _this$config$openid3.root_ca) {
      this.wreckHttpsOption.ca = [fs.readFileSync(this.config.openid.root_ca)];
      this.logger.debug(`Using CA Cert: ${this.config.openid.root_ca}`);
    }
    if ((_this$config$openid4 = this.config.openid) !== null && _this$config$openid4 !== void 0 && _this$config$openid4.pfx) {
      // Use PFX or PKCS12 if provided
      this.logger.debug(`Using PFX or PKCS12: ${this.config.openid.pfx}`);
      this.wreckHttpsOption.pfx = [fs.readFileSync(this.config.openid.pfx)];
    } else if ((_this$config$openid5 = this.config.openid) !== null && _this$config$openid5 !== void 0 && _this$config$openid5.certificate && (_this$config$openid6 = this.config.openid) !== null && _this$config$openid6 !== void 0 && _this$config$openid6.private_key) {
      // Use 'certificate' and 'private_key' if provided
      this.logger.debug(`Using Certificate: ${this.config.openid.certificate}`);
      this.logger.debug(`Using Private Key: ${this.config.openid.private_key}`);
      this.wreckHttpsOption.cert = [fs.readFileSync(this.config.openid.certificate)];
      this.wreckHttpsOption.key = [fs.readFileSync(this.config.openid.private_key)];
    } else {
      this.logger.debug(`Client certificates not provided. Mutual TLS will not be used to obtain endpoints.`);
    }
    // Check if passphrase is provided, use it for 'pfx' and 'key'
    if (((_this$config$openid7 = this.config.openid) === null || _this$config$openid7 === void 0 ? void 0 : _this$config$openid7.passphrase) !== '') {
      var _this$config$openid8;
      this.logger.debug(`Passphrase not provided for private key and/or pfx.`);
      this.wreckHttpsOption.passphrase = (_this$config$openid8 = this.config.openid) === null || _this$config$openid8 === void 0 ? void 0 : _this$config$openid8.passphrase;
    }
    if (((_this$config$openid9 = this.config.openid) === null || _this$config$openid9 === void 0 ? void 0 : _this$config$openid9.verify_hostnames) === false) {
      this.logger.debug(`openId auth 'verify_hostnames' option is off.`);
      this.wreckHttpsOption.checkServerIdentity = (host, cert) => {
        return undefined;
      };
    }
    this.logger.info((0, _object_properties_defined.getObjectProperties)(this.wreckHttpsOption, 'WreckHttpsOptions'));

    // Use proxy agent to allow usage of e.g. http_proxy environment variable
    const httpAgent = new _proxyAgent.ProxyAgent();
    const httpsAllowUnauthorizedAgent = new _proxyAgent.ProxyAgent({
      rejectUnauthorized: false
    });
    let httpsAgent = new _proxyAgent.ProxyAgent();
    if (Object.keys(this.wreckHttpsOption).length > 0) {
      httpsAgent = new _proxyAgent.ProxyAgent(this.wreckHttpsOption);
    }
    return _wreck.default.defaults({
      agents: {
        http: httpAgent,
        https: httpsAgent,
        httpsAllowUnauthorized: httpsAllowUnauthorizedAgent
      }
    });
  }
  getWreckHttpsOptions() {
    return this.wreckHttpsOption;
  }
  createExtraStorage() {
    // @ts-ignore
    const hapiServer = this.sessionStorageFactory.asScoped({}).server;
    const extraCookiePrefix = this.config.openid.extra_storage.cookie_prefix;
    const extraCookieSettings = {
      isSecure: this.config.cookie.secure,
      isSameSite: this.config.cookie.isSameSite,
      password: this.config.cookie.password,
      domain: this.config.cookie.domain,
      path: this.coreSetup.http.basePath.serverBasePath || '/',
      clearInvalid: false,
      isHttpOnly: true,
      ignoreErrors: true,
      encoding: 'iron' // Same as hapi auth cookie
    };

    for (let i = 1; i <= this.config.openid.extra_storage.additional_cookies; i++) {
      hapiServer.states.add(extraCookiePrefix + i, extraCookieSettings);
    }
  }
  getExtraAuthStorageOptions() {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.openid.extra_storage.cookie_prefix,
      additionalCookies: this.config.openid.extra_storage.additional_cookies,
      logger: this.logger
    };
  }
  requestIncludesAuthInfo(request) {
    return request.headers.authorization ? true : false;
  }
  async getAdditionalAuthHeader(request) {
    return {};
  }
  getCookie(request, authInfo) {
    (0, _cookie_splitter.setExtraAuthStorage)(request, request.headers.authorization, this.getExtraAuthStorageOptions());
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValueExtra: true
      },
      authType: this.type,
      expiryTime: Date.now() + this.config.session.ttl
    };
  }
  getKeepAliveExpiry(cookie, request) {
    return Date.now() + this.config.session.ttl;
  }

  // TODO: Add token expiration check here
  async isValidCookie(cookie, request) {
    var _cookie$credentials;
    if (cookie.authType !== this.type || !cookie.username || !cookie.expiryTime || !((_cookie$credentials = cookie.credentials) !== null && _cookie$credentials !== void 0 && _cookie$credentials.authHeaderValue) && !this.getExtraAuthStorageValue(request, cookie)) {
      return false;
    }
    if (cookie.credentials.expiryTime > Date.now()) {
      return true;
    }

    // need to renew id token
    if (cookie.credentials.refresh_token) {
      try {
        var _this$config$openid10, _this$config$openid11;
        const query = {
          grant_type: 'refresh_token',
          client_id: (_this$config$openid10 = this.config.openid) === null || _this$config$openid10 === void 0 ? void 0 : _this$config$openid10.client_id,
          client_secret: (_this$config$openid11 = this.config.openid) === null || _this$config$openid11 === void 0 ? void 0 : _this$config$openid11.client_secret,
          refresh_token: cookie.credentials.refresh_token
        };
        const refreshTokenResponse = await (0, _helper.callTokenEndpoint)(this.openIdAuthConfig.tokenEndpoint, query, this.wreckClient);

        // if no id_token from refresh token call, maybe the Idp doesn't allow refresh id_token
        if (refreshTokenResponse.idToken) {
          cookie.credentials = {
            authHeaderValueExtra: true,
            refresh_token: refreshTokenResponse.refreshToken,
            expiryTime: (0, _helper.getExpirationDate)(refreshTokenResponse)
          };
          (0, _cookie_splitter.setExtraAuthStorage)(request, `Bearer ${refreshTokenResponse.idToken}`, this.getExtraAuthStorageOptions());
          return true;
        } else {
          return false;
        }
      } catch (error) {
        this.logger.error(error);
        return false;
      }
    } else {
      // no refresh token, and current token is expired
      return false;
    }
  }
  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      return this.redirectOIDCCapture(request, toolkit);
    } else {
      return response.unauthorized();
    }
  }
  getExtraAuthStorageValue(request, cookie) {
    var _cookie$credentials2;
    let extraValue = '';
    if (!((_cookie$credentials2 = cookie.credentials) !== null && _cookie$credentials2 !== void 0 && _cookie$credentials2.authHeaderValueExtra)) {
      return extraValue;
    }
    try {
      extraValue = (0, _cookie_splitter.getExtraAuthStorageValue)(request, this.getExtraAuthStorageOptions());
    } catch (error) {
      this.logger.info(error);
    }
    return extraValue;
  }
  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$credentials3;
    const header = {};
    if (cookie.credentials.authHeaderValueExtra) {
      try {
        const extraAuthStorageValue = this.getExtraAuthStorageValue(request, cookie);
        header.authorization = extraAuthStorageValue;
        return header;
      } catch (error) {
        this.logger.error(error);
        // TODO Re-throw?
        // throw error;
      }
    }

    const authHeaderValue = (_cookie$credentials3 = cookie.credentials) === null || _cookie$credentials3 === void 0 ? void 0 : _cookie$credentials3.authHeaderValue;
    if (authHeaderValue) {
      header.authorization = authHeaderValue;
    }
    return header;
  }
}
exports.OpenIdAuthentication = OpenIdAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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