"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.visualReportSchema = exports.triggerSchema = exports.scheduleSchema = exports.reportSchema = exports.reportParamsSchema = exports.reportDefinitionSchema = exports.opensearchDashboardsUserSchema = exports.intervalSchema = exports.deliverySchema = exports.dataReportSchema = exports.cronSchema = exports.channelSchema = void 0;
var _configSchema = require("@osd/config-schema");
var _validationHelper = require("../utils/validationHelper");
var _cronValidator = require("cron-validator");
var _constants = require("../routes/utils/constants");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const dataReportSchema = exports.dataReportSchema = _configSchema.schema.object({
  // Need this to build the links in email
  origin: _configSchema.schema.uri(),
  //e.g. https://xxxxx.com
  base_url: _configSchema.schema.string({
    validate(value) {
      if (!(0, _validationHelper.isValidRelativeUrl)(value)) {
        return `invalid relative url: ${value}`;
      }
    },
    minLength: 1
  }),
  saved_search_id: _configSchema.schema.string(),
  //ISO duration format. 'PT10M' means 10 min
  time_duration: _configSchema.schema.string({
    validate(value) {
      if (!_validationHelper.regexDuration.test(value)) {
        return `invalid time duration: ${value}`;
      }
    }
  }),
  report_format: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.FORMAT.csv), _configSchema.schema.literal(_constants.FORMAT.xlsx)]),
  limit: _configSchema.schema.number({
    defaultValue: _constants.DEFAULT_MAX_SIZE,
    min: 0
  }),
  excel: _configSchema.schema.boolean({
    defaultValue: true
  })
});
const visualReportSchema = exports.visualReportSchema = _configSchema.schema.object({
  // Need this to build the links in email
  origin: _configSchema.schema.uri(),
  //e.g. https://xxxxx.com
  base_url: _configSchema.schema.string({
    validate(value) {
      if (!(0, _validationHelper.isValidRelativeUrl)(value)) {
        return `invalid relative url: ${value}`;
      }
    },
    minLength: 1
  }),
  window_width: _configSchema.schema.number({
    defaultValue: 1600,
    min: 0
  }),
  window_height: _configSchema.schema.number({
    defaultValue: 800,
    min: 0
  }),
  report_format: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.FORMAT.pdf), _configSchema.schema.literal(_constants.FORMAT.png)]),
  header: _configSchema.schema.maybe(_configSchema.schema.string()),
  footer: _configSchema.schema.maybe(_configSchema.schema.string()),
  time_duration: _configSchema.schema.string({
    validate(value) {
      if (!_validationHelper.regexDuration.test(value)) {
        return `invalid time duration: ${value}`;
      }
    }
  })
});
const intervalSchema = exports.intervalSchema = _configSchema.schema.object({
  interval: _configSchema.schema.object({
    period: _configSchema.schema.number({
      min: 0
    }),
    // Refer to job scheduler SPI https://github.com/opensearch-project/job-scheduler/blob/main/spi/src/main/java/org/opensearch/jobscheduler/spi/schedule/IntervalSchedule.java
    unit: _configSchema.schema.oneOf([_configSchema.schema.literal('MINUTES'), _configSchema.schema.literal('HOURS'), _configSchema.schema.literal('DAYS'),
    // Job scheduler in reporting OpenSearch plugin always saves as following format
    _configSchema.schema.literal('Minutes'), _configSchema.schema.literal('Hours'), _configSchema.schema.literal('Days')]),
    // timestamp
    start_time: _configSchema.schema.number()
  })
});
const cronSchema = exports.cronSchema = _configSchema.schema.object({
  cron: _configSchema.schema.object({
    expression: _configSchema.schema.string({
      validate(value) {
        if (!(0, _cronValidator.isValidCron)(value)) {
          return `invalid cron expression: ${value}`;
        }
      }
    }),
    //TODO: add more validation once we add full support of timezone
    timezone: _configSchema.schema.string()
  })
});
const scheduleSchema = exports.scheduleSchema = _configSchema.schema.object({
  schedule_type: _configSchema.schema.oneOf([
  /*
  TODO: Future Date option will be added in the future.
  Currently @osd/config-schema has no support for more than 2 conditions, keep an eye on library update
  */
  _configSchema.schema.literal(_constants.SCHEDULE_TYPE.recurring), _configSchema.schema.literal(_constants.SCHEDULE_TYPE.cron)]),
  schedule: _configSchema.schema.conditional(_configSchema.schema.siblingRef('schedule_type'), _constants.SCHEDULE_TYPE.recurring, intervalSchema, cronSchema),
  enabled_time: _configSchema.schema.number(),
  enabled: _configSchema.schema.boolean()
});
const opensearchDashboardsUserSchema = exports.opensearchDashboardsUserSchema = _configSchema.schema.object({
  opensearch_dashboards_recipients: _configSchema.schema.arrayOf(_configSchema.schema.string())
});
const channelSchema = exports.channelSchema = _configSchema.schema.object({
  recipients: _configSchema.schema.arrayOf(_configSchema.schema.string({
    validate(value) {
      if (!_validationHelper.regexEmailAddress.test(value)) {
        return `invalid email address ${value}`;
      }
    }
  }), {
    minSize: 1
  }),
  title: _configSchema.schema.string(),
  textDescription: _configSchema.schema.string(),
  htmlDescription: _configSchema.schema.maybe(_configSchema.schema.string()),
  configIds: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
});
const triggerSchema = exports.triggerSchema = _configSchema.schema.object({
  trigger_type: _configSchema.schema.oneOf([
  /*
    TODO: Alerting will be added in the future.
    Currently @osd/config-schema has no support for more than 2 conditions, keep an eye on library update
  */
  _configSchema.schema.literal(_constants.TRIGGER_TYPE.schedule), _configSchema.schema.literal(_constants.TRIGGER_TYPE.onDemand)]),
  trigger_params: _configSchema.schema.conditional(_configSchema.schema.siblingRef('trigger_type'), _constants.TRIGGER_TYPE.onDemand, _configSchema.schema.never(), scheduleSchema)
});
const deliverySchema = exports.deliverySchema = _configSchema.schema.object({
  configIds: _configSchema.schema.arrayOf(_configSchema.schema.string()),
  title: _configSchema.schema.string(),
  textDescription: _configSchema.schema.string(),
  htmlDescription: _configSchema.schema.string()
});
const reportParamsSchema = exports.reportParamsSchema = _configSchema.schema.object({
  report_name: _configSchema.schema.string({
    validate(value) {
      if (!_validationHelper.regexReportName.test(value)) {
        return `invald report name ${value}.\nMust be non-empty, allow a-z, A-Z, 0-9, (), [], ',' - and _ and spaces`;
      }
    }
  }),
  report_source: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.REPORT_TYPE.dashboard), _configSchema.schema.literal(_constants.REPORT_TYPE.visualization), _configSchema.schema.literal(_constants.REPORT_TYPE.savedSearch), _configSchema.schema.literal(_constants.REPORT_TYPE.notebook)]),
  description: _configSchema.schema.string(),
  core_params: _configSchema.schema.conditional(_configSchema.schema.siblingRef('report_source'), _constants.REPORT_TYPE.savedSearch, dataReportSchema, visualReportSchema)
});
const reportDefinitionSchema = exports.reportDefinitionSchema = _configSchema.schema.object({
  report_params: reportParamsSchema,
  delivery: deliverySchema,
  trigger: triggerSchema,
  time_created: _configSchema.schema.maybe(_configSchema.schema.number()),
  last_updated: _configSchema.schema.maybe(_configSchema.schema.number()),
  status: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(_constants.REPORT_DEFINITION_STATUS.active), _configSchema.schema.literal(_constants.REPORT_DEFINITION_STATUS.disabled)]))
});
const reportSchema = exports.reportSchema = _configSchema.schema.object({
  query_url: _configSchema.schema.string({
    validate(value) {
      if (!(0, _validationHelper.isValidRelativeUrl)(value)) {
        return `invalid relative url: ${value}`;
      }
    },
    minLength: 1
  }),
  time_from: _configSchema.schema.number(),
  time_to: _configSchema.schema.number(),
  report_definition: reportDefinitionSchema,
  time_created: _configSchema.schema.maybe(_configSchema.schema.number()),
  last_updated: _configSchema.schema.maybe(_configSchema.schema.number()),
  state: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(_constants.REPORT_STATE.created), _configSchema.schema.literal(_constants.REPORT_STATE.error), _configSchema.schema.literal(_constants.REPORT_STATE.pending), _configSchema.schema.literal(_constants.REPORT_STATE.shared)]))
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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