/*
 * Decompiled with CFR 0.152.
 */
package org.logstash.common.io;

import java.io.Closeable;
import java.io.IOException;
import java.nio.channels.FileLock;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.Comparator;
import java.util.LongSummaryStatistics;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.LongAdder;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.logstash.DLQEntry;
import org.logstash.FileLockFactory;
import org.logstash.LockException;
import org.logstash.Timestamp;
import org.logstash.common.io.DeadLetterQueueUtils;
import org.logstash.common.io.RecordIOReader;
import org.logstash.common.io.SegmentListener;

public final class DeadLetterQueueReader
implements Closeable {
    private static final Logger logger = LogManager.getLogger(DeadLetterQueueReader.class);
    public static final String DELETED_SEGMENT_PREFIX = ".deleted_segment";
    private RecordIOReader currentReader;
    private final Path queuePath;
    private final SegmentListener segmentCallback;
    private final ConcurrentSkipListSet<Path> segments;
    private final WatchService watchService;
    private RecordIOReader lastConsumedReader;
    private final LongAdder consumedEvents = new LongAdder();
    private final LongAdder consumedSegments = new LongAdder();
    private final boolean cleanConsumed;
    private FileLock fileLock;

    public DeadLetterQueueReader(Path queuePath) throws IOException {
        this(queuePath, false, null);
    }

    public DeadLetterQueueReader(Path queuePath, boolean cleanConsumed, SegmentListener segmentCallback) throws IOException {
        this.queuePath = queuePath;
        this.watchService = FileSystems.getDefault().newWatchService();
        this.queuePath.register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE);
        this.segments = new ConcurrentSkipListSet<Path>(Comparator.comparingInt(DeadLetterQueueUtils::extractSegmentId));
        this.segments.addAll(DeadLetterQueueUtils.listSegmentPaths(queuePath).filter(p -> p.toFile().length() > 1L).collect(Collectors.toList()));
        this.cleanConsumed = cleanConsumed;
        if (cleanConsumed && segmentCallback == null) {
            throw new IllegalArgumentException("When cleanConsumed is enabled must be passed also a valid segment listener");
        }
        this.segmentCallback = segmentCallback;
        this.lastConsumedReader = null;
        if (cleanConsumed) {
            try {
                this.fileLock = FileLockFactory.obtainLock(queuePath, "dlq_reader.lock");
            }
            catch (LockException ex) {
                throw new LockException("Existing `dlg_reader.lock` file in [" + queuePath + "]. Only one DeadLetterQueueReader with `cleanConsumed` set is allowed per Dead Letter Queue.", ex);
            }
        }
    }

    public void seekToNextEvent(Timestamp timestamp) throws IOException {
        for (Path segment : this.segments) {
            Optional<RecordIOReader> optReader = this.openSegmentReader(segment);
            if (!optReader.isPresent()) continue;
            this.currentReader = optReader.get();
            byte[] event = this.currentReader.seekToNextEventPosition(timestamp, DeadLetterQueueReader::extractEntryTimestamp, Timestamp::compareTo);
            if (event == null) continue;
            return;
        }
        if (this.currentReader != null) {
            this.currentReader.close();
            this.currentReader = null;
        }
    }

    private Optional<RecordIOReader> openSegmentReader(Path segment) throws IOException {
        if (!Files.exists(segment, new LinkOption[0])) {
            this.segments.remove(segment);
            return Optional.empty();
        }
        try {
            return Optional.of(new RecordIOReader(segment));
        }
        catch (NoSuchFileException ex) {
            logger.debug("Segment file {} was deleted by DLQ writer during DLQ reader opening", (Object)segment);
            this.segments.remove(segment);
            return Optional.empty();
        }
    }

    private static Timestamp extractEntryTimestamp(byte[] serialized) {
        try {
            return DLQEntry.deserialize(serialized).getEntryTime();
        }
        catch (IOException e) {
            throw new IllegalStateException(e);
        }
    }

    private long pollNewSegments(long timeout) throws IOException, InterruptedException {
        long startTime = System.currentTimeMillis();
        WatchKey key = this.watchService.poll(timeout, TimeUnit.MILLISECONDS);
        if (key != null) {
            this.pollSegmentsOnWatch(key);
        }
        return System.currentTimeMillis() - startTime;
    }

    private void pollNewSegments() throws IOException {
        WatchKey key = this.watchService.poll();
        if (key != null) {
            this.pollSegmentsOnWatch(key);
        }
    }

    private void pollSegmentsOnWatch(WatchKey key) throws IOException {
        for (WatchEvent<?> watchEvent : key.pollEvents()) {
            if (watchEvent.kind() == StandardWatchEventKinds.ENTRY_CREATE) {
                this.segments.addAll(DeadLetterQueueUtils.listSegmentPaths(this.queuePath).collect(Collectors.toList()));
            } else if (watchEvent.kind() == StandardWatchEventKinds.ENTRY_DELETE) {
                int oldSize = this.segments.size();
                this.segments.clear();
                this.segments.addAll(DeadLetterQueueUtils.listSegmentPaths(this.queuePath).collect(Collectors.toList()));
                logger.debug("Notified of segment removal, switched from {} to {} segments", (Object)oldSize, (Object)this.segments.size());
            }
            key.reset();
        }
    }

    public DLQEntry pollEntry(long timeout) throws IOException, InterruptedException {
        byte[] bytes = this.pollEntryBytes(timeout);
        if (bytes == null) {
            return null;
        }
        return DLQEntry.deserialize(bytes);
    }

    byte[] pollEntryBytes() throws IOException, InterruptedException {
        return this.pollEntryBytes(100L);
    }

    private byte[] pollEntryBytes(long timeout) throws IOException, InterruptedException {
        byte[] event;
        long timeoutRemaining = timeout;
        if (this.currentReader == null) {
            Optional<RecordIOReader> optReader;
            timeoutRemaining -= this.pollNewSegments(timeout);
            if (this.segments.isEmpty()) {
                logger.debug("No entries found: no segment files found in dead-letter-queue directory");
                return null;
            }
            do {
                Path firstSegment;
                try {
                    firstSegment = this.segments.first();
                }
                catch (NoSuchElementException ex) {
                    logger.debug("No entries found: no segment files found in dead-letter-queue directory");
                    return null;
                }
                optReader = this.openSegmentReader(firstSegment);
                if (!optReader.isPresent()) continue;
                this.currentReader = optReader.get();
            } while (!optReader.isPresent());
        }
        if ((event = this.currentReader.readEvent()) == null && this.currentReader.isEndOfStream()) {
            if (this.consumedAllSegments()) {
                this.pollNewSegments(timeoutRemaining);
            } else {
                Optional<RecordIOReader> optReader;
                this.currentReader.close();
                if (this.cleanConsumed) {
                    this.lastConsumedReader = this.currentReader;
                }
                if (!(optReader = this.openNextExistingReader(this.currentReader.getPath())).isPresent()) {
                    this.pollNewSegments(timeoutRemaining);
                } else {
                    this.currentReader = optReader.get();
                    return this.pollEntryBytes(timeoutRemaining);
                }
            }
        }
        return event;
    }

    public void markForDelete() {
        if (!this.cleanConsumed) {
            return;
        }
        if (this.lastConsumedReader == null) {
            return;
        }
        this.segmentCallback.segmentCompleted();
        Path lastConsumedSegmentPath = this.lastConsumedReader.getPath();
        try {
            this.removeSegmentsBefore(lastConsumedSegmentPath);
        }
        catch (IOException ex) {
            logger.warn("Problem occurred in cleaning the segments older than {} ", (Object)lastConsumedSegmentPath, (Object)ex);
        }
        Optional<Long> deletedEvents = this.deleteSegment(lastConsumedSegmentPath);
        if (deletedEvents.isPresent()) {
            this.consumedEvents.add(deletedEvents.get());
            this.consumedSegments.increment();
        }
        this.segmentCallback.segmentsDeleted(this.consumedSegments.intValue(), this.consumedEvents.longValue());
        this.lastConsumedReader = null;
    }

    private boolean consumedAllSegments() {
        try {
            return this.currentReader.getPath().equals(this.segments.last());
        }
        catch (NoSuchElementException ex) {
            logger.debug("No last segment found, poll for new segments");
            return true;
        }
    }

    private Path nextExistingSegmentFile(Path currentSegmentPath) {
        Path nextExpectedSegment;
        boolean skip;
        do {
            if ((nextExpectedSegment = this.segments.higher(currentSegmentPath)) != null && !Files.exists(nextExpectedSegment, new LinkOption[0])) {
                this.segments.remove(nextExpectedSegment);
                skip = true;
                continue;
            }
            skip = false;
        } while (skip);
        return nextExpectedSegment;
    }

    public void setCurrentReaderAndPosition(Path segmentPath, long position) throws IOException {
        Optional<RecordIOReader> optReader;
        if (this.cleanConsumed) {
            this.removeSegmentsBefore(segmentPath);
        }
        if ((optReader = this.openSegmentReader(segmentPath)).isPresent()) {
            this.currentReader = optReader.get();
            this.currentReader.seekToOffset(position);
            return;
        }
        optReader = this.openNextExistingReader(segmentPath);
        if (optReader.isPresent()) {
            this.currentReader = optReader.get();
            return;
        }
        this.pollNewSegments();
        this.openNextExistingReader(segmentPath).ifPresent(reader -> {
            this.currentReader = reader;
        });
    }

    private void removeSegmentsBefore(Path validSegment) throws IOException {
        Comparator<Path> fileTimeAndName = ((Comparator)this::compareByFileTimestamp).thenComparingInt(DeadLetterQueueUtils::extractSegmentId);
        try (Stream<Path> segmentFiles = DeadLetterQueueUtils.listSegmentPaths(this.queuePath);){
            LongSummaryStatistics deletionStats = segmentFiles.filter(p -> fileTimeAndName.compare((Path)p, validSegment) < 0).map(this::deleteSegment).map(o -> o.orElse(0L)).mapToLong(Long::longValue).summaryStatistics();
            this.consumedSegments.add(deletionStats.getCount());
            this.consumedEvents.add(deletionStats.getSum());
        }
        this.createSegmentRemovalFile(validSegment);
    }

    private void createSegmentRemovalFile(Path lastDeletedSegment) {
        Path notificationFile = this.queuePath.resolve(DELETED_SEGMENT_PREFIX);
        byte[] content = (lastDeletedSegment + "\n").getBytes(StandardCharsets.UTF_8);
        if (Files.exists(notificationFile, new LinkOption[0])) {
            DeadLetterQueueReader.updateToExistingNotification(notificationFile, content);
            return;
        }
        this.createNotification(notificationFile, content);
    }

    private void createNotification(Path notificationFile, byte[] content) {
        try {
            Path tmpNotificationFile = Files.createFile(this.queuePath.resolve(".deleted_segment.tmp"), new FileAttribute[0]);
            Files.write(tmpNotificationFile, content, StandardOpenOption.APPEND);
            Files.move(tmpNotificationFile, notificationFile, StandardCopyOption.ATOMIC_MOVE);
            logger.debug("Recreated notification file {}", (Object)notificationFile);
        }
        catch (IOException e) {
            logger.error("Can't create file to notify deletion of segments from DLQ reader in path {}", (Object)notificationFile, (Object)e);
        }
    }

    private static void updateToExistingNotification(Path notificationFile, byte[] contentToAppend) {
        try {
            Files.write(notificationFile, contentToAppend, StandardOpenOption.APPEND);
            logger.debug("Updated existing notification file {}", (Object)notificationFile);
        }
        catch (IOException e) {
            logger.error("Can't update file to notify deletion of segments from DLQ reader in path {}", (Object)notificationFile, (Object)e);
        }
        logger.debug("Notification segments delete file already exists {}", (Object)notificationFile);
    }

    private int compareByFileTimestamp(Path p1, Path p2) {
        Optional<FileTime> timestampResult1 = DeadLetterQueueReader.readLastModifiedTime(p1);
        Optional<FileTime> timestampResult2 = DeadLetterQueueReader.readLastModifiedTime(p2);
        if (!timestampResult1.isPresent() || !timestampResult2.isPresent()) {
            return 0;
        }
        return timestampResult1.get().compareTo(timestampResult2.get());
    }

    private static Optional<FileTime> readLastModifiedTime(Path p) {
        try {
            return Optional.of(Files.getLastModifiedTime(p, new LinkOption[0]));
        }
        catch (NoSuchFileException fileNotFoundEx) {
            logger.debug("File {} doesn't exist", (Object)p);
            return Optional.empty();
        }
        catch (IOException ex) {
            logger.warn("Error reading file's timestamp for {}", (Object)p, (Object)ex);
            return Optional.empty();
        }
    }

    private Optional<Long> deleteSegment(Path segment) {
        this.segments.remove(segment);
        try {
            long eventsInSegment = DeadLetterQueueUtils.countEventsInSegment(segment);
            Files.delete(segment);
            logger.debug("Deleted segment {}", (Object)segment);
            return Optional.of(eventsInSegment);
        }
        catch (NoSuchFileException fileNotFoundEx) {
            logger.debug("Expected file segment {} was already removed by a writer", (Object)segment);
            return Optional.empty();
        }
        catch (IOException ex) {
            logger.warn("Problem occurred in cleaning the segment {} after a repositioning", (Object)segment, (Object)ex);
            return Optional.empty();
        }
    }

    private Optional<RecordIOReader> openNextExistingReader(Path segmentPath) throws IOException {
        Path next;
        while ((next = this.nextExistingSegmentFile(segmentPath)) != null) {
            Optional<RecordIOReader> optReader = this.openSegmentReader(next);
            if (!optReader.isPresent()) continue;
            return optReader;
        }
        return Optional.empty();
    }

    public Path getCurrentSegment() {
        return this.currentReader.getPath();
    }

    public long getCurrentPosition() {
        return this.currentReader.getChannelPosition();
    }

    long getConsumedEvents() {
        return this.consumedEvents.longValue();
    }

    int getConsumedSegments() {
        return this.consumedSegments.intValue();
    }

    @Override
    public void close() throws IOException {
        try {
            if (this.currentReader != null) {
                this.currentReader.close();
            }
            this.watchService.close();
        }
        finally {
            if (this.cleanConsumed) {
                FileLockFactory.releaseLock(this.fileLock);
            }
        }
    }
}

