"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FileSystemCatalogDataAdaptor = void 0;

var fs = _interopRequireWildcard(require("fs/promises"));

var _path = _interopRequireDefault(require("path"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

/**
 * Helper function to compare version numbers.
 * Assumes that the version numbers are valid, produces undefined behavior otherwise.
 *
 * @param a Left-hand number
 * @param b Right-hand number
 * @returns -1 if a > b, 1 if a < b, 0 otherwise.
 */
function compareVersions(a, b) {
  const aParts = a.split('.').map(Number.parseInt);
  const bParts = b.split('.').map(Number.parseInt);

  for (let i = 0; i < Math.max(aParts.length, bParts.length); i++) {
    const aValue = i < aParts.length ? aParts[i] : 0;
    const bValue = i < bParts.length ? bParts[i] : 0;

    if (aValue > bValue) {
      return -1; // a > b
    } else if (aValue < bValue) {
      return 1; // a < b
    }
  }

  return 0; // a == b
}

function tryParseNDJson(content) {
  try {
    const objects = [];

    for (const line of content.split('\n')) {
      if (line.trim() === '') {
        // Other OSD ndjson parsers skip whitespace lines
        continue;
      }

      objects.push(JSON.parse(line));
    }

    return objects;
  } catch (err) {
    return null;
  }
}
/**
 * A CatalogDataAdaptor that reads from the local filesystem.
 * Used to read Integration information when the user uploads their own catalog.
 */


class FileSystemCatalogDataAdaptor {
  /**
   * Creates a new FileSystemCatalogDataAdaptor instance.
   *
   * @param directory The base directory from which to read files. This is not sanitized.
   */
  constructor(directory) {
    _defineProperty(this, "directory", void 0);

    this.directory = directory;
  }

  async readFile(filename, type) {
    let content;

    try {
      content = await fs.readFile(_path.default.join(this.directory, type !== null && type !== void 0 ? type : '.', filename), {
        encoding: 'utf-8'
      });
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    } // First try to parse as JSON, then NDJSON, then fail.


    try {
      const parsed = JSON.parse(content);
      return {
        ok: true,
        value: parsed
      };
    } catch (err) {
      const parsed = tryParseNDJson(content);

      if (parsed) {
        return {
          ok: true,
          value: parsed
        };
      }

      return {
        ok: false,
        error: new Error('Unable to parse file as JSON or NDJson', {
          cause: err
        })
      };
    }
  }

  async readFileRaw(filename, type) {
    try {
      const buffer = await fs.readFile(_path.default.join(this.directory, type !== null && type !== void 0 ? type : '.', filename));
      return {
        ok: true,
        value: buffer
      };
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
  }

  async findIntegrations(dirname = '.') {
    try {
      const files = await fs.readdir(_path.default.join(this.directory, dirname));
      return {
        ok: true,
        value: files
      };
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
  }

  async findIntegrationVersions(dirname = '.') {
    let files;

    const integPath = _path.default.join(this.directory, dirname);

    try {
      files = await fs.readdir(integPath);
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }

    const versions = [];

    for (const file of files) {
      // TODO handle nested repositories -- assumes integrations are 1 level deep
      if (_path.default.extname(file) === '.json' && file.startsWith(`${_path.default.basename(integPath)}-`)) {
        const version = file.substring(_path.default.basename(integPath).length + 1, file.length - 5);

        if (!version.match(/^\d+(\.\d+)*$/)) {
          continue;
        }

        versions.push(version);
      }
    }

    versions.sort((a, b) => compareVersions(a, b));
    return {
      ok: true,
      value: versions
    };
  }

  async getDirectoryType(dirname) {
    const isDir = (await fs.lstat(_path.default.join(this.directory, dirname !== null && dirname !== void 0 ? dirname : '.'))).isDirectory();

    if (!isDir) {
      return 'unknown';
    } // Sloppily just check for one mandatory integration directory to distinguish.
    // Improve if we need to distinguish a repository with an integration named "schemas".


    const hasSchemas = (await fs.lstat(_path.default.join(this.directory, dirname !== null && dirname !== void 0 ? dirname : '.', 'schemas'))).isDirectory();
    return hasSchemas ? 'integration' : 'repository';
  }

  join(filename) {
    return new FileSystemCatalogDataAdaptor(_path.default.join(this.directory, filename));
  }

}

exports.FileSystemCatalogDataAdaptor = FileSystemCatalogDataAdaptor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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