"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

class TransformService {
  constructor(esDriver) {
    _defineProperty(this, "esDriver", void 0);

    _defineProperty(this, "getTransforms", async (context, request, response) => {
      try {
        const {
          from,
          size,
          search,
          sortDirection,
          sortField
        } = request.query;
        const transformSortMap = {
          _id: "transform.transform_id.keyword",
          "transform.source_index": "transform.source_index.keyword",
          "transform.target_index": "transform.target_index.keyword",
          "transform.transform.enabled": "transform.enabled"
        };
        const params = {
          from: parseInt(from, 10),
          size: parseInt(size, 10),
          search,
          sortField: transformSortMap[sortField] || transformSortMap._id,
          sortDirection
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.esDriver.asScoped(request);
        const getTransformsResponse = await callWithRequest("ism.getTransforms", params);
        const totalTransforms = getTransformsResponse.total_transforms;
        const transforms = getTransformsResponse.transforms.map(transform => ({
          _seqNo: transform._seqNo,
          _primaryTerm: transform._primaryTerm,
          _id: transform._id,
          transform: transform.transform,
          metadata: null
        }));

        if (totalTransforms) {
          const ids = transforms.map(transform => transform._id).join(",");
          const explainResponse = await callWithRequest("ism.explainTransform", {
            transformId: ids
          });

          if (!explainResponse.error) {
            transforms.map(transform => {
              transform.metadata = explainResponse[transform._id];
            });
            return response.custom({
              statusCode: 200,
              body: {
                ok: true,
                response: {
                  transforms: transforms,
                  totalTransforms: totalTransforms,
                  metadata: explainResponse
                }
              }
            });
          } else {
            return response.custom({
              statusCode: 200,
              body: {
                ok: false,
                error: explainResponse ? explainResponse.error : "An error occurred when calling getExplain API."
              }
            });
          }
        }

        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              transforms: transforms,
              totalTransforms: totalTransforms,
              metadata: {}
            }
          }
        });
      } catch (err) {
        if (err.statusCode === 404 && err.body.error.type === "index_not_found_exception") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                transforms: [],
                totalTransforms: 0,
                metadata: null
              }
            }
          });
        }

        console.error("Index Management - TransformService - getTransforms", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: "Error in getTransforms" + err.message
          }
        });
      }
    });

    _defineProperty(this, "getTransform", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          transformId: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.esDriver.asScoped(request);
        const getResponse = await callWithRequest("ism.getTransform", params);
        const metadata = await callWithRequest("ism.explainTransform", params);

        const transform = _lodash.default.get(getResponse, "transform", null);

        const seqNo = _lodash.default.get(getResponse, "_seq_no", null);

        const primaryTerm = _lodash.default.get(getResponse, "_primary_term", null);

        if (transform) {
          if (metadata) {
            return response.custom({
              statusCode: 200,
              body: {
                ok: true,
                response: {
                  _id: id,
                  _seqNo: seqNo,
                  _primaryTerm: primaryTerm,
                  transform: transform,
                  metadata: metadata
                }
              }
            });
          } else {
            return response.custom({
              statusCode: 200,
              body: {
                ok: false,
                error: "Failed to load metadata for transform"
              }
            });
          }
        } else {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to load transform"
            }
          });
        }
      } catch (err) {
        console.error("Index Management - TransformService - getTransform:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "startTransform", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          transformId: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.esDriver.asScoped(request);
        const startResponse = await callWithRequest("ism.startTransform", params);

        const acknowledged = _lodash.default.get(startResponse, "acknowledged");

        if (acknowledged) {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: true
            }
          });
        } else {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to start transform"
            }
          });
        }
      } catch (err) {
        console.error("Index Management - TransformService - startTransform", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "stopTransform", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          transformId: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.esDriver.asScoped(request);
        const stopResponse = await callWithRequest("ism.stopTransform", params);

        const acknowledged = _lodash.default.get(stopResponse, "acknowledged");

        if (acknowledged) {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: true
            }
          });
        } else {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to stop transform"
            }
          });
        }
      } catch (err) {
        console.error("Index Management - TransformService - stopTransform", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "deleteTransform", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          transformId: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.esDriver.asScoped(request);
        const deleteResponse = await callWithRequest("ism.deleteTransform", params);

        if (!deleteResponse.errors) {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: true
            }
          });
        } else {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to delete transform"
            }
          });
        }
      } catch (err) {
        console.error("Index Management - TransformService - deleteTransform", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "putTransform", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const {
          seqNo,
          primaryTerm
        } = request.query;
        let method = "ism.putTransform";
        let params = {
          transformId: id,
          if_seq_no: seqNo,
          if_primary_term: primaryTerm,
          body: JSON.stringify(request.body)
        };

        if (seqNo === undefined || primaryTerm === undefined) {
          method = "ism.putTransform";
          params = {
            transformId: id,
            body: JSON.stringify(request.body)
          };
        }

        const {
          callAsCurrentUser: callWithRequest
        } = this.esDriver.asScoped(request);
        const putTransformResponse = await callWithRequest(method, params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: putTransformResponse
          }
        });
      } catch (err) {
        console.error("Index Management - TransformService - putTransform", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "searchSampleData", async (context, request, response) => {
      try {
        const {
          from,
          size
        } = request.query;
        const {
          index
        } = request.params;
        let params = {
          index: index,
          from: from,
          size: size,
          body: request.body ? JSON.stringify({
            query: request.body
          }) : {}
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.esDriver.asScoped(request);
        const searchResponse = await callWithRequest("search", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              total: searchResponse.hits.total,
              data: searchResponse.hits.hits
            }
          }
        });
      } catch (err) {
        if (err.statusCode === 404 && err.body.error.type === "index_not_found_exception") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                total: 0,
                data: []
              }
            }
          });
        }

        console.error("Index Management - TransformService - searchSampleData", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "previewTransform", async (context, request, response) => {
      try {
        let params = {
          body: JSON.stringify(request.body)
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.esDriver.asScoped(request);
        const previewResponse = await callWithRequest("ism.previewTransform", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: previewResponse
          }
        });
      } catch (err) {
        console.error("Index Management - TransformService - previewTransform", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    this.esDriver = esDriver;
  }

}

exports.default = TransformService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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