#!/usr/bin/python
from __future__ import absolute_import, division, print_function

# Copyright: (c) 2022 Fortinet
# GNU General Public License v3.0+ (see LICENSE or https://www.gnu.org/licenses/gpl-3.0.txt)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

__metaclass__ = type

ANSIBLE_METADATA = {
    "status": ["preview"],
    "supported_by": "community",
    "metadata_version": "1.1",
}

DOCUMENTATION = """
---
module: fortios_endpoint_control_fctems_override
short_description: Configure FortiClient Enterprise Management Server (EMS) entries in Fortinet's FortiOS and FortiGate.
description:
    - This module is able to configure a FortiGate or FortiOS (FOS) device by allowing the
      user to set and modify endpoint_control feature and fctems_override category.
      Examples include all parameters and values need to be adjusted to datasources before usage.
      Tested with FOS v6.0.0
version_added: "2.0.0"
author:
    - Link Zheng (@chillancezen)
    - Jie Xue (@JieX19)
    - Hongbin Lu (@fgtdev-hblu)
    - Frank Shen (@frankshen01)
    - Miguel Angel Munoz (@mamunozgonzalez)
    - Nicolas Thomas (@thomnico)
notes:
    - We highly recommend using your own value as the ems_id instead of 0, while '0' is a special placeholder that allows the backend to assign the latest
       available number for the object, it does have limitations. Please find more details in Q&A.
    - Legacy fortiosapi has been deprecated, httpapi is the preferred way to run playbooks

requirements:
    - ansible>=2.9
options:
    access_token:
        description:
            - Token-based authentication.
              Generated from GUI of Fortigate.
        type: str
        required: false
    enable_log:
        description:
            - Enable/Disable logging for task.
        type: bool
        required: false
        default: false
    vdom:
        description:
            - Virtual domain, among those defined previously. A vdom is a
              virtual instance of the FortiGate that can be configured and
              used as a different unit.
        type: str
        default: root
    member_path:
        type: str
        description:
            - Member attribute path to operate on.
            - Delimited by a slash character if there are more than one attribute.
            - Parameter marked with member_path is legitimate for doing member operation.
    member_state:
        type: str
        description:
            - Add or delete a member under specified attribute path.
            - When member_state is specified, the state option is ignored.
        choices:
            - 'present'
            - 'absent'

    state:
        description:
            - Indicates whether to create or remove the object.
        type: str
        required: true
        choices:
            - 'present'
            - 'absent'
    endpoint_control_fctems_override:
        description:
            - Configure FortiClient Enterprise Management Server (EMS) entries.
        default: null
        type: dict
        suboptions:
            call_timeout:
                description:
                    - FortiClient EMS call timeout in seconds (1 - 180 seconds).
                type: int
            capabilities:
                description:
                    - List of EMS capabilities.
                type: list
                elements: str
                choices:
                    - 'fabric-auth'
                    - 'silent-approval'
                    - 'websocket'
                    - 'websocket-malware'
                    - 'push-ca-certs'
                    - 'common-tags-api'
                    - 'tenant-id'
                    - 'single-vdom-connector'
            cloud_server_type:
                description:
                    - Cloud server type.
                type: str
                choices:
                    - 'production'
                    - 'alpha'
                    - 'beta'
            dirty_reason:
                description:
                    - Dirty Reason for FortiClient EMS.
                type: str
                choices:
                    - 'none'
                    - 'mismatched-ems-sn'
            ems_id:
                description:
                    - EMS ID in order (1 - 7). see <a href='#notes'>Notes</a>.
                required: true
                type: int
            fortinetone_cloud_authentication:
                description:
                    - Enable/disable authentication of FortiClient EMS Cloud through FortiCloud account.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            https_port:
                description:
                    - 'FortiClient EMS HTTPS access port number. (1 - 65535).'
                type: int
            interface:
                description:
                    - Specify outgoing interface to reach server. Source system.interface.name.
                type: str
            interface_select_method:
                description:
                    - Specify how to select outgoing interface to reach server.
                type: str
                choices:
                    - 'auto'
                    - 'sdwan'
                    - 'specify'
            name:
                description:
                    - FortiClient Enterprise Management Server (EMS) name.
                type: str
            out_of_sync_threshold:
                description:
                    - Outdated resource threshold in seconds (10 - 3600).
                type: int
            preserve_ssl_session:
                description:
                    - Enable/disable preservation of EMS SSL session connection. Warning, most users should not touch this setting.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            pull_avatars:
                description:
                    - Enable/disable pulling avatars from EMS.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            pull_malware_hash:
                description:
                    - Enable/disable pulling FortiClient malware hash from EMS.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            pull_sysinfo:
                description:
                    - Enable/disable pulling SysInfo from EMS.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            pull_tags:
                description:
                    - Enable/disable pulling FortiClient user tags from EMS.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            pull_vulnerabilities:
                description:
                    - Enable/disable pulling vulnerabilities from EMS.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            serial_number:
                description:
                    - EMS Serial Number.
                type: str
            server:
                description:
                    - FortiClient EMS FQDN or IPv4 address.
                type: str
            source_ip:
                description:
                    - REST API call source IP.
                type: str
            status:
                description:
                    - Enable or disable this EMS configuration.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            tenant_id:
                description:
                    - EMS Tenant ID.
                type: str
            trust_ca_cn:
                description:
                    - Enable/disable trust of the EMS certificate issuer(CA) and common name(CN) for certificate auto-renewal.
                type: str
                choices:
                    - 'enable'
                    - 'disable'
            websocket_override:
                description:
                    - Enable/disable override behavior for how this FortiGate unit connects to EMS using a WebSocket connection.
                type: str
                choices:
                    - 'disable'
                    - 'enable'
"""

EXAMPLES = """
- hosts: fortigates
  collections:
    - fortinet.fortios
  connection: httpapi
  vars:
   vdom: "root"
   ansible_httpapi_use_ssl: yes
   ansible_httpapi_validate_certs: no
   ansible_httpapi_port: 443
  tasks:
  - name: Configure FortiClient Enterprise Management Server (EMS) entries.
    fortios_endpoint_control_fctems_override:
      vdom:  "{{ vdom }}"
      state: "present"
      access_token: "<your_own_value>"
      endpoint_control_fctems_override:
        call_timeout: "30"
        capabilities: "fabric-auth"
        cloud_server_type: "production"
        dirty_reason: "none"
        ems_id: "<you_own_value>"
        fortinetone_cloud_authentication: "enable"
        https_port: "443"
        interface: "<your_own_value> (source system.interface.name)"
        interface_select_method: "auto"
        name: "default_name_12"
        out_of_sync_threshold: "180"
        preserve_ssl_session: "enable"
        pull_avatars: "enable"
        pull_malware_hash: "enable"
        pull_sysinfo: "enable"
        pull_tags: "enable"
        pull_vulnerabilities: "enable"
        serial_number: "<your_own_value>"
        server: "192.168.100.40"
        source_ip: "84.230.14.43"
        status: "enable"
        tenant_id: "<your_own_value>"
        trust_ca_cn: "enable"
        websocket_override: "disable"

"""

RETURN = """
build:
  description: Build number of the fortigate image
  returned: always
  type: str
  sample: '1547'
http_method:
  description: Last method used to provision the content into FortiGate
  returned: always
  type: str
  sample: 'PUT'
http_status:
  description: Last result given by FortiGate on last operation applied
  returned: always
  type: str
  sample: "200"
mkey:
  description: Master key (id) used in the last call to FortiGate
  returned: success
  type: str
  sample: "id"
name:
  description: Name of the table used to fulfill the request
  returned: always
  type: str
  sample: "urlfilter"
path:
  description: Path of the table used to fulfill the request
  returned: always
  type: str
  sample: "webfilter"
revision:
  description: Internal revision number
  returned: always
  type: str
  sample: "17.0.2.10658"
serial:
  description: Serial number of the unit
  returned: always
  type: str
  sample: "FGVMEVYYQT3AB5352"
status:
  description: Indication of the operation's result
  returned: always
  type: str
  sample: "success"
vdom:
  description: Virtual domain used
  returned: always
  type: str
  sample: "root"
version:
  description: Version of the FortiGate
  returned: always
  type: str
  sample: "v5.6.3"

"""
from ansible.module_utils.basic import AnsibleModule
from ansible.module_utils.connection import Connection
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.fortios import (
    FortiOSHandler,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.fortios import (
    check_legacy_fortiosapi,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.fortios import (
    schema_to_module_spec,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.fortios import (
    check_schema_versioning,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortimanager.common import (
    FAIL_SOCKET_MSG,
)
from ansible_collections.fortinet.fortios.plugins.module_utils.fortios.data_post_processor import (
    remove_invalid_fields,
)


def filter_endpoint_control_fctems_override_data(json):
    option_list = [
        "call_timeout",
        "capabilities",
        "cloud_server_type",
        "dirty_reason",
        "ems_id",
        "fortinetone_cloud_authentication",
        "https_port",
        "interface",
        "interface_select_method",
        "name",
        "out_of_sync_threshold",
        "preserve_ssl_session",
        "pull_avatars",
        "pull_malware_hash",
        "pull_sysinfo",
        "pull_tags",
        "pull_vulnerabilities",
        "serial_number",
        "server",
        "source_ip",
        "status",
        "tenant_id",
        "trust_ca_cn",
        "websocket_override",
    ]

    json = remove_invalid_fields(json)
    dictionary = {}

    for attribute in option_list:
        if attribute in json and json[attribute] is not None:
            dictionary[attribute] = json[attribute]

    return dictionary


def flatten_single_path(data, path, index):
    if (
        not data
        or index == len(path)
        or path[index] not in data
        or not data[path[index]]
    ):
        return

    if index == len(path) - 1:
        data[path[index]] = " ".join(str(elem) for elem in data[path[index]])
    elif isinstance(data[path[index]], list):
        for value in data[path[index]]:
            flatten_single_path(value, path, index + 1)
    else:
        flatten_single_path(data[path[index]], path, index + 1)


def flatten_multilists_attributes(data):
    multilist_attrs = [
        ["capabilities"],
    ]

    for attr in multilist_attrs:
        flatten_single_path(data, attr, 0)

    return data


def underscore_to_hyphen(data):
    if isinstance(data, list):
        for i, elem in enumerate(data):
            data[i] = underscore_to_hyphen(elem)
    elif isinstance(data, dict):
        new_data = {}
        for k, v in data.items():
            new_data[k.replace("_", "-")] = underscore_to_hyphen(v)
        data = new_data

    return data


def endpoint_control_fctems_override(data, fos):
    vdom = data["vdom"]

    state = data["state"]

    endpoint_control_fctems_override_data = data["endpoint_control_fctems_override"]
    endpoint_control_fctems_override_data = flatten_multilists_attributes(
        endpoint_control_fctems_override_data
    )
    filtered_data = underscore_to_hyphen(
        filter_endpoint_control_fctems_override_data(
            endpoint_control_fctems_override_data
        )
    )

    if state == "present" or state is True:
        return fos.set(
            "endpoint-control", "fctems-override", data=filtered_data, vdom=vdom
        )

    elif state == "absent":
        return fos.delete(
            "endpoint-control",
            "fctems-override",
            mkey=filtered_data["ems-id"],
            vdom=vdom,
        )
    else:
        fos._module.fail_json(msg="state must be present or absent!")


def is_successful_status(resp):
    return (
        "status" in resp
        and resp["status"] == "success"
        or "http_status" in resp
        and resp["http_status"] == 200
        or "http_method" in resp
        and resp["http_method"] == "DELETE"
        and resp["http_status"] == 404
    )


def fortios_endpoint_control(data, fos):
    fos.do_member_operation("endpoint-control", "fctems-override")
    if data["endpoint_control_fctems_override"]:
        resp = endpoint_control_fctems_override(data, fos)
    else:
        fos._module.fail_json(
            msg="missing task body: %s" % ("endpoint_control_fctems_override")
        )

    return (
        not is_successful_status(resp),
        is_successful_status(resp)
        and (resp["revision_changed"] if "revision_changed" in resp else True),
        resp,
        {},
    )


versioned_schema = {
    "type": "list",
    "elements": "dict",
    "children": {
        "ems_id": {"revisions": {"v7.4.0": True}, "type": "integer", "required": True},
        "status": {
            "revisions": {"v7.4.0": True},
            "type": "string",
            "options": [
                {"value": "enable", "revisions": {"v7.4.0": True}},
                {"value": "disable", "revisions": {"v7.4.0": True}},
            ],
        },
        "name": {"revisions": {"v7.4.0": True}, "type": "string"},
        "dirty_reason": {
            "revisions": {"v7.4.0": True},
            "type": "string",
            "options": [
                {"value": "none", "revisions": {"v7.4.0": True}},
                {"value": "mismatched-ems-sn", "revisions": {"v7.4.0": True}},
            ],
        },
        "fortinetone_cloud_authentication": {
            "revisions": {"v7.4.0": True},
            "type": "string",
            "options": [
                {"value": "enable", "revisions": {"v7.4.0": True}},
                {"value": "disable", "revisions": {"v7.4.0": True}},
            ],
        },
        "server": {"revisions": {"v7.4.0": True}, "type": "string"},
        "https_port": {"revisions": {"v7.4.0": True}, "type": "integer"},
        "serial_number": {"revisions": {"v7.4.0": True}, "type": "string"},
        "tenant_id": {"revisions": {"v7.4.0": True}, "type": "string"},
        "source_ip": {"revisions": {"v7.4.0": True}, "type": "string"},
        "pull_sysinfo": {
            "revisions": {"v7.4.0": True},
            "type": "string",
            "options": [
                {"value": "enable", "revisions": {"v7.4.0": True}},
                {"value": "disable", "revisions": {"v7.4.0": True}},
            ],
        },
        "pull_vulnerabilities": {
            "revisions": {"v7.4.0": True},
            "type": "string",
            "options": [
                {"value": "enable", "revisions": {"v7.4.0": True}},
                {"value": "disable", "revisions": {"v7.4.0": True}},
            ],
        },
        "pull_avatars": {
            "revisions": {"v7.4.0": True},
            "type": "string",
            "options": [
                {"value": "enable", "revisions": {"v7.4.0": True}},
                {"value": "disable", "revisions": {"v7.4.0": True}},
            ],
        },
        "pull_tags": {
            "revisions": {"v7.4.0": True},
            "type": "string",
            "options": [
                {"value": "enable", "revisions": {"v7.4.0": True}},
                {"value": "disable", "revisions": {"v7.4.0": True}},
            ],
        },
        "pull_malware_hash": {
            "revisions": {"v7.4.0": True},
            "type": "string",
            "options": [
                {"value": "enable", "revisions": {"v7.4.0": True}},
                {"value": "disable", "revisions": {"v7.4.0": True}},
            ],
        },
        "cloud_server_type": {
            "revisions": {"v7.4.0": True},
            "type": "string",
            "options": [
                {"value": "production", "revisions": {"v7.4.0": True}},
                {"value": "alpha", "revisions": {"v7.4.0": True}},
                {"value": "beta", "revisions": {"v7.4.0": True}},
            ],
        },
        "capabilities": {
            "revisions": {"v7.4.0": True},
            "type": "list",
            "options": [
                {"value": "fabric-auth", "revisions": {"v7.4.0": True}},
                {"value": "silent-approval", "revisions": {"v7.4.0": True}},
                {"value": "websocket", "revisions": {"v7.4.0": True}},
                {"value": "websocket-malware", "revisions": {"v7.4.0": True}},
                {"value": "push-ca-certs", "revisions": {"v7.4.0": True}},
                {"value": "common-tags-api", "revisions": {"v7.4.0": True}},
                {"value": "tenant-id", "revisions": {"v7.4.0": True}},
                {"value": "single-vdom-connector", "revisions": {"v7.4.0": True}},
            ],
            "multiple_values": True,
            "elements": "str",
        },
        "call_timeout": {"revisions": {"v7.4.0": True}, "type": "integer"},
        "out_of_sync_threshold": {"revisions": {"v7.4.0": True}, "type": "integer"},
        "websocket_override": {
            "revisions": {"v7.4.0": True},
            "type": "string",
            "options": [
                {"value": "disable", "revisions": {"v7.4.0": True}},
                {"value": "enable", "revisions": {"v7.4.0": True}},
            ],
        },
        "preserve_ssl_session": {
            "revisions": {"v7.4.0": True},
            "type": "string",
            "options": [
                {"value": "enable", "revisions": {"v7.4.0": True}},
                {"value": "disable", "revisions": {"v7.4.0": True}},
            ],
        },
        "interface_select_method": {
            "revisions": {"v7.4.0": True},
            "type": "string",
            "options": [
                {"value": "auto", "revisions": {"v7.4.0": True}},
                {"value": "sdwan", "revisions": {"v7.4.0": True}},
                {"value": "specify", "revisions": {"v7.4.0": True}},
            ],
        },
        "interface": {"revisions": {"v7.4.0": True}, "type": "string"},
        "trust_ca_cn": {
            "revisions": {"v7.4.0": True},
            "type": "string",
            "options": [
                {"value": "enable", "revisions": {"v7.4.0": True}},
                {"value": "disable", "revisions": {"v7.4.0": True}},
            ],
        },
    },
    "revisions": {"v7.4.0": True},
}


def main():
    module_spec = schema_to_module_spec(versioned_schema)
    mkeyname = "ems-id"
    fields = {
        "access_token": {"required": False, "type": "str", "no_log": True},
        "enable_log": {"required": False, "type": "bool", "default": False},
        "vdom": {"required": False, "type": "str", "default": "root"},
        "member_path": {"required": False, "type": "str"},
        "member_state": {
            "type": "str",
            "required": False,
            "choices": ["present", "absent"],
        },
        "state": {"required": True, "type": "str", "choices": ["present", "absent"]},
        "endpoint_control_fctems_override": {
            "required": False,
            "type": "dict",
            "default": None,
            "options": {},
        },
    }
    for attribute_name in module_spec["options"]:
        fields["endpoint_control_fctems_override"]["options"][
            attribute_name
        ] = module_spec["options"][attribute_name]
        if mkeyname and mkeyname == attribute_name:
            fields["endpoint_control_fctems_override"]["options"][attribute_name][
                "required"
            ] = True

    module = AnsibleModule(argument_spec=fields, supports_check_mode=False)
    check_legacy_fortiosapi(module)

    is_error = False
    has_changed = False
    result = None
    diff = None

    versions_check_result = None
    if module._socket_path:
        connection = Connection(module._socket_path)
        if "access_token" in module.params:
            connection.set_option("access_token", module.params["access_token"])

        if "enable_log" in module.params:
            connection.set_option("enable_log", module.params["enable_log"])
        else:
            connection.set_option("enable_log", False)
        fos = FortiOSHandler(connection, module, mkeyname)
        versions_check_result = check_schema_versioning(
            fos, versioned_schema, "endpoint_control_fctems_override"
        )

        is_error, has_changed, result, diff = fortios_endpoint_control(
            module.params, fos
        )

    else:
        module.fail_json(**FAIL_SOCKET_MSG)

    if versions_check_result and versions_check_result["matched"] is False:
        module.warn(
            "Ansible has detected version mismatch between FortOS system and your playbook, see more details by specifying option -vvv"
        )

    if not is_error:
        if versions_check_result and versions_check_result["matched"] is False:
            module.exit_json(
                changed=has_changed,
                version_check_warning=versions_check_result,
                meta=result,
                diff=diff,
            )
        else:
            module.exit_json(changed=has_changed, meta=result, diff=diff)
    else:
        if versions_check_result and versions_check_result["matched"] is False:
            module.fail_json(
                msg="Error in repo",
                version_check_warning=versions_check_result,
                meta=result,
            )
        else:
            module.fail_json(msg="Error in repo", meta=result)


if __name__ == "__main__":
    main()
