'use strict';

const MetricsAggregationBase = require('./metrics-aggregation-base');

/**
 * A single-value metrics aggregation that computes the average of numeric
 * values that are extracted from the aggregated documents. These values can be
 * extracted either from specific numeric fields in the documents, or be
 * generated by a provided script.
 *
 * [Elasticsearch reference](https://www.elastic.co/guide/en/elasticsearch/reference/current/search-aggregations-metrics-avg-aggregation.html)
 *
 * Aggregation that computes the average of numeric values that are extracted
 * from the aggregated documents.
 *
 * @example
 * // Compute the average grade over all documents
 * const agg = esb.avgAggregation('avg_grade', 'grade');
 *
 * @example
 * // Compute the average grade based on a script
 * const agg = esb.avgAggregation('avg_grade').script(
 *     esb.script('inline', "doc['grade'].value").lang('painless')
 * );
 *
 * @example
 * // Value script, apply grade correction
 * const agg = esb.avgAggregation('avg_grade', 'grade').script(
 *     esb.script('inline', '_value * params.correction')
 *         .lang('painless')
 *         .params({ correction: 1.2 })
 * );
 *
 * @example
 * // Missing value
 * const agg = esb.avgAggregation('avg_grade', 'grade').missing(10);
 *
 * @param {string} name The name which will be used to refer to this aggregation.
 * @param {string=} field The field to aggregate on
 *
 * @extends MetricsAggregationBase
 */
class AvgAggregation extends MetricsAggregationBase {
    // eslint-disable-next-line require-jsdoc
    constructor(name, field) {
        super(name, 'avg', field);
    }
}

module.exports = AvgAggregation;
