/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.http;

import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import org.opensearch.common.settings.Settings;
import org.opensearch.common.settings.SettingsException;
import org.opensearch.core.common.Strings;
import org.opensearch.core.common.bytes.BytesArray;
import org.opensearch.core.common.bytes.BytesReference;
import org.opensearch.core.rest.RestStatus;
import org.opensearch.http.HttpRequest;
import org.opensearch.http.HttpResponse;
import org.opensearch.http.HttpTransportSettings;
import org.opensearch.rest.RestRequest;
import org.opensearch.rest.RestUtils;

public class CorsHandler {
    public static final String ANY_ORIGIN = "*";
    public static final String ORIGIN = "origin";
    public static final String DATE = "date";
    public static final String VARY = "vary";
    public static final String HOST = "host";
    public static final String ACCESS_CONTROL_REQUEST_METHOD = "access-control-request-method";
    public static final String ACCESS_CONTROL_ALLOW_HEADERS = "access-control-allow-headers";
    public static final String ACCESS_CONTROL_ALLOW_CREDENTIALS = "access-control-allow-credentials";
    public static final String ACCESS_CONTROL_ALLOW_METHODS = "access-control-allow-methods";
    public static final String ACCESS_CONTROL_ALLOW_ORIGIN = "access-control-allow-origin";
    public static final String ACCESS_CONTROL_MAX_AGE = "access-control-max-age";
    private static final Pattern SCHEME_PATTERN = Pattern.compile("^https?://");
    private static final DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("EEE, dd MMM yyyy HH:mm:ss O", Locale.ENGLISH);
    private final Config config;

    public CorsHandler(Config config) {
        this.config = config;
    }

    public HttpResponse handleInbound(HttpRequest request) {
        if (this.config.isCorsSupportEnabled()) {
            if (CorsHandler.isPreflightRequest(request)) {
                return this.handlePreflight(request);
            }
            if (!this.validateOrigin(request)) {
                return CorsHandler.forbidden(request);
            }
        }
        return null;
    }

    public void setCorsResponseHeaders(HttpRequest httpRequest, HttpResponse httpResponse) {
        if (!this.config.isCorsSupportEnabled()) {
            return;
        }
        if (this.setOrigin(httpRequest, httpResponse)) {
            this.setAllowCredentials(httpResponse);
        }
    }

    private HttpResponse handlePreflight(HttpRequest request) {
        HttpResponse response = request.createResponse(RestStatus.OK, (BytesReference)BytesArray.EMPTY);
        if (this.setOrigin(request, response)) {
            this.setAllowMethods(response);
            this.setAllowHeaders(response);
            this.setAllowCredentials(response);
            this.setMaxAge(response);
            this.setPreflightHeaders(response);
            return response;
        }
        return CorsHandler.forbidden(request);
    }

    private static HttpResponse forbidden(HttpRequest request) {
        HttpResponse response = request.createResponse(RestStatus.FORBIDDEN, (BytesReference)BytesArray.EMPTY);
        response.addHeader("content-length", "0");
        return response;
    }

    private static boolean isSameOrigin(String origin, String host) {
        String originDomain;
        return !Strings.isNullOrEmpty((String)host) && host.equals(originDomain = SCHEME_PATTERN.matcher(origin).replaceFirst(""));
    }

    private void setPreflightHeaders(HttpResponse response) {
        response.addHeader(DATE, dateTimeFormatter.format(ZonedDateTime.now(ZoneOffset.UTC)));
        response.addHeader("content-length", "0");
    }

    private boolean setOrigin(HttpRequest request, HttpResponse response) {
        String origin = CorsHandler.getOrigin(request);
        if (!Strings.isNullOrEmpty((String)origin)) {
            if (this.config.isAnyOriginSupported()) {
                if (this.config.isCredentialsAllowed()) {
                    CorsHandler.setAllowOrigin(response, origin);
                    CorsHandler.setVaryHeader(response);
                } else {
                    CorsHandler.setAllowOrigin(response, ANY_ORIGIN);
                }
                return true;
            }
            if (this.config.isOriginAllowed(origin) || CorsHandler.isSameOrigin(origin, CorsHandler.getHost(request))) {
                CorsHandler.setAllowOrigin(response, origin);
                CorsHandler.setVaryHeader(response);
                return true;
            }
        }
        return false;
    }

    private boolean validateOrigin(HttpRequest request) {
        if (this.config.isAnyOriginSupported()) {
            return true;
        }
        String origin = CorsHandler.getOrigin(request);
        if (Strings.isNullOrEmpty((String)origin)) {
            return true;
        }
        if (CorsHandler.isSameOrigin(origin, CorsHandler.getHost(request))) {
            return true;
        }
        return this.config.isOriginAllowed(origin);
    }

    private static String getOrigin(HttpRequest request) {
        List<String> headers = request.getHeaders().get(ORIGIN);
        if (headers == null || headers.isEmpty()) {
            return null;
        }
        return headers.get(0);
    }

    private static String getHost(HttpRequest request) {
        List<String> headers = request.getHeaders().get(HOST);
        if (headers == null || headers.isEmpty()) {
            return null;
        }
        return headers.get(0);
    }

    private static boolean isPreflightRequest(HttpRequest request) {
        Map<String, List<String>> headers = request.getHeaders();
        return request.method().equals((Object)RestRequest.Method.OPTIONS) && headers.containsKey(ORIGIN) && headers.containsKey(ACCESS_CONTROL_REQUEST_METHOD);
    }

    private static void setVaryHeader(HttpResponse response) {
        response.addHeader(VARY, ORIGIN);
    }

    private static void setAllowOrigin(HttpResponse response, String origin) {
        response.addHeader(ACCESS_CONTROL_ALLOW_ORIGIN, origin);
    }

    private void setAllowMethods(HttpResponse response) {
        for (RestRequest.Method method : this.config.allowedRequestMethods()) {
            response.addHeader(ACCESS_CONTROL_ALLOW_METHODS, method.name().trim());
        }
    }

    private void setAllowHeaders(HttpResponse response) {
        for (String header : this.config.allowedRequestHeaders) {
            response.addHeader(ACCESS_CONTROL_ALLOW_HEADERS, header);
        }
    }

    private void setAllowCredentials(HttpResponse response) {
        if (this.config.isCredentialsAllowed()) {
            response.addHeader(ACCESS_CONTROL_ALLOW_CREDENTIALS, "true");
        }
    }

    private void setMaxAge(HttpResponse response) {
        response.addHeader(ACCESS_CONTROL_MAX_AGE, Long.toString(this.config.maxAge));
    }

    public static CorsHandler disabled() {
        Config.Builder builder = new Config.Builder();
        builder.enabled = false;
        return new CorsHandler(new Config(builder));
    }

    public static Config buildConfig(Settings settings) {
        Config.Builder builder;
        if (!HttpTransportSettings.SETTING_CORS_ENABLED.get(settings).booleanValue()) {
            Config.Builder builder2 = new Config.Builder();
            builder2.enabled = false;
            return new Config(builder2);
        }
        String origin = HttpTransportSettings.SETTING_CORS_ALLOW_ORIGIN.get(settings);
        if (Strings.isNullOrEmpty((String)origin)) {
            builder = Config.Builder.forOrigins(new String[0]);
        } else if (origin.equals(ANY_ORIGIN)) {
            builder = Config.Builder.forAnyOrigin();
        } else {
            try {
                Pattern p = RestUtils.checkCorsSettingForRegex(origin);
                builder = p == null ? Config.Builder.forOrigins(RestUtils.corsSettingAsArray(origin)) : Config.Builder.forPattern(p);
            }
            catch (PatternSyntaxException e) {
                throw new SettingsException("Bad regex in [" + HttpTransportSettings.SETTING_CORS_ALLOW_ORIGIN.getKey() + "]: [" + origin + "]", e);
            }
        }
        if (HttpTransportSettings.SETTING_CORS_ALLOW_CREDENTIALS.get(settings).booleanValue()) {
            builder.allowCredentials();
        }
        String[] strMethods = Strings.tokenizeToStringArray((String)HttpTransportSettings.SETTING_CORS_ALLOW_METHODS.get(settings), (String)",");
        RestRequest.Method[] methods = (RestRequest.Method[])Arrays.stream(strMethods).map(s -> s.toUpperCase(Locale.ENGLISH)).map(RestRequest.Method::valueOf).toArray(RestRequest.Method[]::new);
        Config config = builder.allowedRequestMethods(methods).maxAge(HttpTransportSettings.SETTING_CORS_MAX_AGE.get(settings)).allowedRequestHeaders(Strings.tokenizeToStringArray((String)HttpTransportSettings.SETTING_CORS_ALLOW_HEADERS.get(settings), (String)",")).build();
        return config;
    }

    public static CorsHandler fromSettings(Settings settings) {
        return new CorsHandler(CorsHandler.buildConfig(settings));
    }

    public static class Config {
        private final boolean enabled;
        private final Optional<Set<String>> origins;
        private final Optional<Pattern> pattern;
        private final boolean anyOrigin;
        private final boolean credentialsAllowed;
        private final Set<RestRequest.Method> allowedRequestMethods;
        private final Set<String> allowedRequestHeaders;
        private final long maxAge;

        public Config(Builder builder) {
            this.enabled = builder.enabled;
            this.origins = builder.origins.map(HashSet::new);
            this.pattern = builder.pattern;
            this.anyOrigin = builder.anyOrigin;
            this.credentialsAllowed = builder.allowCredentials;
            this.allowedRequestMethods = Collections.unmodifiableSet(builder.requestMethods);
            this.allowedRequestHeaders = Collections.unmodifiableSet(builder.requestHeaders);
            this.maxAge = builder.maxAge;
        }

        public boolean isCorsSupportEnabled() {
            return this.enabled;
        }

        public boolean isAnyOriginSupported() {
            return this.anyOrigin;
        }

        public boolean isOriginAllowed(String origin) {
            if (this.origins.isPresent()) {
                return this.origins.get().contains(origin);
            }
            if (this.pattern.isPresent()) {
                return this.pattern.get().matcher(origin).matches();
            }
            return false;
        }

        public boolean isCredentialsAllowed() {
            return this.credentialsAllowed;
        }

        public Set<RestRequest.Method> allowedRequestMethods() {
            return this.allowedRequestMethods;
        }

        public Set<String> allowedRequestHeaders() {
            return this.allowedRequestHeaders;
        }

        public long maxAge() {
            return this.maxAge;
        }

        public Optional<Set<String>> origins() {
            return this.origins;
        }

        public String toString() {
            return "Config{enabled=" + this.enabled + ", origins=" + this.origins + ", pattern=" + this.pattern + ", anyOrigin=" + this.anyOrigin + ", credentialsAllowed=" + this.credentialsAllowed + ", allowedRequestMethods=" + this.allowedRequestMethods + ", allowedRequestHeaders=" + this.allowedRequestHeaders + ", maxAge=" + this.maxAge + "}";
        }

        private static class Builder {
            private boolean enabled = true;
            private Optional<Set<String>> origins;
            private Optional<Pattern> pattern;
            private final boolean anyOrigin;
            private boolean allowCredentials = false;
            long maxAge;
            private final Set<RestRequest.Method> requestMethods = new HashSet<RestRequest.Method>();
            private final Set<String> requestHeaders = new HashSet<String>();

            private Builder() {
                this.anyOrigin = true;
                this.origins = Optional.empty();
                this.pattern = Optional.empty();
            }

            private Builder(String ... origins) {
                this.origins = Optional.of(new LinkedHashSet<String>(Arrays.asList(origins)));
                this.pattern = Optional.empty();
                this.anyOrigin = false;
            }

            private Builder(Pattern pattern) {
                this.pattern = Optional.of(pattern);
                this.origins = Optional.empty();
                this.anyOrigin = false;
            }

            static Builder forOrigins(String ... origins) {
                return new Builder(origins);
            }

            static Builder forAnyOrigin() {
                return new Builder();
            }

            static Builder forPattern(Pattern pattern) {
                return new Builder(pattern);
            }

            Builder allowCredentials() {
                this.allowCredentials = true;
                return this;
            }

            public Builder allowedRequestMethods(RestRequest.Method[] methods) {
                this.requestMethods.addAll(Arrays.asList(methods));
                return this;
            }

            public Builder maxAge(int maxAge) {
                this.maxAge = maxAge;
                return this;
            }

            public Builder allowedRequestHeaders(String[] headers) {
                this.requestHeaders.addAll(Arrays.asList(headers));
                return this;
            }

            public Config build() {
                return new Config(this);
            }
        }
    }
}

