/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.util.bkd;

import java.io.Closeable;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.IntFunction;
import org.apache.lucene.codecs.CodecUtil;
import org.apache.lucene.codecs.MutablePointTree;
import org.apache.lucene.index.MergeState;
import org.apache.lucene.index.PointValues;
import org.apache.lucene.store.ByteBuffersDataOutput;
import org.apache.lucene.store.ChecksumIndexInput;
import org.apache.lucene.store.DataOutput;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.TrackingDirectoryWrapper;
import org.apache.lucene.util.ArrayUtil;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.BytesRefBuilder;
import org.apache.lucene.util.FixedBitSet;
import org.apache.lucene.util.IOUtils;
import org.apache.lucene.util.NumericUtils;
import org.apache.lucene.util.PriorityQueue;
import org.apache.lucene.util.bkd.BKDConfig;
import org.apache.lucene.util.bkd.BKDRadixSelector;
import org.apache.lucene.util.bkd.BKDUtil;
import org.apache.lucene.util.bkd.DocIdsWriter;
import org.apache.lucene.util.bkd.HeapPointWriter;
import org.apache.lucene.util.bkd.MutablePointTreeReaderUtils;
import org.apache.lucene.util.bkd.OfflinePointWriter;
import org.apache.lucene.util.bkd.PointReader;
import org.apache.lucene.util.bkd.PointValue;
import org.apache.lucene.util.bkd.PointWriter;

public class BKDWriter
implements Closeable {
    public static final String CODEC_NAME = "BKD";
    public static final int VERSION_START = 4;
    public static final int VERSION_LEAF_STORES_BOUNDS = 5;
    public static final int VERSION_SELECTIVE_INDEXING = 6;
    public static final int VERSION_LOW_CARDINALITY_LEAVES = 7;
    public static final int VERSION_META_FILE = 9;
    public static final int VERSION_CURRENT = 9;
    private static final int SPLITS_BEFORE_EXACT_BOUNDS = 4;
    public static final float DEFAULT_MAX_MB_SORT_IN_HEAP = 16.0f;
    protected final BKDConfig config;
    private final ArrayUtil.ByteArrayComparator comparator;
    private final BKDUtil.ByteArrayPredicate equalsPredicate;
    private final ArrayUtil.ByteArrayComparator commonPrefixComparator;
    final TrackingDirectoryWrapper tempDir;
    final String tempFileNamePrefix;
    final double maxMBSortInHeap;
    final byte[] scratchDiff;
    final byte[] scratch1;
    final byte[] scratch2;
    final BytesRef scratchBytesRef1 = new BytesRef();
    final BytesRef scratchBytesRef2 = new BytesRef();
    final int[] commonPrefixLengths;
    protected final FixedBitSet docsSeen;
    private PointWriter pointWriter;
    private boolean finished;
    private IndexOutput tempInput;
    private final int maxPointsSortInHeap;
    protected final byte[] minPackedValue;
    protected final byte[] maxPackedValue;
    protected long pointCount;
    private final long totalPointCount;
    private final int maxDoc;
    private final DocIdsWriter docIdsWriter;

    public BKDWriter(int maxDoc, Directory tempDir, String tempFileNamePrefix, BKDConfig config, double maxMBSortInHeap, long totalPointCount) {
        BKDWriter.verifyParams(maxMBSortInHeap, totalPointCount);
        this.tempDir = new TrackingDirectoryWrapper(tempDir);
        this.tempFileNamePrefix = tempFileNamePrefix;
        this.maxMBSortInHeap = maxMBSortInHeap;
        this.totalPointCount = totalPointCount;
        this.maxDoc = maxDoc;
        this.config = config;
        this.comparator = ArrayUtil.getUnsignedComparator(config.bytesPerDim);
        this.equalsPredicate = BKDUtil.getEqualsPredicate(config.bytesPerDim);
        this.commonPrefixComparator = BKDUtil.getPrefixLengthComparator(config.bytesPerDim);
        this.docsSeen = new FixedBitSet(maxDoc);
        this.scratchDiff = new byte[config.bytesPerDim];
        this.scratch1 = new byte[config.packedBytesLength];
        this.scratch2 = new byte[config.packedBytesLength];
        this.commonPrefixLengths = new int[config.numDims];
        this.minPackedValue = new byte[config.packedIndexBytesLength];
        this.maxPackedValue = new byte[config.packedIndexBytesLength];
        this.maxPointsSortInHeap = (int)(maxMBSortInHeap * 1024.0 * 1024.0 / (double)config.bytesPerDoc);
        this.docIdsWriter = new DocIdsWriter(config.maxPointsInLeafNode);
        if (this.maxPointsSortInHeap < config.maxPointsInLeafNode) {
            throw new IllegalArgumentException("maxMBSortInHeap=" + maxMBSortInHeap + " only allows for maxPointsSortInHeap=" + this.maxPointsSortInHeap + ", but this is less than maxPointsInLeafNode=" + config.maxPointsInLeafNode + "; either increase maxMBSortInHeap or decrease maxPointsInLeafNode");
        }
    }

    private static void verifyParams(double maxMBSortInHeap, long totalPointCount) {
        if (maxMBSortInHeap < 0.0) {
            throw new IllegalArgumentException("maxMBSortInHeap must be >= 0.0 (got: " + maxMBSortInHeap + ")");
        }
        if (totalPointCount < 0L) {
            throw new IllegalArgumentException("totalPointCount must be >=0 (got: " + totalPointCount + ")");
        }
    }

    private void initPointWriter() throws IOException {
        assert (this.pointWriter == null) : "Point writer is already initialized";
        if (this.totalPointCount > (long)this.maxPointsSortInHeap) {
            this.pointWriter = new OfflinePointWriter(this.config, this.tempDir, this.tempFileNamePrefix, "spill", 0L);
            this.tempInput = ((OfflinePointWriter)this.pointWriter).out;
        } else {
            this.pointWriter = new HeapPointWriter(this.config, Math.toIntExact(this.totalPointCount));
        }
    }

    public void add(byte[] packedValue, int docID) throws IOException {
        if (packedValue.length != this.config.packedBytesLength) {
            throw new IllegalArgumentException("packedValue should be length=" + this.config.packedBytesLength + " (got: " + packedValue.length + ")");
        }
        if (this.pointCount >= this.totalPointCount) {
            throw new IllegalStateException("totalPointCount=" + this.totalPointCount + " was passed when we were created, but we just hit " + (this.pointCount + 1L) + " values");
        }
        if (this.pointCount == 0L) {
            this.initPointWriter();
            System.arraycopy(packedValue, 0, this.minPackedValue, 0, this.config.packedIndexBytesLength);
            System.arraycopy(packedValue, 0, this.maxPackedValue, 0, this.config.packedIndexBytesLength);
        } else {
            for (int dim = 0; dim < this.config.numIndexDims; ++dim) {
                int offset = dim * this.config.bytesPerDim;
                if (this.comparator.compare(packedValue, offset, this.minPackedValue, offset) < 0) {
                    System.arraycopy(packedValue, offset, this.minPackedValue, offset, this.config.bytesPerDim);
                    continue;
                }
                if (this.comparator.compare(packedValue, offset, this.maxPackedValue, offset) <= 0) continue;
                System.arraycopy(packedValue, offset, this.maxPackedValue, offset, this.config.bytesPerDim);
            }
        }
        this.pointWriter.append(packedValue, docID);
        ++this.pointCount;
        this.docsSeen.set(docID);
    }

    public Runnable writeField(IndexOutput metaOut, IndexOutput indexOut, IndexOutput dataOut, String fieldName, MutablePointTree reader) throws IOException {
        if (this.config.numDims == 1) {
            return this.writeField1Dim(metaOut, indexOut, dataOut, fieldName, reader);
        }
        return this.writeFieldNDims(metaOut, indexOut, dataOut, fieldName, reader);
    }

    private void computePackedValueBounds(MutablePointTree values, int from, int to, byte[] minPackedValue, byte[] maxPackedValue, BytesRef scratch) {
        if (from == to) {
            return;
        }
        values.getValue(from, scratch);
        System.arraycopy(scratch.bytes, scratch.offset, minPackedValue, 0, this.config.packedIndexBytesLength);
        System.arraycopy(scratch.bytes, scratch.offset, maxPackedValue, 0, this.config.packedIndexBytesLength);
        for (int i = from + 1; i < to; ++i) {
            values.getValue(i, scratch);
            for (int dim = 0; dim < this.config.numIndexDims; ++dim) {
                int startOffset = dim * this.config.bytesPerDim;
                int endOffset = startOffset + this.config.bytesPerDim;
                if (Arrays.compareUnsigned(scratch.bytes, scratch.offset + startOffset, scratch.offset + endOffset, minPackedValue, startOffset, endOffset) < 0) {
                    System.arraycopy(scratch.bytes, scratch.offset + startOffset, minPackedValue, startOffset, this.config.bytesPerDim);
                    continue;
                }
                if (Arrays.compareUnsigned(scratch.bytes, scratch.offset + startOffset, scratch.offset + endOffset, maxPackedValue, startOffset, endOffset) <= 0) continue;
                System.arraycopy(scratch.bytes, scratch.offset + startOffset, maxPackedValue, startOffset, this.config.bytesPerDim);
            }
        }
    }

    private Runnable writeFieldNDims(IndexOutput metaOut, IndexOutput indexOut, IndexOutput dataOut, String fieldName, MutablePointTree values) throws IOException {
        if (this.pointCount != 0L) {
            throw new IllegalStateException("cannot mix add and writeField");
        }
        if (this.finished) {
            throw new IllegalStateException("already finished");
        }
        this.finished = true;
        this.pointCount = values.size();
        int numLeaves = Math.toIntExact((this.pointCount + (long)this.config.maxPointsInLeafNode - 1L) / (long)this.config.maxPointsInLeafNode);
        int numSplits = numLeaves - 1;
        this.checkMaxLeafNodeCount(numLeaves);
        byte[] splitPackedValues = new byte[numSplits * this.config.bytesPerDim];
        final byte[] splitDimensionValues = new byte[numSplits];
        final long[] leafBlockFPs = new long[numLeaves];
        this.computePackedValueBounds(values, 0, Math.toIntExact(this.pointCount), this.minPackedValue, this.maxPackedValue, this.scratchBytesRef1);
        for (int i = 0; i < Math.toIntExact(this.pointCount); ++i) {
            this.docsSeen.set(values.getDocID(i));
        }
        long dataStartFP = dataOut.getFilePointer();
        int[] parentSplits = new int[this.config.numIndexDims];
        this.build(0, numLeaves, values, 0, Math.toIntExact(this.pointCount), dataOut, (byte[])this.minPackedValue.clone(), (byte[])this.maxPackedValue.clone(), parentSplits, splitPackedValues, splitDimensionValues, leafBlockFPs, new int[this.config.maxPointsInLeafNode]);
        assert (Arrays.equals(parentSplits, new int[this.config.numIndexDims]));
        this.scratchBytesRef1.length = this.config.bytesPerDim;
        this.scratchBytesRef1.bytes = splitPackedValues;
        BKDTreeLeafNodes leafNodes = new BKDTreeLeafNodes(){

            @Override
            public long getLeafLP(int index) {
                return leafBlockFPs[index];
            }

            @Override
            public BytesRef getSplitValue(int index) {
                BKDWriter.this.scratchBytesRef1.offset = index * BKDWriter.this.config.bytesPerDim;
                return BKDWriter.this.scratchBytesRef1;
            }

            @Override
            public int getSplitDimension(int index) {
                return splitDimensionValues[index] & 0xFF;
            }

            @Override
            public int numLeaves() {
                return leafBlockFPs.length;
            }
        };
        return () -> {
            try {
                this.writeIndex(metaOut, indexOut, this.config.maxPointsInLeafNode, leafNodes, dataStartFP);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        };
    }

    private Runnable writeField1Dim(IndexOutput metaOut, IndexOutput indexOut, IndexOutput dataOut, String fieldName, MutablePointTree reader) throws IOException {
        MutablePointTreeReaderUtils.sort(this.config, this.maxDoc, reader, 0, Math.toIntExact(reader.size()));
        final OneDimensionBKDWriter oneDimWriter = new OneDimensionBKDWriter(metaOut, indexOut, dataOut);
        reader.visitDocValues(new PointValues.IntersectVisitor(){

            @Override
            public void visit(int docID, byte[] packedValue) throws IOException {
                oneDimWriter.add(packedValue, docID);
            }

            @Override
            public void visit(int docID) {
                throw new IllegalStateException();
            }

            @Override
            public PointValues.Relation compare(byte[] minPackedValue, byte[] maxPackedValue) {
                return PointValues.Relation.CELL_CROSSES_QUERY;
            }
        });
        return oneDimWriter.finish();
    }

    public Runnable merge(IndexOutput metaOut, IndexOutput indexOut, IndexOutput dataOut, List<MergeState.DocMap> docMaps, List<PointValues> readers) throws IOException {
        assert (docMaps == null || readers.size() == docMaps.size());
        BKDMergeQueue queue = new BKDMergeQueue(this.config.bytesPerDim, readers.size());
        for (int i = 0; i < readers.size(); ++i) {
            PointValues pointValues = readers.get(i);
            assert (pointValues.getNumDimensions() == this.config.numDims && pointValues.getBytesPerDimension() == this.config.bytesPerDim && pointValues.getNumIndexDimensions() == this.config.numIndexDims);
            MergeState.DocMap docMap = docMaps == null ? null : docMaps.get(i);
            MergeReader reader = new MergeReader(pointValues, docMap);
            if (!reader.next()) continue;
            queue.add(reader);
        }
        OneDimensionBKDWriter oneDimWriter = new OneDimensionBKDWriter(metaOut, indexOut, dataOut);
        while (queue.size() != 0) {
            MergeReader reader = (MergeReader)queue.top();
            oneDimWriter.add(reader.packedValue, reader.docID);
            if (reader.next()) {
                queue.updateTop();
                continue;
            }
            queue.pop();
        }
        return oneDimWriter.finish();
    }

    private int getNumLeftLeafNodes(int numLeaves) {
        assert (numLeaves > 1) : "getNumLeftLeaveNodes() called with " + numLeaves;
        int lastFullLevel = 31 - Integer.numberOfLeadingZeros(numLeaves);
        int leavesFullLevel = 1 << lastFullLevel;
        int numLeftLeafNodes = leavesFullLevel / 2;
        int unbalancedLeafNodes = numLeaves - leavesFullLevel;
        numLeftLeafNodes += Math.min(unbalancedLeafNodes, numLeftLeafNodes);
        assert (numLeftLeafNodes >= numLeaves - numLeftLeafNodes && (long)numLeftLeafNodes <= 2L * (long)(numLeaves - numLeftLeafNodes));
        return numLeftLeafNodes;
    }

    private void checkMaxLeafNodeCount(int numLeaves) {
        if ((long)this.config.bytesPerDim * (long)numLeaves > (long)ArrayUtil.MAX_ARRAY_LENGTH) {
            throw new IllegalStateException("too many nodes; increase config.maxPointsInLeafNode (currently " + this.config.maxPointsInLeafNode + ") and reindex");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Runnable finish(IndexOutput metaOut, IndexOutput indexOut, IndexOutput dataOut) throws IOException {
        if (this.finished) {
            throw new IllegalStateException("already finished");
        }
        if (this.pointCount == 0L) {
            return null;
        }
        this.finished = true;
        this.pointWriter.close();
        BKDRadixSelector.PathSlice points = new BKDRadixSelector.PathSlice(this.pointWriter, 0L, this.pointCount);
        this.tempInput = null;
        this.pointWriter = null;
        int numLeaves = Math.toIntExact((this.pointCount + (long)this.config.maxPointsInLeafNode - 1L) / (long)this.config.maxPointsInLeafNode);
        int numSplits = numLeaves - 1;
        this.checkMaxLeafNodeCount(numLeaves);
        byte[] splitPackedValues = new byte[Math.toIntExact(numSplits * this.config.bytesPerDim)];
        final byte[] splitDimensionValues = new byte[numSplits];
        final long[] leafBlockFPs = new long[numLeaves];
        assert (this.pointCount / (long)numLeaves <= (long)this.config.maxPointsInLeafNode) : "pointCount=" + this.pointCount + " numLeaves=" + numLeaves + " config.maxPointsInLeafNode=" + this.config.maxPointsInLeafNode;
        BKDRadixSelector radixSelector = new BKDRadixSelector(this.config, this.maxPointsSortInHeap, this.tempDir, this.tempFileNamePrefix);
        long dataStartFP = dataOut.getFilePointer();
        boolean success = false;
        try {
            int[] parentSplits = new int[this.config.numIndexDims];
            this.build(0, numLeaves, points, dataOut, radixSelector, (byte[])this.minPackedValue.clone(), (byte[])this.maxPackedValue.clone(), parentSplits, splitPackedValues, splitDimensionValues, leafBlockFPs, new int[this.config.maxPointsInLeafNode]);
            assert (Arrays.equals(parentSplits, new int[this.config.numIndexDims]));
            assert (this.tempDir.getCreatedFiles().isEmpty());
            success = true;
        }
        finally {
            if (!success) {
                IOUtils.deleteFilesIgnoringExceptions((Directory)this.tempDir, this.tempDir.getCreatedFiles());
            }
        }
        this.scratchBytesRef1.bytes = splitPackedValues;
        this.scratchBytesRef1.length = this.config.bytesPerDim;
        BKDTreeLeafNodes leafNodes = new BKDTreeLeafNodes(){

            @Override
            public long getLeafLP(int index) {
                return leafBlockFPs[index];
            }

            @Override
            public BytesRef getSplitValue(int index) {
                BKDWriter.this.scratchBytesRef1.offset = index * BKDWriter.this.config.bytesPerDim;
                return BKDWriter.this.scratchBytesRef1;
            }

            @Override
            public int getSplitDimension(int index) {
                return splitDimensionValues[index] & 0xFF;
            }

            @Override
            public int numLeaves() {
                return leafBlockFPs.length;
            }
        };
        return () -> {
            try {
                this.writeIndex(metaOut, indexOut, this.config.maxPointsInLeafNode, leafNodes, dataStartFP);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        };
    }

    private byte[] packIndex(BKDTreeLeafNodes leafNodes) throws IOException {
        ByteBuffersDataOutput writeBuffer = ByteBuffersDataOutput.newResettableInstance();
        ArrayList<byte[]> blocks = new ArrayList<byte[]>();
        byte[] lastSplitValues = new byte[this.config.bytesPerDim * this.config.numIndexDims];
        int totalSize = this.recursePackIndex(writeBuffer, leafNodes, 0L, blocks, lastSplitValues, new boolean[this.config.numIndexDims], false, 0, leafNodes.numLeaves());
        byte[] index = new byte[totalSize];
        int upto = 0;
        for (byte[] block : blocks) {
            System.arraycopy(block, 0, index, upto, block.length);
            upto += block.length;
        }
        assert (upto == totalSize);
        return index;
    }

    private int appendBlock(ByteBuffersDataOutput writeBuffer, List<byte[]> blocks) {
        byte[] block = writeBuffer.toArrayCopy();
        blocks.add(block);
        writeBuffer.reset();
        return block.length;
    }

    private int recursePackIndex(ByteBuffersDataOutput writeBuffer, BKDTreeLeafNodes leafNodes, long minBlockFP, List<byte[]> blocks, byte[] lastSplitValues, boolean[] negativeDeltas, boolean isLeft, int leavesOffset, int numLeaves) throws IOException {
        int firstDiffByteDelta;
        long leftBlockFP;
        if (numLeaves == 1) {
            if (isLeft) {
                assert (leafNodes.getLeafLP(leavesOffset) - minBlockFP == 0L);
                return 0;
            }
            long delta = leafNodes.getLeafLP(leavesOffset) - minBlockFP;
            assert (leafNodes.numLeaves() == numLeaves || delta > 0L) : "expected delta > 0; got numLeaves =" + numLeaves + " and delta=" + delta;
            writeBuffer.writeVLong(delta);
            return this.appendBlock(writeBuffer, blocks);
        }
        if (isLeft) {
            assert (leafNodes.getLeafLP(leavesOffset) == minBlockFP);
            leftBlockFP = minBlockFP;
        } else {
            leftBlockFP = leafNodes.getLeafLP(leavesOffset);
            long delta = leftBlockFP - minBlockFP;
            assert (leafNodes.numLeaves() == numLeaves || delta > 0L) : "expected delta > 0; got numLeaves =" + numLeaves + " and delta=" + delta;
            writeBuffer.writeVLong(delta);
        }
        int numLeftLeafNodes = this.getNumLeftLeafNodes(numLeaves);
        int rightOffset = leavesOffset + numLeftLeafNodes;
        int splitOffset = rightOffset - 1;
        int splitDim = leafNodes.getSplitDimension(splitOffset);
        BytesRef splitValue = leafNodes.getSplitValue(splitOffset);
        int address = splitValue.offset;
        int prefix = this.commonPrefixComparator.compare(splitValue.bytes, address, lastSplitValues, splitDim * this.config.bytesPerDim);
        if (prefix < this.config.bytesPerDim) {
            firstDiffByteDelta = (splitValue.bytes[address + prefix] & 0xFF) - (lastSplitValues[splitDim * this.config.bytesPerDim + prefix] & 0xFF);
            if (negativeDeltas[splitDim]) {
                firstDiffByteDelta = -firstDiffByteDelta;
            }
            assert (firstDiffByteDelta > 0);
        } else {
            firstDiffByteDelta = 0;
        }
        int code = (firstDiffByteDelta * (1 + this.config.bytesPerDim) + prefix) * this.config.numIndexDims + splitDim;
        writeBuffer.writeVInt(code);
        int suffix = this.config.bytesPerDim - prefix;
        byte[] savSplitValue = new byte[suffix];
        if (suffix > 1) {
            writeBuffer.writeBytes(splitValue.bytes, address + prefix + 1, suffix - 1);
        }
        byte[] cmp = (byte[])lastSplitValues.clone();
        System.arraycopy(lastSplitValues, splitDim * this.config.bytesPerDim + prefix, savSplitValue, 0, suffix);
        System.arraycopy(splitValue.bytes, address + prefix, lastSplitValues, splitDim * this.config.bytesPerDim + prefix, suffix);
        int numBytes = this.appendBlock(writeBuffer, blocks);
        int idxSav = blocks.size();
        blocks.add(null);
        boolean savNegativeDelta = negativeDeltas[splitDim];
        negativeDeltas[splitDim] = true;
        int leftNumBytes = this.recursePackIndex(writeBuffer, leafNodes, leftBlockFP, blocks, lastSplitValues, negativeDeltas, true, leavesOffset, numLeftLeafNodes);
        if (numLeftLeafNodes != 1) {
            writeBuffer.writeVInt(leftNumBytes);
        } else assert (leftNumBytes == 0) : "leftNumBytes=" + leftNumBytes;
        byte[] bytes2 = writeBuffer.toArrayCopy();
        writeBuffer.reset();
        blocks.set(idxSav, bytes2);
        negativeDeltas[splitDim] = false;
        int rightNumBytes = this.recursePackIndex(writeBuffer, leafNodes, leftBlockFP, blocks, lastSplitValues, negativeDeltas, false, rightOffset, numLeaves - numLeftLeafNodes);
        negativeDeltas[splitDim] = savNegativeDelta;
        System.arraycopy(savSplitValue, 0, lastSplitValues, splitDim * this.config.bytesPerDim + prefix, suffix);
        assert (Arrays.equals(lastSplitValues, cmp));
        return numBytes + bytes2.length + leftNumBytes + rightNumBytes;
    }

    private void writeIndex(IndexOutput metaOut, IndexOutput indexOut, int countPerLeaf, BKDTreeLeafNodes leafNodes, long dataStartFP) throws IOException {
        byte[] packedIndex = this.packIndex(leafNodes);
        this.writeIndex(metaOut, indexOut, countPerLeaf, leafNodes.numLeaves(), packedIndex, dataStartFP);
    }

    private void writeIndex(IndexOutput metaOut, IndexOutput indexOut, int countPerLeaf, int numLeaves, byte[] packedIndex, long dataStartFP) throws IOException {
        CodecUtil.writeHeader(metaOut, CODEC_NAME, 9);
        metaOut.writeVInt(this.config.numDims);
        metaOut.writeVInt(this.config.numIndexDims);
        metaOut.writeVInt(countPerLeaf);
        metaOut.writeVInt(this.config.bytesPerDim);
        assert (numLeaves > 0);
        metaOut.writeVInt(numLeaves);
        metaOut.writeBytes(this.minPackedValue, 0, this.config.packedIndexBytesLength);
        metaOut.writeBytes(this.maxPackedValue, 0, this.config.packedIndexBytesLength);
        metaOut.writeVLong(this.pointCount);
        metaOut.writeVInt(this.docsSeen.cardinality());
        metaOut.writeVInt(packedIndex.length);
        metaOut.writeLong(dataStartFP);
        metaOut.writeLong(indexOut.getFilePointer() + (long)(metaOut == indexOut ? 8 : 0));
        indexOut.writeBytes(packedIndex, 0, packedIndex.length);
    }

    private void writeLeafBlockDocs(DataOutput out, int[] docIDs, int start, int count) throws IOException {
        assert (count > 0) : "config.maxPointsInLeafNode=" + this.config.maxPointsInLeafNode;
        out.writeVInt(count);
        this.docIdsWriter.writeDocIds(docIDs, start, count, out);
    }

    private void writeLeafBlockPackedValues(DataOutput out, int[] commonPrefixLengths, int count, int sortedDim, IntFunction<BytesRef> packedValues, int leafCardinality) throws IOException {
        int prefixLenSum = Arrays.stream(commonPrefixLengths).sum();
        if (prefixLenSum == this.config.packedBytesLength) {
            out.writeByte((byte)-1);
        } else {
            int lowCardinalityCost;
            int highCardinalityCost;
            assert (commonPrefixLengths[sortedDim] < this.config.bytesPerDim);
            int compressedByteOffset = sortedDim * this.config.bytesPerDim + commonPrefixLengths[sortedDim];
            if (count == leafCardinality) {
                highCardinalityCost = 0;
                lowCardinalityCost = 1;
            } else {
                int runLen;
                int numRunLens = 0;
                for (int i = 0; i < count; i += runLen) {
                    runLen = BKDWriter.runLen(packedValues, i, Math.min(i + 255, count), compressedByteOffset);
                    assert (runLen <= 255);
                    ++numRunLens;
                }
                highCardinalityCost = count * (this.config.packedBytesLength - prefixLenSum - 1) + 2 * numRunLens;
                lowCardinalityCost = leafCardinality * (this.config.packedBytesLength - prefixLenSum + 1);
            }
            if (lowCardinalityCost <= highCardinalityCost) {
                out.writeByte((byte)-2);
                this.writeLowCardinalityLeafBlockPackedValues(out, commonPrefixLengths, count, packedValues);
            } else {
                out.writeByte((byte)sortedDim);
                this.writeHighCardinalityLeafBlockPackedValues(out, commonPrefixLengths, count, sortedDim, packedValues, compressedByteOffset);
            }
        }
    }

    private void writeLowCardinalityLeafBlockPackedValues(DataOutput out, int[] commonPrefixLengths, int count, IntFunction<BytesRef> packedValues) throws IOException {
        int i;
        if (this.config.numIndexDims != 1) {
            this.writeActualBounds(out, commonPrefixLengths, count, packedValues);
        }
        BytesRef value = packedValues.apply(0);
        System.arraycopy(value.bytes, value.offset, this.scratch1, 0, this.config.packedBytesLength);
        int cardinality = 1;
        block0: for (i = 1; i < count; ++i) {
            value = packedValues.apply(i);
            for (int dim = 0; dim < this.config.numDims; ++dim) {
                int start = dim * this.config.bytesPerDim;
                if (!this.equalsPredicate.test(value.bytes, value.offset + start, this.scratch1, start)) {
                    out.writeVInt(cardinality);
                    for (int j = 0; j < this.config.numDims; ++j) {
                        out.writeBytes(this.scratch1, j * this.config.bytesPerDim + commonPrefixLengths[j], this.config.bytesPerDim - commonPrefixLengths[j]);
                    }
                    System.arraycopy(value.bytes, value.offset, this.scratch1, 0, this.config.packedBytesLength);
                    cardinality = 1;
                    continue block0;
                }
                if (dim != this.config.numDims - 1) continue;
                ++cardinality;
            }
        }
        out.writeVInt(cardinality);
        for (i = 0; i < this.config.numDims; ++i) {
            out.writeBytes(this.scratch1, i * this.config.bytesPerDim + commonPrefixLengths[i], this.config.bytesPerDim - commonPrefixLengths[i]);
        }
    }

    private void writeHighCardinalityLeafBlockPackedValues(DataOutput out, int[] commonPrefixLengths, int count, int sortedDim, IntFunction<BytesRef> packedValues, int compressedByteOffset) throws IOException {
        if (this.config.numIndexDims != 1) {
            this.writeActualBounds(out, commonPrefixLengths, count, packedValues);
        }
        int n = sortedDim;
        commonPrefixLengths[n] = commonPrefixLengths[n] + 1;
        int i = 0;
        while (i < count) {
            int runLen = BKDWriter.runLen(packedValues, i, Math.min(i + 255, count), compressedByteOffset);
            assert (runLen <= 255);
            BytesRef first = packedValues.apply(i);
            byte prefixByte = first.bytes[first.offset + compressedByteOffset];
            out.writeByte(prefixByte);
            out.writeByte((byte)runLen);
            this.writeLeafBlockPackedValuesRange(out, commonPrefixLengths, i, i + runLen, packedValues);
            assert ((i += runLen) <= count);
        }
    }

    private void writeActualBounds(DataOutput out, int[] commonPrefixLengths, int count, IntFunction<BytesRef> packedValues) throws IOException {
        for (int dim = 0; dim < this.config.numIndexDims; ++dim) {
            int commonPrefixLength = commonPrefixLengths[dim];
            int suffixLength = this.config.bytesPerDim - commonPrefixLength;
            if (suffixLength <= 0) continue;
            BytesRef[] minMax = BKDWriter.computeMinMax(count, packedValues, dim * this.config.bytesPerDim + commonPrefixLength, suffixLength);
            BytesRef min = minMax[0];
            BytesRef max = minMax[1];
            out.writeBytes(min.bytes, min.offset, min.length);
            out.writeBytes(max.bytes, max.offset, max.length);
        }
    }

    private static BytesRef[] computeMinMax(int count, IntFunction<BytesRef> packedValues, int offset, int length) {
        assert (length > 0);
        BytesRefBuilder min = new BytesRefBuilder();
        BytesRefBuilder max = new BytesRefBuilder();
        BytesRef first = packedValues.apply(0);
        min.copyBytes(first.bytes, first.offset + offset, length);
        max.copyBytes(first.bytes, first.offset + offset, length);
        for (int i = 1; i < count; ++i) {
            BytesRef candidate = packedValues.apply(i);
            if (Arrays.compareUnsigned(min.bytes(), 0, length, candidate.bytes, candidate.offset + offset, candidate.offset + offset + length) > 0) {
                min.copyBytes(candidate.bytes, candidate.offset + offset, length);
                continue;
            }
            if (Arrays.compareUnsigned(max.bytes(), 0, length, candidate.bytes, candidate.offset + offset, candidate.offset + offset + length) >= 0) continue;
            max.copyBytes(candidate.bytes, candidate.offset + offset, length);
        }
        return new BytesRef[]{min.get(), max.get()};
    }

    private void writeLeafBlockPackedValuesRange(DataOutput out, int[] commonPrefixLengths, int start, int end, IntFunction<BytesRef> packedValues) throws IOException {
        for (int i = start; i < end; ++i) {
            BytesRef ref = packedValues.apply(i);
            assert (ref.length == this.config.packedBytesLength);
            for (int dim = 0; dim < this.config.numDims; ++dim) {
                int prefix = commonPrefixLengths[dim];
                out.writeBytes(ref.bytes, ref.offset + dim * this.config.bytesPerDim + prefix, this.config.bytesPerDim - prefix);
            }
        }
    }

    private static int runLen(IntFunction<BytesRef> packedValues, int start, int end, int byteOffset) {
        BytesRef first = packedValues.apply(start);
        byte b = first.bytes[first.offset + byteOffset];
        for (int i = start + 1; i < end; ++i) {
            BytesRef ref = packedValues.apply(i);
            byte b2 = ref.bytes[ref.offset + byteOffset];
            assert (Byte.toUnsignedInt(b2) >= Byte.toUnsignedInt(b));
            if (b == b2) continue;
            return i - start;
        }
        return end - start;
    }

    private void writeCommonPrefixes(DataOutput out, int[] commonPrefixes, byte[] packedValue) throws IOException {
        for (int dim = 0; dim < this.config.numDims; ++dim) {
            out.writeVInt(commonPrefixes[dim]);
            out.writeBytes(packedValue, dim * this.config.bytesPerDim, commonPrefixes[dim]);
        }
    }

    @Override
    public void close() throws IOException {
        this.finished = true;
        if (this.tempInput != null) {
            try {
                this.tempInput.close();
            }
            finally {
                this.tempDir.deleteFile(this.tempInput.getName());
                this.tempInput = null;
            }
        }
    }

    private Error verifyChecksum(Throwable priorException, PointWriter writer) throws IOException {
        assert (priorException != null);
        if (writer instanceof OfflinePointWriter) {
            String tempFileName = ((OfflinePointWriter)writer).name;
            if (this.tempDir.getCreatedFiles().contains(tempFileName)) {
                try (ChecksumIndexInput in = this.tempDir.openChecksumInput(tempFileName, IOContext.READONCE);){
                    CodecUtil.checkFooter(in, priorException);
                }
            }
        }
        throw IOUtils.rethrowAlways(priorException);
    }

    protected int split(byte[] minPackedValue, byte[] maxPackedValue, int[] parentSplits) {
        int maxNumSplits = 0;
        for (int numSplits : parentSplits) {
            maxNumSplits = Math.max(maxNumSplits, numSplits);
        }
        for (int dim = 0; dim < this.config.numIndexDims; ++dim) {
            int offset = dim * this.config.bytesPerDim;
            if (parentSplits[dim] >= maxNumSplits / 2 || this.comparator.compare(minPackedValue, offset, maxPackedValue, offset) == 0) continue;
            return dim;
        }
        int splitDim = -1;
        for (int dim = 0; dim < this.config.numIndexDims; ++dim) {
            NumericUtils.subtract(this.config.bytesPerDim, dim, maxPackedValue, minPackedValue, this.scratchDiff);
            if (splitDim != -1 && this.comparator.compare(this.scratchDiff, 0, this.scratch1, 0) <= 0) continue;
            System.arraycopy(this.scratchDiff, 0, this.scratch1, 0, this.config.bytesPerDim);
            splitDim = dim;
        }
        return splitDim;
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private HeapPointWriter switchToHeap(PointWriter source) throws IOException {
        int count = Math.toIntExact(source.count());
        try (PointReader reader = source.getReader(0L, source.count());){
            HeapPointWriter writer = new HeapPointWriter(this.config, count);
            try {
                for (int i = 0; i < count; ++i) {
                    boolean hasNext = reader.next();
                    assert (hasNext);
                    writer.append(reader.pointValue());
                }
                source.destroy();
                HeapPointWriter heapPointWriter = writer;
                writer.close();
                return heapPointWriter;
            }
            catch (Throwable throwable) {
                try {
                    writer.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (Throwable t) {
            throw this.verifyChecksum(t, source);
        }
    }

    private void build(int leavesOffset, int numLeaves, final MutablePointTree reader, final int from, int to, IndexOutput out, byte[] minPackedValue, byte[] maxPackedValue, int[] parentSplits, byte[] splitPackedValues, byte[] splitDimensionValues, long[] leafBlockFPs, int[] spareDocIds) throws IOException {
        if (numLeaves == 1) {
            int dim;
            int count = to - from;
            assert (count <= this.config.maxPointsInLeafNode);
            Arrays.fill(this.commonPrefixLengths, this.config.bytesPerDim);
            reader.getValue(from, this.scratchBytesRef1);
            for (int i = from + 1; i < to; ++i) {
                reader.getValue(i, this.scratchBytesRef2);
                for (dim = 0; dim < this.config.numDims; ++dim) {
                    int offset = dim * this.config.bytesPerDim;
                    int dimensionPrefixLength = this.commonPrefixLengths[dim];
                    this.commonPrefixLengths[dim] = Math.min(dimensionPrefixLength, this.commonPrefixComparator.compare(this.scratchBytesRef1.bytes, this.scratchBytesRef1.offset + offset, this.scratchBytesRef2.bytes, this.scratchBytesRef2.offset + offset));
                }
            }
            FixedBitSet[] usedBytes = new FixedBitSet[this.config.numDims];
            for (dim = 0; dim < this.config.numDims; ++dim) {
                if (this.commonPrefixLengths[dim] >= this.config.bytesPerDim) continue;
                usedBytes[dim] = new FixedBitSet(256);
            }
            for (int i = from + 1; i < to; ++i) {
                for (int dim2 = 0; dim2 < this.config.numDims; ++dim2) {
                    if (usedBytes[dim2] == null) continue;
                    byte b = reader.getByteAt(i, dim2 * this.config.bytesPerDim + this.commonPrefixLengths[dim2]);
                    usedBytes[dim2].set(Byte.toUnsignedInt(b));
                }
            }
            int sortedDim = 0;
            int sortedDimCardinality = Integer.MAX_VALUE;
            for (int dim3 = 0; dim3 < this.config.numDims; ++dim3) {
                int cardinality;
                if (usedBytes[dim3] == null || (cardinality = usedBytes[dim3].cardinality()) >= sortedDimCardinality) continue;
                sortedDim = dim3;
                sortedDimCardinality = cardinality;
            }
            MutablePointTreeReaderUtils.sortByDim(this.config, sortedDim, this.commonPrefixLengths, reader, from, to, this.scratchBytesRef1, this.scratchBytesRef2);
            BytesRef comparator = this.scratchBytesRef1;
            BytesRef collector = this.scratchBytesRef2;
            reader.getValue(from, comparator);
            int leafCardinality = 1;
            block6: for (int i = from + 1; i < to; ++i) {
                reader.getValue(i, collector);
                for (int dim4 = 0; dim4 < this.config.numDims; ++dim4) {
                    int start = dim4 * this.config.bytesPerDim;
                    if (this.equalsPredicate.test(collector.bytes, collector.offset + start, comparator.bytes, comparator.offset + start)) continue;
                    ++leafCardinality;
                    BytesRef scratch = collector;
                    collector = comparator;
                    comparator = scratch;
                    continue block6;
                }
            }
            leafBlockFPs[leavesOffset] = out.getFilePointer();
            int[] docIDs = spareDocIds;
            for (int i = from; i < to; ++i) {
                docIDs[i - from] = reader.getDocID(i);
            }
            this.writeLeafBlockDocs(out, docIDs, 0, count);
            reader.getValue(from, this.scratchBytesRef1);
            System.arraycopy(this.scratchBytesRef1.bytes, this.scratchBytesRef1.offset, this.scratch1, 0, this.config.packedBytesLength);
            this.writeCommonPrefixes(out, this.commonPrefixLengths, this.scratch1);
            IntFunction<BytesRef> packedValues = new IntFunction<BytesRef>(){

                @Override
                public BytesRef apply(int i) {
                    reader.getValue(from + i, BKDWriter.this.scratchBytesRef1);
                    return BKDWriter.this.scratchBytesRef1;
                }
            };
            assert (BKDWriter.valuesInOrderAndBounds(this.config, count, sortedDim, minPackedValue, maxPackedValue, packedValues, docIDs, 0));
            this.writeLeafBlockPackedValues(out, this.commonPrefixLengths, count, sortedDim, packedValues, leafCardinality);
        } else {
            int splitDim;
            if (this.config.numIndexDims == 1) {
                splitDim = 0;
            } else {
                if (numLeaves != leafBlockFPs.length && this.config.numIndexDims > 2 && Arrays.stream(parentSplits).sum() % 4 == 0) {
                    this.computePackedValueBounds(reader, from, to, minPackedValue, maxPackedValue, this.scratchBytesRef1);
                }
                splitDim = this.split(minPackedValue, maxPackedValue, parentSplits);
            }
            int numLeftLeafNodes = this.getNumLeftLeafNodes(numLeaves);
            int mid = from + numLeftLeafNodes * this.config.maxPointsInLeafNode;
            int commonPrefixLen = this.commonPrefixComparator.compare(minPackedValue, splitDim * this.config.bytesPerDim, maxPackedValue, splitDim * this.config.bytesPerDim);
            MutablePointTreeReaderUtils.partition(this.config, this.maxDoc, splitDim, commonPrefixLen, reader, from, to, mid, this.scratchBytesRef1, this.scratchBytesRef2);
            int rightOffset = leavesOffset + numLeftLeafNodes;
            int splitOffset = rightOffset - 1;
            int address = splitOffset * this.config.bytesPerDim;
            splitDimensionValues[splitOffset] = (byte)splitDim;
            reader.getValue(mid, this.scratchBytesRef1);
            System.arraycopy(this.scratchBytesRef1.bytes, this.scratchBytesRef1.offset + splitDim * this.config.bytesPerDim, splitPackedValues, address, this.config.bytesPerDim);
            byte[] minSplitPackedValue = ArrayUtil.copyOfSubArray(minPackedValue, 0, this.config.packedIndexBytesLength);
            byte[] maxSplitPackedValue = ArrayUtil.copyOfSubArray(maxPackedValue, 0, this.config.packedIndexBytesLength);
            System.arraycopy(this.scratchBytesRef1.bytes, this.scratchBytesRef1.offset + splitDim * this.config.bytesPerDim, minSplitPackedValue, splitDim * this.config.bytesPerDim, this.config.bytesPerDim);
            System.arraycopy(this.scratchBytesRef1.bytes, this.scratchBytesRef1.offset + splitDim * this.config.bytesPerDim, maxSplitPackedValue, splitDim * this.config.bytesPerDim, this.config.bytesPerDim);
            int n = splitDim;
            parentSplits[n] = parentSplits[n] + 1;
            this.build(leavesOffset, numLeftLeafNodes, reader, from, mid, out, minPackedValue, maxSplitPackedValue, parentSplits, splitPackedValues, splitDimensionValues, leafBlockFPs, spareDocIds);
            this.build(rightOffset, numLeaves - numLeftLeafNodes, reader, mid, to, out, minSplitPackedValue, maxPackedValue, parentSplits, splitPackedValues, splitDimensionValues, leafBlockFPs, spareDocIds);
            int n2 = splitDim;
            parentSplits[n2] = parentSplits[n2] - 1;
        }
    }

    private void computePackedValueBounds(BKDRadixSelector.PathSlice slice, byte[] minPackedValue, byte[] maxPackedValue) throws IOException {
        try (PointReader reader = slice.writer.getReader(slice.start, slice.count);){
            if (!reader.next()) {
                return;
            }
            BytesRef value = reader.pointValue().packedValue();
            System.arraycopy(value.bytes, value.offset, minPackedValue, 0, this.config.packedIndexBytesLength);
            System.arraycopy(value.bytes, value.offset, maxPackedValue, 0, this.config.packedIndexBytesLength);
            while (reader.next()) {
                value = reader.pointValue().packedValue();
                for (int dim = 0; dim < this.config.numIndexDims; ++dim) {
                    int startOffset = dim * this.config.bytesPerDim;
                    if (this.comparator.compare(value.bytes, value.offset + startOffset, minPackedValue, startOffset) < 0) {
                        System.arraycopy(value.bytes, value.offset + startOffset, minPackedValue, startOffset, this.config.bytesPerDim);
                        continue;
                    }
                    if (this.comparator.compare(value.bytes, value.offset + startOffset, maxPackedValue, startOffset) <= 0) continue;
                    System.arraycopy(value.bytes, value.offset + startOffset, maxPackedValue, startOffset, this.config.bytesPerDim);
                }
            }
        }
    }

    private void build(int leavesOffset, int numLeaves, BKDRadixSelector.PathSlice points, IndexOutput out, BKDRadixSelector radixSelector, byte[] minPackedValue, byte[] maxPackedValue, int[] parentSplits, byte[] splitPackedValues, byte[] splitDimensionValues, long[] leafBlockFPs, int[] spareDocIds) throws IOException {
        if (numLeaves == 1) {
            int i;
            int dim;
            final HeapPointWriter heapSource = !(points.writer instanceof HeapPointWriter) ? this.switchToHeap(points.writer) : (HeapPointWriter)points.writer;
            final int from = Math.toIntExact(points.start);
            int to = Math.toIntExact(points.start + points.count);
            this.computeCommonPrefixLength(heapSource, this.scratch1, from, to);
            int sortedDim = 0;
            int sortedDimCardinality = Integer.MAX_VALUE;
            FixedBitSet[] usedBytes = new FixedBitSet[this.config.numDims];
            for (dim = 0; dim < this.config.numDims; ++dim) {
                if (this.commonPrefixLengths[dim] >= this.config.bytesPerDim) continue;
                usedBytes[dim] = new FixedBitSet(256);
            }
            for (dim = 0; dim < this.config.numDims; ++dim) {
                int prefix = this.commonPrefixLengths[dim];
                if (prefix >= this.config.bytesPerDim) continue;
                int offset = dim * this.config.bytesPerDim;
                for (i = from; i < to; ++i) {
                    PointValue value = heapSource.getPackedValueSlice(i);
                    BytesRef packedValue = value.packedValue();
                    int bucket = packedValue.bytes[packedValue.offset + offset + prefix] & 0xFF;
                    usedBytes[dim].set(bucket);
                }
                int cardinality = usedBytes[dim].cardinality();
                if (cardinality >= sortedDimCardinality) continue;
                sortedDim = dim;
                sortedDimCardinality = cardinality;
            }
            radixSelector.heapRadixSort(heapSource, from, to, sortedDim, this.commonPrefixLengths[sortedDim]);
            int leafCardinality = heapSource.computeCardinality(from, to, this.commonPrefixLengths);
            leafBlockFPs[leavesOffset] = out.getFilePointer();
            int count = to - from;
            assert (count > 0) : "numLeaves=" + numLeaves + " leavesOffset=" + leavesOffset;
            assert (count <= spareDocIds.length) : "count=" + count + " > length=" + spareDocIds.length;
            int[] docIDs = spareDocIds;
            for (i = 0; i < count; ++i) {
                docIDs[i] = heapSource.getPackedValueSlice(from + i).docID();
            }
            this.writeLeafBlockDocs(out, docIDs, 0, count);
            this.writeCommonPrefixes(out, this.commonPrefixLengths, this.scratch1);
            IntFunction<BytesRef> packedValues = new IntFunction<BytesRef>(){
                final BytesRef scratch = new BytesRef();
                {
                    this.scratch.length = BKDWriter.this.config.packedBytesLength;
                }

                @Override
                public BytesRef apply(int i) {
                    PointValue value = heapSource.getPackedValueSlice(from + i);
                    return value.packedValue();
                }
            };
            assert (BKDWriter.valuesInOrderAndBounds(this.config, count, sortedDim, minPackedValue, maxPackedValue, packedValues, docIDs, 0));
            this.writeLeafBlockPackedValues(out, this.commonPrefixLengths, count, sortedDim, packedValues, leafCardinality);
        } else {
            int splitDim;
            if (this.config.numIndexDims == 1) {
                splitDim = 0;
            } else {
                if (numLeaves != leafBlockFPs.length && this.config.numIndexDims > 2 && Arrays.stream(parentSplits).sum() % 4 == 0) {
                    this.computePackedValueBounds(points, minPackedValue, maxPackedValue);
                }
                splitDim = this.split(minPackedValue, maxPackedValue, parentSplits);
            }
            assert (numLeaves <= leafBlockFPs.length) : "numLeaves=" + numLeaves + " leafBlockFPs.length=" + leafBlockFPs.length;
            int numLeftLeafNodes = this.getNumLeftLeafNodes(numLeaves);
            long leftCount = numLeftLeafNodes * this.config.maxPointsInLeafNode;
            BKDRadixSelector.PathSlice[] slices = new BKDRadixSelector.PathSlice[2];
            int commonPrefixLen = this.commonPrefixComparator.compare(minPackedValue, splitDim * this.config.bytesPerDim, maxPackedValue, splitDim * this.config.bytesPerDim);
            byte[] splitValue = radixSelector.select(points, slices, points.start, points.start + points.count, points.start + leftCount, splitDim, commonPrefixLen);
            int rightOffset = leavesOffset + numLeftLeafNodes;
            int splitValueOffset = rightOffset - 1;
            splitDimensionValues[splitValueOffset] = (byte)splitDim;
            int address = splitValueOffset * this.config.bytesPerDim;
            System.arraycopy(splitValue, 0, splitPackedValues, address, this.config.bytesPerDim);
            byte[] minSplitPackedValue = new byte[this.config.packedIndexBytesLength];
            System.arraycopy(minPackedValue, 0, minSplitPackedValue, 0, this.config.packedIndexBytesLength);
            byte[] maxSplitPackedValue = new byte[this.config.packedIndexBytesLength];
            System.arraycopy(maxPackedValue, 0, maxSplitPackedValue, 0, this.config.packedIndexBytesLength);
            System.arraycopy(splitValue, 0, minSplitPackedValue, splitDim * this.config.bytesPerDim, this.config.bytesPerDim);
            System.arraycopy(splitValue, 0, maxSplitPackedValue, splitDim * this.config.bytesPerDim, this.config.bytesPerDim);
            int n = splitDim;
            parentSplits[n] = parentSplits[n] + 1;
            this.build(leavesOffset, numLeftLeafNodes, slices[0], out, radixSelector, minPackedValue, maxSplitPackedValue, parentSplits, splitPackedValues, splitDimensionValues, leafBlockFPs, spareDocIds);
            this.build(rightOffset, numLeaves - numLeftLeafNodes, slices[1], out, radixSelector, minSplitPackedValue, maxPackedValue, parentSplits, splitPackedValues, splitDimensionValues, leafBlockFPs, spareDocIds);
            int n2 = splitDim;
            parentSplits[n2] = parentSplits[n2] - 1;
        }
    }

    private void computeCommonPrefixLength(HeapPointWriter heapPointWriter, byte[] commonPrefix, int from, int to) {
        Arrays.fill(this.commonPrefixLengths, this.config.bytesPerDim);
        PointValue value = heapPointWriter.getPackedValueSlice(from);
        BytesRef packedValue = value.packedValue();
        for (int dim = 0; dim < this.config.numDims; ++dim) {
            System.arraycopy(packedValue.bytes, packedValue.offset + dim * this.config.bytesPerDim, commonPrefix, dim * this.config.bytesPerDim, this.config.bytesPerDim);
        }
        for (int i = from + 1; i < to; ++i) {
            value = heapPointWriter.getPackedValueSlice(i);
            packedValue = value.packedValue();
            for (int dim = 0; dim < this.config.numDims; ++dim) {
                if (this.commonPrefixLengths[dim] == 0) continue;
                this.commonPrefixLengths[dim] = Math.min(this.commonPrefixLengths[dim], this.commonPrefixComparator.compare(commonPrefix, dim * this.config.bytesPerDim, packedValue.bytes, packedValue.offset + dim * this.config.bytesPerDim));
            }
        }
    }

    private static boolean valuesInOrderAndBounds(BKDConfig config, int count, int sortedDim, byte[] minPackedValue, byte[] maxPackedValue, IntFunction<BytesRef> values, int[] docs, int docsOffset) {
        byte[] lastPackedValue = new byte[config.packedBytesLength];
        int lastDoc = -1;
        for (int i = 0; i < count; ++i) {
            BytesRef packedValue = values.apply(i);
            assert (packedValue.length == config.packedBytesLength);
            assert (BKDWriter.valueInOrder(config, i, sortedDim, lastPackedValue, packedValue.bytes, packedValue.offset, docs[docsOffset + i], lastDoc));
            lastDoc = docs[docsOffset + i];
            assert (BKDWriter.valueInBounds(config, packedValue, minPackedValue, maxPackedValue));
        }
        return true;
    }

    private static boolean valueInOrder(BKDConfig config, long ord, int sortedDim, byte[] lastPackedValue, byte[] packedValue, int packedValueOffset, int doc, int lastDoc) {
        int dimOffset = sortedDim * config.bytesPerDim;
        if (ord > 0L) {
            int cmp = Arrays.compareUnsigned(lastPackedValue, dimOffset, dimOffset + config.bytesPerDim, packedValue, packedValueOffset + dimOffset, packedValueOffset + dimOffset + config.bytesPerDim);
            if (cmp > 0) {
                throw new AssertionError((Object)("values out of order: last value=" + new BytesRef(lastPackedValue) + " current value=" + new BytesRef(packedValue, packedValueOffset, config.packedBytesLength) + " ord=" + ord));
            }
            if (cmp == 0 && config.numDims > config.numIndexDims && (cmp = Arrays.compareUnsigned(lastPackedValue, config.packedIndexBytesLength, config.packedBytesLength, packedValue, packedValueOffset + config.packedIndexBytesLength, packedValueOffset + config.packedBytesLength)) > 0) {
                throw new AssertionError((Object)("data values out of order: last value=" + new BytesRef(lastPackedValue) + " current value=" + new BytesRef(packedValue, packedValueOffset, config.packedBytesLength) + " ord=" + ord));
            }
            if (cmp == 0 && doc < lastDoc) {
                throw new AssertionError((Object)("docs out of order: last doc=" + lastDoc + " current doc=" + doc + " ord=" + ord));
            }
        }
        System.arraycopy(packedValue, packedValueOffset, lastPackedValue, 0, config.packedBytesLength);
        return true;
    }

    private static boolean valueInBounds(BKDConfig config, BytesRef packedValue, byte[] minPackedValue, byte[] maxPackedValue) {
        for (int dim = 0; dim < config.numIndexDims; ++dim) {
            int offset = config.bytesPerDim * dim;
            if (Arrays.compareUnsigned(packedValue.bytes, packedValue.offset + offset, packedValue.offset + offset + config.bytesPerDim, minPackedValue, offset, offset + config.bytesPerDim) < 0) {
                return false;
            }
            if (Arrays.compareUnsigned(packedValue.bytes, packedValue.offset + offset, packedValue.offset + offset + config.bytesPerDim, maxPackedValue, offset, offset + config.bytesPerDim) <= 0) continue;
            return false;
        }
        return true;
    }

    private class OneDimensionBKDWriter {
        final IndexOutput metaOut;
        final IndexOutput indexOut;
        final IndexOutput dataOut;
        final long dataStartFP;
        final List<Long> leafBlockFPs = new ArrayList<Long>();
        final List<byte[]> leafBlockStartValues = new ArrayList<byte[]>();
        final byte[] leafValues;
        final int[] leafDocs;
        private long valueCount;
        private int leafCount;
        private int leafCardinality;
        final byte[] lastPackedValue;
        private int lastDocID;

        OneDimensionBKDWriter(IndexOutput metaOut, IndexOutput indexOut, IndexOutput dataOut) {
            this.leafValues = new byte[BKDWriter.this.config.maxPointsInLeafNode * BKDWriter.this.config.packedBytesLength];
            this.leafDocs = new int[BKDWriter.this.config.maxPointsInLeafNode];
            if (BKDWriter.this.config.numIndexDims != 1) {
                throw new UnsupportedOperationException("config.numIndexDims must be 1 but got " + BKDWriter.this.config.numIndexDims);
            }
            if (BKDWriter.this.pointCount != 0L) {
                throw new IllegalStateException("cannot mix add and merge");
            }
            if (BKDWriter.this.finished) {
                throw new IllegalStateException("already finished");
            }
            BKDWriter.this.finished = true;
            this.metaOut = metaOut;
            this.indexOut = indexOut;
            this.dataOut = dataOut;
            this.dataStartFP = dataOut.getFilePointer();
            this.lastPackedValue = new byte[BKDWriter.this.config.packedBytesLength];
        }

        void add(byte[] packedValue, int docID) throws IOException {
            assert (BKDWriter.valueInOrder(BKDWriter.this.config, this.valueCount + (long)this.leafCount, 0, this.lastPackedValue, packedValue, 0, docID, this.lastDocID));
            if (this.leafCount == 0 || !BKDWriter.this.equalsPredicate.test(this.leafValues, (this.leafCount - 1) * BKDWriter.this.config.bytesPerDim, packedValue, 0)) {
                ++this.leafCardinality;
            }
            System.arraycopy(packedValue, 0, this.leafValues, this.leafCount * BKDWriter.this.config.packedBytesLength, BKDWriter.this.config.packedBytesLength);
            this.leafDocs[this.leafCount] = docID;
            BKDWriter.this.docsSeen.set(docID);
            ++this.leafCount;
            if (this.valueCount + (long)this.leafCount > BKDWriter.this.totalPointCount) {
                throw new IllegalStateException("totalPointCount=" + BKDWriter.this.totalPointCount + " was passed when we were created, but we just hit " + (this.valueCount + (long)this.leafCount) + " values");
            }
            if (this.leafCount == BKDWriter.this.config.maxPointsInLeafNode) {
                this.writeLeafBlock(this.leafCardinality);
                this.leafCardinality = 0;
                this.leafCount = 0;
            }
            assert ((this.lastDocID = docID) >= 0);
        }

        public Runnable finish() throws IOException {
            if (this.leafCount > 0) {
                this.writeLeafBlock(this.leafCardinality);
                this.leafCardinality = 0;
                this.leafCount = 0;
            }
            if (this.valueCount == 0L) {
                return null;
            }
            BKDWriter.this.pointCount = this.valueCount;
            BKDWriter.this.scratchBytesRef1.length = BKDWriter.this.config.bytesPerDim;
            BKDWriter.this.scratchBytesRef1.offset = 0;
            assert (this.leafBlockStartValues.size() + 1 == this.leafBlockFPs.size());
            BKDTreeLeafNodes leafNodes = new BKDTreeLeafNodes(){

                @Override
                public long getLeafLP(int index) {
                    return OneDimensionBKDWriter.this.leafBlockFPs.get(index);
                }

                @Override
                public BytesRef getSplitValue(int index) {
                    BKDWriter.this.scratchBytesRef1.bytes = OneDimensionBKDWriter.this.leafBlockStartValues.get(index);
                    return BKDWriter.this.scratchBytesRef1;
                }

                @Override
                public int getSplitDimension(int index) {
                    return 0;
                }

                @Override
                public int numLeaves() {
                    return OneDimensionBKDWriter.this.leafBlockFPs.size();
                }
            };
            return () -> {
                try {
                    BKDWriter.this.writeIndex(this.metaOut, this.indexOut, BKDWriter.this.config.maxPointsInLeafNode, leafNodes, this.dataStartFP);
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            };
        }

        private void writeLeafBlock(int leafCardinality) throws IOException {
            assert (this.leafCount != 0);
            if (this.valueCount == 0L) {
                System.arraycopy(this.leafValues, 0, BKDWriter.this.minPackedValue, 0, BKDWriter.this.config.packedIndexBytesLength);
            }
            System.arraycopy(this.leafValues, (this.leafCount - 1) * BKDWriter.this.config.packedBytesLength, BKDWriter.this.maxPackedValue, 0, BKDWriter.this.config.packedIndexBytesLength);
            this.valueCount += (long)this.leafCount;
            if (this.leafBlockFPs.size() > 0) {
                this.leafBlockStartValues.add(ArrayUtil.copyOfSubArray(this.leafValues, 0, BKDWriter.this.config.packedBytesLength));
            }
            this.leafBlockFPs.add(this.dataOut.getFilePointer());
            BKDWriter.this.checkMaxLeafNodeCount(this.leafBlockFPs.size());
            BKDWriter.this.commonPrefixLengths[0] = BKDWriter.this.commonPrefixComparator.compare(this.leafValues, 0, this.leafValues, (this.leafCount - 1) * BKDWriter.this.config.packedBytesLength);
            BKDWriter.this.writeLeafBlockDocs(this.dataOut, this.leafDocs, 0, this.leafCount);
            BKDWriter.this.writeCommonPrefixes(this.dataOut, BKDWriter.this.commonPrefixLengths, this.leafValues);
            BKDWriter.this.scratchBytesRef1.length = BKDWriter.this.config.packedBytesLength;
            BKDWriter.this.scratchBytesRef1.bytes = this.leafValues;
            IntFunction<BytesRef> packedValues = new IntFunction<BytesRef>(){

                @Override
                public BytesRef apply(int i) {
                    BKDWriter.this.scratchBytesRef1.offset = BKDWriter.this.config.packedBytesLength * i;
                    return BKDWriter.this.scratchBytesRef1;
                }
            };
            assert (BKDWriter.valuesInOrderAndBounds(BKDWriter.this.config, this.leafCount, 0, ArrayUtil.copyOfSubArray(this.leafValues, 0, BKDWriter.this.config.packedBytesLength), ArrayUtil.copyOfSubArray(this.leafValues, (this.leafCount - 1) * BKDWriter.this.config.packedBytesLength, this.leafCount * BKDWriter.this.config.packedBytesLength), packedValues, this.leafDocs, 0));
            BKDWriter.this.writeLeafBlockPackedValues(this.dataOut, BKDWriter.this.commonPrefixLengths, this.leafCount, 0, packedValues, leafCardinality);
        }
    }

    private static interface BKDTreeLeafNodes {
        public int numLeaves();

        public long getLeafLP(int var1);

        public BytesRef getSplitValue(int var1);

        public int getSplitDimension(int var1);
    }

    private static class BKDMergeQueue
    extends PriorityQueue<MergeReader> {
        private final int bytesPerDim;

        public BKDMergeQueue(int bytesPerDim, int maxSize) {
            super(maxSize);
            this.bytesPerDim = bytesPerDim;
        }

        @Override
        public boolean lessThan(MergeReader a, MergeReader b) {
            assert (a != b);
            int cmp = Arrays.compareUnsigned(a.packedValue, 0, this.bytesPerDim, b.packedValue, 0, this.bytesPerDim);
            if (cmp < 0) {
                return true;
            }
            if (cmp > 0) {
                return false;
            }
            return a.docID < b.docID;
        }
    }

    private static class MergeIntersectsVisitor
    implements PointValues.IntersectVisitor {
        int docsInBlock = 0;
        byte[] packedValues;
        int[] docIDs = new int[0];
        private final int packedBytesLength;

        MergeIntersectsVisitor(int packedBytesLength) {
            this.packedValues = new byte[0];
            this.packedBytesLength = packedBytesLength;
        }

        void reset() {
            this.docsInBlock = 0;
        }

        @Override
        public void grow(int count) {
            assert (this.docsInBlock == 0);
            if (this.docIDs.length < count) {
                this.docIDs = ArrayUtil.grow(this.docIDs, count);
                int packedValuesSize = Math.toIntExact((long)this.docIDs.length * (long)this.packedBytesLength);
                if (packedValuesSize > ArrayUtil.MAX_ARRAY_LENGTH) {
                    throw new IllegalStateException("array length must be <= to " + ArrayUtil.MAX_ARRAY_LENGTH + " but was: " + packedValuesSize);
                }
                this.packedValues = ArrayUtil.growExact(this.packedValues, packedValuesSize);
            }
        }

        @Override
        public void visit(int docID) {
            throw new UnsupportedOperationException();
        }

        @Override
        public void visit(int docID, byte[] packedValue) {
            System.arraycopy(packedValue, 0, this.packedValues, this.docsInBlock * this.packedBytesLength, this.packedBytesLength);
            this.docIDs[this.docsInBlock++] = docID;
        }

        @Override
        public PointValues.Relation compare(byte[] minPackedValue, byte[] maxPackedValue) {
            return PointValues.Relation.CELL_CROSSES_QUERY;
        }
    }

    private static class MergeReader {
        private final PointValues.PointTree pointTree;
        private final int packedBytesLength;
        private final MergeState.DocMap docMap;
        private final MergeIntersectsVisitor mergeIntersectsVisitor;
        private int docBlockUpto;
        public int docID;
        public final byte[] packedValue;

        public MergeReader(PointValues pointValues, MergeState.DocMap docMap) throws IOException {
            this.packedBytesLength = pointValues.getBytesPerDimension() * pointValues.getNumDimensions();
            this.pointTree = pointValues.getPointTree();
            this.mergeIntersectsVisitor = new MergeIntersectsVisitor(this.packedBytesLength);
            while (this.pointTree.moveToChild()) {
            }
            this.pointTree.visitDocValues(this.mergeIntersectsVisitor);
            this.docMap = docMap;
            this.packedValue = new byte[this.packedBytesLength];
        }

        public boolean next() throws IOException {
            int index;
            int oldDocID;
            int mappedDocID;
            do {
                if (this.docBlockUpto == this.mergeIntersectsVisitor.docsInBlock) {
                    if (!this.collectNextLeaf()) {
                        assert (this.mergeIntersectsVisitor.docsInBlock == 0);
                        return false;
                    }
                    assert (this.mergeIntersectsVisitor.docsInBlock > 0);
                    this.docBlockUpto = 0;
                }
                ++this.docBlockUpto;
                oldDocID = this.mergeIntersectsVisitor.docIDs[index];
            } while ((mappedDocID = this.docMap == null ? oldDocID : this.docMap.get(oldDocID)) == -1);
            this.docID = mappedDocID;
            System.arraycopy(this.mergeIntersectsVisitor.packedValues, index * this.packedBytesLength, this.packedValue, 0, this.packedBytesLength);
            return true;
        }

        private boolean collectNextLeaf() throws IOException {
            assert (!this.pointTree.moveToChild());
            this.mergeIntersectsVisitor.reset();
            do {
                if (!this.pointTree.moveToSibling()) continue;
                while (this.pointTree.moveToChild()) {
                }
                this.pointTree.visitDocValues(this.mergeIntersectsVisitor);
                return true;
            } while (this.pointTree.moveToParent());
            return false;
        }
    }
}

