// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
/*
 * This file contains information about particular algorithm suites used
 * within the encryption SDK.  In most cases, end-users don't need to
 * manipulate this structure, but it can occasionally be needed for more
 * advanced use cases, such as writing keyrings.
 *
 * These are the WebCrypto specific values the AWS Encryption SDK for JavaScript
 * Algorithm Suites.
 */
import { AlgorithmSuite, AlgorithmSuiteIdentifier, MessageFormat, } from './algorithm_suites';
import { needs } from './needs';
/* References to https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/algorithms-reference.html
 * These are the composed parameters for each algorithm suite specification for
 * for the WebCrypto environment.
 */
const webCryptoAlgAes128GcmIv12Tag16 = {
    id: AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16,
    messageFormat: MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 128,
    ivLength: 12,
    tagLength: 128,
    cacheSafe: false,
    commitment: 'NONE',
};
/* Web browsers do not support 192 bit key lengths at this time. */
const webCryptoAlgAes192GcmIv12Tag16 = {
    id: AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16,
    messageFormat: MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 192,
    ivLength: 12,
    tagLength: 128,
    cacheSafe: false,
    commitment: 'NONE',
};
const webCryptoAlgAes256GcmIv12Tag16 = {
    id: AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16,
    messageFormat: MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    cacheSafe: false,
    commitment: 'NONE',
};
const webCryptoAlgAes128GcmIv12Tag16HkdfSha256 = {
    id: AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256,
    messageFormat: MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 128,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'SHA-256',
    cacheSafe: true,
    commitment: 'NONE',
};
/* Web browsers do not support 192 bit key lengths at this time. */
const webCryptoAlgAes192GcmIv12Tag16HkdfSha256 = {
    id: AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256,
    messageFormat: MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 192,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'SHA-256',
    cacheSafe: true,
    commitment: 'NONE',
};
const webCryptoAlgAes256GcmIv12Tag16HkdfSha256 = {
    id: AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256,
    messageFormat: MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'SHA-256',
    cacheSafe: true,
    commitment: 'NONE',
};
const webCryptoAlgAes128GcmIv12Tag16HkdfSha256EcdsaP256 = {
    id: AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256,
    messageFormat: MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 128,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'SHA-256',
    cacheSafe: true,
    signatureCurve: 'P-256',
    signatureHash: 'SHA-256',
    commitment: 'NONE',
};
/* Web browsers do not support 192 bit key lengths at this time. */
const webCryptoAlgAes192GcmIv12Tag16HkdfSha384EcdsaP384 = {
    id: AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384,
    messageFormat: MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 192,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'SHA-384',
    cacheSafe: true,
    signatureCurve: 'P-384',
    signatureHash: 'SHA-384',
    commitment: 'NONE',
};
const webCryptoAlgAes256GcmIv12Tag16HkdfSha384EcdsaP384 = {
    id: AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384,
    messageFormat: MessageFormat.V1,
    encryption: 'AES-GCM',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'SHA-384',
    cacheSafe: true,
    signatureCurve: 'P-384',
    signatureHash: 'SHA-384',
    commitment: 'NONE',
};
const webCryptoAlgAes256GcmHkdfSha512Committing = {
    id: AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY,
    messageFormat: MessageFormat.V2,
    encryption: 'AES-GCM',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'SHA-512',
    cacheSafe: true,
    commitment: 'KEY',
    commitmentHash: 'SHA-512',
    suiteDataLength: 32,
    commitmentLength: 256,
    saltLengthBytes: 32,
};
const webCryptoAlgAes256GcmHkdfSha512CommittingEcdsaP384 = {
    id: AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384,
    messageFormat: MessageFormat.V2,
    encryption: 'AES-GCM',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'SHA-512',
    cacheSafe: true,
    signatureCurve: 'P-384',
    signatureHash: 'SHA-384',
    commitment: 'KEY',
    commitmentHash: 'SHA-512',
    suiteDataLength: 32,
    commitmentLength: 256,
    saltLengthBytes: 32,
};
const webCryptoAlgorithms = Object.freeze({
    [AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16]: Object.freeze(webCryptoAlgAes128GcmIv12Tag16),
    [AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16]: Object.freeze(webCryptoAlgAes192GcmIv12Tag16),
    [AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16]: Object.freeze(webCryptoAlgAes256GcmIv12Tag16),
    [AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(webCryptoAlgAes128GcmIv12Tag16HkdfSha256),
    [AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(webCryptoAlgAes192GcmIv12Tag16HkdfSha256),
    [AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(webCryptoAlgAes256GcmIv12Tag16HkdfSha256),
    [AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256]: Object.freeze(webCryptoAlgAes128GcmIv12Tag16HkdfSha256EcdsaP256),
    [AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384]: Object.freeze(webCryptoAlgAes192GcmIv12Tag16HkdfSha384EcdsaP384),
    [AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384]: Object.freeze(webCryptoAlgAes256GcmIv12Tag16HkdfSha384EcdsaP384),
    [AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY]: Object.freeze(webCryptoAlgAes256GcmHkdfSha512Committing),
    [AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384]: Object.freeze(webCryptoAlgAes256GcmHkdfSha512CommittingEcdsaP384),
});
const supportedWebCryptoAlgorithms = Object.freeze({
    [AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16]: Object.freeze(webCryptoAlgAes128GcmIv12Tag16),
    // [AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16]: Object.freeze(webCryptoAlgAes192GcmIv12Tag16),
    [AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16]: Object.freeze(webCryptoAlgAes256GcmIv12Tag16),
    [AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(webCryptoAlgAes128GcmIv12Tag16HkdfSha256),
    // [AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(webCryptoAlgAes192GcmIv12Tag16HkdfSha256),
    [AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(webCryptoAlgAes256GcmIv12Tag16HkdfSha256),
    [AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256]: Object.freeze(webCryptoAlgAes128GcmIv12Tag16HkdfSha256EcdsaP256),
    // [AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384]: Object.freeze(webCryptoAlgAes192GcmIv12Tag16HkdfSha384EcdsaP384),
    [AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384]: Object.freeze(webCryptoAlgAes256GcmIv12Tag16HkdfSha384EcdsaP384),
    [AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY]: Object.freeze(webCryptoAlgAes256GcmHkdfSha512Committing),
    [AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384]: Object.freeze(webCryptoAlgAes256GcmHkdfSha512CommittingEcdsaP384),
});
export class WebCryptoAlgorithmSuite extends AlgorithmSuite {
    type = 'webCrypto';
    constructor(id) {
        super(webCryptoAlgorithms[id]);
        /* Precondition: Browsers do not support 192 bit keys so the AlgorithmSuiteIdentifier is removed.
         * This is primarily an error in decrypt but this make it clear.
         * The error can manifest deep in the decrypt loop making it hard to debug.
         */
        needs(Object.prototype.hasOwnProperty.call(supportedWebCryptoAlgorithms, id), '192-bit AES keys are not supported');
        Object.setPrototypeOf(this, WebCryptoAlgorithmSuite.prototype);
        Object.freeze(this);
    }
}
Object.freeze(WebCryptoAlgorithmSuite.prototype);
Object.freeze(WebCryptoAlgorithmSuite);
//# sourceMappingURL=data:application/json;base64,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