/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.security;

import com.fasterxml.jackson.annotation.JsonIgnore;
import java.lang.invoke.MethodHandles;
import java.security.Principal;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.servlet.http.HttpServletRequest;
import org.apache.solr.common.SolrException;
import org.apache.solr.security.AuthorizationContext;
import org.apache.solr.security.AuthorizationResponse;
import org.apache.solr.servlet.ServletUtils;
import org.apache.solr.servlet.SolrRequestParsers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

public class AuditEvent {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private String baseUrl;
    private String nodeName;
    private String message;
    private Level level;
    private Date date;
    private String username;
    private String session;
    private String clientIp;
    private List<String> collections;
    private Map<String, Object> context;
    private Map<String, String> headers;
    private Map<String, List<String>> solrParams = new HashMap<String, List<String>>();
    private String solrHost;
    private int solrPort;
    private String solrIp;
    private String resource;
    private String httpMethod;
    private String httpQueryString;
    private EventType eventType;
    private AuthorizationResponse autResponse;
    private RequestType requestType;
    private double qTime = -1.0;
    private int status = -1;
    private Throwable exception;
    private static final List<String> ADMIN_PATH_REGEXES = Arrays.asList("^/admin/.*", "^/api/(c|collections)$", "^/api/(c|collections)/[^/]+/config$", "^/api/(c|collections)/[^/]+/schema$", "^/api/(c|collections)/[^/]+/shards.*", "^/api/cores.*$", "^/api/node.*$", "^/api/cluster.*$");
    private static final List<String> STREAMING_PATH_REGEXES = Collections.singletonList(".*/stream.*");
    private static final List<String> INDEXING_PATH_REGEXES = Collections.singletonList(".*/update.*");
    private static final List<String> SEARCH_PATH_REGEXES = Arrays.asList(".*/select.*", ".*/query.*");
    private static final List<Pattern> ADMIN_PATH_PATTERNS = ADMIN_PATH_REGEXES.stream().map(Pattern::compile).collect(Collectors.toList());
    private static final List<Pattern> STREAMING_PATH_PATTERNS = STREAMING_PATH_REGEXES.stream().map(Pattern::compile).collect(Collectors.toList());
    private static final List<Pattern> INDEXING_PATH_PATTERNS = INDEXING_PATH_REGEXES.stream().map(Pattern::compile).collect(Collectors.toList());
    private static final List<Pattern> SEARCH_PATH_PATTERNS = SEARCH_PATH_REGEXES.stream().map(Pattern::compile).collect(Collectors.toList());

    public AuditEvent(EventType eventType) {
        this.date = new Date();
        this.eventType = eventType;
        this.status = eventType.status;
        this.level = eventType.level;
        this.message = eventType.message;
    }

    public AuditEvent(EventType eventType, HttpServletRequest httpRequest) {
        this(eventType, null, httpRequest);
    }

    protected AuditEvent() {
    }

    public AuditEvent(EventType eventType, Throwable exception, HttpServletRequest httpRequest) {
        this(eventType);
        Principal principal;
        this.solrHost = httpRequest.getLocalName();
        this.solrPort = httpRequest.getLocalPort();
        this.solrIp = httpRequest.getLocalAddr();
        this.clientIp = httpRequest.getRemoteAddr();
        this.httpMethod = httpRequest.getMethod();
        this.httpQueryString = httpRequest.getQueryString();
        this.headers = this.getHeadersFromRequest(httpRequest);
        this.baseUrl = httpRequest.getRequestURL().toString();
        this.nodeName = MDC.get((String)"node_name");
        SolrRequestParsers.parseQueryString(this.httpQueryString).forEach(sp -> this.solrParams.put((String)sp.getKey(), Arrays.asList((String[])sp.getValue())));
        this.setResource(ServletUtils.getPathAfterContext(httpRequest));
        this.setRequestType(this.findRequestType());
        if (exception != null) {
            this.setException(exception);
        }
        if ((principal = httpRequest.getUserPrincipal()) != null) {
            this.username = httpRequest.getUserPrincipal().getName();
        } else if (eventType.equals((Object)EventType.AUTHENTICATED)) {
            this.eventType = EventType.ANONYMOUS;
            this.message = EventType.ANONYMOUS.message;
            this.level = EventType.ANONYMOUS.level;
            log.debug("Audit event type changed from AUTHENTICATED to ANONYMOUS since no Principal found on request");
        }
    }

    public AuditEvent(EventType eventType, HttpServletRequest httpRequest, AuthorizationContext authorizationContext) {
        this(eventType, httpRequest);
        this.collections = authorizationContext.getCollectionRequests().stream().map(r -> r.collectionName).collect(Collectors.toList());
        this.setResource(authorizationContext.getResource());
        this.requestType = RequestType.convertType(authorizationContext.getRequestType());
        if (authorizationContext.getParams() != null) {
            authorizationContext.getParams().forEach(p -> this.solrParams.put((String)p.getKey(), Arrays.asList((String[])p.getValue())));
        }
    }

    public AuditEvent(EventType eventType, HttpServletRequest httpRequest, AuthorizationContext authorizationContext, double qTime, Throwable exception) {
        this(eventType, httpRequest, authorizationContext);
        this.setQTime(qTime);
        this.setException(exception);
    }

    private HashMap<String, String> getHeadersFromRequest(HttpServletRequest httpRequest) {
        HashMap<String, String> h = new HashMap<String, String>();
        Enumeration headersEnum = httpRequest.getHeaderNames();
        while (headersEnum != null && headersEnum.hasMoreElements()) {
            String name = (String)headersEnum.nextElement();
            h.put(name, httpRequest.getHeader(name));
        }
        return h;
    }

    public String getMessage() {
        return this.message;
    }

    public Level getLevel() {
        return this.level;
    }

    public Date getDate() {
        return this.date;
    }

    public String getUsername() {
        return this.username;
    }

    public String getSession() {
        return this.session;
    }

    public String getClientIp() {
        return this.clientIp;
    }

    public Map<String, Object> getContext() {
        return this.context;
    }

    public List<String> getCollections() {
        return this.collections;
    }

    public String getResource() {
        return this.resource;
    }

    public String getHttpMethod() {
        return this.httpMethod;
    }

    public String getHttpQueryString() {
        return this.httpQueryString;
    }

    public EventType getEventType() {
        return this.eventType;
    }

    public String getSolrHost() {
        return this.solrHost;
    }

    public String getSolrIp() {
        return this.solrIp;
    }

    public int getSolrPort() {
        return this.solrPort;
    }

    public Map<String, String> getHeaders() {
        return this.headers;
    }

    public Map<String, List<String>> getSolrParams() {
        return this.solrParams;
    }

    public String getSolrParamAsString(String key) {
        List<String> v = this.getSolrParams().get(key);
        if (v != null && v.size() > 0) {
            return String.valueOf(v.get(0));
        }
        return null;
    }

    public AuthorizationResponse getAutResponse() {
        return this.autResponse;
    }

    public String getNodeName() {
        return this.nodeName;
    }

    public RequestType getRequestType() {
        return this.requestType;
    }

    public int getStatus() {
        return this.status;
    }

    public double getQTime() {
        return this.qTime;
    }

    public Throwable getException() {
        return this.exception;
    }

    @Deprecated
    @JsonIgnore
    public StringBuffer getRequestUrl() {
        return new StringBuffer(this.baseUrl);
    }

    public String getUrl() {
        if (this.baseUrl == null) {
            return null;
        }
        return this.baseUrl + (String)(this.httpQueryString != null ? "?" + this.httpQueryString : "");
    }

    public String getBaseUrl() {
        return this.baseUrl;
    }

    public AuditEvent setBaseUrl(String baseUrl) {
        this.baseUrl = baseUrl;
        return this;
    }

    public AuditEvent setSession(String session) {
        this.session = session;
        return this;
    }

    public AuditEvent setClientIp(String clientIp) {
        this.clientIp = clientIp;
        return this;
    }

    public AuditEvent setContext(Map<String, Object> context) {
        this.context = context;
        return this;
    }

    public AuditEvent setContextEntry(String key, Object value) {
        this.context.put(key, value);
        return this;
    }

    public AuditEvent setMessage(String message) {
        this.message = message;
        return this;
    }

    public AuditEvent setLevel(Level level) {
        this.level = level;
        return this;
    }

    public AuditEvent setDate(Date date) {
        this.date = date;
        return this;
    }

    public AuditEvent setUsername(String username) {
        this.username = username;
        return this;
    }

    public AuditEvent setCollections(List<String> collections) {
        this.collections = collections;
        return this;
    }

    public AuditEvent setResource(String resource) {
        this.resource = this.normalizeResourcePath(resource);
        return this;
    }

    public AuditEvent setHttpMethod(String httpMethod) {
        this.httpMethod = httpMethod;
        return this;
    }

    public AuditEvent setHttpQueryString(String httpQueryString) {
        this.httpQueryString = httpQueryString;
        return this;
    }

    public AuditEvent setSolrHost(String solrHost) {
        this.solrHost = solrHost;
        return this;
    }

    public AuditEvent setSolrPort(int solrPort) {
        this.solrPort = solrPort;
        return this;
    }

    public AuditEvent setSolrIp(String solrIp) {
        this.solrIp = solrIp;
        return this;
    }

    public AuditEvent setHeaders(Map<String, String> headers) {
        this.headers = headers;
        return this;
    }

    public AuditEvent setSolrParams(Map<String, List<String>> solrParams) {
        this.solrParams = solrParams;
        return this;
    }

    public AuditEvent setAutResponse(AuthorizationResponse autResponse) {
        this.autResponse = autResponse;
        return this;
    }

    public AuditEvent setRequestType(RequestType requestType) {
        this.requestType = requestType;
        return this;
    }

    public AuditEvent setQTime(double qTime) {
        this.qTime = qTime;
        return this;
    }

    public AuditEvent setStatus(int status) {
        this.status = status;
        return this;
    }

    public AuditEvent setException(Throwable exception) {
        this.exception = exception;
        if (exception != null) {
            this.eventType = EventType.ERROR;
            this.level = EventType.ERROR.level;
            this.message = EventType.ERROR.message;
            if (exception instanceof SolrException) {
                this.status = ((SolrException)exception).code();
            }
        }
        return this;
    }

    private RequestType findRequestType() {
        if (this.resource == null) {
            return RequestType.UNKNOWN;
        }
        if (SEARCH_PATH_PATTERNS.stream().anyMatch(p -> p.matcher(this.resource).matches())) {
            return RequestType.SEARCH;
        }
        if (INDEXING_PATH_PATTERNS.stream().anyMatch(p -> p.matcher(this.resource).matches())) {
            return RequestType.UPDATE;
        }
        if (STREAMING_PATH_PATTERNS.stream().anyMatch(p -> p.matcher(this.resource).matches())) {
            return RequestType.STREAMING;
        }
        if (ADMIN_PATH_PATTERNS.stream().anyMatch(p -> p.matcher(this.resource).matches())) {
            return RequestType.ADMIN;
        }
        return RequestType.UNKNOWN;
    }

    protected String normalizeResourcePath(String resourcePath) {
        if (resourcePath == null) {
            return "";
        }
        return resourcePath.replaceFirst("^/____v2", "/api");
    }

    public static enum RequestType {
        ADMIN,
        SEARCH,
        UPDATE,
        STREAMING,
        UNKNOWN;


        static RequestType convertType(AuthorizationContext.RequestType ctxReqType) {
            switch (ctxReqType) {
                case ADMIN: {
                    return ADMIN;
                }
                case READ: {
                    return SEARCH;
                }
                case WRITE: {
                    return UPDATE;
                }
            }
            return UNKNOWN;
        }
    }

    public static enum Level {
        INFO,
        WARN,
        ERROR;

    }

    public static enum EventType {
        AUTHENTICATED("Authenticated", "User successfully authenticated", Level.INFO, -1),
        REJECTED("Rejected", "Authentication request rejected", Level.WARN, 401),
        ANONYMOUS("Anonymous", "Request proceeds with unknown user", Level.INFO, -1),
        ANONYMOUS_REJECTED("AnonymousRejected", "Request from unknown user rejected", Level.WARN, 401),
        AUTHORIZED("Authorized", "Authorization succeeded", Level.INFO, -1),
        UNAUTHORIZED("Unauthorized", "Authorization failed", Level.WARN, 403),
        COMPLETED("Completed", "Request completed", Level.INFO, 200),
        ERROR("Error", "Request was not executed due to an error", Level.ERROR, 500);

        public final String message;
        public String explanation;
        public final Level level;
        public int status;

        private EventType(String message, String explanation, Level level, int status) {
            this.message = message;
            this.explanation = explanation;
            this.level = level;
            this.status = status;
        }
    }
}

