<?php

declare(strict_types=1);

/*
 * Copyright (c) 2017-2022 François Kooman <fkooman@tuxed.net>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

namespace fkooman\SeCookie;

use DateTimeImmutable;

class ActiveSession
{
    private string $sessionId;
    private DateTimeImmutable $expiresAt;
    private array $sessionData;
    private bool $isNew = false;
    private bool $hasBeenModified = false;

    public function __construct(string $sessionId, DateTimeImmutable $expiresAt, ?array $sessionData = null)
    {
        $this->sessionId = $sessionId;
        $this->expiresAt = $expiresAt;
        if (null === $sessionData) {
            // new sessions, i.e. not resumed from previous requests are
            // created without the "sessionData" parameter set, this is our
            // hint that we are dealing with a new session... this is important
            // because we do not want to "rewrite" new sessions right away...
            $sessionData = [];
            $this->isNew = true;
            $this->hasBeenModified = true;
        }
        $this->sessionData = $sessionData;
    }

    public function sessionId(): string
    {
        return $this->sessionId;
    }

    public function expiresAt(): DateTimeImmutable
    {
        return $this->expiresAt;
    }

    public function sessionData(): array
    {
        return $this->sessionData;
    }

    public function remove(string $sessionKey): void
    {
        $this->hasBeenModified = true;
        if (\array_key_exists($sessionKey, $this->sessionData)) {
            unset($this->sessionData[$sessionKey]);
        }
    }

    public function set(string $sessionKey, string $sessionValue): void
    {
        $this->hasBeenModified = true;
        $this->sessionData[$sessionKey] = $sessionValue;
    }

    public function get(string $sessionKey): ?string
    {
        if (!\array_key_exists($sessionKey, $this->sessionData) || !\is_string($this->sessionData[$sessionKey])) {
            return null;
        }

        return $this->sessionData[$sessionKey];
    }

    public function isNew(): bool
    {
        return $this->isNew;
    }

    public function hasBeenModified(): bool
    {
        return $this->hasBeenModified;
    }
}
