"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
Object.defineProperty(exports, "SecurityPluginSetup", {
  enumerable: true,
  get: function () {
    return _types.SecurityPluginSetup;
  }
});
Object.defineProperty(exports, "SecurityPluginStart", {
  enumerable: true,
  get: function () {
    return _types.SecurityPluginStart;
  }
});
exports.configSchema = exports.config = void 0;
exports.plugin = plugin;

var _configSchema = require("@osd/config-schema");

var _plugin = require("./plugin");

var _types = require("./types");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
const validateAuthType = value => {
  const supportedAuthTypes = ['', 'basicauth', 'jwt', 'openid', 'saml', 'proxy', 'kerberos', 'proxycache'];
  value.forEach(authVal => {
    if (!supportedAuthTypes.includes(authVal.toLowerCase())) {
      throw new Error(`Unsupported authentication type: ${authVal}. Allowed auth.type are ${supportedAuthTypes}.`);
    }
  });
};

const configSchema = _configSchema.schema.object({
  enabled: _configSchema.schema.boolean({
    defaultValue: true
  }),
  allow_client_certificates: _configSchema.schema.boolean({
    defaultValue: false
  }),
  readonly_mode: _configSchema.schema.object({
    roles: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  clusterPermissions: _configSchema.schema.object({
    include: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  indexPermissions: _configSchema.schema.object({
    include: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  disabledTransportCategories: _configSchema.schema.object({
    exclude: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  disabledRestCategories: _configSchema.schema.object({
    exclude: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  cookie: _configSchema.schema.object({
    secure: _configSchema.schema.boolean({
      defaultValue: false
    }),
    name: _configSchema.schema.string({
      defaultValue: 'security_authentication'
    }),
    password: _configSchema.schema.string({
      defaultValue: 'security_cookie_default_password',
      minLength: 32
    }),
    ttl: _configSchema.schema.number({
      defaultValue: 60 * 60 * 1000
    }),
    domain: _configSchema.schema.nullable(_configSchema.schema.string()),
    isSameSite: _configSchema.schema.oneOf([_configSchema.schema.literal('Strict'), _configSchema.schema.literal('Lax'), _configSchema.schema.literal('None'), _configSchema.schema.literal(false)], {
      defaultValue: false
    })
  }),
  session: _configSchema.schema.object({
    ttl: _configSchema.schema.number({
      defaultValue: 60 * 60 * 1000
    }),
    keepalive: _configSchema.schema.boolean({
      defaultValue: true
    })
  }),
  auth: _configSchema.schema.object({
    type: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: [''],

      validate(value) {
        if (!value || value.length === 0) {
          return `Authentication type is not configured properly. At least one authentication type must be selected.`;
        }

        if (value.length > 1) {
          const includeBasicAuth = value.find(element => {
            return element.toLowerCase() === 'basicauth';
          });

          if (!includeBasicAuth) {
            return `Authentication type is not configured properly. basicauth is mandatory.`;
          }
        }

        validateAuthType(value);
      }

    }), _configSchema.schema.string({
      defaultValue: '',

      validate(value) {
        const valArray = [];
        valArray.push(value);
        validateAuthType(valArray);
      }

    })], {
      defaultValue: ''
    }),
    anonymous_auth_enabled: _configSchema.schema.boolean({
      defaultValue: false
    }),
    unauthenticated_routes: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: ['/api/reporting/stats']
    }),
    forbidden_usernames: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    logout_url: _configSchema.schema.string({
      defaultValue: ''
    }),
    multiple_auth_enabled: _configSchema.schema.boolean({
      defaultValue: false
    })
  }),
  basicauth: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: true
    }),
    unauthenticated_routes: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    forbidden_usernames: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    header_trumps_session: _configSchema.schema.boolean({
      defaultValue: false
    }),
    alternative_login: _configSchema.schema.object({
      headers: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
        defaultValue: []
      }),
      show_for_parameter: _configSchema.schema.string({
        defaultValue: ''
      }),
      valid_redirects: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
        defaultValue: []
      }),
      button_text: _configSchema.schema.string({
        defaultValue: 'Log in with provider'
      }),
      buttonstyle: _configSchema.schema.string({
        defaultValue: ''
      })
    }),
    loadbalancer_url: _configSchema.schema.maybe(_configSchema.schema.string()),
    login: _configSchema.schema.object({
      title: _configSchema.schema.string({
        defaultValue: 'Log in to OpenSearch Dashboards'
      }),
      subtitle: _configSchema.schema.string({
        defaultValue: 'If you have forgotten your username or password, contact your system administrator.'
      }),
      showbrandimage: _configSchema.schema.boolean({
        defaultValue: true
      }),
      brandimage: _configSchema.schema.string({
        defaultValue: ''
      }),
      // TODO: update brand image
      buttonstyle: _configSchema.schema.string({
        defaultValue: ''
      })
    })
  }),
  multitenancy: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: false
    }),
    show_roles: _configSchema.schema.boolean({
      defaultValue: false
    }),
    enable_filter: _configSchema.schema.boolean({
      defaultValue: false
    }),
    debug: _configSchema.schema.boolean({
      defaultValue: false
    }),
    enable_aggregation_view: _configSchema.schema.boolean({
      defaultValue: false
    }),
    tenants: _configSchema.schema.object({
      enable_private: _configSchema.schema.boolean({
        defaultValue: true
      }),
      enable_global: _configSchema.schema.boolean({
        defaultValue: true
      }),
      preferred: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
        defaultValue: []
      })
    })
  }),
  configuration: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: true
    })
  }),
  accountinfo: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: false
    })
  }),
  openid: _configSchema.schema.maybe(_configSchema.schema.object({
    connect_url: _configSchema.schema.maybe(_configSchema.schema.string()),
    header: _configSchema.schema.string({
      defaultValue: 'Authorization'
    }),
    // TODO: test if siblingRef() works here
    // client_id is required when auth.type is openid
    client_id: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth.type'), 'openid', _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())),
    client_secret: _configSchema.schema.string({
      defaultValue: ''
    }),
    scope: _configSchema.schema.string({
      defaultValue: 'openid profile email address phone'
    }),
    base_redirect_url: _configSchema.schema.string({
      defaultValue: ''
    }),
    logout_url: _configSchema.schema.string({
      defaultValue: ''
    }),
    root_ca: _configSchema.schema.string({
      defaultValue: ''
    }),
    verify_hostnames: _configSchema.schema.boolean({
      defaultValue: true
    }),
    refresh_tokens: _configSchema.schema.boolean({
      defaultValue: true
    }),
    trust_dynamic_headers: _configSchema.schema.boolean({
      defaultValue: false
    })
  })),
  proxycache: _configSchema.schema.maybe(_configSchema.schema.object({
    // when auth.type is proxycache, user_header, roles_header and proxy_header_ip are required
    user_header: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth.type'), 'proxycache', _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())),
    roles_header: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth.type'), 'proxycache', _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())),
    proxy_header: _configSchema.schema.maybe(_configSchema.schema.string({
      defaultValue: 'x-forwarded-for'
    })),
    proxy_header_ip: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth.type'), 'proxycache', _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())),
    login_endpoint: _configSchema.schema.maybe(_configSchema.schema.string({
      defaultValue: ''
    }))
  })),
  jwt: _configSchema.schema.maybe(_configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: false
    }),
    login_endpoint: _configSchema.schema.maybe(_configSchema.schema.string()),
    url_param: _configSchema.schema.string({
      defaultValue: 'authorization'
    }),
    header: _configSchema.schema.string({
      defaultValue: 'Authorization'
    })
  })),
  ui: _configSchema.schema.object({
    basicauth: _configSchema.schema.object({
      // the login config here is the same as old config `_security.basicauth.login`
      // Since we are now rendering login page to browser app, so move these config to browser side.
      login: _configSchema.schema.object({
        title: _configSchema.schema.string({
          defaultValue: 'Log in to OpenSearch Dashboards'
        }),
        subtitle: _configSchema.schema.string({
          defaultValue: 'If you have forgotten your username or password, contact your system administrator.'
        }),
        showbrandimage: _configSchema.schema.boolean({
          defaultValue: true
        }),
        brandimage: _configSchema.schema.string({
          defaultValue: ''
        }),
        buttonstyle: _configSchema.schema.string({
          defaultValue: ''
        })
      })
    }),
    anonymous: _configSchema.schema.object({
      login: _configSchema.schema.object({
        buttonname: _configSchema.schema.string({
          defaultValue: 'Log in as anonymous'
        }),
        showbrandimage: _configSchema.schema.boolean({
          defaultValue: false
        }),
        brandimage: _configSchema.schema.string({
          defaultValue: ''
        }),
        buttonstyle: _configSchema.schema.string({
          defaultValue: ''
        })
      })
    }),
    openid: _configSchema.schema.object({
      login: _configSchema.schema.object({
        buttonname: _configSchema.schema.string({
          defaultValue: 'Log in with single sign-on'
        }),
        showbrandimage: _configSchema.schema.boolean({
          defaultValue: false
        }),
        brandimage: _configSchema.schema.string({
          defaultValue: ''
        }),
        buttonstyle: _configSchema.schema.string({
          defaultValue: ''
        })
      })
    }),
    saml: _configSchema.schema.object({
      login: _configSchema.schema.object({
        buttonname: _configSchema.schema.string({
          defaultValue: 'Log in with single sign-on'
        }),
        showbrandimage: _configSchema.schema.boolean({
          defaultValue: false
        }),
        brandimage: _configSchema.schema.string({
          defaultValue: ''
        }),
        buttonstyle: _configSchema.schema.string({
          defaultValue: ''
        })
      })
    }),
    autologout: _configSchema.schema.boolean({
      defaultValue: true
    }),
    backend_configurable: _configSchema.schema.boolean({
      defaultValue: true
    })
  })
});

exports.configSchema = configSchema;
const config = {
  exposeToBrowser: {
    enabled: true,
    auth: true,
    ui: true,
    multitenancy: true,
    readonly_mode: true,
    clusterPermissions: true,
    indexPermissions: true,
    disabledTransportCategories: true,
    disabledRestCategories: true
  },
  schema: configSchema,
  deprecations: ({
    rename,
    unused
  }) => [rename('basicauth.login.title', 'ui.basicauth.login.title'), rename('basicauth.login.subtitle', 'ui.basicauth.login.subtitle'), rename('basicauth.login.showbrandimage', 'ui.basicauth.login.showbrandimage'), rename('basicauth.login.brandimage', 'ui.basicauth.login.brandimage'), rename('basicauth.login.buttonstyle', 'ui.basicauth.login.buttonstyle')]
}; //  This exports static code and TypeScript types,
//  as well as, OpenSearchDashboards Platform `plugin()` initializer.

exports.config = config;

function plugin(initializerContext) {
  return new _plugin.SecurityPlugin(initializerContext);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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