"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.callTokenEndpoint = callTokenEndpoint;
exports.composeLogoutUrl = composeLogoutUrl;
exports.getBaseRedirectUrl = getBaseRedirectUrl;
exports.getExpirationDate = getExpirationDate;
exports.getRootUrl = getRootUrl;
exports.parseTokenResponse = parseTokenResponse;

var _querystring = require("querystring");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
function parseTokenResponse(payload) {
  const payloadString = payload.toString();

  if (payloadString.trim()[0] === '{') {
    try {
      return JSON.parse(payloadString);
    } catch (error) {
      throw Error(`Invalid JSON payload: ${error}`);
    }
  }

  return (0, _querystring.parse)(payloadString);
}

function getRootUrl(config, core, request) {
  var _config$openid;

  const host = core.http.getServerInfo().hostname;
  const port = core.http.getServerInfo().port;
  let protocol = core.http.getServerInfo().protocol;
  let httpHost = `${host}:${port}`;

  if ((_config$openid = config.openid) !== null && _config$openid !== void 0 && _config$openid.trust_dynamic_headers) {
    const xForwardedHost = request.headers['x-forwarded-host'] || undefined;
    const xForwardedProto = request.headers['x-forwarded-proto'] || undefined;

    if (xForwardedHost) {
      httpHost = xForwardedHost;
    }

    if (xForwardedProto) {
      protocol = xForwardedProto;
    }
  }

  return `${protocol}://${httpHost}`;
}

function getBaseRedirectUrl(config, core, request) {
  var _config$openid2;

  if ((_config$openid2 = config.openid) !== null && _config$openid2 !== void 0 && _config$openid2.base_redirect_url) {
    const baseRedirectUrl = config.openid.base_redirect_url;
    return baseRedirectUrl.endsWith('/') ? baseRedirectUrl.slice(0, -1) : baseRedirectUrl;
  }

  const rootUrl = getRootUrl(config, core, request);

  if (core.http.basePath.serverBasePath) {
    return `${rootUrl}${core.http.basePath.serverBasePath}`;
  }

  return rootUrl;
}

async function callTokenEndpoint(tokenEndpoint, query, wreckClient) {
  var _tokenResponse$res;

  const tokenResponse = await wreckClient.post(tokenEndpoint, {
    payload: (0, _querystring.stringify)(query),
    headers: {
      'Content-Type': 'application/x-www-form-urlencoded'
    }
  });

  if (!((_tokenResponse$res = tokenResponse.res) !== null && _tokenResponse$res !== void 0 && _tokenResponse$res.statusCode) || tokenResponse.res.statusCode < 200 || tokenResponse.res.statusCode > 299) {
    throw new Error(`Failed calling token endpoint: ${tokenResponse.res.statusCode} ${tokenResponse.res.statusMessage}`);
  }

  const tokenPayload = parseTokenResponse(tokenResponse.payload);
  return {
    idToken: tokenPayload.id_token,
    accessToken: tokenPayload.access_token,
    refreshToken: tokenPayload.refresh_token,
    expiresIn: tokenPayload.expires_in
  };
}

function composeLogoutUrl(customLogoutUrl, idpEndsessionEndpoint, additionalQueryParams) {
  const logoutEndpont = customLogoutUrl || idpEndsessionEndpoint;
  const logoutUrl = new URL(logoutEndpont);
  Object.keys(additionalQueryParams).forEach(key => {
    logoutUrl.searchParams.append(key, additionalQueryParams[key]);
  });
  return logoutUrl.toString();
}

function getExpirationDate(tokenResponse) {
  if (!tokenResponse) {
    throw new Error('Invalid token');
  } else if (tokenResponse.idToken) {
    const idToken = tokenResponse.idToken;
    const parts = idToken.split('.');

    if (parts.length !== 3) {
      throw new Error('Invalid token');
    }

    const claim = JSON.parse(Buffer.from(parts[1], 'base64').toString());
    return claim.exp * 1000;
  } else {
    return Date.now() + tokenResponse.expiresIn * 1000;
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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