"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = _default;

var _configSchema = require("@osd/config-schema");

var _common = require("../../common");

var _createReport = require("./lib/createReport");

var _helpers = require("./utils/helpers");

var _constants = require("./utils/constants");

var _backendToUi = require("./utils/converters/backendToUi");

var _metricHelper = require("./utils/metricHelper");

var _validationHelper = require("../../server/utils/validationHelper");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
function _default(router, config) {
  const protocol = config.get('osd_server', 'protocol');
  const hostname = config.get('osd_server', 'hostname');
  const port = config.get('osd_server', 'port');
  const basePath = config.osdConfig.get('server', 'basePath'); // generate report (with provided metadata)

  router.post({
    path: `${_common.API_PREFIX}/generateReport`,
    validate: {
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        timezone: _configSchema.schema.maybe(_configSchema.schema.string()),
        dateFormat: _configSchema.schema.maybe(_configSchema.schema.string()),
        csvSeparator: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report', 'create', 'count'); //@ts-ignore

    const logger = context.reporting_plugin.logger;
    let report = request.body; // input validation

    try {
      report.report_definition.report_params.core_params.origin = `${protocol}://${hostname}:${port}${basePath}`;
      report = await (0, _validationHelper.validateReport)(context.core.opensearch.legacy.client, report, basePath);
    } catch (error) {
      logger.error(`Failed input validation for create report ${error}`);
      (0, _metricHelper.addToMetric)('report', 'create', 'user_error');
      return response.badRequest({
        body: error
      });
    }

    try {
      const reportData = await (0, _createReport.createReport)(request, context, report, config); // if not deliver to user himself , no need to send actual file data to client

      const delivery = report.report_definition.delivery;
      (0, _metricHelper.addToMetric)('report', 'create', 'count', report);
      return response.ok({
        body: {
          data: reportData.dataUrl,
          filename: reportData.fileName
        }
      });
    } catch (error) {
      // TODO: better error handling for delivery and stages in generating report, pass logger to deeper level
      logger.error(`Failed to generate report: ${error}`);
      logger.error(error);
      (0, _metricHelper.addToMetric)('report', 'create', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  }); // generate report from report id

  router.get({
    path: `${_common.API_PREFIX}/generateReport/{reportId}`,
    validate: {
      params: _configSchema.schema.object({
        reportId: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        timezone: _configSchema.schema.string(),
        dateFormat: _configSchema.schema.string(),
        csvSeparator: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report', 'download', 'count'); //@ts-ignore

    const logger = context.reporting_plugin.logger;

    try {
      const savedReportId = request.params.reportId; // @ts-ignore

      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request); // get report

      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.getReportById', {
        reportInstanceId: savedReportId
      }); // convert report to use UI model

      const report = (0, _backendToUi.backendToUiReport)(opensearchResp.reportInstance, basePath); // generate report

      const reportData = await (0, _createReport.createReport)(request, context, report, config, savedReportId);
      (0, _metricHelper.addToMetric)('report', 'download', 'count', report);
      return response.ok({
        body: {
          data: reportData.dataUrl,
          filename: reportData.fileName
        }
      });
    } catch (error) {
      logger.error(`Failed to generate report by id: ${error}`);
      logger.error(error);
      (0, _metricHelper.addToMetric)('report', 'download', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  }); // create report from existing report definition

  router.post({
    path: `${_common.API_PREFIX}/generateReport/{reportDefinitionId}`,
    validate: {
      params: _configSchema.schema.object({
        reportDefinitionId: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        timezone: _configSchema.schema.string(),
        dateFormat: _configSchema.schema.string(),
        csvSeparator: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report', 'create_from_definition', 'count'); //@ts-ignore

    const logger = context.reporting_plugin.logger;
    const reportDefinitionId = request.params.reportDefinitionId;
    let report;

    try {
      // @ts-ignore
      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request); // call OpenSearch API to create report from definition

      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.createReportFromDefinition', {
        reportDefinitionId: reportDefinitionId,
        body: {
          reportDefinitionId: reportDefinitionId
        }
      });
      const reportId = opensearchResp.reportInstance.id; // convert report to use UI model

      const report = (0, _backendToUi.backendToUiReport)(opensearchResp.reportInstance, basePath); // generate report

      const reportData = await (0, _createReport.createReport)(request, context, report, config, reportId);
      (0, _metricHelper.addToMetric)('report', 'create_from_definition', 'count', report);
      return response.ok({
        body: {
          data: reportData.dataUrl,
          filename: reportData.fileName
        }
      });
    } catch (error) {
      logger.error(`Failed to generate report from reportDefinition id ${reportDefinitionId} : ${error}`);
      logger.error(error);
      (0, _metricHelper.addToMetric)('report', 'create_from_definition', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  }); // get all reports details

  router.get({
    path: `${_common.API_PREFIX}/reports`,
    validate: {
      query: _configSchema.schema.object({
        fromIndex: _configSchema.schema.maybe(_configSchema.schema.number()),
        maxItems: _configSchema.schema.maybe(_configSchema.schema.number())
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report', 'list', 'count');
    const {
      fromIndex,
      maxItems
    } = request.query;

    try {
      // @ts-ignore
      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request);
      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.getReports', {
        fromIndex: fromIndex,
        maxItems: maxItems || _constants.DEFAULT_MAX_SIZE
      });
      const reportsList = (0, _backendToUi.backendToUiReportsList)(opensearchResp.reportInstanceList, basePath);
      return response.ok({
        body: {
          data: reportsList
        }
      });
    } catch (error) {
      //@ts-ignore
      context.reporting_plugin.logger.error(`Failed to get reports details: ${error}`);
      (0, _metricHelper.addToMetric)('report', 'list', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  }); // get single report details by id

  router.get({
    path: `${_common.API_PREFIX}/reports/{reportId}`,
    validate: {
      params: _configSchema.schema.object({
        reportId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report', 'info', 'count');

    try {
      // @ts-ignore
      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request);
      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.getReportById', {
        reportInstanceId: request.params.reportId
      });
      const report = (0, _backendToUi.backendToUiReport)(opensearchResp.reportInstance, basePath);
      return response.ok({
        body: report
      });
    } catch (error) {
      //@ts-ignore
      context.reporting_plugin.logger.error(`Failed to get single report details: ${error}`);
      (0, _metricHelper.addToMetric)('report', 'info', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  });
}

module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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