"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PanelsRouter = PanelsRouter;

var _configSchema = require("@osd/config-schema");

var _custom_panel_adaptor = require("../../adaptors/custom_panels/custom_panel_adaptor");

var _custom_panels = require("../../../common/constants/custom_panels");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
function PanelsRouter(router) {
  const customPanelBackend = new _custom_panel_adaptor.CustomPanelsAdaptor(); // Fetch all the custom panels available

  router.get({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels`,
    validate: {}
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const panelsList = await customPanelBackend.viewPanelList(opensearchNotebooksClient);
      return response.ok({
        body: {
          panels: panelsList
        }
      });
    } catch (error) {
      console.error('Issue in fetching panel list:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Fetch the required panel by id

  router.get({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/{panelId}`,
    validate: {
      params: _configSchema.schema.object({
        panelId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const panelObject = await customPanelBackend.getPanel(opensearchNotebooksClient, request.params.panelId);
      return response.ok({
        body: panelObject
      });
    } catch (error) {
      console.error('Issue in fetching panel:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Create a new panel

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels`,
    validate: {
      body: _configSchema.schema.object({
        panelName: _configSchema.schema.string(),
        applicationId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const newPanelId = await customPanelBackend.createNewPanel(opensearchNotebooksClient, request.body.panelName, request.body.applicationId || '');
      return response.ok({
        body: {
          message: 'Panel Created',
          newPanelId
        }
      });
    } catch (error) {
      console.error('Issue in creating new panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // rename an existing panel

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/rename`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        panelName: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const responseBody = await customPanelBackend.renamePanel(opensearchNotebooksClient, request.body.panelId, request.body.panelName);
      return response.ok({
        body: {
          message: 'Panel Renamed'
        }
      });
    } catch (error) {
      console.error('Issue in renaming panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // clones an existing panel
  // returns new panel Id

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/clone`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        panelName: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const cloneResponse = await customPanelBackend.clonePanel(opensearchNotebooksClient, request.body.panelId, request.body.panelName);
      return response.ok({
        body: {
          message: 'Panel Cloned',
          clonePanelId: cloneResponse.clonePanelId,
          dateCreated: cloneResponse.dateCreated,
          dateModified: cloneResponse.dateModified
        }
      });
    } catch (error) {
      console.error('Issue in cloning panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // delete an existing panel

  router.delete({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/{panelId}`,
    validate: {
      params: _configSchema.schema.object({
        panelId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const deleteResponse = await customPanelBackend.deletePanel(opensearchNotebooksClient, request.params.panelId);
      return response.ok({
        body: {
          message: 'Panel Deleted'
        }
      });
    } catch (error) {
      console.error('Issue in deleting panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // delete an existing panel(s)

  router.delete({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panelList/{panelIdList}`,
    validate: {
      params: _configSchema.schema.object({
        panelIdList: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const deleteResponse = await customPanelBackend.deletePanelList(opensearchNotebooksClient, request.params.panelIdList);
      return response.ok({
        body: {
          message: 'Panel Deleted'
        }
      });
    } catch (error) {
      console.error('Issue in deleting panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // replaces the ppl query filter in panel

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/filter`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        query: _configSchema.schema.string(),
        language: _configSchema.schema.string(),
        to: _configSchema.schema.string(),
        from: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const panelFilterResponse = await customPanelBackend.addPanelFilter(opensearchNotebooksClient, request.body.panelId, request.body.query, request.body.language, request.body.to, request.body.from);
      return response.ok({
        body: {
          message: 'Panel PPL Filter Changed'
        }
      });
    } catch (error) {
      console.error('Issue in adding query filter', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Add Sample Panels

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/addSamplePanels`,
    validate: {
      body: _configSchema.schema.object({
        savedVisualizationIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const panelsData = await customPanelBackend.addSamplePanels(opensearchNotebooksClient, request.body.savedVisualizationIds);
      return response.ok({
        body: {
          demoPanelsData: panelsData
        }
      });
    } catch (error) {
      console.error('Issue in fetching panel list:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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