"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CustomPanelsAdaptor = void 0;

var _uuid = require("uuid");

var _sample_panels = require("../../common/helpers/custom_panels/sample_panels");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class CustomPanelsAdaptor {
  constructor() {
    _defineProperty(this, "indexPanel", async function (client, panelBody) {
      try {
        const response = await client.callAsCurrentUser('observability.createObject', {
          body: {
            operationalPanel: panelBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Index Panel Error:' + error);
      }
    });

    _defineProperty(this, "updatePanel", async function (client, panelId, updatePanelBody) {
      try {
        const response = await client.callAsCurrentUser('observability.updateObjectById', {
          objectId: panelId,
          body: {
            operationalPanel: updatePanelBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Update Panel Error:' + error);
      }
    });

    _defineProperty(this, "getPanel", async function (client, panelId) {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: panelId
        });
        return response.observabilityObjectList[0];
      } catch (error) {
        throw new Error('Get Panel Error:' + error);
      }
    });

    _defineProperty(this, "viewPanelList", async function (client) {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'operationalPanel',
          maxItems: 10000
        });
        return response.observabilityObjectList.filter(panel => !panel.operationalPanel.applicationId).map(panel => ({
          name: panel.operationalPanel.name,
          id: panel.objectId,
          dateCreated: panel.createdTimeMs,
          dateModified: panel.lastUpdatedTimeMs
        }));
      } catch (error) {
        throw new Error('View Panel List Error:' + error);
      }
    });

    _defineProperty(this, "deletePanel", async function (client, panelId) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectById', {
          objectId: panelId
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Delete Panel Error:' + error);
      }
    });

    _defineProperty(this, "deletePanelList", async function (client, panelIdList) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectByIdList', {
          objectIdList: panelIdList
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Delete Panel List Error:' + error);
      }
    });

    _defineProperty(this, "createNewPanel", async (client, panelName, appId) => {
      const panelBody = {
        name: panelName,
        visualizations: [],
        timeRange: {
          to: 'now',
          from: 'now-1d'
        },
        queryFilter: {
          query: '',
          language: 'ppl'
        }
      };

      if (appId) {
        panelBody.applicationId = appId;
        panelBody.timeRange = {
          to: 'now',
          from: 'now-24h'
        };
      }

      try {
        const response = await this.indexPanel(client, panelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Create Panel Error:' + error);
      }
    });

    _defineProperty(this, "renamePanel", async (client, panelId, panelName) => {
      const updatePanelBody = {
        name: panelName
      };

      try {
        const response = await this.updatePanel(client, panelId, updatePanelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Rename Panel Error:' + error);
      }
    });

    _defineProperty(this, "clonePanel", async (client, panelId, panelName) => {
      const updatePanelBody = {
        name: panelName
      };

      try {
        const getPanel = await this.getPanel(client, panelId);
        const clonePanelBody = {
          name: panelName,
          visualizations: getPanel.operationalPanel.visualizations,
          timeRange: getPanel.operationalPanel.timeRange,
          queryFilter: getPanel.operationalPanel.queryFilter
        };
        const indexResponse = await this.indexPanel(client, clonePanelBody);
        const getClonedPanel = await this.getPanel(client, indexResponse.objectId);
        return {
          clonePanelId: getClonedPanel.objectId,
          dateCreated: getClonedPanel.createdTimeMs,
          dateModified: getClonedPanel.lastUpdatedTimeMs
        };
      } catch (error) {
        throw new Error('Clone Panel Error:' + error);
      }
    });

    _defineProperty(this, "addPanelFilter", async (client, panelId, query, language, to, from) => {
      const updatePanelBody = {
        timeRange: {
          to,
          from
        },
        queryFilter: {
          query,
          language
        }
      };

      try {
        const response = await this.updatePanel(client, panelId, updatePanelBody);
        return response.objectId;
      } catch (error) {
        throw new Error('Add Panel Filter Error:' + error);
      }
    });

    _defineProperty(this, "parseSavedVisualizations", visualization => {
      return {
        id: visualization.objectId,
        name: visualization.savedVisualization.name,
        query: visualization.savedVisualization.query,
        type: visualization.savedVisualization.type,
        timeField: visualization.savedVisualization.selected_timestamp.name,
        selected_date_range: visualization.savedVisualization.selected_date_range,
        selected_fields: visualization.savedVisualization.selected_fields,
        user_configs: visualization.savedVisualization.hasOwnProperty('user_configs') ? JSON.parse(visualization.savedVisualization.user_configs) : {},
        ...(visualization.savedVisualization.application_id ? {
          application_id: visualization.savedVisualization.application_id
        } : {})
      };
    });

    _defineProperty(this, "getAllSavedVisualizations", async client => {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'savedVisualization'
        });
        return response.observabilityObjectList.map(visualization => this.parseSavedVisualizations(visualization));
      } catch (error) {
        throw new Error('View Saved Visualizations Error:' + error);
      }
    });

    _defineProperty(this, "getSavedVisualizationById", async (client, savedVisualizationId) => {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: savedVisualizationId
        });
        const visualization = response.observabilityObjectList[0];
        return this.parseSavedVisualizations(visualization);
      } catch (error) {
        throw new Error('Fetch Saved Visualizations By Id Error:' + error);
      }
    });

    _defineProperty(this, "getVisualizations", async (client, panelId) => {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: panelId
        });
        return response.observabilityObjectList[0].operationalPanel.visualizations;
      } catch (error) {
        throw new Error('Get Visualizations Error:' + error);
      }
    });

    _defineProperty(this, "calculatOverlapArea", (bb1, bb2) => {
      const x_left = Math.max(bb1.x1, bb2.x1);
      const y_top = Math.max(bb1.y1, bb2.y1);
      const x_right = Math.min(bb1.x2, bb2.x2);
      const y_bottom = Math.min(bb1.y2, bb2.y2);
      if (x_right < x_left || y_bottom < y_top) return 0;
      return (x_right - x_left) * (y_bottom - y_top);
    });

    _defineProperty(this, "getTotalOverlapArea", panelVisualizations => {
      const newVizBox = {
        x1: 0,
        y1: 0,
        x2: 6,
        y2: 4
      };
      const currentVizBoxes = panelVisualizations.map(visualization => {
        return {
          x1: visualization.x,
          y1: visualization.y,
          x2: visualization.x + visualization.w,
          y2: visualization.y + visualization.h
        };
      });
      let isOverlapping = 0;
      currentVizBoxes.map(viz => {
        isOverlapping += this.calculatOverlapArea(viz, newVizBox);
      });
      return isOverlapping;
    });

    _defineProperty(this, "getNewVizDimensions", panelVisualizations => {
      let maxY = 0;
      let maxYH = 0; // check if we can place the new visualization at default location

      if (this.getTotalOverlapArea(panelVisualizations) === 0) {
        return {
          x: 0,
          y: 0,
          w: 6,
          h: 4
        };
      } // else place the new visualization at the bottom of the panel


      panelVisualizations.map(panelVisualization => {
        if (panelVisualization.y >= maxY) {
          maxY = panelVisualization.y;
          maxYH = panelVisualization.h;
        }
      });
      return {
        x: 0,
        y: maxY + maxYH,
        w: 6,
        h: 4
      };
    });

    _defineProperty(this, "addVisualization", async (client, panelId, savedVisualizationId, oldVisualizationId) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        let newDimensions;
        let visualizationsList = [];

        if (oldVisualizationId === undefined) {
          newDimensions = this.getNewVizDimensions(allPanelVisualizations);
          visualizationsList = allPanelVisualizations;
        } else {
          allPanelVisualizations.map(visualization => {
            if (visualization.id !== oldVisualizationId) {
              visualizationsList.push(visualization);
            } else {
              newDimensions = {
                x: visualization.x,
                y: visualization.y,
                w: visualization.w,
                h: visualization.h
              };
            }
          });
        }

        const newPanelVisualizations = [...visualizationsList, {
          id: 'panel_viz_' + (0, _uuid.v4)(),
          savedVisualizationId,
          ...newDimensions
        }];
        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: newPanelVisualizations
        });
        return newPanelVisualizations;
      } catch (error) {
        throw new Error('Add/Replace Visualization Error:' + error);
      }
    });

    _defineProperty(this, "editVisualization", async (client, panelId, visualizationParams) => {
      try {
        const allPanelVisualizations = await this.getVisualizations(client, panelId);
        const filteredPanelVisualizations = [];

        for (let i = 0; i < allPanelVisualizations.length; i++) {
          for (let j = 0; j < visualizationParams.length; j++) {
            if (allPanelVisualizations[i].id === visualizationParams[j].i) {
              filteredPanelVisualizations.push({ ...allPanelVisualizations[i],
                x: visualizationParams[j].x,
                y: visualizationParams[j].y,
                w: visualizationParams[j].w,
                h: visualizationParams[j].h
              });
            }
          }
        }

        const updatePanelResponse = await this.updatePanel(client, panelId, {
          visualizations: filteredPanelVisualizations
        });
        return filteredPanelVisualizations;
      } catch (error) {
        throw new Error('Edit Visualizations Error:' + error);
      }
    });

    _defineProperty(this, "addSamplePanels", async (client, savedVisualizationIds) => {
      try {
        const panelBody = (0, _sample_panels.createDemoPanel)(savedVisualizationIds);
        const indexResponse = await this.indexPanel(client, panelBody);
        const fetchPanel = await this.getPanel(client, indexResponse.objectId);
        const fetchResponse = {
          name: fetchPanel.operationalPanel.name,
          id: fetchPanel.objectId,
          dateCreated: fetchPanel.createdTimeMs,
          dateModified: fetchPanel.lastUpdatedTimeMs
        };
        return [fetchResponse];
      } catch (error) {
        throw new Error('Create Panel Error:' + error);
      }
    });
  }

}

exports.CustomPanelsAdaptor = CustomPanelsAdaptor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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