"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createVisualReport = void 0;

var _puppeteerCore = _interopRequireDefault(require("puppeteer-core"));

var _dompurify = _interopRequireDefault(require("dompurify"));

var _jsdom = require("jsdom");

var _constants = require("../constants");

var _helpers = require("../helpers");

var _fs = _interopRequireDefault(require("fs"));

var _lodash = _interopRequireDefault(require("lodash"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const createVisualReport = async (reportParams, queryUrl, logger, extraHeaders, timezone, validRequestProtocol = /^(data:image)/) => {
  const {
    core_params,
    report_name: reportName,
    report_source: reportSource
  } = reportParams;
  const coreParams = core_params;
  const {
    header,
    footer,
    window_height: windowHeight,
    window_width: windowWidth,
    report_format: reportFormat
  } = coreParams;
  const window = new _jsdom.JSDOM('').window;
  const DOMPurify = (0, _dompurify.default)(window);
  let keywordFilteredHeader = header ? _constants.converter.makeHtml(header) : _constants.DEFAULT_REPORT_HEADER;
  let keywordFilteredFooter = footer ? _constants.converter.makeHtml(footer) : '';
  keywordFilteredHeader = DOMPurify.sanitize(keywordFilteredHeader);
  keywordFilteredFooter = DOMPurify.sanitize(keywordFilteredFooter); // filter blocked keywords in header and footer

  if (keywordFilteredHeader !== '') {
    keywordFilteredHeader = (0, _constants.replaceBlockedKeywords)(keywordFilteredHeader);
  }

  if (keywordFilteredFooter !== '') {
    keywordFilteredFooter = (0, _constants.replaceBlockedKeywords)(keywordFilteredFooter);
  } // set up puppeteer


  const browser = await _puppeteerCore.default.launch({
    headless: true,

    /**
     * TODO: temp fix to disable sandbox when launching chromium on Linux instance
     * https://github.com/puppeteer/puppeteer/blob/main/docs/troubleshooting.md#setting-up-chrome-linux-sandbox
     */
    args: ['--no-sandbox', '--disable-setuid-sandbox', '--disable-gpu', '--no-zygote', '--single-process', '--font-render-hinting=none', '--js-flags="--jitless --no-opt"', '--disable-features=V8OptimizeJavascript'],
    executablePath: _constants.CHROMIUM_PATH,
    ignoreHTTPSErrors: true,
    env: {
      TZ: timezone || 'UTC'
    }
  });
  const page = await browser.newPage();
  await page.setRequestInterception(true);
  let localStorageAvailable = true;
  page.on('request', req => {
    // disallow non-allowlisted connections. urls with valid protocols do not need ALLOWED_HOSTS check
    if (!validRequestProtocol.test(req.url()) && !_constants.ALLOWED_HOSTS.test(new URL(req.url()).hostname)) {
      if (req.isNavigationRequest() && req.redirectChain().length > 0) {
        localStorageAvailable = false;
        logger.error('Reporting does not allow redirections to outside of localhost, aborting. URL received: ' + req.url());
      } else {
        logger.warn('Disabled connection to non-allowlist domains: ' + req.url());
      }

      req.abort();
    } else {
      req.continue();
    }
  });
  page.setDefaultNavigationTimeout(0);
  page.setDefaultTimeout(100000); // use 100s timeout instead of default 30s
  // Set extra headers that are needed

  if (!_lodash.default.isEmpty(extraHeaders)) {
    await page.setExtraHTTPHeaders(extraHeaders);
  }

  logger.info(`original queryUrl ${queryUrl}`);
  await page.goto(queryUrl, {
    waitUntil: 'networkidle0'
  }); // should add to local storage after page.goto, then access the page again - browser must have an url to register local storage item on it

  try {
    await page.evaluate(
    /* istanbul ignore next */
    key => {
      try {
        if (localStorageAvailable && typeof localStorage !== 'undefined' && localStorage !== null) {
          localStorage.setItem(key, 'false');
        }
      } catch (err) {}
    }, _constants.SECURITY_CONSTANTS.TENANT_LOCAL_STORAGE_KEY);
  } catch (err) {
    logger.error(err);
  }

  await page.goto(queryUrl, {
    waitUntil: 'networkidle0'
  });
  logger.info(`page url ${page.url()}`);
  await page.setViewport({
    width: windowWidth,
    height: windowHeight
  });
  let buffer; // remove unwanted elements

  await page.evaluate(
  /* istanbul ignore next */
  (reportSource, REPORT_TYPE) => {
    // remove buttons
    document.querySelectorAll("[class^='euiButton']").forEach(e => e.remove()); // remove top navBar

    document.querySelectorAll("[class^='euiHeader']").forEach(e => e.remove()); // remove visualization editor

    if (reportSource === REPORT_TYPE.visualization) {
      var _document$querySelect, _document$querySelect2;

      (_document$querySelect = document.querySelector('[data-test-subj="splitPanelResizer"]')) === null || _document$querySelect === void 0 ? void 0 : _document$querySelect.remove();
      (_document$querySelect2 = document.querySelector('.visEditor__collapsibleSidebar')) === null || _document$querySelect2 === void 0 ? void 0 : _document$querySelect2.remove();
    }

    document.body.style.paddingTop = '0px';
  }, reportSource, _constants.REPORT_TYPE); // force wait for any resize to load after the above DOM modification

  await page.waitFor(1000); // crop content

  switch (reportSource) {
    case _constants.REPORT_TYPE.dashboard:
      await page.waitForSelector(_constants.SELECTOR.dashboard, {
        visible: true
      });
      break;

    case _constants.REPORT_TYPE.visualization:
      await page.waitForSelector(_constants.SELECTOR.visualization, {
        visible: true
      });
      break;

    case _constants.REPORT_TYPE.notebook:
      await page.waitForSelector(_constants.SELECTOR.notebook, {
        visible: true
      });
      break;

    default:
      throw Error(`report source can only be one of [Dashboard, Visualization]`);
  } // wait for dynamic page content to render


  await waitForDynamicContent(page);
  await addReportHeader(page, keywordFilteredHeader);
  await addReportFooter(page, keywordFilteredFooter);
  await addReportStyle(page); // this causes UT to fail in github CI but works locally

  try {
    const numDisallowedTags = await page.evaluate(() => document.getElementsByTagName('iframe').length + document.getElementsByTagName('embed').length + document.getElementsByTagName('object').length);

    if (numDisallowedTags > 0) {
      throw Error('Reporting does not support "iframe", "embed", or "object" tags, aborting');
    }
  } catch (error) {
    logger.error(error);
  } // create pdf or png accordingly


  if (reportFormat === _constants.FORMAT.pdf) {
    const scrollHeight = await page.evaluate(
    /* istanbul ignore next */
    () => document.documentElement.scrollHeight);
    buffer = await page.pdf({
      margin: undefined,
      width: windowWidth,
      height: scrollHeight + 'px',
      printBackground: true,
      pageRanges: '1'
    });
  } else if (reportFormat === _constants.FORMAT.png) {
    buffer = await page.screenshot({
      fullPage: true
    });
  }

  const curTime = new Date();
  const timeCreated = curTime.valueOf();
  const fileName = `${(0, _helpers.getFileName)(reportName, curTime)}.${reportFormat}`;
  await browser.close();
  return {
    timeCreated,
    dataUrl: buffer.toString('base64'),
    fileName
  };
};

exports.createVisualReport = createVisualReport;

const addReportStyle = async page => {
  const css = _fs.default.readFileSync(`${__dirname}/style.css`).toString();

  await page.evaluate(
  /* istanbul ignore next */
  style => {
    const styleElement = document.createElement('style');
    styleElement.innerHTML = style;
    document.getElementsByTagName('head')[0].appendChild(styleElement);
  }, css);
};

const addReportHeader = async (page, header) => {
  const headerHtml = _fs.default.readFileSync(`${__dirname}/header_template.html`).toString().replace('<!--CONTENT-->', header);

  await page.evaluate(
  /* istanbul ignore next */
  headerHtml => {
    var _content$parentNode;

    const content = document.body.firstChild;
    const headerContainer = document.createElement('div');
    headerContainer.className = 'reportWrapper';
    headerContainer.innerHTML = headerHtml;
    content === null || content === void 0 ? void 0 : (_content$parentNode = content.parentNode) === null || _content$parentNode === void 0 ? void 0 : _content$parentNode.insertBefore(headerContainer, content);
  }, headerHtml);
};

const addReportFooter = async (page, footer) => {
  const headerHtml = _fs.default.readFileSync(`${__dirname}/footer_template.html`).toString().replace('<!--CONTENT-->', footer);

  await page.evaluate(
  /* istanbul ignore next */
  headerHtml => {
    var _content$parentNode2;

    const content = document.body.firstChild;
    const headerContainer = document.createElement('div');
    headerContainer.className = 'reportWrapper';
    headerContainer.innerHTML = headerHtml;
    content === null || content === void 0 ? void 0 : (_content$parentNode2 = content.parentNode) === null || _content$parentNode2 === void 0 ? void 0 : _content$parentNode2.insertBefore(headerContainer, null);
  }, headerHtml);
}; // add waitForDynamicContent function


const waitForDynamicContent = async (page, timeout = 30000, interval = 1000, checks = 5) => {
  const maxChecks = timeout / interval;
  let passedChecks = 0;
  let previousLength = 0;
  let i = 0;

  while (i++ <= maxChecks) {
    let pageContent = await page.content();
    let currentLength = pageContent.length;
    previousLength === 0 || previousLength != currentLength ? passedChecks = 0 : passedChecks++;

    if (passedChecks >= checks) {
      break;
    }

    previousLength = currentLength;
    await page.waitFor(interval);
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInZpc3VhbFJlcG9ydEhlbHBlci50cyJdLCJuYW1lcyI6WyJjcmVhdGVWaXN1YWxSZXBvcnQiLCJyZXBvcnRQYXJhbXMiLCJxdWVyeVVybCIsImxvZ2dlciIsImV4dHJhSGVhZGVycyIsInRpbWV6b25lIiwidmFsaWRSZXF1ZXN0UHJvdG9jb2wiLCJjb3JlX3BhcmFtcyIsInJlcG9ydF9uYW1lIiwicmVwb3J0TmFtZSIsInJlcG9ydF9zb3VyY2UiLCJyZXBvcnRTb3VyY2UiLCJjb3JlUGFyYW1zIiwiaGVhZGVyIiwiZm9vdGVyIiwid2luZG93X2hlaWdodCIsIndpbmRvd0hlaWdodCIsIndpbmRvd193aWR0aCIsIndpbmRvd1dpZHRoIiwicmVwb3J0X2Zvcm1hdCIsInJlcG9ydEZvcm1hdCIsIndpbmRvdyIsIkpTRE9NIiwiRE9NUHVyaWZ5Iiwia2V5d29yZEZpbHRlcmVkSGVhZGVyIiwiY29udmVydGVyIiwibWFrZUh0bWwiLCJERUZBVUxUX1JFUE9SVF9IRUFERVIiLCJrZXl3b3JkRmlsdGVyZWRGb290ZXIiLCJzYW5pdGl6ZSIsImJyb3dzZXIiLCJwdXBwZXRlZXIiLCJsYXVuY2giLCJoZWFkbGVzcyIsImFyZ3MiLCJleGVjdXRhYmxlUGF0aCIsIkNIUk9NSVVNX1BBVEgiLCJpZ25vcmVIVFRQU0Vycm9ycyIsImVudiIsIlRaIiwicGFnZSIsIm5ld1BhZ2UiLCJzZXRSZXF1ZXN0SW50ZXJjZXB0aW9uIiwibG9jYWxTdG9yYWdlQXZhaWxhYmxlIiwib24iLCJyZXEiLCJ0ZXN0IiwidXJsIiwiQUxMT1dFRF9IT1NUUyIsIlVSTCIsImhvc3RuYW1lIiwiaXNOYXZpZ2F0aW9uUmVxdWVzdCIsInJlZGlyZWN0Q2hhaW4iLCJsZW5ndGgiLCJlcnJvciIsIndhcm4iLCJhYm9ydCIsImNvbnRpbnVlIiwic2V0RGVmYXVsdE5hdmlnYXRpb25UaW1lb3V0Iiwic2V0RGVmYXVsdFRpbWVvdXQiLCJfIiwiaXNFbXB0eSIsInNldEV4dHJhSFRUUEhlYWRlcnMiLCJpbmZvIiwiZ290byIsIndhaXRVbnRpbCIsImV2YWx1YXRlIiwia2V5IiwibG9jYWxTdG9yYWdlIiwic2V0SXRlbSIsImVyciIsIlNFQ1VSSVRZX0NPTlNUQU5UUyIsIlRFTkFOVF9MT0NBTF9TVE9SQUdFX0tFWSIsInNldFZpZXdwb3J0Iiwid2lkdGgiLCJoZWlnaHQiLCJidWZmZXIiLCJSRVBPUlRfVFlQRSIsImRvY3VtZW50IiwicXVlcnlTZWxlY3RvckFsbCIsImZvckVhY2giLCJlIiwicmVtb3ZlIiwidmlzdWFsaXphdGlvbiIsInF1ZXJ5U2VsZWN0b3IiLCJib2R5Iiwic3R5bGUiLCJwYWRkaW5nVG9wIiwid2FpdEZvciIsImRhc2hib2FyZCIsIndhaXRGb3JTZWxlY3RvciIsIlNFTEVDVE9SIiwidmlzaWJsZSIsIm5vdGVib29rIiwiRXJyb3IiLCJ3YWl0Rm9yRHluYW1pY0NvbnRlbnQiLCJhZGRSZXBvcnRIZWFkZXIiLCJhZGRSZXBvcnRGb290ZXIiLCJhZGRSZXBvcnRTdHlsZSIsIm51bURpc2FsbG93ZWRUYWdzIiwiZ2V0RWxlbWVudHNCeVRhZ05hbWUiLCJGT1JNQVQiLCJwZGYiLCJzY3JvbGxIZWlnaHQiLCJkb2N1bWVudEVsZW1lbnQiLCJtYXJnaW4iLCJ1bmRlZmluZWQiLCJwcmludEJhY2tncm91bmQiLCJwYWdlUmFuZ2VzIiwicG5nIiwic2NyZWVuc2hvdCIsImZ1bGxQYWdlIiwiY3VyVGltZSIsIkRhdGUiLCJ0aW1lQ3JlYXRlZCIsInZhbHVlT2YiLCJmaWxlTmFtZSIsImNsb3NlIiwiZGF0YVVybCIsInRvU3RyaW5nIiwiY3NzIiwiZnMiLCJyZWFkRmlsZVN5bmMiLCJfX2Rpcm5hbWUiLCJzdHlsZUVsZW1lbnQiLCJjcmVhdGVFbGVtZW50IiwiaW5uZXJIVE1MIiwiYXBwZW5kQ2hpbGQiLCJoZWFkZXJIdG1sIiwicmVwbGFjZSIsImNvbnRlbnQiLCJmaXJzdENoaWxkIiwiaGVhZGVyQ29udGFpbmVyIiwiY2xhc3NOYW1lIiwicGFyZW50Tm9kZSIsImluc2VydEJlZm9yZSIsInRpbWVvdXQiLCJpbnRlcnZhbCIsImNoZWNrcyIsIm1heENoZWNrcyIsInBhc3NlZENoZWNrcyIsInByZXZpb3VzTGVuZ3RoIiwiaSIsInBhZ2VDb250ZW50IiwiY3VycmVudExlbmd0aCJdLCJtYXBwaW5ncyI6Ijs7Ozs7OztBQUtBOztBQUNBOztBQUNBOztBQUVBOztBQVNBOztBQUlBOztBQUNBOzs7O0FBdkJBO0FBQ0E7QUFDQTtBQUNBO0FBc0JPLE1BQU1BLGtCQUFrQixHQUFHLE9BQ2hDQyxZQURnQyxFQUVoQ0MsUUFGZ0MsRUFHaENDLE1BSGdDLEVBSWhDQyxZQUpnQyxFQUtoQ0MsUUFMZ0MsRUFNaENDLG9CQUFvQixHQUFHLGVBTlMsS0FPSTtBQUNwQyxRQUFNO0FBQ0pDLElBQUFBLFdBREk7QUFFSkMsSUFBQUEsV0FBVyxFQUFFQyxVQUZUO0FBR0pDLElBQUFBLGFBQWEsRUFBRUM7QUFIWCxNQUlGVixZQUpKO0FBS0EsUUFBTVcsVUFBVSxHQUFHTCxXQUFuQjtBQUNBLFFBQU07QUFDSk0sSUFBQUEsTUFESTtBQUVKQyxJQUFBQSxNQUZJO0FBR0pDLElBQUFBLGFBQWEsRUFBRUMsWUFIWDtBQUlKQyxJQUFBQSxZQUFZLEVBQUVDLFdBSlY7QUFLSkMsSUFBQUEsYUFBYSxFQUFFQztBQUxYLE1BTUZSLFVBTko7QUFRQSxRQUFNUyxNQUFNLEdBQUcsSUFBSUMsWUFBSixDQUFVLEVBQVYsRUFBY0QsTUFBN0I7QUFDQSxRQUFNRSxTQUFTLEdBQUcsd0JBQWdCRixNQUFoQixDQUFsQjtBQUVBLE1BQUlHLHFCQUFxQixHQUFHWCxNQUFNLEdBQzlCWSxxQkFBVUMsUUFBVixDQUFtQmIsTUFBbkIsQ0FEOEIsR0FFOUJjLGdDQUZKO0FBR0EsTUFBSUMscUJBQXFCLEdBQUdkLE1BQU0sR0FBR1cscUJBQVVDLFFBQVYsQ0FBbUJaLE1BQW5CLENBQUgsR0FBZ0MsRUFBbEU7QUFFQVUsRUFBQUEscUJBQXFCLEdBQUdELFNBQVMsQ0FBQ00sUUFBVixDQUFtQkwscUJBQW5CLENBQXhCO0FBQ0FJLEVBQUFBLHFCQUFxQixHQUFHTCxTQUFTLENBQUNNLFFBQVYsQ0FBbUJELHFCQUFuQixDQUF4QixDQXhCb0MsQ0EwQnBDOztBQUNBLE1BQUlKLHFCQUFxQixLQUFLLEVBQTlCLEVBQWtDO0FBQ2hDQSxJQUFBQSxxQkFBcUIsR0FBRyx1Q0FBdUJBLHFCQUF2QixDQUF4QjtBQUNEOztBQUNELE1BQUlJLHFCQUFxQixLQUFLLEVBQTlCLEVBQWtDO0FBQ2hDQSxJQUFBQSxxQkFBcUIsR0FBRyx1Q0FBdUJBLHFCQUF2QixDQUF4QjtBQUNELEdBaENtQyxDQWtDcEM7OztBQUNBLFFBQU1FLE9BQU8sR0FBRyxNQUFNQyx1QkFBVUMsTUFBVixDQUFpQjtBQUNyQ0MsSUFBQUEsUUFBUSxFQUFFLElBRDJCOztBQUVyQztBQUNKO0FBQ0E7QUFDQTtBQUNJQyxJQUFBQSxJQUFJLEVBQUUsQ0FDSixjQURJLEVBRUosMEJBRkksRUFHSixlQUhJLEVBSUosYUFKSSxFQUtKLGtCQUxJLEVBTUosNEJBTkksRUFPSixpQ0FQSSxFQVFKLHlDQVJJLENBTitCO0FBZ0JyQ0MsSUFBQUEsY0FBYyxFQUFFQyx3QkFoQnFCO0FBaUJyQ0MsSUFBQUEsaUJBQWlCLEVBQUUsSUFqQmtCO0FBa0JyQ0MsSUFBQUEsR0FBRyxFQUFFO0FBQ0hDLE1BQUFBLEVBQUUsRUFBRWxDLFFBQVEsSUFBSTtBQURiO0FBbEJnQyxHQUFqQixDQUF0QjtBQXNCQSxRQUFNbUMsSUFBSSxHQUFHLE1BQU1WLE9BQU8sQ0FBQ1csT0FBUixFQUFuQjtBQUVBLFFBQU1ELElBQUksQ0FBQ0Usc0JBQUwsQ0FBNEIsSUFBNUIsQ0FBTjtBQUNBLE1BQUlDLHFCQUFxQixHQUFHLElBQTVCO0FBQ0FILEVBQUFBLElBQUksQ0FBQ0ksRUFBTCxDQUFRLFNBQVIsRUFBb0JDLEdBQUQsSUFBUztBQUMxQjtBQUNBLFFBQ0UsQ0FBQ3ZDLG9CQUFvQixDQUFDd0MsSUFBckIsQ0FBMEJELEdBQUcsQ0FBQ0UsR0FBSixFQUExQixDQUFELElBQ0EsQ0FBQ0MseUJBQWNGLElBQWQsQ0FBbUIsSUFBSUcsR0FBSixDQUFRSixHQUFHLENBQUNFLEdBQUosRUFBUixFQUFtQkcsUUFBdEMsQ0FGSCxFQUdFO0FBQ0EsVUFBSUwsR0FBRyxDQUFDTSxtQkFBSixNQUE2Qk4sR0FBRyxDQUFDTyxhQUFKLEdBQW9CQyxNQUFwQixHQUE2QixDQUE5RCxFQUFpRTtBQUMvRFYsUUFBQUEscUJBQXFCLEdBQUcsS0FBeEI7QUFDQXhDLFFBQUFBLE1BQU0sQ0FBQ21ELEtBQVAsQ0FDRSw0RkFDRVQsR0FBRyxDQUFDRSxHQUFKLEVBRko7QUFJRCxPQU5ELE1BTU87QUFDTDVDLFFBQUFBLE1BQU0sQ0FBQ29ELElBQVAsQ0FDRSxtREFBbURWLEdBQUcsQ0FBQ0UsR0FBSixFQURyRDtBQUdEOztBQUNERixNQUFBQSxHQUFHLENBQUNXLEtBQUo7QUFDRCxLQWhCRCxNQWdCTztBQUNMWCxNQUFBQSxHQUFHLENBQUNZLFFBQUo7QUFDRDtBQUNGLEdBckJEO0FBdUJBakIsRUFBQUEsSUFBSSxDQUFDa0IsMkJBQUwsQ0FBaUMsQ0FBakM7QUFDQWxCLEVBQUFBLElBQUksQ0FBQ21CLGlCQUFMLENBQXVCLE1BQXZCLEVBckZvQyxDQXFGSjtBQUNoQzs7QUFDQSxNQUFJLENBQUNDLGdCQUFFQyxPQUFGLENBQVV6RCxZQUFWLENBQUwsRUFBOEI7QUFDNUIsVUFBTW9DLElBQUksQ0FBQ3NCLG1CQUFMLENBQXlCMUQsWUFBekIsQ0FBTjtBQUNEOztBQUNERCxFQUFBQSxNQUFNLENBQUM0RCxJQUFQLENBQWEscUJBQW9CN0QsUUFBUyxFQUExQztBQUNBLFFBQU1zQyxJQUFJLENBQUN3QixJQUFMLENBQVU5RCxRQUFWLEVBQW9CO0FBQUUrRCxJQUFBQSxTQUFTLEVBQUU7QUFBYixHQUFwQixDQUFOLENBM0ZvQyxDQTRGcEM7O0FBQ0EsTUFBSTtBQUNGLFVBQU16QixJQUFJLENBQUMwQixRQUFMO0FBQ0o7QUFDQ0MsSUFBQUEsR0FBRCxJQUFTO0FBQ1AsVUFBSTtBQUNGLFlBQ0V4QixxQkFBcUIsSUFDckIsT0FBT3lCLFlBQVAsS0FBd0IsV0FEeEIsSUFFQUEsWUFBWSxLQUFLLElBSG5CLEVBSUU7QUFDQUEsVUFBQUEsWUFBWSxDQUFDQyxPQUFiLENBQXFCRixHQUFyQixFQUEwQixPQUExQjtBQUNEO0FBQ0YsT0FSRCxDQVFFLE9BQU9HLEdBQVAsRUFBWSxDQUFFO0FBQ2pCLEtBWkcsRUFhSkMsOEJBQW1CQyx3QkFiZixDQUFOO0FBZUQsR0FoQkQsQ0FnQkUsT0FBT0YsR0FBUCxFQUFZO0FBQ1puRSxJQUFBQSxNQUFNLENBQUNtRCxLQUFQLENBQWFnQixHQUFiO0FBQ0Q7O0FBQ0QsUUFBTTlCLElBQUksQ0FBQ3dCLElBQUwsQ0FBVTlELFFBQVYsRUFBb0I7QUFBRStELElBQUFBLFNBQVMsRUFBRTtBQUFiLEdBQXBCLENBQU47QUFDQTlELEVBQUFBLE1BQU0sQ0FBQzRELElBQVAsQ0FBYSxZQUFXdkIsSUFBSSxDQUFDTyxHQUFMLEVBQVcsRUFBbkM7QUFFQSxRQUFNUCxJQUFJLENBQUNpQyxXQUFMLENBQWlCO0FBQ3JCQyxJQUFBQSxLQUFLLEVBQUV4RCxXQURjO0FBRXJCeUQsSUFBQUEsTUFBTSxFQUFFM0Q7QUFGYSxHQUFqQixDQUFOO0FBS0EsTUFBSTRELE1BQUosQ0F4SG9DLENBeUhwQzs7QUFDQSxRQUFNcEMsSUFBSSxDQUFDMEIsUUFBTDtBQUNKO0FBQ0EsR0FBQ3ZELFlBQUQsRUFBZWtFLFdBQWYsS0FBK0I7QUFDN0I7QUFDQUMsSUFBQUEsUUFBUSxDQUNMQyxnQkFESCxDQUNvQixzQkFEcEIsRUFFR0MsT0FGSCxDQUVZQyxDQUFELElBQU9BLENBQUMsQ0FBQ0MsTUFBRixFQUZsQixFQUY2QixDQUs3Qjs7QUFDQUosSUFBQUEsUUFBUSxDQUNMQyxnQkFESCxDQUNvQixzQkFEcEIsRUFFR0MsT0FGSCxDQUVZQyxDQUFELElBQU9BLENBQUMsQ0FBQ0MsTUFBRixFQUZsQixFQU42QixDQVM3Qjs7QUFDQSxRQUFJdkUsWUFBWSxLQUFLa0UsV0FBVyxDQUFDTSxhQUFqQyxFQUFnRDtBQUFBOztBQUM5QywrQkFBQUwsUUFBUSxDQUNMTSxhQURILENBQ2lCLHNDQURqQixpRkFFSUYsTUFGSjtBQUdBLGdDQUFBSixRQUFRLENBQUNNLGFBQVQsQ0FBdUIsZ0NBQXZCLG1GQUEwREYsTUFBMUQ7QUFDRDs7QUFDREosSUFBQUEsUUFBUSxDQUFDTyxJQUFULENBQWNDLEtBQWQsQ0FBb0JDLFVBQXBCLEdBQWlDLEtBQWpDO0FBQ0QsR0FuQkcsRUFvQko1RSxZQXBCSSxFQXFCSmtFLHNCQXJCSSxDQUFOLENBMUhvQyxDQWtKcEM7O0FBQ0EsUUFBTXJDLElBQUksQ0FBQ2dELE9BQUwsQ0FBYSxJQUFiLENBQU4sQ0FuSm9DLENBb0pwQzs7QUFDQSxVQUFRN0UsWUFBUjtBQUNFLFNBQUtrRSx1QkFBWVksU0FBakI7QUFDRSxZQUFNakQsSUFBSSxDQUFDa0QsZUFBTCxDQUFxQkMsb0JBQVNGLFNBQTlCLEVBQXlDO0FBQzdDRyxRQUFBQSxPQUFPLEVBQUU7QUFEb0MsT0FBekMsQ0FBTjtBQUdBOztBQUNGLFNBQUtmLHVCQUFZTSxhQUFqQjtBQUNFLFlBQU0zQyxJQUFJLENBQUNrRCxlQUFMLENBQXFCQyxvQkFBU1IsYUFBOUIsRUFBNkM7QUFDakRTLFFBQUFBLE9BQU8sRUFBRTtBQUR3QyxPQUE3QyxDQUFOO0FBR0E7O0FBQ0YsU0FBS2YsdUJBQVlnQixRQUFqQjtBQUNFLFlBQU1yRCxJQUFJLENBQUNrRCxlQUFMLENBQXFCQyxvQkFBU0UsUUFBOUIsRUFBd0M7QUFDNUNELFFBQUFBLE9BQU8sRUFBRTtBQURtQyxPQUF4QyxDQUFOO0FBR0E7O0FBQ0Y7QUFDRSxZQUFNRSxLQUFLLENBQ1IsNkRBRFEsQ0FBWDtBQWpCSixHQXJKb0MsQ0EyS3BDOzs7QUFDQSxRQUFNQyxxQkFBcUIsQ0FBQ3ZELElBQUQsQ0FBM0I7QUFFQSxRQUFNd0QsZUFBZSxDQUFDeEQsSUFBRCxFQUFPaEIscUJBQVAsQ0FBckI7QUFDQSxRQUFNeUUsZUFBZSxDQUFDekQsSUFBRCxFQUFPWixxQkFBUCxDQUFyQjtBQUNBLFFBQU1zRSxjQUFjLENBQUMxRCxJQUFELENBQXBCLENBaExvQyxDQWtMcEM7O0FBQ0EsTUFBSTtBQUNGLFVBQU0yRCxpQkFBaUIsR0FBRyxNQUFNM0QsSUFBSSxDQUFDMEIsUUFBTCxDQUM5QixNQUNFWSxRQUFRLENBQUNzQixvQkFBVCxDQUE4QixRQUE5QixFQUF3Qy9DLE1BQXhDLEdBQ0F5QixRQUFRLENBQUNzQixvQkFBVCxDQUE4QixPQUE5QixFQUF1Qy9DLE1BRHZDLEdBRUF5QixRQUFRLENBQUNzQixvQkFBVCxDQUE4QixRQUE5QixFQUF3Qy9DLE1BSlosQ0FBaEM7O0FBTUEsUUFBSThDLGlCQUFpQixHQUFHLENBQXhCLEVBQTJCO0FBQ3pCLFlBQU1MLEtBQUssQ0FBQywwRUFBRCxDQUFYO0FBQ0Q7QUFDRixHQVZELENBVUUsT0FBT3hDLEtBQVAsRUFBYztBQUNkbkQsSUFBQUEsTUFBTSxDQUFDbUQsS0FBUCxDQUFhQSxLQUFiO0FBQ0QsR0EvTG1DLENBaU1wQzs7O0FBQ0EsTUFBSWxDLFlBQVksS0FBS2lGLGtCQUFPQyxHQUE1QixFQUFpQztBQUMvQixVQUFNQyxZQUFZLEdBQUcsTUFBTS9ELElBQUksQ0FBQzBCLFFBQUw7QUFDekI7QUFDQSxVQUFNWSxRQUFRLENBQUMwQixlQUFULENBQXlCRCxZQUZOLENBQTNCO0FBS0EzQixJQUFBQSxNQUFNLEdBQUcsTUFBTXBDLElBQUksQ0FBQzhELEdBQUwsQ0FBUztBQUN0QkcsTUFBQUEsTUFBTSxFQUFFQyxTQURjO0FBRXRCaEMsTUFBQUEsS0FBSyxFQUFFeEQsV0FGZTtBQUd0QnlELE1BQUFBLE1BQU0sRUFBRTRCLFlBQVksR0FBRyxJQUhEO0FBSXRCSSxNQUFBQSxlQUFlLEVBQUUsSUFKSztBQUt0QkMsTUFBQUEsVUFBVSxFQUFFO0FBTFUsS0FBVCxDQUFmO0FBT0QsR0FiRCxNQWFPLElBQUl4RixZQUFZLEtBQUtpRixrQkFBT1EsR0FBNUIsRUFBaUM7QUFDdENqQyxJQUFBQSxNQUFNLEdBQUcsTUFBTXBDLElBQUksQ0FBQ3NFLFVBQUwsQ0FBZ0I7QUFDN0JDLE1BQUFBLFFBQVEsRUFBRTtBQURtQixLQUFoQixDQUFmO0FBR0Q7O0FBRUQsUUFBTUMsT0FBTyxHQUFHLElBQUlDLElBQUosRUFBaEI7QUFDQSxRQUFNQyxXQUFXLEdBQUdGLE9BQU8sQ0FBQ0csT0FBUixFQUFwQjtBQUNBLFFBQU1DLFFBQVEsR0FBSSxHQUFFLDBCQUFZM0csVUFBWixFQUF3QnVHLE9BQXhCLENBQWlDLElBQUc1RixZQUFhLEVBQXJFO0FBQ0EsUUFBTVUsT0FBTyxDQUFDdUYsS0FBUixFQUFOO0FBRUEsU0FBTztBQUFFSCxJQUFBQSxXQUFGO0FBQWVJLElBQUFBLE9BQU8sRUFBRTFDLE1BQU0sQ0FBQzJDLFFBQVAsQ0FBZ0IsUUFBaEIsQ0FBeEI7QUFBbURILElBQUFBO0FBQW5ELEdBQVA7QUFDRCxDQWxPTTs7OztBQW9PUCxNQUFNbEIsY0FBYyxHQUFHLE1BQU8xRCxJQUFQLElBQWdDO0FBQ3JELFFBQU1nRixHQUFHLEdBQUdDLFlBQUdDLFlBQUgsQ0FBaUIsR0FBRUMsU0FBVSxZQUE3QixFQUEwQ0osUUFBMUMsRUFBWjs7QUFFQSxRQUFNL0UsSUFBSSxDQUFDMEIsUUFBTDtBQUNKO0FBQ0NvQixFQUFBQSxLQUFELElBQW1CO0FBQ2pCLFVBQU1zQyxZQUFZLEdBQUc5QyxRQUFRLENBQUMrQyxhQUFULENBQXVCLE9BQXZCLENBQXJCO0FBQ0FELElBQUFBLFlBQVksQ0FBQ0UsU0FBYixHQUF5QnhDLEtBQXpCO0FBQ0FSLElBQUFBLFFBQVEsQ0FBQ3NCLG9CQUFULENBQThCLE1BQTlCLEVBQXNDLENBQXRDLEVBQXlDMkIsV0FBekMsQ0FBcURILFlBQXJEO0FBQ0QsR0FORyxFQU9KSixHQVBJLENBQU47QUFTRCxDQVpEOztBQWNBLE1BQU14QixlQUFlLEdBQUcsT0FBT3hELElBQVAsRUFBNkIzQixNQUE3QixLQUFnRDtBQUN0RSxRQUFNbUgsVUFBVSxHQUFHUCxZQUNoQkMsWUFEZ0IsQ0FDRixHQUFFQyxTQUFVLHVCQURWLEVBRWhCSixRQUZnQixHQUdoQlUsT0FIZ0IsQ0FHUixnQkFIUSxFQUdVcEgsTUFIVixDQUFuQjs7QUFLQSxRQUFNMkIsSUFBSSxDQUFDMEIsUUFBTDtBQUNKO0FBQ0M4RCxFQUFBQSxVQUFELElBQXdCO0FBQUE7O0FBQ3RCLFVBQU1FLE9BQU8sR0FBR3BELFFBQVEsQ0FBQ08sSUFBVCxDQUFjOEMsVUFBOUI7QUFDQSxVQUFNQyxlQUFlLEdBQUd0RCxRQUFRLENBQUMrQyxhQUFULENBQXVCLEtBQXZCLENBQXhCO0FBQ0FPLElBQUFBLGVBQWUsQ0FBQ0MsU0FBaEIsR0FBNEIsZUFBNUI7QUFDQUQsSUFBQUEsZUFBZSxDQUFDTixTQUFoQixHQUE0QkUsVUFBNUI7QUFDQUUsSUFBQUEsT0FBTyxTQUFQLElBQUFBLE9BQU8sV0FBUCxtQ0FBQUEsT0FBTyxDQUFFSSxVQUFULDRFQUFxQkMsWUFBckIsQ0FBa0NILGVBQWxDLEVBQW1ERixPQUFuRDtBQUNELEdBUkcsRUFTSkYsVUFUSSxDQUFOO0FBV0QsQ0FqQkQ7O0FBbUJBLE1BQU0vQixlQUFlLEdBQUcsT0FBT3pELElBQVAsRUFBNkIxQixNQUE3QixLQUFnRDtBQUN0RSxRQUFNa0gsVUFBVSxHQUFHUCxZQUNoQkMsWUFEZ0IsQ0FDRixHQUFFQyxTQUFVLHVCQURWLEVBRWhCSixRQUZnQixHQUdoQlUsT0FIZ0IsQ0FHUixnQkFIUSxFQUdVbkgsTUFIVixDQUFuQjs7QUFLQSxRQUFNMEIsSUFBSSxDQUFDMEIsUUFBTDtBQUNKO0FBQ0M4RCxFQUFBQSxVQUFELElBQXdCO0FBQUE7O0FBQ3RCLFVBQU1FLE9BQU8sR0FBR3BELFFBQVEsQ0FBQ08sSUFBVCxDQUFjOEMsVUFBOUI7QUFDQSxVQUFNQyxlQUFlLEdBQUd0RCxRQUFRLENBQUMrQyxhQUFULENBQXVCLEtBQXZCLENBQXhCO0FBQ0FPLElBQUFBLGVBQWUsQ0FBQ0MsU0FBaEIsR0FBNEIsZUFBNUI7QUFDQUQsSUFBQUEsZUFBZSxDQUFDTixTQUFoQixHQUE0QkUsVUFBNUI7QUFDQUUsSUFBQUEsT0FBTyxTQUFQLElBQUFBLE9BQU8sV0FBUCxvQ0FBQUEsT0FBTyxDQUFFSSxVQUFULDhFQUFxQkMsWUFBckIsQ0FBa0NILGVBQWxDLEVBQW1ELElBQW5EO0FBQ0QsR0FSRyxFQVNKSixVQVRJLENBQU47QUFXRCxDQWpCRCxDLENBbUJBOzs7QUFDQSxNQUFNakMscUJBQXFCLEdBQUcsT0FDNUJ2RCxJQUQ0QixFQUU1QmdHLE9BQU8sR0FBRyxLQUZrQixFQUc1QkMsUUFBUSxHQUFHLElBSGlCLEVBSTVCQyxNQUFNLEdBQUcsQ0FKbUIsS0FLekI7QUFDSCxRQUFNQyxTQUFTLEdBQUdILE9BQU8sR0FBR0MsUUFBNUI7QUFDQSxNQUFJRyxZQUFZLEdBQUcsQ0FBbkI7QUFDQSxNQUFJQyxjQUFjLEdBQUcsQ0FBckI7QUFFQSxNQUFJQyxDQUFDLEdBQUcsQ0FBUjs7QUFDQSxTQUFPQSxDQUFDLE1BQU1ILFNBQWQsRUFBeUI7QUFDdkIsUUFBSUksV0FBVyxHQUFHLE1BQU12RyxJQUFJLENBQUMwRixPQUFMLEVBQXhCO0FBQ0EsUUFBSWMsYUFBYSxHQUFHRCxXQUFXLENBQUMxRixNQUFoQztBQUVBd0YsSUFBQUEsY0FBYyxLQUFLLENBQW5CLElBQXdCQSxjQUFjLElBQUlHLGFBQTFDLEdBQ0tKLFlBQVksR0FBRyxDQURwQixHQUVJQSxZQUFZLEVBRmhCOztBQUdBLFFBQUlBLFlBQVksSUFBSUYsTUFBcEIsRUFBNEI7QUFDMUI7QUFDRDs7QUFFREcsSUFBQUEsY0FBYyxHQUFHRyxhQUFqQjtBQUNBLFVBQU14RyxJQUFJLENBQUNnRCxPQUFMLENBQWFpRCxRQUFiLENBQU47QUFDRDtBQUNGLENBekJEIiwic291cmNlc0NvbnRlbnQiOlsiLypcbiAqIENvcHlyaWdodCBPcGVuU2VhcmNoIENvbnRyaWJ1dG9yc1xuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqL1xuXG5pbXBvcnQgcHVwcGV0ZWVyLCB7IEhlYWRlcnMgfSBmcm9tICdwdXBwZXRlZXItY29yZSc7XG5pbXBvcnQgY3JlYXRlRE9NUHVyaWZ5IGZyb20gJ2RvbXB1cmlmeSc7XG5pbXBvcnQgeyBKU0RPTSB9IGZyb20gJ2pzZG9tJztcbmltcG9ydCB7IExvZ2dlciB9IGZyb20gJy4uLy4uLy4uLy4uLy4uLy4uL3NyYy9jb3JlL3NlcnZlcic7XG5pbXBvcnQge1xuICBERUZBVUxUX1JFUE9SVF9IRUFERVIsXG4gIFJFUE9SVF9UWVBFLFxuICBGT1JNQVQsXG4gIFNFTEVDVE9SLFxuICBDSFJPTUlVTV9QQVRILFxuICBTRUNVUklUWV9DT05TVEFOVFMsXG4gIEFMTE9XRURfSE9TVFMsXG59IGZyb20gJy4uL2NvbnN0YW50cyc7XG5pbXBvcnQgeyBnZXRGaWxlTmFtZSB9IGZyb20gJy4uL2hlbHBlcnMnO1xuaW1wb3J0IHsgQ3JlYXRlUmVwb3J0UmVzdWx0VHlwZSB9IGZyb20gJy4uL3R5cGVzJztcbmltcG9ydCB7IFJlcG9ydFBhcmFtc1NjaGVtYVR5cGUsIFZpc3VhbFJlcG9ydFNjaGVtYVR5cGUgfSBmcm9tICdzZXJ2ZXIvbW9kZWwnO1xuaW1wb3J0IHsgY29udmVydGVyLCByZXBsYWNlQmxvY2tlZEtleXdvcmRzIH0gZnJvbSAnLi4vY29uc3RhbnRzJztcbmltcG9ydCBmcyBmcm9tICdmcyc7XG5pbXBvcnQgXyBmcm9tICdsb2Rhc2gnO1xuXG5leHBvcnQgY29uc3QgY3JlYXRlVmlzdWFsUmVwb3J0ID0gYXN5bmMgKFxuICByZXBvcnRQYXJhbXM6IFJlcG9ydFBhcmFtc1NjaGVtYVR5cGUsXG4gIHF1ZXJ5VXJsOiBzdHJpbmcsXG4gIGxvZ2dlcjogTG9nZ2VyLFxuICBleHRyYUhlYWRlcnM6IEhlYWRlcnMsXG4gIHRpbWV6b25lPzogc3RyaW5nLFxuICB2YWxpZFJlcXVlc3RQcm90b2NvbCA9IC9eKGRhdGE6aW1hZ2UpLyxcbik6IFByb21pc2U8Q3JlYXRlUmVwb3J0UmVzdWx0VHlwZT4gPT4ge1xuICBjb25zdCB7XG4gICAgY29yZV9wYXJhbXMsXG4gICAgcmVwb3J0X25hbWU6IHJlcG9ydE5hbWUsXG4gICAgcmVwb3J0X3NvdXJjZTogcmVwb3J0U291cmNlLFxuICB9ID0gcmVwb3J0UGFyYW1zO1xuICBjb25zdCBjb3JlUGFyYW1zID0gY29yZV9wYXJhbXMgYXMgVmlzdWFsUmVwb3J0U2NoZW1hVHlwZTtcbiAgY29uc3Qge1xuICAgIGhlYWRlcixcbiAgICBmb290ZXIsXG4gICAgd2luZG93X2hlaWdodDogd2luZG93SGVpZ2h0LFxuICAgIHdpbmRvd193aWR0aDogd2luZG93V2lkdGgsXG4gICAgcmVwb3J0X2Zvcm1hdDogcmVwb3J0Rm9ybWF0LFxuICB9ID0gY29yZVBhcmFtcztcblxuICBjb25zdCB3aW5kb3cgPSBuZXcgSlNET00oJycpLndpbmRvdztcbiAgY29uc3QgRE9NUHVyaWZ5ID0gY3JlYXRlRE9NUHVyaWZ5KHdpbmRvdyk7XG5cbiAgbGV0IGtleXdvcmRGaWx0ZXJlZEhlYWRlciA9IGhlYWRlciBcbiAgICA/IGNvbnZlcnRlci5tYWtlSHRtbChoZWFkZXIpIFxuICAgIDogREVGQVVMVF9SRVBPUlRfSEVBREVSO1xuICBsZXQga2V5d29yZEZpbHRlcmVkRm9vdGVyID0gZm9vdGVyID8gY29udmVydGVyLm1ha2VIdG1sKGZvb3RlcikgOiAnJztcblxuICBrZXl3b3JkRmlsdGVyZWRIZWFkZXIgPSBET01QdXJpZnkuc2FuaXRpemUoa2V5d29yZEZpbHRlcmVkSGVhZGVyKTtcbiAga2V5d29yZEZpbHRlcmVkRm9vdGVyID0gRE9NUHVyaWZ5LnNhbml0aXplKGtleXdvcmRGaWx0ZXJlZEZvb3Rlcik7XG5cbiAgLy8gZmlsdGVyIGJsb2NrZWQga2V5d29yZHMgaW4gaGVhZGVyIGFuZCBmb290ZXJcbiAgaWYgKGtleXdvcmRGaWx0ZXJlZEhlYWRlciAhPT0gJycpIHtcbiAgICBrZXl3b3JkRmlsdGVyZWRIZWFkZXIgPSByZXBsYWNlQmxvY2tlZEtleXdvcmRzKGtleXdvcmRGaWx0ZXJlZEhlYWRlcik7XG4gIH1cbiAgaWYgKGtleXdvcmRGaWx0ZXJlZEZvb3RlciAhPT0gJycpIHtcbiAgICBrZXl3b3JkRmlsdGVyZWRGb290ZXIgPSByZXBsYWNlQmxvY2tlZEtleXdvcmRzKGtleXdvcmRGaWx0ZXJlZEZvb3Rlcik7XG4gIH1cblxuICAvLyBzZXQgdXAgcHVwcGV0ZWVyXG4gIGNvbnN0IGJyb3dzZXIgPSBhd2FpdCBwdXBwZXRlZXIubGF1bmNoKHtcbiAgICBoZWFkbGVzczogdHJ1ZSxcbiAgICAvKipcbiAgICAgKiBUT0RPOiB0ZW1wIGZpeCB0byBkaXNhYmxlIHNhbmRib3ggd2hlbiBsYXVuY2hpbmcgY2hyb21pdW0gb24gTGludXggaW5zdGFuY2VcbiAgICAgKiBodHRwczovL2dpdGh1Yi5jb20vcHVwcGV0ZWVyL3B1cHBldGVlci9ibG9iL21haW4vZG9jcy90cm91Ymxlc2hvb3RpbmcubWQjc2V0dGluZy11cC1jaHJvbWUtbGludXgtc2FuZGJveFxuICAgICAqL1xuICAgIGFyZ3M6IFtcbiAgICAgICctLW5vLXNhbmRib3gnLFxuICAgICAgJy0tZGlzYWJsZS1zZXR1aWQtc2FuZGJveCcsXG4gICAgICAnLS1kaXNhYmxlLWdwdScsXG4gICAgICAnLS1uby16eWdvdGUnLFxuICAgICAgJy0tc2luZ2xlLXByb2Nlc3MnLFxuICAgICAgJy0tZm9udC1yZW5kZXItaGludGluZz1ub25lJyxcbiAgICAgICctLWpzLWZsYWdzPVwiLS1qaXRsZXNzIC0tbm8tb3B0XCInLFxuICAgICAgJy0tZGlzYWJsZS1mZWF0dXJlcz1WOE9wdGltaXplSmF2YXNjcmlwdCcsXG4gICAgXSxcbiAgICBleGVjdXRhYmxlUGF0aDogQ0hST01JVU1fUEFUSCxcbiAgICBpZ25vcmVIVFRQU0Vycm9yczogdHJ1ZSxcbiAgICBlbnY6IHtcbiAgICAgIFRaOiB0aW1lem9uZSB8fCAnVVRDJyxcbiAgICB9LFxuICB9KTtcbiAgY29uc3QgcGFnZSA9IGF3YWl0IGJyb3dzZXIubmV3UGFnZSgpO1xuXG4gIGF3YWl0IHBhZ2Uuc2V0UmVxdWVzdEludGVyY2VwdGlvbih0cnVlKTtcbiAgbGV0IGxvY2FsU3RvcmFnZUF2YWlsYWJsZSA9IHRydWU7XG4gIHBhZ2Uub24oJ3JlcXVlc3QnLCAocmVxKSA9PiB7XG4gICAgLy8gZGlzYWxsb3cgbm9uLWFsbG93bGlzdGVkIGNvbm5lY3Rpb25zLiB1cmxzIHdpdGggdmFsaWQgcHJvdG9jb2xzIGRvIG5vdCBuZWVkIEFMTE9XRURfSE9TVFMgY2hlY2tcbiAgICBpZiAoXG4gICAgICAhdmFsaWRSZXF1ZXN0UHJvdG9jb2wudGVzdChyZXEudXJsKCkpICYmXG4gICAgICAhQUxMT1dFRF9IT1NUUy50ZXN0KG5ldyBVUkwocmVxLnVybCgpKS5ob3N0bmFtZSlcbiAgICApIHtcbiAgICAgIGlmIChyZXEuaXNOYXZpZ2F0aW9uUmVxdWVzdCgpICYmIHJlcS5yZWRpcmVjdENoYWluKCkubGVuZ3RoID4gMCkge1xuICAgICAgICBsb2NhbFN0b3JhZ2VBdmFpbGFibGUgPSBmYWxzZTtcbiAgICAgICAgbG9nZ2VyLmVycm9yKFxuICAgICAgICAgICdSZXBvcnRpbmcgZG9lcyBub3QgYWxsb3cgcmVkaXJlY3Rpb25zIHRvIG91dHNpZGUgb2YgbG9jYWxob3N0LCBhYm9ydGluZy4gVVJMIHJlY2VpdmVkOiAnICtcbiAgICAgICAgICAgIHJlcS51cmwoKVxuICAgICAgICApO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgbG9nZ2VyLndhcm4oXG4gICAgICAgICAgJ0Rpc2FibGVkIGNvbm5lY3Rpb24gdG8gbm9uLWFsbG93bGlzdCBkb21haW5zOiAnICsgcmVxLnVybCgpXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgICByZXEuYWJvcnQoKTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmVxLmNvbnRpbnVlKCk7XG4gICAgfVxuICB9KTtcblxuICBwYWdlLnNldERlZmF1bHROYXZpZ2F0aW9uVGltZW91dCgwKTtcbiAgcGFnZS5zZXREZWZhdWx0VGltZW91dCgxMDAwMDApOyAvLyB1c2UgMTAwcyB0aW1lb3V0IGluc3RlYWQgb2YgZGVmYXVsdCAzMHNcbiAgLy8gU2V0IGV4dHJhIGhlYWRlcnMgdGhhdCBhcmUgbmVlZGVkXG4gIGlmICghXy5pc0VtcHR5KGV4dHJhSGVhZGVycykpIHtcbiAgICBhd2FpdCBwYWdlLnNldEV4dHJhSFRUUEhlYWRlcnMoZXh0cmFIZWFkZXJzKTtcbiAgfVxuICBsb2dnZXIuaW5mbyhgb3JpZ2luYWwgcXVlcnlVcmwgJHtxdWVyeVVybH1gKTtcbiAgYXdhaXQgcGFnZS5nb3RvKHF1ZXJ5VXJsLCB7IHdhaXRVbnRpbDogJ25ldHdvcmtpZGxlMCcgfSk7XG4gIC8vIHNob3VsZCBhZGQgdG8gbG9jYWwgc3RvcmFnZSBhZnRlciBwYWdlLmdvdG8sIHRoZW4gYWNjZXNzIHRoZSBwYWdlIGFnYWluIC0gYnJvd3NlciBtdXN0IGhhdmUgYW4gdXJsIHRvIHJlZ2lzdGVyIGxvY2FsIHN0b3JhZ2UgaXRlbSBvbiBpdFxuICB0cnkge1xuICAgIGF3YWl0IHBhZ2UuZXZhbHVhdGUoXG4gICAgICAvKiBpc3RhbmJ1bCBpZ25vcmUgbmV4dCAqL1xuICAgICAgKGtleSkgPT4ge1xuICAgICAgICB0cnkge1xuICAgICAgICAgIGlmIChcbiAgICAgICAgICAgIGxvY2FsU3RvcmFnZUF2YWlsYWJsZSAmJlxuICAgICAgICAgICAgdHlwZW9mIGxvY2FsU3RvcmFnZSAhPT0gJ3VuZGVmaW5lZCcgJiZcbiAgICAgICAgICAgIGxvY2FsU3RvcmFnZSAhPT0gbnVsbFxuICAgICAgICAgICkge1xuICAgICAgICAgICAgbG9jYWxTdG9yYWdlLnNldEl0ZW0oa2V5LCAnZmFsc2UnKTtcbiAgICAgICAgICB9XG4gICAgICAgIH0gY2F0Y2ggKGVycikge31cbiAgICAgIH0sXG4gICAgICBTRUNVUklUWV9DT05TVEFOVFMuVEVOQU5UX0xPQ0FMX1NUT1JBR0VfS0VZXG4gICAgKTtcbiAgfSBjYXRjaCAoZXJyKSB7XG4gICAgbG9nZ2VyLmVycm9yKGVycik7XG4gIH1cbiAgYXdhaXQgcGFnZS5nb3RvKHF1ZXJ5VXJsLCB7IHdhaXRVbnRpbDogJ25ldHdvcmtpZGxlMCcgfSk7XG4gIGxvZ2dlci5pbmZvKGBwYWdlIHVybCAke3BhZ2UudXJsKCl9YCk7XG5cbiAgYXdhaXQgcGFnZS5zZXRWaWV3cG9ydCh7XG4gICAgd2lkdGg6IHdpbmRvd1dpZHRoLFxuICAgIGhlaWdodDogd2luZG93SGVpZ2h0LFxuICB9KTtcblxuICBsZXQgYnVmZmVyOiBCdWZmZXI7XG4gIC8vIHJlbW92ZSB1bndhbnRlZCBlbGVtZW50c1xuICBhd2FpdCBwYWdlLmV2YWx1YXRlKFxuICAgIC8qIGlzdGFuYnVsIGlnbm9yZSBuZXh0ICovXG4gICAgKHJlcG9ydFNvdXJjZSwgUkVQT1JUX1RZUEUpID0+IHtcbiAgICAgIC8vIHJlbW92ZSBidXR0b25zXG4gICAgICBkb2N1bWVudFxuICAgICAgICAucXVlcnlTZWxlY3RvckFsbChcIltjbGFzc149J2V1aUJ1dHRvbiddXCIpXG4gICAgICAgIC5mb3JFYWNoKChlKSA9PiBlLnJlbW92ZSgpKTtcbiAgICAgIC8vIHJlbW92ZSB0b3AgbmF2QmFyXG4gICAgICBkb2N1bWVudFxuICAgICAgICAucXVlcnlTZWxlY3RvckFsbChcIltjbGFzc149J2V1aUhlYWRlciddXCIpXG4gICAgICAgIC5mb3JFYWNoKChlKSA9PiBlLnJlbW92ZSgpKTtcbiAgICAgIC8vIHJlbW92ZSB2aXN1YWxpemF0aW9uIGVkaXRvclxuICAgICAgaWYgKHJlcG9ydFNvdXJjZSA9PT0gUkVQT1JUX1RZUEUudmlzdWFsaXphdGlvbikge1xuICAgICAgICBkb2N1bWVudFxuICAgICAgICAgIC5xdWVyeVNlbGVjdG9yKCdbZGF0YS10ZXN0LXN1Ymo9XCJzcGxpdFBhbmVsUmVzaXplclwiXScpXG4gICAgICAgICAgPy5yZW1vdmUoKTtcbiAgICAgICAgZG9jdW1lbnQucXVlcnlTZWxlY3RvcignLnZpc0VkaXRvcl9fY29sbGFwc2libGVTaWRlYmFyJyk/LnJlbW92ZSgpO1xuICAgICAgfVxuICAgICAgZG9jdW1lbnQuYm9keS5zdHlsZS5wYWRkaW5nVG9wID0gJzBweCc7XG4gICAgfSxcbiAgICByZXBvcnRTb3VyY2UsXG4gICAgUkVQT1JUX1RZUEVcbiAgKTtcblxuICAvLyBmb3JjZSB3YWl0IGZvciBhbnkgcmVzaXplIHRvIGxvYWQgYWZ0ZXIgdGhlIGFib3ZlIERPTSBtb2RpZmljYXRpb25cbiAgYXdhaXQgcGFnZS53YWl0Rm9yKDEwMDApO1xuICAvLyBjcm9wIGNvbnRlbnRcbiAgc3dpdGNoIChyZXBvcnRTb3VyY2UpIHtcbiAgICBjYXNlIFJFUE9SVF9UWVBFLmRhc2hib2FyZDpcbiAgICAgIGF3YWl0IHBhZ2Uud2FpdEZvclNlbGVjdG9yKFNFTEVDVE9SLmRhc2hib2FyZCwge1xuICAgICAgICB2aXNpYmxlOiB0cnVlLFxuICAgICAgfSk7XG4gICAgICBicmVhaztcbiAgICBjYXNlIFJFUE9SVF9UWVBFLnZpc3VhbGl6YXRpb246XG4gICAgICBhd2FpdCBwYWdlLndhaXRGb3JTZWxlY3RvcihTRUxFQ1RPUi52aXN1YWxpemF0aW9uLCB7XG4gICAgICAgIHZpc2libGU6IHRydWUsXG4gICAgICB9KTtcbiAgICAgIGJyZWFrO1xuICAgIGNhc2UgUkVQT1JUX1RZUEUubm90ZWJvb2s6XG4gICAgICBhd2FpdCBwYWdlLndhaXRGb3JTZWxlY3RvcihTRUxFQ1RPUi5ub3RlYm9vaywge1xuICAgICAgICB2aXNpYmxlOiB0cnVlLFxuICAgICAgfSk7XG4gICAgICBicmVhaztcbiAgICBkZWZhdWx0OlxuICAgICAgdGhyb3cgRXJyb3IoXG4gICAgICAgIGByZXBvcnQgc291cmNlIGNhbiBvbmx5IGJlIG9uZSBvZiBbRGFzaGJvYXJkLCBWaXN1YWxpemF0aW9uXWBcbiAgICAgICk7XG4gIH1cblxuICAvLyB3YWl0IGZvciBkeW5hbWljIHBhZ2UgY29udGVudCB0byByZW5kZXJcbiAgYXdhaXQgd2FpdEZvckR5bmFtaWNDb250ZW50KHBhZ2UpO1xuXG4gIGF3YWl0IGFkZFJlcG9ydEhlYWRlcihwYWdlLCBrZXl3b3JkRmlsdGVyZWRIZWFkZXIpO1xuICBhd2FpdCBhZGRSZXBvcnRGb290ZXIocGFnZSwga2V5d29yZEZpbHRlcmVkRm9vdGVyKTtcbiAgYXdhaXQgYWRkUmVwb3J0U3R5bGUocGFnZSk7XG5cbiAgLy8gdGhpcyBjYXVzZXMgVVQgdG8gZmFpbCBpbiBnaXRodWIgQ0kgYnV0IHdvcmtzIGxvY2FsbHlcbiAgdHJ5IHtcbiAgICBjb25zdCBudW1EaXNhbGxvd2VkVGFncyA9IGF3YWl0IHBhZ2UuZXZhbHVhdGUoXG4gICAgICAoKSA9PlxuICAgICAgICBkb2N1bWVudC5nZXRFbGVtZW50c0J5VGFnTmFtZSgnaWZyYW1lJykubGVuZ3RoICtcbiAgICAgICAgZG9jdW1lbnQuZ2V0RWxlbWVudHNCeVRhZ05hbWUoJ2VtYmVkJykubGVuZ3RoICtcbiAgICAgICAgZG9jdW1lbnQuZ2V0RWxlbWVudHNCeVRhZ05hbWUoJ29iamVjdCcpLmxlbmd0aFxuICAgICk7XG4gICAgaWYgKG51bURpc2FsbG93ZWRUYWdzID4gMCkge1xuICAgICAgdGhyb3cgRXJyb3IoJ1JlcG9ydGluZyBkb2VzIG5vdCBzdXBwb3J0IFwiaWZyYW1lXCIsIFwiZW1iZWRcIiwgb3IgXCJvYmplY3RcIiB0YWdzLCBhYm9ydGluZycpO1xuICAgIH1cbiAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICBsb2dnZXIuZXJyb3IoZXJyb3IpO1xuICB9XG5cbiAgLy8gY3JlYXRlIHBkZiBvciBwbmcgYWNjb3JkaW5nbHlcbiAgaWYgKHJlcG9ydEZvcm1hdCA9PT0gRk9STUFULnBkZikge1xuICAgIGNvbnN0IHNjcm9sbEhlaWdodCA9IGF3YWl0IHBhZ2UuZXZhbHVhdGUoXG4gICAgICAvKiBpc3RhbmJ1bCBpZ25vcmUgbmV4dCAqL1xuICAgICAgKCkgPT4gZG9jdW1lbnQuZG9jdW1lbnRFbGVtZW50LnNjcm9sbEhlaWdodFxuICAgICk7XG5cbiAgICBidWZmZXIgPSBhd2FpdCBwYWdlLnBkZih7XG4gICAgICBtYXJnaW46IHVuZGVmaW5lZCxcbiAgICAgIHdpZHRoOiB3aW5kb3dXaWR0aCxcbiAgICAgIGhlaWdodDogc2Nyb2xsSGVpZ2h0ICsgJ3B4JyxcbiAgICAgIHByaW50QmFja2dyb3VuZDogdHJ1ZSxcbiAgICAgIHBhZ2VSYW5nZXM6ICcxJyxcbiAgICB9KTtcbiAgfSBlbHNlIGlmIChyZXBvcnRGb3JtYXQgPT09IEZPUk1BVC5wbmcpIHtcbiAgICBidWZmZXIgPSBhd2FpdCBwYWdlLnNjcmVlbnNob3Qoe1xuICAgICAgZnVsbFBhZ2U6IHRydWUsXG4gICAgfSk7XG4gIH1cblxuICBjb25zdCBjdXJUaW1lID0gbmV3IERhdGUoKTtcbiAgY29uc3QgdGltZUNyZWF0ZWQgPSBjdXJUaW1lLnZhbHVlT2YoKTtcbiAgY29uc3QgZmlsZU5hbWUgPSBgJHtnZXRGaWxlTmFtZShyZXBvcnROYW1lLCBjdXJUaW1lKX0uJHtyZXBvcnRGb3JtYXR9YDtcbiAgYXdhaXQgYnJvd3Nlci5jbG9zZSgpO1xuXG4gIHJldHVybiB7IHRpbWVDcmVhdGVkLCBkYXRhVXJsOiBidWZmZXIudG9TdHJpbmcoJ2Jhc2U2NCcpLCBmaWxlTmFtZSB9O1xufTtcblxuY29uc3QgYWRkUmVwb3J0U3R5bGUgPSBhc3luYyAocGFnZTogcHVwcGV0ZWVyLlBhZ2UpID0+IHtcbiAgY29uc3QgY3NzID0gZnMucmVhZEZpbGVTeW5jKGAke19fZGlybmFtZX0vc3R5bGUuY3NzYCkudG9TdHJpbmcoKTtcblxuICBhd2FpdCBwYWdlLmV2YWx1YXRlKFxuICAgIC8qIGlzdGFuYnVsIGlnbm9yZSBuZXh0ICovXG4gICAgKHN0eWxlOiBzdHJpbmcpID0+IHtcbiAgICAgIGNvbnN0IHN0eWxlRWxlbWVudCA9IGRvY3VtZW50LmNyZWF0ZUVsZW1lbnQoJ3N0eWxlJyk7XG4gICAgICBzdHlsZUVsZW1lbnQuaW5uZXJIVE1MID0gc3R5bGU7XG4gICAgICBkb2N1bWVudC5nZXRFbGVtZW50c0J5VGFnTmFtZSgnaGVhZCcpWzBdLmFwcGVuZENoaWxkKHN0eWxlRWxlbWVudCk7XG4gICAgfSxcbiAgICBjc3NcbiAgKTtcbn07XG5cbmNvbnN0IGFkZFJlcG9ydEhlYWRlciA9IGFzeW5jIChwYWdlOiBwdXBwZXRlZXIuUGFnZSwgaGVhZGVyOiBzdHJpbmcpID0+IHtcbiAgY29uc3QgaGVhZGVySHRtbCA9IGZzXG4gICAgLnJlYWRGaWxlU3luYyhgJHtfX2Rpcm5hbWV9L2hlYWRlcl90ZW1wbGF0ZS5odG1sYClcbiAgICAudG9TdHJpbmcoKVxuICAgIC5yZXBsYWNlKCc8IS0tQ09OVEVOVC0tPicsIGhlYWRlcik7XG5cbiAgYXdhaXQgcGFnZS5ldmFsdWF0ZShcbiAgICAvKiBpc3RhbmJ1bCBpZ25vcmUgbmV4dCAqL1xuICAgIChoZWFkZXJIdG1sOiBzdHJpbmcpID0+IHtcbiAgICAgIGNvbnN0IGNvbnRlbnQgPSBkb2N1bWVudC5ib2R5LmZpcnN0Q2hpbGQ7XG4gICAgICBjb25zdCBoZWFkZXJDb250YWluZXIgPSBkb2N1bWVudC5jcmVhdGVFbGVtZW50KCdkaXYnKTtcbiAgICAgIGhlYWRlckNvbnRhaW5lci5jbGFzc05hbWUgPSAncmVwb3J0V3JhcHBlcic7XG4gICAgICBoZWFkZXJDb250YWluZXIuaW5uZXJIVE1MID0gaGVhZGVySHRtbDtcbiAgICAgIGNvbnRlbnQ/LnBhcmVudE5vZGU/Lmluc2VydEJlZm9yZShoZWFkZXJDb250YWluZXIsIGNvbnRlbnQpO1xuICAgIH0sXG4gICAgaGVhZGVySHRtbFxuICApO1xufTtcblxuY29uc3QgYWRkUmVwb3J0Rm9vdGVyID0gYXN5bmMgKHBhZ2U6IHB1cHBldGVlci5QYWdlLCBmb290ZXI6IHN0cmluZykgPT4ge1xuICBjb25zdCBoZWFkZXJIdG1sID0gZnNcbiAgICAucmVhZEZpbGVTeW5jKGAke19fZGlybmFtZX0vZm9vdGVyX3RlbXBsYXRlLmh0bWxgKVxuICAgIC50b1N0cmluZygpXG4gICAgLnJlcGxhY2UoJzwhLS1DT05URU5ULS0+JywgZm9vdGVyKTtcblxuICBhd2FpdCBwYWdlLmV2YWx1YXRlKFxuICAgIC8qIGlzdGFuYnVsIGlnbm9yZSBuZXh0ICovXG4gICAgKGhlYWRlckh0bWw6IHN0cmluZykgPT4ge1xuICAgICAgY29uc3QgY29udGVudCA9IGRvY3VtZW50LmJvZHkuZmlyc3RDaGlsZDtcbiAgICAgIGNvbnN0IGhlYWRlckNvbnRhaW5lciA9IGRvY3VtZW50LmNyZWF0ZUVsZW1lbnQoJ2RpdicpO1xuICAgICAgaGVhZGVyQ29udGFpbmVyLmNsYXNzTmFtZSA9ICdyZXBvcnRXcmFwcGVyJztcbiAgICAgIGhlYWRlckNvbnRhaW5lci5pbm5lckhUTUwgPSBoZWFkZXJIdG1sO1xuICAgICAgY29udGVudD8ucGFyZW50Tm9kZT8uaW5zZXJ0QmVmb3JlKGhlYWRlckNvbnRhaW5lciwgbnVsbCk7XG4gICAgfSxcbiAgICBoZWFkZXJIdG1sXG4gICk7XG59O1xuXG4vLyBhZGQgd2FpdEZvckR5bmFtaWNDb250ZW50IGZ1bmN0aW9uXG5jb25zdCB3YWl0Rm9yRHluYW1pY0NvbnRlbnQgPSBhc3luYyAoXG4gIHBhZ2UsXG4gIHRpbWVvdXQgPSAzMDAwMCxcbiAgaW50ZXJ2YWwgPSAxMDAwLFxuICBjaGVja3MgPSA1XG4pID0+IHtcbiAgY29uc3QgbWF4Q2hlY2tzID0gdGltZW91dCAvIGludGVydmFsO1xuICBsZXQgcGFzc2VkQ2hlY2tzID0gMDtcbiAgbGV0IHByZXZpb3VzTGVuZ3RoID0gMDtcblxuICBsZXQgaSA9IDA7XG4gIHdoaWxlIChpKysgPD0gbWF4Q2hlY2tzKSB7XG4gICAgbGV0IHBhZ2VDb250ZW50ID0gYXdhaXQgcGFnZS5jb250ZW50KCk7XG4gICAgbGV0IGN1cnJlbnRMZW5ndGggPSBwYWdlQ29udGVudC5sZW5ndGg7XG5cbiAgICBwcmV2aW91c0xlbmd0aCA9PT0gMCB8fCBwcmV2aW91c0xlbmd0aCAhPSBjdXJyZW50TGVuZ3RoXG4gICAgICA/IChwYXNzZWRDaGVja3MgPSAwKVxuICAgICAgOiBwYXNzZWRDaGVja3MrKztcbiAgICBpZiAocGFzc2VkQ2hlY2tzID49IGNoZWNrcykge1xuICAgICAgYnJlYWs7XG4gICAgfVxuXG4gICAgcHJldmlvdXNMZW5ndGggPSBjdXJyZW50TGVuZ3RoO1xuICAgIGF3YWl0IHBhZ2Uud2FpdEZvcihpbnRlcnZhbCk7XG4gIH1cbn07XG4iXX0=