"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerNoteRoute = registerNoteRoute;

var _configSchema = require("@osd/config-schema");

var _notebooks = require("../../../common/constants/notebooks");

var _notebooks2 = _interopRequireDefault(require("../../adaptors/notebooks"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
function registerNoteRoute(router) {
  // Fetch all the notebooks available
  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/`,
    validate: {}
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    let notebooksData = [];

    try {
      notebooksData = await _notebooks2.default.viewNotes(opensearchNotebooksClient, _notebooks.wreckOptions);
      return response.ok({
        body: {
          data: notebooksData
        }
      });
    } catch (error) {
      console.log('Notebook:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Get all paragraphs of notebooks

  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/{noteId}`,
    validate: {
      params: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const notebookinfo = await _notebooks2.default.fetchNote(opensearchNotebooksClient, request.params.noteId, _notebooks.wreckOptions);
      return response.ok({
        body: notebookinfo
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Add a Notebook

  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const addResponse = await _notebooks2.default.addNote(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: addResponse.message.objectId
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Rename a notebook

  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/rename`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const renameResponse = await _notebooks2.default.renameNote(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: renameResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Clone a notebook

  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/clone`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const cloneResponse = await _notebooks2.default.cloneNote(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: cloneResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Delete notebooks

  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/{noteList}`,
    validate: {
      params: _configSchema.schema.object({
        noteList: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const delResponse = await _notebooks2.default.deleteNote(opensearchNotebooksClient, request.params.noteList, _notebooks.wreckOptions);
      return response.ok({
        body: delResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Add sample notebooks

  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/addSampleNotebooks`,
    validate: {
      body: _configSchema.schema.object({
        visIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const addSampleNotesResponse = await _notebooks2.default.addSampleNotes(opensearchNotebooksClient, request.body.visIds, _notebooks.wreckOptions);
      return response.ok({
        body: addSampleNotesResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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