"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _constants = require("../utils/constants");

var _helpers = require("../utils/helpers");

var _DataStreamService = require("./DataStreamService");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class IndexService {
  constructor(osDriver) {
    _defineProperty(this, "osDriver", void 0);

    _defineProperty(this, "getIndices", async (context, request, response) => {
      try {
        // @ts-ignore
        const {
          from,
          size,
          sortField,
          sortDirection,
          terms,
          indices,
          dataStreams,
          showDataStreams
        } = request.query;
        const params = {
          index: (0, _helpers.getSearchString)(terms, indices, dataStreams),
          format: "json",
          s: `${sortField}:${sortDirection}`
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const [indicesResponse, indexToDataStreamMapping] = await Promise.all([callWithRequest("cat.indices", params), (0, _DataStreamService.getIndexToDataStreamMapping)({
          callAsCurrentUser: callWithRequest
        })]); // Augment the indices with their parent data stream name.

        indicesResponse.forEach(index => {
          index.data_stream = indexToDataStreamMapping[index.index] || null;
        }); // Filtering out indices that belong to a data stream. This must be done before pagination.

        const filteredIndices = showDataStreams ? indicesResponse : indicesResponse.filter(index => index.data_stream === null); // _cat doesn't support pagination, do our own in server pagination to at least reduce network bandwidth

        const fromNumber = parseInt(from, 10);
        const sizeNumber = parseInt(size, 10);
        const paginatedIndices = filteredIndices.slice(fromNumber, fromNumber + sizeNumber);
        const indexNames = paginatedIndices.map(value => value.index);
        const managedStatus = await this._getManagedStatus(request, indexNames); // NOTE: Cannot use response.ok due to typescript type checking

        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              indices: paginatedIndices.map(catIndex => ({ ...catIndex,
                managed: managedStatus[catIndex.index] || "N/A"
              })),
              totalIndices: filteredIndices.length
            }
          }
        });
      } catch (err) {
        // Throws an error if there is no index matching pattern
        if (err.statusCode === 404 && err.body.error.type === "index_not_found_exception") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                indices: [],
                totalIndices: 0
              }
            }
          });
        }

        console.error("Index Management - IndexService - getIndices:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "_getManagedStatus", async (request, indexNames) => {
      try {
        const explainParamas = {
          index: indexNames.toString()
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const explainResponse = await callWithRequest("ism.explain", explainParamas);
        const managed = {};

        for (const indexName in explainResponse) {
          if (indexName === "total_managed_indices") continue;
          const explain = explainResponse[indexName];
          managed[indexName] = explain["index.plugins.index_state_management.policy_id"] === null ? "No" : "Yes";
        }

        return managed;
      } catch (err) {
        // otherwise it could be an unauthorized access error to config index or some other error
        // in which case we will return managed status N/A
        console.error("Index Management - IndexService - _getManagedStatus:", err);
        return indexNames.reduce((accu, value) => ({ ...accu,
          [value]: "N/A"
        }), {});
      }
    });

    _defineProperty(this, "applyPolicy", async (context, request, response) => {
      try {
        const {
          indices,
          policyId
        } = request.body;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const params = {
          index: indices.join(","),
          body: {
            policy_id: policyId
          }
        };
        const addResponse = await callWithRequest("ism.add", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              failures: addResponse.failures,
              updatedIndices: addResponse.updated_indices,
              failedIndices: addResponse.failed_indices.map(failedIndex => ({
                indexName: failedIndex.index_name,
                indexUuid: failedIndex.index_uuid,
                reason: failedIndex.reason
              }))
            }
          }
        });
      } catch (err) {
        console.error("Index Management - IndexService - applyPolicy:", err); // return { ok: false, error: err.message };

        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "editRolloverAlias", async (context, request, response) => {
      try {
        const {
          alias,
          index
        } = request.body;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const params = {
          index,
          body: {
            [_constants.Setting.RolloverAlias]: alias
          }
        };
        const rollOverResponse = await callWithRequest("indices.putSettings", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: rollOverResponse
          }
        });
      } catch (err) {
        console.error("Index Management - IndexService - editRolloverAlias", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    this.osDriver = osDriver;
  }

}

exports.default = IndexService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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