/*
 * Decompiled with CFR 0.152.
 */
package org.gradle.internal.file.impl;

import java.io.File;
import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.function.LongSupplier;
import java.util.function.Predicate;
import org.gradle.internal.file.Deleter;
import org.gradle.internal.impldep.com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultDeleter
implements Deleter {
    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultDeleter.class);
    private final LongSupplier timeProvider;
    private final Predicate<? super File> isSymlink;
    private final boolean runGcOnFailedDelete;
    private static final int DELETE_RETRY_SLEEP_MILLIS = 10;
    @VisibleForTesting
    static final int MAX_REPORTED_PATHS = 16;
    @VisibleForTesting
    static final String HELP_FAILED_DELETE_CHILDREN = "Failed to delete some children. This might happen because a process has files open or has its working directory set in the target directory.";
    @VisibleForTesting
    static final String HELP_NEW_CHILDREN = "New files were found. This might happen because a process is still writing to the target directory.";

    public DefaultDeleter(LongSupplier timeProvider, Predicate<? super File> isSymlink, boolean runGcOnFailedDelete) {
        this.timeProvider = timeProvider;
        this.isSymlink = isSymlink;
        this.runGcOnFailedDelete = runGcOnFailedDelete;
    }

    @Override
    public boolean deleteRecursively(File target) throws IOException {
        return this.deleteRecursively(target, false);
    }

    @Override
    public boolean deleteRecursively(File root, boolean followSymlinks) throws IOException {
        if (root.exists()) {
            return this.deleteRecursively(root, followSymlinks ? Handling.FOLLOW_SYMLINKED_DIRECTORIES : Handling.DO_NOT_FOLLOW_SYMLINKS);
        }
        return false;
    }

    @Override
    public boolean ensureEmptyDirectory(File target) throws IOException {
        return this.ensureEmptyDirectory(target, false);
    }

    @Override
    public boolean ensureEmptyDirectory(File root, boolean followSymlinks) throws IOException {
        if (root.exists()) {
            if (root.isDirectory() && (followSymlinks || !this.isSymlink.test(root))) {
                return this.deleteRecursively(root, followSymlinks ? Handling.KEEP_AND_FOLLOW_SYMLINKED_DIRECTORIES : Handling.KEEP_AND_DO_NOT_FOLLOW_CHILD_SYMLINKS);
            }
            if (!this.tryHardToDelete(root)) {
                throw new IOException("Couldn't delete " + root);
            }
        }
        if (!root.mkdirs()) {
            throw new IOException("Couldn't create directory: " + root);
        }
        return true;
    }

    @Override
    public boolean delete(File target) throws IOException {
        if (!target.exists()) {
            return false;
        }
        if (!this.tryHardToDelete(target)) {
            throw new IOException("Couldn't delete " + target);
        }
        return true;
    }

    private boolean deleteRecursively(File root, Handling handling) throws IOException {
        LOGGER.debug("Deleting {}", (Object)root);
        long startTime = this.timeProvider.getAsLong();
        ArrayList<String> failedPaths = new ArrayList<String>();
        boolean attemptedToRemoveAnything = this.deleteRecursively(startTime, root, root, handling, failedPaths);
        if (!failedPaths.isEmpty()) {
            this.throwWithHelpMessage(startTime, root, handling, failedPaths, false);
        }
        return attemptedToRemoveAnything;
    }

    private boolean deleteRecursively(long startTime, File baseDir, File file, Handling handling, Collection<String> failedPaths) throws IOException {
        if (this.shouldRemoveContentsOf(file, handling)) {
            File[] contents = file.listFiles();
            if (contents == null) {
                return false;
            }
            boolean attemptedToDeleteAnything = false;
            for (File item : contents) {
                this.deleteRecursively(startTime, baseDir, item, handling.getDescendantHandling(), failedPaths);
                attemptedToDeleteAnything = true;
            }
            if (handling.shouldKeepEntry()) {
                return attemptedToDeleteAnything;
            }
        }
        if (!this.tryHardToDelete(file)) {
            failedPaths.add(file.getAbsolutePath());
            if (failedPaths.size() == 16) {
                this.throwWithHelpMessage(startTime, baseDir, handling, failedPaths, true);
            }
        }
        return true;
    }

    private boolean shouldRemoveContentsOf(File file, Handling handling) {
        return file.isDirectory() && (handling.shouldFollowLinkedDirectory() || !this.isSymlink.test(file));
    }

    protected boolean deleteFile(File file) {
        return file.delete() && !file.exists();
    }

    private boolean tryHardToDelete(File file) {
        if (this.deleteFile(file)) {
            return true;
        }
        if (this.runGcOnFailedDelete) {
            System.gc();
        }
        try {
            Thread.sleep(10L);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
        }
        return this.deleteFile(file);
    }

    private void throwWithHelpMessage(long startTime, File file, Handling handling, Collection<String> failedPaths, boolean more) throws IOException {
        throw new IOException(this.buildHelpMessageForFailedDelete(startTime, file, handling, failedPaths, more));
    }

    private String buildHelpMessageForFailedDelete(long startTime, File file, Handling handling, Collection<String> failedPaths, boolean more) {
        StringBuilder help = new StringBuilder("Unable to delete ");
        if (this.isSymlink.test(file)) {
            help.append("symlink to ");
        }
        if (file.isDirectory()) {
            help.append("directory ");
        } else {
            help.append("file ");
        }
        help.append('\'').append(file).append('\'');
        if (this.shouldRemoveContentsOf(file, handling)) {
            Collection<String> newPaths;
            String absolutePath = file.getAbsolutePath();
            failedPaths.remove(absolutePath);
            if (!failedPaths.isEmpty()) {
                help.append("\n  ").append(HELP_FAILED_DELETE_CHILDREN);
                for (String failed : failedPaths) {
                    help.append("\n  - ").append(failed);
                }
                if (more) {
                    help.append("\n  - and more ...");
                }
            }
            if (!(newPaths = DefaultDeleter.listNewPaths(startTime, file, failedPaths)).isEmpty()) {
                help.append("\n  ").append(HELP_NEW_CHILDREN);
                for (String newPath : newPaths) {
                    help.append("\n  - ").append(newPath);
                }
                if (newPaths.size() == 16) {
                    help.append("\n  - and more ...");
                }
            }
        }
        return help.toString();
    }

    private static Collection<String> listNewPaths(long startTime, File directory, Collection<String> failedPaths) {
        ArrayList<String> paths = new ArrayList<String>(16);
        ArrayDeque<File> stack = new ArrayDeque<File>();
        stack.push(directory);
        while (!stack.isEmpty() && paths.size() < 16) {
            File[] children;
            File current = (File)stack.pop();
            String absolutePath = current.getAbsolutePath();
            if (!current.equals(directory) && !failedPaths.contains(absolutePath) && current.lastModified() >= startTime) {
                paths.add(absolutePath);
            }
            if (!current.isDirectory() || (children = current.listFiles()) == null) continue;
            for (File child : children) {
                stack.push(child);
            }
        }
        return paths;
    }

    private static enum Handling {
        KEEP_AND_FOLLOW_SYMLINKED_DIRECTORIES(true, true){

            @Override
            public Handling getDescendantHandling() {
                return FOLLOW_SYMLINKED_DIRECTORIES;
            }
        }
        ,
        KEEP_AND_DO_NOT_FOLLOW_CHILD_SYMLINKS(true, true){

            @Override
            public Handling getDescendantHandling() {
                return DO_NOT_FOLLOW_SYMLINKS;
            }
        }
        ,
        FOLLOW_SYMLINKED_DIRECTORIES(false, true){

            @Override
            public Handling getDescendantHandling() {
                return FOLLOW_SYMLINKED_DIRECTORIES;
            }
        }
        ,
        DO_NOT_FOLLOW_SYMLINKS(false, false){

            @Override
            public Handling getDescendantHandling() {
                return DO_NOT_FOLLOW_SYMLINKS;
            }
        };

        private final boolean shouldKeepEntry;
        private final boolean shouldFollowLinkedDirectory;

        private Handling(boolean shouldKeepEntry, boolean shouldFollowLinkedDirectory) {
            this.shouldKeepEntry = shouldKeepEntry;
            this.shouldFollowLinkedDirectory = shouldFollowLinkedDirectory;
        }

        public boolean shouldKeepEntry() {
            return this.shouldKeepEntry;
        }

        public boolean shouldFollowLinkedDirectory() {
            return this.shouldFollowLinkedDirectory;
        }

        public abstract Handling getDescendantHandling();
    }
}

