/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import React, { memo } from 'react';
import classNames from 'classnames';
import { i18n } from '@kbn/i18n';
import { FormattedMessage } from '@kbn/i18n/react';
import { EuiPopover, keys, EuiIcon, EuiSpacer, EuiButtonEmpty, EuiButtonGroup } from '@elastic/eui';
import { legendColors } from './models';

var VisLegendItemComponent = function VisLegendItemComponent(_ref) {
  var item = _ref.item,
      legendId = _ref.legendId,
      selected = _ref.selected,
      canFilter = _ref.canFilter,
      anchorPosition = _ref.anchorPosition,
      onFilter = _ref.onFilter,
      onSelect = _ref.onSelect,
      onHighlight = _ref.onHighlight,
      onUnhighlight = _ref.onUnhighlight,
      setColor = _ref.setColor,
      getColor = _ref.getColor;

  /**
   * Keydown listener for a legend entry.
   * This will close the details panel of this legend entry when pressing Escape.
   */
  var onLegendEntryKeydown = function onLegendEntryKeydown(event) {
    if (event.key === keys.ESCAPE) {
      event.preventDefault();
      event.stopPropagation();
      onSelect(null)();
    }
  };

  var filterOptions = [{
    id: 'filterIn',
    label: i18n.translate('visTypeVislib.vislib.legend.filterForValueButtonAriaLabel', {
      defaultMessage: 'Filter for value {legendDataLabel}',
      values: {
        legendDataLabel: item.label
      }
    }),
    iconType: 'plusInCircle',
    'data-test-subj': "legend-".concat(item.label, "-filterIn")
  }, {
    id: 'filterOut',
    label: i18n.translate('visTypeVislib.vislib.legend.filterOutValueButtonAriaLabel', {
      defaultMessage: 'Filter out value {legendDataLabel}',
      values: {
        legendDataLabel: item.label
      }
    }),
    iconType: 'minusInCircle',
    'data-test-subj': "legend-".concat(item.label, "-filterOut")
  }];

  var handleFilterChange = function handleFilterChange(id) {
    onFilter(item, id !== 'filterIn');
  };

  var renderFilterBar = function renderFilterBar() {
    return /*#__PURE__*/React.createElement(React.Fragment, null, /*#__PURE__*/React.createElement(EuiButtonGroup, {
      type: "multi",
      isIconOnly: true,
      isFullWidth: true,
      legend: i18n.translate('visTypeVislib.vislib.legend.filterOptionsLegend', {
        defaultMessage: '{legendDataLabel}, filter options',
        values: {
          legendDataLabel: item.label
        }
      }),
      options: filterOptions,
      onChange: handleFilterChange,
      "data-test-subj": "legend-".concat(item.label, "-filters")
    }), /*#__PURE__*/React.createElement(EuiSpacer, {
      size: "s"
    }));
  };

  var button = /*#__PURE__*/React.createElement(EuiButtonEmpty, {
    size: "xs",
    color: "text",
    flush: "left",
    className: "visLegend__button",
    onKeyDown: onLegendEntryKeydown,
    onMouseEnter: onHighlight,
    onFocus: onHighlight,
    onClick: onSelect(item.label),
    onMouseLeave: onUnhighlight,
    onBlur: onUnhighlight,
    "data-label": item.label,
    title: item.label,
    "aria-label": i18n.translate('visTypeVislib.vislib.legend.toggleOptionsButtonAriaLabel', {
      defaultMessage: '{legendDataLabel}, toggle options',
      values: {
        legendDataLabel: item.label
      }
    }),
    "data-test-subj": "legend-".concat(item.label)
  }, /*#__PURE__*/React.createElement(EuiIcon, {
    size: "l",
    type: "dot",
    color: getColor(item.label),
    "data-test-subj": "legendSelectedColor-".concat(getColor(item.label))
  }), /*#__PURE__*/React.createElement("span", {
    className: "visLegend__valueTitle"
  }, item.label));

  var renderDetails = function renderDetails() {
    return /*#__PURE__*/React.createElement(EuiPopover, {
      ownFocus: true,
      display: "block",
      button: button,
      isOpen: selected,
      anchorPosition: anchorPosition,
      closePopover: onSelect(null),
      panelPaddingSize: "s"
    }, /*#__PURE__*/React.createElement("div", {
      className: "visLegend__valueDetails"
    }, canFilter && renderFilterBar(), /*#__PURE__*/React.createElement("div", {
      className: "visLegend__valueColorPicker",
      role: "listbox"
    }, /*#__PURE__*/React.createElement("span", {
      id: "".concat(legendId, "ColorPickerDesc"),
      className: "euiScreenReaderOnly"
    }, /*#__PURE__*/React.createElement(FormattedMessage, {
      id: "visTypeVislib.vislib.legend.setColorScreenReaderDescription",
      defaultMessage: "Set color for value {legendDataLabel}",
      values: {
        legendDataLabel: item.label
      }
    })), legendColors.map(function (color) {
      return /*#__PURE__*/React.createElement(EuiIcon, {
        role: "option",
        tabIndex: 0,
        type: "dot",
        size: "l",
        color: getColor(item.label),
        key: color,
        "aria-label": color,
        "aria-describedby": "".concat(legendId, "ColorPickerDesc"),
        "aria-selected": color === getColor(item.label),
        onClick: setColor(item.label, color),
        onKeyPress: setColor(item.label, color),
        className: classNames('visLegend__valueColorPickerDot', {
          'visLegend__valueColorPickerDot-isSelected': color === getColor(item.label)
        }),
        style: {
          color: color
        },
        "data-test-subj": "legendSelectColor-".concat(color)
      });
    }))));
  };

  return /*#__PURE__*/React.createElement("li", {
    key: item.label,
    className: "visLegend__value"
  }, renderDetails());
};

export var VisLegendItem = /*#__PURE__*/memo(VisLegendItemComponent);