"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spread = (this && this.__spread) || function () {
    for (var ar = [], i = 0; i < arguments.length; i++) ar = ar.concat(__read(arguments[i]));
    return ar;
};
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
Object.defineProperty(exports, "__esModule", { value: true });
var axis_utils_1 = require("../utils/axis_utils");
var curves_1 = require("../../../utils/curves");
var x_domain_1 = require("../domains/x_domain");
var y_domain_1 = require("../domains/y_domain");
var rendering_1 = require("../rendering/rendering");
var scales_1 = require("../utils/scales");
var series_1 = require("../utils/series");
var specs_1 = require("../utils/specs");
var commons_1 = require("../../../utils/commons");
var MAX_ANIMATABLE_BARS = 300;
var MAX_ANIMATABLE_LINES_AREA_POINTS = 600;
/**
 * Adds or removes series from array or series
 * @param series
 * @param target
 */
function updateDeselectedDataSeries(series, target) {
    var seriesIndex = series_1.getSeriesIndex(series, target);
    var updatedSeries = series ? __spread(series) : [];
    if (seriesIndex > -1) {
        updatedSeries.splice(seriesIndex, 1);
    }
    else {
        updatedSeries.push(target);
    }
    return updatedSeries;
}
exports.updateDeselectedDataSeries = updateDeselectedDataSeries;
/**
 * Return map assocition between `seriesKey` and only the custom colors string
 * @param seriesSpecs
 * @param seriesCollection
 * @param seriesColorOverrides color override from legend
 */
function getCustomSeriesColors(seriesSpecs, seriesCollection, seriesColorOverrides) {
    if (seriesColorOverrides === void 0) { seriesColorOverrides = new Map(); }
    var updatedCustomSeriesColors = new Map();
    var counters = new Map();
    seriesCollection.forEach(function (_a, seriesKey) {
        var seriesIdentifier = _a.seriesIdentifier;
        var spec = getSpecsById(seriesSpecs, seriesIdentifier.specId);
        if (!spec || !(spec.customSeriesColors || seriesColorOverrides.size > 0)) {
            return;
        }
        var color;
        if (seriesColorOverrides.has(seriesKey)) {
            color = seriesColorOverrides.get(seriesKey);
        }
        if (!color && spec.customSeriesColors) {
            var counter = counters.get(seriesIdentifier.specId) || 0;
            color = Array.isArray(spec.customSeriesColors)
                ? spec.customSeriesColors[counter % spec.customSeriesColors.length]
                : spec.customSeriesColors(seriesIdentifier);
            counters.set(seriesIdentifier.specId, counter + 1);
        }
        if (color) {
            updatedCustomSeriesColors.set(seriesKey, color);
        }
    });
    return updatedCustomSeriesColors;
}
exports.getCustomSeriesColors = getCustomSeriesColors;
function getLastValues(formattedDataSeries) {
    var lastValues = new Map();
    // we need to get the latest
    formattedDataSeries.stacked.forEach(function (ds) {
        ds.dataSeries.forEach(function (series) {
            var seriesKey = series_1.getSeriesKey(series);
            if (series.data.length > 0) {
                var last = series.data[series.data.length - 1];
                if (last !== null) {
                    var y1 = last.initialY1, y0 = last.initialY0;
                    if (!last.filled && (y1 !== null || y0 !== null)) {
                        lastValues.set(seriesKey, { y0: y0, y1: y1 });
                    }
                }
            }
        });
    });
    formattedDataSeries.nonStacked.forEach(function (ds) {
        ds.dataSeries.forEach(function (series) {
            var seriesKey = series_1.getSeriesKey(series);
            if (series.data.length > 0) {
                var last = series.data[series.data.length - 1];
                if (last !== null) {
                    var y1 = last.initialY1, y0 = last.initialY0;
                    if (y1 !== null || y0 !== null) {
                        lastValues.set(seriesKey, { y0: y0, y1: y1 });
                    }
                }
            }
        });
    });
    return lastValues;
}
/**
 * Compute data domains for all specified specs.
 * @param seriesSpecs a map of all the series specs
 * @param customYDomainsByGroupId custom Y domains grouped by GroupId
 * @param customXDomain if specified in <Settings />, the custom X domain
 * @param deselectedDataSeries is optional; if not supplied,
 * then all series will be factored into computations. Otherwise, selectedDataSeries
 * is used to restrict the computation for just the selected series
 * @returns `SeriesDomainsAndData`
 */
function computeSeriesDomains(seriesSpecs, customYDomainsByGroupId, deselectedDataSeries, customXDomain) {
    if (customYDomainsByGroupId === void 0) { customYDomainsByGroupId = new Map(); }
    if (deselectedDataSeries === void 0) { deselectedDataSeries = []; }
    var _a = deselectedDataSeries
        ? series_1.getSplittedSeries(seriesSpecs, deselectedDataSeries)
        : series_1.getSplittedSeries(seriesSpecs, []), splittedSeries = _a.splittedSeries, xValues = _a.xValues, seriesCollection = _a.seriesCollection;
    var splittedDataSeries = __spread(splittedSeries.values());
    var specsArray = __spread(seriesSpecs.values());
    var xDomain = x_domain_1.mergeXDomain(specsArray, xValues, customXDomain);
    var yDomain = y_domain_1.mergeYDomain(splittedSeries, specsArray, customYDomainsByGroupId);
    var formattedDataSeries = series_1.getFormattedDataseries(specsArray, splittedSeries, xValues, xDomain.scaleType, seriesSpecs);
    // we need to get the last values from the formatted dataseries
    // because we change the format if we are on percentage mode
    var lastValues = getLastValues(formattedDataSeries);
    var updatedSeriesCollection = new Map();
    seriesCollection.forEach(function (value, key) {
        var lastValue = lastValues.get(key);
        var updatedColorSet = __assign(__assign({}, value), { lastValue: lastValue });
        updatedSeriesCollection.set(key, updatedColorSet);
    });
    return {
        xDomain: xDomain,
        yDomain: yDomain,
        splittedDataSeries: splittedDataSeries,
        formattedDataSeries: formattedDataSeries,
        seriesCollection: updatedSeriesCollection,
    };
}
exports.computeSeriesDomains = computeSeriesDomains;
function computeSeriesGeometries(seriesSpecs, xDomain, yDomain, formattedDataSeries, seriesColorMap, chartTheme, chartDims, chartRotation, axesSpecs, enableHistogramMode) {
    var chartColors = chartTheme.colors;
    var barsPadding = enableHistogramMode ? chartTheme.scales.histogramPadding : chartTheme.scales.barsPadding;
    var width = [0, 180].includes(chartRotation) ? chartDims.width : chartDims.height;
    var height = [0, 180].includes(chartRotation) ? chartDims.height : chartDims.width;
    // const { width, height } = chartDims;
    var stacked = formattedDataSeries.stacked, nonStacked = formattedDataSeries.nonStacked;
    // compute how many series are clustered
    var _a = scales_1.countBarsInCluster(stacked, nonStacked), stackedBarsInCluster = _a.stackedBarsInCluster, totalBarsInCluster = _a.totalBarsInCluster;
    // compute scales
    var xScale = scales_1.computeXScale({ xDomain: xDomain, totalBarsInCluster: totalBarsInCluster, range: [0, width], barsPadding: barsPadding, enableHistogramMode: enableHistogramMode });
    var yScales = scales_1.computeYScales({ yDomains: yDomain, range: [height, 0] });
    // compute colors
    // compute geometries
    var points = [];
    var areas = [];
    var bars = [];
    var lines = [];
    var stackedGeometriesIndex = new Map();
    var nonStackedGeometriesIndex = new Map();
    var orderIndex = 0;
    var geometriesCounts = {
        points: 0,
        bars: 0,
        areas: 0,
        areasPoints: 0,
        lines: 0,
        linePoints: 0,
    };
    formattedDataSeries.stacked.forEach(function (dataSeriesGroup) {
        var groupId = dataSeriesGroup.groupId, dataSeries = dataSeriesGroup.dataSeries, counts = dataSeriesGroup.counts;
        var yScale = yScales.get(groupId);
        if (!yScale) {
            return;
        }
        var geometries = renderGeometries(orderIndex, totalBarsInCluster, true, dataSeries, xScale, yScale, seriesSpecs, seriesColorMap, chartColors.defaultVizColor, axesSpecs, chartTheme, enableHistogramMode);
        orderIndex = counts.barSeries > 0 ? orderIndex + 1 : orderIndex;
        areas.push.apply(areas, __spread(geometries.areas));
        lines.push.apply(lines, __spread(geometries.lines));
        bars.push.apply(bars, __spread(geometries.bars));
        points.push.apply(points, __spread(geometries.points));
        stackedGeometriesIndex = mergeGeometriesIndexes(stackedGeometriesIndex, geometries.geometriesIndex);
        // update counts
        geometriesCounts.points += geometries.geometriesCounts.points;
        geometriesCounts.bars += geometries.geometriesCounts.bars;
        geometriesCounts.areas += geometries.geometriesCounts.areas;
        geometriesCounts.areasPoints += geometries.geometriesCounts.areasPoints;
        geometriesCounts.lines += geometries.geometriesCounts.lines;
        geometriesCounts.linePoints += geometries.geometriesCounts.linePoints;
    });
    formattedDataSeries.nonStacked.map(function (dataSeriesGroup) {
        var groupId = dataSeriesGroup.groupId, dataSeries = dataSeriesGroup.dataSeries;
        var yScale = yScales.get(groupId);
        if (!yScale) {
            return;
        }
        var geometries = renderGeometries(stackedBarsInCluster, totalBarsInCluster, false, dataSeries, xScale, yScale, seriesSpecs, seriesColorMap, chartColors.defaultVizColor, axesSpecs, chartTheme, enableHistogramMode);
        areas.push.apply(areas, __spread(geometries.areas));
        lines.push.apply(lines, __spread(geometries.lines));
        bars.push.apply(bars, __spread(geometries.bars));
        points.push.apply(points, __spread(geometries.points));
        nonStackedGeometriesIndex = mergeGeometriesIndexes(nonStackedGeometriesIndex, geometries.geometriesIndex);
        // update counts
        geometriesCounts.points += geometries.geometriesCounts.points;
        geometriesCounts.bars += geometries.geometriesCounts.bars;
        geometriesCounts.areas += geometries.geometriesCounts.areas;
        geometriesCounts.areasPoints += geometries.geometriesCounts.areasPoints;
        geometriesCounts.lines += geometries.geometriesCounts.lines;
        geometriesCounts.linePoints += geometries.geometriesCounts.linePoints;
    });
    var geometriesIndex = mergeGeometriesIndexes(stackedGeometriesIndex, nonStackedGeometriesIndex);
    return {
        scales: {
            xScale: xScale,
            yScales: yScales,
        },
        geometries: {
            points: points,
            areas: areas,
            bars: bars,
            lines: lines,
        },
        geometriesIndex: geometriesIndex,
        geometriesCounts: geometriesCounts,
    };
}
exports.computeSeriesGeometries = computeSeriesGeometries;
function setBarSeriesAccessors(isHistogramMode, seriesSpecs) {
    var e_1, _a;
    if (!isHistogramMode) {
        return;
    }
    try {
        for (var seriesSpecs_1 = __values(seriesSpecs), seriesSpecs_1_1 = seriesSpecs_1.next(); !seriesSpecs_1_1.done; seriesSpecs_1_1 = seriesSpecs_1.next()) {
            var _b = __read(seriesSpecs_1_1.value, 2), spec = _b[1];
            if (specs_1.isBarSeriesSpec(spec)) {
                var stackAccessors = spec.stackAccessors ? __spread(spec.stackAccessors) : spec.yAccessors;
                if (spec.splitSeriesAccessors) {
                    stackAccessors = __spread(stackAccessors, spec.splitSeriesAccessors);
                }
                spec.stackAccessors = stackAccessors;
            }
        }
    }
    catch (e_1_1) { e_1 = { error: e_1_1 }; }
    finally {
        try {
            if (seriesSpecs_1_1 && !seriesSpecs_1_1.done && (_a = seriesSpecs_1.return)) _a.call(seriesSpecs_1);
        }
        finally { if (e_1) throw e_1.error; }
    }
    return;
}
exports.setBarSeriesAccessors = setBarSeriesAccessors;
function isHistogramModeEnabled(seriesSpecs) {
    return seriesSpecs.some(function (spec) {
        return specs_1.isBarSeriesSpec(spec) && spec.enableHistogramMode;
    });
}
exports.isHistogramModeEnabled = isHistogramModeEnabled;
function computeXScaleOffset(xScale, enableHistogramMode, histogramModeAlignment) {
    if (histogramModeAlignment === void 0) { histogramModeAlignment = specs_1.HistogramModeAlignments.Start; }
    if (!enableHistogramMode) {
        return 0;
    }
    var bandwidth = xScale.bandwidth, barsPadding = xScale.barsPadding;
    var band = bandwidth / (1 - barsPadding);
    var halfPadding = (band - bandwidth) / 2;
    var startAlignmentOffset = bandwidth / 2 + halfPadding;
    switch (histogramModeAlignment) {
        case specs_1.HistogramModeAlignments.Center:
            return 0;
        case specs_1.HistogramModeAlignments.End:
            return -startAlignmentOffset;
        default:
            return startAlignmentOffset;
    }
}
exports.computeXScaleOffset = computeXScaleOffset;
function renderGeometries(indexOffset, clusteredCount, isStacked, dataSeries, xScale, yScale, seriesSpecs, seriesColorsMap, defaultColor, axesSpecs, chartTheme, enableHistogramMode) {
    var len = dataSeries.length;
    var i;
    var points = [];
    var bars = [];
    var areas = [];
    var lines = [];
    var pointGeometriesIndex = new Map();
    var barGeometriesIndex = new Map();
    var lineGeometriesIndex = new Map();
    var areaGeometriesIndex = new Map();
    var geometriesCounts = {
        points: 0,
        bars: 0,
        areas: 0,
        areasPoints: 0,
        lines: 0,
        linePoints: 0,
    };
    var barIndexOffset = 0;
    for (i = 0; i < len; i++) {
        var ds = dataSeries[i];
        var spec = getSpecsById(seriesSpecs, ds.specId);
        if (spec === undefined) {
            continue;
        }
        var color = seriesColorsMap.get(series_1.getSeriesKey(ds)) || defaultColor;
        if (specs_1.isBarSeriesSpec(spec)) {
            var shift = isStacked ? indexOffset : indexOffset + barIndexOffset;
            var barSeriesStyle = commons_1.mergePartial(chartTheme.barSeriesStyle, spec.barSeriesStyle, {
                mergeOptionalPartialValues: true,
            });
            var yAxis = getAxesSpecForSpecId(axesSpecs, spec.groupId).yAxis;
            var valueFormatter = yAxis && yAxis.tickFormat ? yAxis.tickFormat : commons_1.identity;
            var displayValueSettings = spec.displayValueSettings
                ? __assign({ valueFormatter: valueFormatter }, spec.displayValueSettings) : undefined;
            var renderedBars = rendering_1.renderBars(shift, ds, xScale, yScale, color, barSeriesStyle, displayValueSettings, spec.styleAccessor, spec.minBarHeight);
            barGeometriesIndex = mergeGeometriesIndexes(barGeometriesIndex, renderedBars.indexedGeometries);
            bars.push.apply(bars, __spread(renderedBars.barGeometries));
            geometriesCounts.bars += renderedBars.barGeometries.length;
            barIndexOffset += 1;
        }
        else if (specs_1.isLineSeriesSpec(spec)) {
            var lineShift = clusteredCount > 0 ? clusteredCount : 1;
            var lineSeriesStyle = spec.lineSeriesStyle
                ? commons_1.mergePartial(chartTheme.lineSeriesStyle, spec.lineSeriesStyle, { mergeOptionalPartialValues: true })
                : chartTheme.lineSeriesStyle;
            var xScaleOffset = computeXScaleOffset(xScale, enableHistogramMode, spec.histogramModeAlignment);
            var renderedLines = rendering_1.renderLine(
            // move the point on half of the bandwidth if we have mixed bars/lines
            (xScale.bandwidth * lineShift) / 2, ds, xScale, yScale, color, spec.curve || curves_1.CurveType.LINEAR, specs_1.isBandedSpec(spec.y0Accessors), xScaleOffset, lineSeriesStyle, spec.pointStyleAccessor, Boolean(spec.fit && (spec.fit.type || spec.fit) !== specs_1.Fit.None));
            lineGeometriesIndex = mergeGeometriesIndexes(lineGeometriesIndex, renderedLines.indexedGeometries);
            lines.push(renderedLines.lineGeometry);
            geometriesCounts.linePoints += renderedLines.lineGeometry.points.length;
            geometriesCounts.lines += 1;
        }
        else if (specs_1.isAreaSeriesSpec(spec)) {
            var areaShift = clusteredCount > 0 ? clusteredCount : 1;
            var areaSeriesStyle = spec.areaSeriesStyle
                ? commons_1.mergePartial(chartTheme.areaSeriesStyle, spec.areaSeriesStyle, { mergeOptionalPartialValues: true })
                : chartTheme.areaSeriesStyle;
            var xScaleOffset = computeXScaleOffset(xScale, enableHistogramMode, spec.histogramModeAlignment);
            var renderedAreas = rendering_1.renderArea(
            // move the point on half of the bandwidth if we have mixed bars/lines
            (xScale.bandwidth * areaShift) / 2, ds, xScale, yScale, color, spec.curve || curves_1.CurveType.LINEAR, specs_1.isBandedSpec(spec.y0Accessors), xScaleOffset, areaSeriesStyle, isStacked, spec.pointStyleAccessor, Boolean(spec.fit && (spec.fit.type || spec.fit) !== specs_1.Fit.None));
            areaGeometriesIndex = mergeGeometriesIndexes(areaGeometriesIndex, renderedAreas.indexedGeometries);
            areas.push(renderedAreas.areaGeometry);
            geometriesCounts.areasPoints += renderedAreas.areaGeometry.points.length;
            geometriesCounts.areas += 1;
        }
    }
    var geometriesIndex = mergeGeometriesIndexes(pointGeometriesIndex, lineGeometriesIndex, areaGeometriesIndex, barGeometriesIndex);
    return {
        points: points,
        bars: bars,
        areas: areas,
        lines: lines,
        geometriesIndex: geometriesIndex,
        geometriesCounts: geometriesCounts,
    };
}
function getSpecsById(specs, id) {
    return specs.find(function (spec) { return spec.id === id; });
}
exports.getSpecsById = getSpecsById;
function getAxesSpecForSpecId(axesSpecs, groupId) {
    var e_2, _a;
    var xAxis;
    var yAxis;
    try {
        for (var axesSpecs_1 = __values(axesSpecs), axesSpecs_1_1 = axesSpecs_1.next(); !axesSpecs_1_1.done; axesSpecs_1_1 = axesSpecs_1.next()) {
            var axisSpec = axesSpecs_1_1.value;
            if (axisSpec.groupId !== groupId) {
                continue;
            }
            if (axis_utils_1.isVerticalAxis(axisSpec.position)) {
                yAxis = axisSpec;
            }
            else {
                xAxis = axisSpec;
            }
        }
    }
    catch (e_2_1) { e_2 = { error: e_2_1 }; }
    finally {
        try {
            if (axesSpecs_1_1 && !axesSpecs_1_1.done && (_a = axesSpecs_1.return)) _a.call(axesSpecs_1);
        }
        finally { if (e_2) throw e_2.error; }
    }
    return {
        xAxis: xAxis,
        yAxis: yAxis,
    };
}
exports.getAxesSpecForSpecId = getAxesSpecForSpecId;
function computeChartTransform(chartDimensions, chartRotation) {
    if (chartRotation === 90) {
        return {
            x: chartDimensions.width,
            y: 0,
            rotate: 90,
        };
    }
    else if (chartRotation === -90) {
        return {
            x: 0,
            y: chartDimensions.height,
            rotate: -90,
        };
    }
    else if (chartRotation === 180) {
        return {
            x: chartDimensions.width,
            y: chartDimensions.height,
            rotate: 180,
        };
    }
    else {
        return {
            x: 0,
            y: 0,
            rotate: 0,
        };
    }
}
exports.computeChartTransform = computeChartTransform;
/**
 * Merge multiple geometry indexes maps together.
 * @param iterables a set of maps to be merged
 * @returns a new Map where each element with the same key are concatenated on a single
 * IndexedGemoetry array for that key
 */
function mergeGeometriesIndexes() {
    var e_3, _a, e_4, _b;
    var iterables = [];
    for (var _i = 0; _i < arguments.length; _i++) {
        iterables[_i] = arguments[_i];
    }
    var geometriesIndex = new Map();
    try {
        for (var iterables_1 = __values(iterables), iterables_1_1 = iterables_1.next(); !iterables_1_1.done; iterables_1_1 = iterables_1.next()) {
            var iterable = iterables_1_1.value;
            try {
                for (var iterable_1 = (e_4 = void 0, __values(iterable)), iterable_1_1 = iterable_1.next(); !iterable_1_1.done; iterable_1_1 = iterable_1.next()) {
                    var item = iterable_1_1.value;
                    rendering_1.mutableIndexedGeometryMapUpsert(geometriesIndex, item[0], item[1]);
                }
            }
            catch (e_4_1) { e_4 = { error: e_4_1 }; }
            finally {
                try {
                    if (iterable_1_1 && !iterable_1_1.done && (_b = iterable_1.return)) _b.call(iterable_1);
                }
                finally { if (e_4) throw e_4.error; }
            }
        }
    }
    catch (e_3_1) { e_3 = { error: e_3_1 }; }
    finally {
        try {
            if (iterables_1_1 && !iterables_1_1.done && (_a = iterables_1.return)) _a.call(iterables_1);
        }
        finally { if (e_3) throw e_3.error; }
    }
    return geometriesIndex;
}
exports.mergeGeometriesIndexes = mergeGeometriesIndexes;
function isHorizontalRotation(chartRotation) {
    return chartRotation === 0 || chartRotation === 180;
}
exports.isHorizontalRotation = isHorizontalRotation;
function isVerticalRotation(chartRotation) {
    return chartRotation === -90 || chartRotation === 90;
}
exports.isVerticalRotation = isVerticalRotation;
/**
 * Check if a specs map contains only line or area specs
 * @param specs Map<SpecId, BasicSeriesSpec>
 */
function isLineAreaOnlyChart(specs) {
    return !specs.some(function (spec) {
        return spec.seriesType === specs_1.SeriesTypes.Bar;
    });
}
exports.isLineAreaOnlyChart = isLineAreaOnlyChart;
function isChartAnimatable(geometriesCounts, animationEnabled) {
    if (!animationEnabled) {
        return false;
    }
    var bars = geometriesCounts.bars, linePoints = geometriesCounts.linePoints, areasPoints = geometriesCounts.areasPoints;
    var isBarsAnimatable = bars <= MAX_ANIMATABLE_BARS;
    var isLinesAndAreasAnimatable = linePoints + areasPoints <= MAX_ANIMATABLE_LINES_AREA_POINTS;
    return isBarsAnimatable && isLinesAndAreasAnimatable;
}
exports.isChartAnimatable = isChartAnimatable;
function isAllSeriesDeselected(legendItems) {
    var e_5, _a;
    try {
        for (var legendItems_1 = __values(legendItems), legendItems_1_1 = legendItems_1.next(); !legendItems_1_1.done; legendItems_1_1 = legendItems_1.next()) {
            var _b = __read(legendItems_1_1.value, 2), legendItem = _b[1];
            if (legendItem.isSeriesVisible) {
                return false;
            }
        }
    }
    catch (e_5_1) { e_5 = { error: e_5_1 }; }
    finally {
        try {
            if (legendItems_1_1 && !legendItems_1_1.done && (_a = legendItems_1.return)) _a.call(legendItems_1);
        }
        finally { if (e_5) throw e_5.error; }
    }
    return true;
}
exports.isAllSeriesDeselected = isAllSeriesDeselected;
//# sourceMappingURL=utils.js.map