"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ReportManager = void 0;

var _util = require("./util");

var _metrics = require("./metrics");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const REPORT_VERSION = 1;

class ReportManager {
  constructor(report) {
    _defineProperty(this, "report", void 0);

    this.report = report || ReportManager.createReport();
  }

  static createReport() {
    return {
      reportVersion: REPORT_VERSION
    };
  }

  clearReport() {
    this.report = ReportManager.createReport();
  }

  isReportEmpty() {
    const {
      uiStatsMetrics,
      userAgent
    } = this.report;
    const noUiStats = !uiStatsMetrics || Object.keys(uiStatsMetrics).length === 0;
    const noUserAgent = !userAgent || Object.keys(userAgent).length === 0;
    return noUiStats && noUserAgent;
  }

  incrementStats(count, stats) {
    const {
      min = 0,
      max = 0,
      sum = 0
    } = stats || {};
    const newMin = Math.min(min, count);
    const newMax = Math.max(max, count);
    const newAvg = newMin + newMax / 2;
    const newSum = sum + count;
    return {
      min: newMin,
      max: newMax,
      avg: newAvg,
      sum: newSum
    };
  }

  assignReports(newMetrics) {
    (0, _util.wrapArray)(newMetrics).forEach(newMetric => this.assignReport(this.report, newMetric));
    return {
      report: this.report
    };
  }

  static createMetricKey(metric) {
    switch (metric.type) {
      case _metrics.METRIC_TYPE.USER_AGENT:
        {
          const {
            appName,
            type
          } = metric;
          return `${appName}-${type}`;
        }

      case _metrics.METRIC_TYPE.CLICK:
      case _metrics.METRIC_TYPE.LOADED:
      case _metrics.METRIC_TYPE.COUNT:
        {
          const {
            appName,
            eventName,
            type
          } = metric;
          return `${appName}-${type}-${eventName}`;
        }

      default:
        throw new _util.UnreachableCaseError(metric);
    }
  }

  assignReport(report, metric) {
    const key = ReportManager.createMetricKey(metric);

    switch (metric.type) {
      case _metrics.METRIC_TYPE.USER_AGENT:
        {
          const {
            appName,
            type,
            userAgent
          } = metric;

          if (userAgent) {
            report.userAgent = {
              [key]: {
                key,
                appName,
                type,
                userAgent: metric.userAgent
              }
            };
          }

          return;
        }

      case _metrics.METRIC_TYPE.CLICK:
      case _metrics.METRIC_TYPE.LOADED:
      case _metrics.METRIC_TYPE.COUNT:
        {
          const {
            appName,
            type,
            eventName,
            count
          } = metric;
          report.uiStatsMetrics = report.uiStatsMetrics || {};
          const existingStats = (report.uiStatsMetrics[key] || {}).stats;
          report.uiStatsMetrics[key] = {
            key,
            appName,
            eventName,
            type,
            stats: this.incrementStats(count, existingStats)
          };
          return;
        }

      default:
        throw new _util.UnreachableCaseError(metric);
    }
  }

}

exports.ReportManager = ReportManager;

_defineProperty(ReportManager, "REPORT_VERSION", REPORT_VERSION);